/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.server.http;

import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.io.Reader;
import java.io.StringReader;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.UnavailableException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.openrdf.util.http.HttpServerUtil;
import org.openrdf.util.log.ThreadLog;

import org.openrdf.sesame.config.SystemConfig;
import org.openrdf.sesame.config.SystemConfigHandler;
import org.openrdf.sesame.config.handlers.SystemConfigFileHandler;
import org.openrdf.sesame.server.SesameServer;

public class ConfigServlet extends SesameServlet {

/*--------------------------------------------+
|	Variables                                 |
+--------------------------------------------*/

	private SystemConfigHandler _configHandler;

/*--------------------------------------------+
|	Methods                                   |
+--------------------------------------------*/

	// Implements SesameServlet._init(...)
	protected void _init(ServletConfig config)
		throws ServletException
	{
		// Log errors and warning to stderr until the system configuration has been read
		SesameServer.setDefaultLogFile(null);

		String baseDir = config.getServletContext().getRealPath("");
		SesameServer.setBaseDir(new File(baseDir));
		ThreadLog.trace("baseDir = " + baseDir);

		// SAX Driver
		String saxDriver = getInitParameter("org.xml.sax.driver");
		if (saxDriver != null) {
			System.setProperty("org.xml.sax.driver", saxDriver);
		}

		// System configuration handler & parameters
		String sysConfigHandler = getInitParameter("systemConfigHandler");
		if (sysConfigHandler == null) {
			String msg = "init-param 'systemConfigHandler' not defined; please specify full classname of SystemConfigHandler implementation.";
			ThreadLog.error(msg);
			throw new UnavailableException(msg);
		}

		try {
			// Create a handler for the system configuration
			Class configHandlerClass = Class.forName(sysConfigHandler);
			_configHandler = (SystemConfigHandler)configHandlerClass.newInstance();

			// Get and set all parameters for the created handler
			Map configParameters = new HashMap();

			Iterator iter = _configHandler.getParameterNames().iterator();
			while (iter.hasNext()) {
				String paramName = (String)iter.next();
				String paramValue = getInitParameter(paramName);

				if (paramValue != null) {
					configParameters.put(paramName, paramValue);
				}
				else {
					String msg = "init-param '"+paramName+"' not defined. This parameter must be defined when using "+sysConfigHandler+".";
					ThreadLog.error(msg);
					throw new UnavailableException(msg);
				}
			}
			_configHandler.setParameters(configParameters);
		}
		catch (ClassNotFoundException e) {
			ThreadLog.error("SystemConfigHandler class not found", e);
			throw new UnavailableException(e.getMessage());
		}
		catch (Exception e) {
			ThreadLog.error("Unable to create configuration handler", e);
			throw new UnavailableException(e.getMessage());
		}

		// Load system configuration and initialize the Sesame server
		try {
			SystemConfig sysConfig = _configHandler.loadConfig();
			SesameServer.setSystemConfig(sysConfig);
		}
		catch (Exception e) {
			ThreadLog.error("Unable to load system config", e);
		}

		// create static LocalService object
		SesameServer.createLocalService();

		// Set default log file:
		SesameServer.setDefaultLogFile("general.log");

		try {
			HttpServerUtil.setTmpDir(SesameServer.getLocalService().getTmpDir());
		}
		catch (IOException e) {
			ThreadLog.warning("Unable to set tmp dir for request parameter values, using default java tmp dir", e);
		}
	}

	// Implements SesameServlet._destroy()
	protected void _destroy() {
		ThreadLog.trace("ConfigServlet: Clearing SesameServer");
		SesameServer.clear();
		ThreadLog.trace("SesameServer cleared");
		SesameServer.unsetDefaultLogFile();
	}

	// Implements SesameServlet._doGet(...)
	protected void _doGet(HttpServletRequest request, HttpServletResponse response)
		throws ServletException, IOException
	{
		_handleFormURLEncodedRequest(request, response);
	}

	// Implements SesameServlet._doPost(...)
	protected void _doPost(HttpServletRequest request, HttpServletResponse response)
		throws ServletException, IOException
	{
		if (HttpServerUtil.isMultipartFormRequest(request)) {
			_handleMultipartFormRequest(request, response);
		}
		else {
			_handleFormURLEncodedRequest(request, response);
		}
	}

	private void _handleFormURLEncodedRequest(HttpServletRequest request, HttpServletResponse response)
		throws IOException
	{
		String action = HttpServerUtil.getParameter(request, "action");
		String password = HttpServerUtil.getParameter(request, "password");
		String config = HttpServerUtil.getParameter(request, "config");

		_handleRequest(request, response, action, password, config);
	}

	private void _handleMultipartFormRequest(HttpServletRequest request, HttpServletResponse response)
		throws IOException
	{
		Map fileItemMap = HttpServerUtil.parseMultipartFormRequest(request);

		String action = HttpServerUtil.getParameter(fileItemMap, "action");
		String password = HttpServerUtil.getParameter(fileItemMap, "password");
		String config = HttpServerUtil.getParameter(fileItemMap, "config");

		_handleRequest(request, response, action, password, config);
	}

	private void _handleRequest(HttpServletRequest request, HttpServletResponse response,
		String action, String password, String config)
		throws IOException
	{
		SesameServer.setThreadLogFile("config.log");
		_logIP(request);

		// form data trace output
		ThreadLog.trace("action = " + action);
		ThreadLog.trace("password set = " + (password != null));
		ThreadLog.trace("config set = " + (config != null));

		// Check action
		if (action == null) {
			_sendBadRequest("No action specified", response);
			return;
		}

		if (!action.equals("getConfig") &&
			!action.equals("setConfig") &&
			!action.equals("sys-refresh"))
		{
			_sendBadRequest("Unknown action specified: " + action, response);
			return;
		}

		if (action.equals("setConfig") && config == null) {
			_sendBadRequest("'config' parameter is missing", response);
			return;
		}

		// Check password
		String adminPassword = SesameServer.getSystemConfig().getAdminPassword();
		if (adminPassword != null && // not null
			adminPassword.trim().length() > 0 && // not an empty string
			!adminPassword.equals(password)) // not equal
		{
			// incorrect password
			if (password == null) {
				_sendBadRequest("Password required", response);
			} else {
				_sendBadRequest("Password incorrect", response);
			}
			return;
		}

		// Prevent the browsers from caching the results
		HttpServerUtil.setNoCacheHeaders(response);

		if (action.equals("getConfig")) {
			_sendConfig(response);
		}
		else if (action.equals("setConfig")) {
			_setNewConfig(config, response);
		}
		else if (action.equals("sys-refresh")) {
			_refreshSystemConfig(response);
		}
	}

	private void _sendConfig(HttpServletResponse response)
		throws IOException
	{
		try {
			ThreadLog.trace("sending system configuration");

			response.setStatus(HttpServletResponse.SC_OK);
			OutputStream out = response.getOutputStream();

			SystemConfig sysConfig = SesameServer.getSystemConfig();
			SystemConfigFileHandler.writeConfiguration(sysConfig, out);
			out.close();

			ThreadLog.trace("system configuration sent");
		}
		catch (Exception e) {
			_sendInternalError("Sending of system configuration failed", e, response);
		}
	}

	private void _setNewConfig(String config, HttpServletResponse response)
		throws IOException
	{
		SystemConfig newConfig = null;

		// Parse received system configuration
		try {
			ThreadLog.trace("setting new system configuration");
			Reader reader = new StringReader(config);
			newConfig = SystemConfigFileHandler.readConfiguration(reader);
		}
		catch (IOException e) {
			_sendBadRequest("Unable to read received system configuration: " + e.getMessage(), response);
			return;
		}

		// Set new system configuration
		try {
			SesameServer.setSystemConfig(newConfig);

			// Store to disk
			_configHandler.storeConfig(newConfig);

			ThreadLog.trace("new system configuration set");

			// Send acknowledgement
			response.setStatus(HttpServletResponse.SC_OK);
			PrintWriter pw = response.getWriter();
			pw.write("System configuration successfully updated");
			pw.close();
		}
		catch (Exception e) {
			_sendInternalError("Unable to store new system configuration", e, response);
		}
	}

	private void _refreshSystemConfig(HttpServletResponse response)
		throws IOException
	{
		try {
			ThreadLog.trace("Refresing SesameServer configuration");
			SystemConfig config = _configHandler.loadConfig();
			SesameServer.setSystemConfig(config);
			ThreadLog.trace("SesameServer configuration refreshed");

			// Send acknowledgement
			response.setStatus(HttpServletResponse.SC_OK);
			PrintWriter pw = response.getWriter();
			pw.write("system configuration is refreshed");
			pw.close();
		}
		catch (Exception e) {
			_sendInternalError("Unable to refresh system configuration", e, response);
		}
	}
}
