/* OMM - Ontology Middleware Module
 * Copyright (C) 2002 OntoText Lab, Sirma AI OOD
 *
 * Contact:
 * Sirma AI OOD, OntoText Lab.
 * 38A, Christo Botev Blvd.
 * 1000 Sofia, Bulgaria
 * tel. +359(2)981 00 18
 * fax. +359(2)981 90 58
 * info@ontotext.com
 *
 *  http://www.ontotext.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	02111-1307	USA
 */
package org.openrdf.sesame.server;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.openrdf.sesame.config.SystemConfigHandler;
import org.openrdf.sesame.omm.SessionContext;
import org.openrdf.sesame.omm.SessionKilled;

/**
 * Title:       Sample standalone Sesame Application
 * Description: Demonstrates an standalone Sesame Application
 * Company:     Sirma AI, OntoText Lab/
 * @author Damyan Ognyanoff
 * @version 1.0
 */
final public class SesameStartup {

/*----------------+
| Constants       |
+----------------*/

	private static final String PARSER_KEY = "org.xml.sax.driver";
	private static final String DEFAULT_CONFIG_HANDLER = "org.openrdf.sesame.config.handlers.SystemConfigFileHandler";

/*----------------+
| Methods         |
+----------------*/

	/**
	 * Initializes Sesame and loads the configuration.
	 * <p>
	 * Note: special handling takes place when the Handler parses the
	 * configuration from an XML file: this requires that the System.property
	 * <tt>org.xml.sax.driver</tt> to be set. If the parser is not set it is assumed
	 * that this is the <tt>org.apache.xerces.parsers.SAXParser</tt> and is set
	 * accordingly.
	 *
	 * @param configHandlerClass The name of the configHandlerClass it must
	 * implement SystemConfigHandler interface.
	 * @param configParams A map containing the parameters required by the Handler.
	 */
	public static void initialize(String configHandlerClass, Map configParams)
		throws Exception
	{
		String parserValue = (String)configParams.get(PARSER_KEY);
		if (parserValue != null && parserValue.length() > 0) {
			System.setProperty(PARSER_KEY, parserValue);
		}

		SystemConfigHandler sch = null;

		sch = (SystemConfigHandler)Class.forName(configHandlerClass).newInstance();

		Iterator iter = sch.getParameterNames().iterator();
		while (iter.hasNext()) {
			Object key = iter.next();
			if (!configParams.containsKey(key)) {
				throw new IllegalArgumentException(key+" is not found in the Params Map");
			}
		}

		sch.setParameters(configParams);

		SesameServer.setSystemConfig(sch.loadConfig());

		// initilize the 'main' session to the current thread
		SessionContext sc = SessionContext.getContext();
		if (sc == null) {
			sc = new SessionContext();
			SessionContext.put("main", sc);
			SessionContext.setContext(sc);
			new ThreadLocal().set(new SessionKilled("main"));
		}
	}

	/**
	 * Initializes Sesame/OMM given a configuration file.
	 * @param fileName the file name of the configuration
	 * @throws Exception
	 */
	public static void initialize(String fileName)
		throws Exception
	{
		HashMap params = new HashMap(1);
		params.put("systemConfigFile", fileName);

		initialize(DEFAULT_CONFIG_HANDLER, params);
	}

	/**
	 * Initializes Sesame/OMM given a configuration file and a parser class.
	 * @param fileName the configuration file name.
	 * @param parser XML parser class name to be used with the configuration file.
	 * @throws Exception
	 */
	public static void initialize(String fileName, String parser)
		throws Exception
	{
		HashMap params = new HashMap(2);
		params.put("systemConfigFile", fileName);
		params.put(PARSER_KEY, parser);

		initialize(DEFAULT_CONFIG_HANDLER , params);
	}
}
