/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.server;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;

import org.openrdf.util.io.IOUtil;
import org.openrdf.util.log.ThreadLog;

import org.openrdf.sesame.config.SystemConfig;
import org.openrdf.sesame.config.SystemConfigCenterListener;
import org.openrdf.sesame.repository.RepositoryList;
import org.openrdf.sesame.repository.local.LocalService;

/**
 * Static object that represents a locally running Sesame server.
 *
 * @author Jeen Broekstra
 * @version $Revision: 1.20.4.3 $
 */
public class SesameServer {

/*-------------+
| Variables    |
+-------------*/

	private static LocalService _localService;

	// Default: no users, no repositories
	private static SystemConfig _sysConfig = new SystemConfig();

	/** The base dir to resolve any relative paths against. **/
	private static File _baseDir = null;

	/** The version of this code base. **/
	private static String _version = null;

	private static List _listeners = new ArrayList();

/*-------------+
| Methods      |
+-------------*/

	/**
	 * Creates a LocalService object with the supplied SystemConfig. This
	 * LocalService can subsequently be shared across the JVM, to allow
	 * different services and protocol handlers to access the local
	 * repositories.
	 */
	public static void createLocalService() {
		_localService = new LocalService(_sysConfig);
	}

	/**
	 * Retrieves a LocalService object that can be used to access local
	 * repositories automatically
	 */
	public static LocalService getLocalService() {
		return _localService;
	}

	/**
	 * Gets the current system configuration of the system.
	 **/
	public static SystemConfig getSystemConfig() {
		return _sysConfig;
	}

	/**
	 * Sets a new system configuration for the system.
	 * Loads a new configuration and refreshes Sesame.
	 * @param newConfig the new SystemConfig
	 */
	public static void setSystemConfig(SystemConfig newConfig) {
		// Check if something changed in the RMI-related parameters
		boolean oldEnabled = false;
		String oldFac = null;
		int oldPort = -1;

		if (_sysConfig != null) {
			oldEnabled = _sysConfig.isRMIEnabled();
			oldFac = _sysConfig.getRMIFactoryClass();
			oldPort = _sysConfig.getRMIPort();
		}

		boolean newEnabled = newConfig.isRMIEnabled();
		String newFac = newConfig.getRMIFactoryClass();
		int newPort = newConfig.getRMIPort();

		if (oldEnabled != newEnabled ||
			(oldFac == null) != (newFac == null) || // one is null, the other isn't
			oldFac != null && !oldFac.equals(newFac) ||
			oldPort != newPort)
		{
			// something changed in the RMI settings

			// unbind any old RMI server
			if (oldEnabled) {
				_unbindRMIServer(oldFac, oldPort);
			}

			// bind the new RMI server
			if (newEnabled) {
				_bindRMIServer(newFac, newPort);
			}
		}

		// Unset all old system properties
		Properties oldSystemProps = _sysConfig.getSystemProps();
		Iterator keys = oldSystemProps.keySet().iterator();
		while (keys.hasNext()) {
			String key = (String)keys.next();
			System.setProperty(key, "");
		}

		// Set all new system properties
		Properties newSystemProps = newConfig.getSystemProps();
		keys = newSystemProps.keySet().iterator();
		while (keys.hasNext()) {
			String key = (String)keys.next();
			String value = newSystemProps.getProperty(key);
			System.setProperty(key, value);
		}

		_sysConfig = newConfig;

		if (_localService == null) {
			createLocalService();
		}
		else {
			_localService.setSystemConfig(newConfig);
		}

		_notifyListeners();
	}

	/**
	 * Sets the base dir against which to resolve relative paths.
	 **/
	public static void setBaseDir(File baseDir) {
		_baseDir = baseDir;
	}

	/**
	 * Gets the base dir against which to resolve relative paths (null if not set).
	 **/
	public static File getBaseDir() {
		return _baseDir;
	}

	/**
	 * Sets the default log file for all threads that have not registered
	 * themselves with ThreadLog.
	 **/
	public static void setDefaultLogFile(String fileName) {
		if (_sysConfig.getLogDir() == null || fileName == null) {
			// logDir not yet initialized or no filename, log to System.err
			ThreadLog.setDefaultLog(null, ThreadLog.WARNING);
		}
		else {
			File logDir = new File(_baseDir, _sysConfig.getLogDir());
			String logFile = new File(logDir, fileName).getPath();
			int logLevel = _sysConfig.getLogLevel();
			ThreadLog.setDefaultLog(logFile, logLevel);
		}
	}

	/**
	 * Unsets the default log file.
	 **/
	public static void unsetDefaultLogFile() {
		ThreadLog.unsetDefaultLog();
	}

	/**
	 * Sets the log file for the thread calling this method. All
	 * logging calls to ThreadLog will be logged in this file.
	 **/
	public static void setThreadLogFile(String fileName) {
		if (_sysConfig.getLogDir() == null || fileName == null) {
			// logDir not yet initialized or no filename, log to System.err
			ThreadLog.registerThread(null, ThreadLog.ALL);
		}
		else {
			File logDir = new File(_baseDir, _sysConfig.getLogDir());
			String logFile = new File(logDir, fileName).getPath();
			int logLevel = _sysConfig.getLogLevel();
			ThreadLog.registerThread(logFile, logLevel);
		}
	}

	/**
	 * Sets the log file for the thread calling this method. All
	 * logging calls to ThreadLog will be logged in this file. The
	 * calling thread is supposed to do stuff with the indicated
	 * <tt>repositoryID</tt>.
	 **/
	public static void setThreadLogFileForRepository(String repositoryID) {
		
		RepositoryList list = _localService.getRepositoryList();
		
		if (list.getRepository(repositoryID) != null) {
			// only set if the specified repository is known. if not, the thread will still log
			// in the default log file.
			setThreadLogFile("repositories/" + repositoryID + ".log");
		}
	}

	/**
	 * Unsets the log file for the thread calling this method.
	 **/
	public static void unsetThreadLogFile() {
		ThreadLog.deregisterThread();
	}

	/**
	 * Gets the version of this code base.
	 *
	 * @return A version String, e.g. "1.0".
	 **/
	public static String getVersion() {
		if (_version == null) {
			InputStream in = SesameServer.class.getClassLoader().getResourceAsStream("VERSION");

			if (in == null) {
				_version = "[VERSION FILE MISSING]";
			}
			else {
				try {
					_version = new String(IOUtil.readFully(in)).trim();
					in.close();
				}
				catch (IOException e) {
					ThreadLog.error("Unable to read version from file", e);
				}
			}
		}

		return _version;
	}

	/**
	 * Gets a Sail for the specified repositoryId. The Sail that is returned
	 * is shared with everyone else calling this method with the same
	 * repositoryId.
	 *
	 * @param repositoryId the ID of the repository to get the SAIL for
	 * @throws UnknownRepositoryException If the SystemConfigCenter does
	 * not know the specified repository ID.
	 */
/*
	public static Sail getSail(String repositoryId)
		throws UnknownRepositoryException
	{
		synchronized (_sharedSails) {
			// First check wether the Sail is already created.
			Sail result = (Sail)_sharedSails.get(repositoryId);
			if (result != null) {
				return result;
			}

			// Get the info on the repository
			RepositoryConfig repConfig = _sysConfig.getRepositoryConfig(repositoryId);
			if (repConfig == null) {
				throw new UnknownRepositoryException("Unknown repository: " + repositoryId);
			}

			// Create a new Sail for the bottom of the Sail stack
			List sailList = repConfig.getSailList();
			SailConfig sailConfig = (SailConfig)sailList.get(sailList.size() - 1);
			String className = sailConfig.getSailClass();

			try {
				Class sailClass = Class.forName(className);

				ThreadLog.trace("Creating new Sail: " + className);
				result = (Sail)sailClass.newInstance();

				ThreadLog.trace("Initializing Sail");
				result.initialize(sailConfig.getConfigParameters());
				ThreadLog.trace("Sail initialized");
			}
			catch (Exception e) {
				//FIXME: throw some other type of exception?
				if (e instanceof SailInternalException) {
					throw (SailInternalException)e;
				}
				else {
					throw new SailInternalException(e);
				}
			}

			// Add any stacked Sails on top
			for (int i = sailList.size() - 2; i >= 0; i--) {
				sailConfig = (SailConfig)sailList.get(i);
				className = sailConfig.getSailClass();

				try {
					Class sailClass = Class.forName(className);

					ThreadLog.trace("Creating new stacked Sail: " + className);
					StackedSail stackedSail = (StackedSail)sailClass.newInstance();

					stackedSail.setBaseSail(result);

					ThreadLog.trace("Initializing stacked Sail");
					stackedSail.initialize(sailConfig.getConfigParameters());
					ThreadLog.trace("Stacked Sail initialized");

					result = stackedSail;
				}
				catch (Exception e) {
					ThreadLog.error("Error initializing Sail:", e);
					
					//FIXME: throw some other type of exception?
					if (e instanceof SailInternalException) {
						throw (SailInternalException)e;
					}
					else {
						throw new SailInternalException(e);
					}
				}
			}

			// Register the new Sail
			_sharedSails.put(repositoryId, result); 
			return result;
		}
	}
*/

	/**
	 * Gets (or creates) a Sail for the specified repositoryId.
	 *
	 * @param repositoryId the ID of the repository to get the SAIL for
	 * @param user the user to authenticate as
	 * @param password the password of the user
	 * @param wantsReadAccess true if read access is required, otherwise - false
	 * @param wantsWriteAccess true if write access is required, otherwise - false
	 * @return the first SAIL from the stacked SAILs.
	 * @throws AccessDeniedException if no permission are granted to the user
	 * @throws UnknownRepositoryException if no such repository is found
	 */
/*
	public static Sail getSail(
		String repositoryId, String user, String password,
		boolean wantsReadAccess, boolean wantsWriteAccess)
		throws AccessDeniedException, UnknownRepositoryException
	{
		UserAccessManager.checkAccessPermissions(_sysConfig, repositoryId,
				user, password, wantsReadAccess, wantsWriteAccess);

		return getSail(repositoryId);
	}
*/

	/**
	 * Retrieves a list of repositories given a user and a password
	 * @param user the user to login as
	 * @param password the password of the user
	 * @param wantsReadAccess true if read access is required, otherwise - false
	 * @param wantsWriteAccess true if write access is required, otherwise - false
	 * @return a list of the available repositories
	 */
/*
	public static List getRepositoryList(String user, String password,
		boolean wantsReadAccess, boolean wantsWriteAccess)
	{
		List result = new ArrayList();

		Iterator iter = _sysConfig.getRepositoryConfigList().iterator();
		while (iter.hasNext()) {
			RepositoryConfig repConfig = (RepositoryConfig)iter.next();
			try {
				UserAccessManager.checkAccessPermissions(
						_sysConfig, repConfig.getRepositoryId(),
						user, password,
						wantsReadAccess, wantsWriteAccess);

				// access permissions are OK
				result.add(repConfig);
			}
			catch (AccessDeniedException e) {
				// ignore
			}
			catch (UnknownRepositoryException e) {
				// never thrown
			}
		}

		return result;
	}
*/

	/**
	 * Clears the LocalService
	 */
	public static void clear() {
		_localService.shutDown();
		_unbindRMIServer(_sysConfig.getRMIFactoryClass(), _sysConfig.getRMIPort());
	}

	/**
	 * Adds a SystemConfigCenterListener
	 *
	 * @param listener SystemConfigCenterListener
	 */
	public static void addListener(SystemConfigCenterListener listener) {
		_listeners.add(listener);
	}

	/**
	 * Removes a SystemConfigCenterListener
	 *
	 * @param listener SystemConfigCenterListener
	 */
	public static void removeListener(SystemConfigCenterListener listener) {
		_listeners.remove(listener);
	}

	protected static void _notifyListeners() {
		Iterator i = _listeners.iterator();

		while (i.hasNext()) {
			((SystemConfigCenterListener)i.next()).configurationRefreshed();
		}
	}

/*---------------------------------------------+
| (un)binding RMI servers                      |
+---------------------------------------------*/


	protected static void _bindRMIServer(String rmiFactoryClass, int port) {
		if (rmiFactoryClass == null) {
			return;
		}

		// let's bind that rmi-factory
		Class rmi_factory = null;
		java.lang.reflect.Method rmi_factory_bind = null;

		try {
			rmi_factory = Class.forName(rmiFactoryClass);
			rmi_factory_bind = rmi_factory.getDeclaredMethod("bind", new Class[]{Integer.class});
			rmi_factory_bind.invoke(null, new Object[] { new Integer(port) });
		}
		catch (ClassNotFoundException e) {
			ThreadLog.error("RMI factory class '" + rmiFactoryClass + "' not found");
		}
		catch (NoSuchMethodException e) {
			ThreadLog.error("Method 'static void bind(Integer)' not found in RMI factory class '" + rmiFactoryClass + "'");
		}
		catch (IllegalAccessException e) {
			ThreadLog.error("Unable to bind RMI server: " + e.getMessage());
		}
		catch (java.lang.reflect.InvocationTargetException e) {
			ThreadLog.error("Unable to bind RMI server: " + e.getMessage());
		}
	}

	protected static void _unbindRMIServer(String rmiFactoryClass, int port) {
		if (rmiFactoryClass == null) {
			return;
		}

		// let's unbind that rmi-factory
		Class rmi_factory = null;
		java.lang.reflect.Method rmi_factory_unbind = null;

		try {
			rmi_factory = Class.forName(rmiFactoryClass);
			rmi_factory_unbind = rmi_factory.getDeclaredMethod("unbind", new Class[] {Integer.class});
			rmi_factory_unbind.invoke(null, new Object[] { new Integer(port) });
		}
		catch (ClassNotFoundException e) {
			ThreadLog.error("RMI factory class '" + rmiFactoryClass + "' not found");
		}
		catch (NoSuchMethodException e) {
			ThreadLog.error("Method 'static void unbind(Integer)' not found in RMI factory class '" + rmiFactoryClass + "'");
		}
		catch (IllegalAccessException e) {
			ThreadLog.error("Unable to unbind RMI server: " + e.getMessage());
		}
		catch (java.lang.reflect.InvocationTargetException e) {
			ThreadLog.error("Unable to unbind RMI server: " + e.getMessage());
		}

		// clean up on a separate thread
		Thread gcThread = new Thread(new Runnable() {
			public void run() {
				synchronized (this) {
					try {
						wait(500); System.gc();
						wait(500); System.gc();
						wait(500); System.gc();
					}
					catch (InterruptedException e) {
						notifyAll();
					}
				}
			}
		});

		gcThread.start();
	}
}
	
