/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2003 OntoText Lab, Sirma AI OOD
 *
 *  Contact:
 *	Sirma AI OOD, OntoText Lab.
 *	38A, Christo Botev Blvd.
 *  1000 Sofia, Bulgaria
 *	tel. +359(2)981 00 18
 *	fax. +359(2)981 90 58
 *	info@ontotext.com
 *
 * 	http://www.ontotext.com/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sailimpl.rdbms.rules;

/**
 * <p>Title: Custom Inference Rules</p>
 * <p>Description:</p>
 * The class is used to produce SQL queries used for statement and dependancy inferencing.
 * It keeps several TripleTemplates defining relationships between triples that lead
 * to the inference of new ones. The templates consist of three Component instances for
 * subject, predicate and object and can represent an URI, variable or regular expression.
 *
 * <p>Company: Ontotext Lab. Sirma AI</p>
 * @author Damyan Ognyanoff
 * @version 1.0
 */
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;

import org.openrdf.sesame.sailimpl.rdbms.CustomInferenceServices;
import org.openrdf.sesame.sailimpl.rdbms.TableNames;

public class Rule implements TableNames {

	final static String C_SELECT = "SELECT DISTINCT ";

	/**
	 * the map that hold id's (as Integer variables) fro each non-variable string found in the rules.
	 * These ids are afterwards matchd against the real ids of those resources with
	 * same URIs and used within generated SQL queries.
	 */
	static HashMap idMap = new HashMap();

	/**
	 * collection of all triple templates for the rule
	 */
	ArrayList premiseList;

	/**
	 * The consequent template
	 */
	TripleTemplate consequent;

	/**
	 * The name of the Rule
	 */
	String ruleName;

	ArrayList triggersRule = null;

	HashMap regExpByVars;

	void normalizeRegExpVar(Component var) {
		if (var.isRegExp()) {
			if (regExpByVars == null) {
				regExpByVars = new HashMap();
			}
			regExpByVars.put(var.value(), var);
		}
	}

	static String getFROM() {
		return " FROM " + NEW_TRIPLES_TABLE + " nt ";
	}

	static String getLEFT_JOIN(int id) {
		return "LEFT JOIN " + TRIPLES_TABLE + " t" + id + " ON ";
	}

	public static Iterator constantsIter() {
		return idMap.keySet().iterator();
	}

	public static String getId(Component constant) {
		Integer val = (Integer)idMap.get(constant.value());
		if (val == null) {
			if (!constant.isVar()) {
				val = setId(constant, idMap.size());
			}
			else {
				return null;
			}
		}
		return val.toString();
	}

	public static int getIntId(Component constant) {
		Integer val = (Integer)idMap.get(constant.value());
		if (val == null) {
			if (!constant.isVar()) {
				val = setId(constant, idMap.size());
			}
			else {
				return -1;
			}
		}
		return val.intValue();
	}

	public static Integer setId(Component constant, int id) {
		if (!constant.isVar()) {
			return setId(constant.value(), id);
		}
		return new Integer(id);
	}

	public static Integer setId(String constant, int id) {
		Integer i = new Integer(id);
		idMap.put(constant, i);
		return i;
	}

	public Rule(String name) {
		premiseList = new ArrayList();
		consequent = null;
		ruleName = name;
	}

	public void addPremise(TripleTemplate t) {
		if (!t.subject.isVar()) {
			getId(t.subject);
		}
		if (!t.predicate.isVar()) {
			getId(t.predicate);
		}
		if (!t.object.isVar()) {
			getId(t.object);
		}

		normalizeRegExpVar(t.subject);
		normalizeRegExpVar(t.predicate);
		normalizeRegExpVar(t.object);

		premiseList.add(t);
	}

	public void setConsequent(TripleTemplate t) {
		if (!t.subject.isVar()) {
			getId(t.subject);
		}
		if (!t.predicate.isVar()) {
			getId(t.predicate);
		}
		if (!t.object.isVar()) {
			getId(t.object);
		}

		normalizeRegExpVar(t.subject);
		normalizeRegExpVar(t.predicate);
		normalizeRegExpVar(t.object);

		consequent = t;
	}

	private boolean matchwasbyobject = false;

	String match(Component component, int until) {
		matchwasbyobject = false;
		if (!component.isVar()) {
			return Rule.getId(component);
		}
		int count = 0;
		Iterator iter = premiseList.iterator();
		while (iter.hasNext()) {
			if (count >= until) {
				break;
			}
			TripleTemplate tt = (TripleTemplate)iter.next();
			if (0 == component.compareTo(tt.subject)) {
				return "" + ((count == 0) ? "nt" : ("t" + count)) + ".subj";
			}
			if (0 == component.compareTo(tt.predicate)) {
				return "" + ((count == 0) ? "nt" : ("t" + count)) + ".pred";
			}
			if (0 == component.compareTo(tt.object)) {
				matchwasbyobject = true;
				return "" + ((count == 0) ? "nt" : ("t" + count)) + ".obj";
			}
			count++;
		}
		return null;
	}

	private String where = "";

	void processJoin(StringBuffer buffer, int id, TripleTemplate tt) {
		buffer.append(' ');
		buffer.append(Rule.getLEFT_JOIN(id));
		boolean bFirst = false;
		String s = match(tt.subject, id);
		if (s != null) {
			buffer.append("t" + id);
			buffer.append(".subj = ");
			buffer.append(s);
			if (matchwasbyobject) {
				if (where.length() != 0) {
					where += " AND ";
				}
				where += s + " > 0"; // non-literal restriction
			}
			bFirst = true;
		}
		s = match(tt.predicate, id);
		if (s != null) {
			if (bFirst) {
				buffer.append(" AND ");
			}
			buffer.append("t" + id);
			buffer.append(".pred= ");
			buffer.append(s);
			bFirst = true;
		}
		s = match(tt.object, id);
		if (s != null) {
			if (bFirst) {
				buffer.append(" AND ");
			}
			buffer.append("t" + id);
			buffer.append(".obj= ");
			buffer.append(s);
		}
		processjoinWithRegExpr(buffer, tt.subject, "t" + id, "subj");
		processjoinWithRegExpr(buffer, tt.predicate, "t" + id, "pred");
		processjoinWithRegExpr(buffer, tt.object, "t" + id, "obj");
	}

	public ArrayList getAllSQLs() {
		ArrayList result = new ArrayList();
		result.add(getSQL());
		if (premiseList.size() < 2) {
			return result;
		}
		for (int i = 1; i < premiseList.size(); i++) {
			Object f = premiseList.remove(0);
			premiseList.add(f);
			result.add(getSQL());
		}
		//restore
		Object f = premiseList.remove(0);
		premiseList.add(f);
		return result;
	}

	String getSQL() {
		alreadyJoined = null;
		where = "";
		StringBuffer buffer = new StringBuffer();
		buffer.append(C_SELECT);
		buffer.append(match(consequent.subject, premiseList.size()));
		buffer.append(',');
		buffer.append(match(consequent.predicate, premiseList.size()));
		buffer.append(',');
		buffer.append(match(consequent.object, premiseList.size()));
		buffer.append('\n');
		buffer.append(Rule.getFROM());
		buffer.append('\n');
		//    processjoinWithRegExpr(buffer, consequent.subj, "nt", "subj");
		//    processjoinWithRegExpr(buffer, consequent.pred, "nt", "pred");
		//    processjoinWithRegExpr(buffer, consequent.obj, "nt", "obj");
		for (int i = 1; i < premiseList.size(); i++) {
			TripleTemplate tt = (TripleTemplate)premiseList.get(i);
			processJoin(buffer, i, tt);
		}

		processJoin(buffer, premiseList.size(), consequent);

		TripleTemplate ttFirst = (TripleTemplate)premiseList.get(0);
		buffer.append(" WHERE ");
		boolean ifFirst = false;
		if (where.length() > 0) {
			buffer.append(where);
			ifFirst = true;
		}
		if (!ttFirst.subject.isVar()) {
			if (ifFirst) {
				buffer.append(" AND ");
			}
			buffer.append("nt.subj= ");
			buffer.append(getId(ttFirst.subject));
			ifFirst = true;
		}
		if (!ttFirst.predicate.isVar()) {
			if (ifFirst) {
				buffer.append(" AND ");
			}
			buffer.append("nt.pred = ");
			buffer.append(getId(ttFirst.predicate));
			ifFirst = true;
		}
		if (!ttFirst.object.isVar()) {
			if (ifFirst) {
				buffer.append(" AND ");
			}
			buffer.append("nt.obj = ");
			buffer.append(getId(ttFirst.object));
			ifFirst = true;
		}
		for (int i = 1; i < premiseList.size(); i++) {
			if (ifFirst) {
				buffer.append(" AND ");
			}
			buffer.append("t" + i + ".subj is not NULL");
			ifFirst = true;
		}
		if (ifFirst) {
			buffer.append(" AND ");
		}
		buffer.append("t" + premiseList.size() + ".subj is NULL");

		return buffer.toString();
	}

	//  10. 	xxx rdf:type rdfs:ContainerMembershipProperty      -->  (dep1)
	//    xxx rdfs:subPropertyOf rdfs:member			(t)
	/**
	 *   Algorithm outline:
	 * 0. add insert into clause+join of t with the first template (dep1)
	 * 1. fill all constant restrictions on t component (the inferred statement)
	 * 2. add each match with dep1 components to the join
	 * 3. add next template restriction (dep2) from the remaining list
	 * 4. fiil its contsnt restrictions and add any matches with the t or dep1
	 * 5. add WHERE clause
	 */
	private boolean appendConstaints(StringBuffer buffer, TripleTemplate t,
			boolean bFirst, String tablePrefix)
	{
		if (!t.subject.isVar()) {
			if (bFirst) {
				buffer.append(" AND ");
			}
			buffer.append(tablePrefix);
			buffer.append(".subj = ");
			buffer.append(getId(t.subject));
			bFirst = true;
		}
		if (!t.predicate.isVar()) {
			if (bFirst) {
				buffer.append(" AND ");
			}	
			buffer.append(tablePrefix);
			buffer.append(".pred = ");
			buffer.append(getId(t.predicate));
			bFirst = true;
		}
		if (!t.object.isVar()) {
			if (bFirst) {
				buffer.append(" AND ");
			}
			buffer.append(tablePrefix);
			buffer.append(".obj = ");
			buffer.append(getId(t.object));
			bFirst = true;
		}
		return bFirst;
	}

	private String matchVarComponent(Component var, TripleTemplate t) {
		if (var.compareTo(t.subject) == 0) {
			return ".subj";
		}
		if (var.compareTo(t.predicate) == 0) {
			return ".pred";
		}
		if (var.compareTo(t.object) == 0) {
			return ".obj";
		}
		return null;
	}

	private boolean matchAgainst(StringBuffer buffer, boolean bFirst,
			TripleTemplate from, String fromPrefix, TripleTemplate to, String toPrefix)
	{
		if (bFirst) {
			buffer.append(" AND ");
		}

		buffer.append(fromPrefix);
		buffer.append(".id <> ");
		buffer.append(toPrefix);
		buffer.append(".id ");
		bFirst = true;

		String foundMatch = null;
		foundMatch = matchVarComponent(from.subject, to);
		if (foundMatch != null) {
			if (bFirst) {
				buffer.append(" AND ");
			}
			buffer.append(fromPrefix);
			buffer.append(".subj = ");
			buffer.append(toPrefix);
			buffer.append(foundMatch);
			foundMatch = null;
			bFirst = true;
		}
		foundMatch = matchVarComponent(from.predicate, to);
		if (foundMatch != null) {
			if (bFirst) {
				buffer.append(" AND ");
			}
			buffer.append(fromPrefix);
			buffer.append(".pred = ");
			buffer.append(toPrefix);
			buffer.append(foundMatch);
			foundMatch = null;
			bFirst = true;
		}
		foundMatch = matchVarComponent(from.object, to);
		if (foundMatch != null) {
			if (bFirst) {
				buffer.append(" AND ");
			}
			buffer.append(fromPrefix);
			buffer.append(".obj = ");
			buffer.append(toPrefix);
			buffer.append(foundMatch);
			foundMatch = null;
			bFirst = true;
		}
		return bFirst;
	}

	public ArrayList getDependSQL() {
		alreadyJoined = null;
		StringBuffer bufferPrefix = new StringBuffer();
		bufferPrefix.append("insert into ");
		bufferPrefix.append(DEPEND_TABLE);
		bufferPrefix.append("\n\r SELECT t.id ");

		for (int i = 1; i <= CustomInferenceServices.getMaxTemplateCountsPerRule(); i++) {
			if (i <= premiseList.size()) {
				bufferPrefix.append(", dep" + (i) + ".id ");
			}
			else {
				bufferPrefix.append(", 0");
			}
		}
		bufferPrefix.append(" FROM ");

		StringBuffer bufferMidSuffix = new StringBuffer();
		StringBuffer bufferSuffix = new StringBuffer();
		bufferSuffix.append(" WHERE ");
		boolean bFirst = false;
		bFirst = appendConstaints(bufferSuffix, consequent, bFirst, "t");
		for (int i = 0; i < premiseList.size(); i++) {
			TripleTemplate tr = (TripleTemplate)premiseList.get(i);
			bFirst = appendConstaints(bufferSuffix, tr, bFirst, "dep" + (i + 1));
			bFirst = matchAgainst(bufferSuffix, bFirst, consequent, "t", tr, "dep" + (i + 1));
			for (int j = 0; j < i; j++) {
				bFirst = matchAgainst(bufferSuffix, bFirst, tr, "dep" + (i + 1),
						(TripleTemplate)premiseList.get(j), "dep" + (j + 1));
			}
			processjoinWithRegExpr(bufferMidSuffix, tr.subject, "dep" + (i + 1), "subj");
			processjoinWithRegExpr(bufferMidSuffix, tr.predicate, "dep" + (i + 1), "pred");
			processjoinWithRegExpr(bufferMidSuffix, tr.object, "dep" + (i + 1), "obj");
		}
		
		ArrayList ret = new ArrayList();
		for (int i = 0; i < premiseList.size(); i++) {
			String middle = "";
			for (int j = 0; j < premiseList.size(); j++) {
				if (i == j) {
					middle += ALL_NEW_TRIPLES_TABLE + " dep" + (j + 1) + ",";
				}
				else {
					middle += TRIPLES_TABLE + " dep" + (j + 1) + ",";
				}
			}
			middle += TRIPLES_TABLE + " t ";
			ret.add(bufferPrefix.toString() + middle + bufferMidSuffix + bufferSuffix.toString());
		}
		return ret;
	}

	public void initTriggersRuleCollection() {
		triggersRule = new ArrayList();
	}

	public void triggersRule(String ruleRef) {
		triggersRule.add(ruleRef);
	}

	public boolean hasTriggersRuleCollection() {
		return triggersRule != null;
	}

	public String getName() {
		return ruleName;
	}

	protected String escapeRegExpression(String expression, String esc) {
		if (esc == null) {
			esc = "\\";
		}
		if (0 == esc.compareTo("\\")) {
			esc = esc + esc;
		}
		expression = expression.replaceAll("[',_,%," + esc + "]", esc + "$0");
		expression = expression.replaceAll("([^" + esc + "])([?])", "$1_");
		return expression.replaceAll("([^" + esc + "])([\\*])", "$1%");
	}

	HashMap alreadyJoined;

	protected void processjoinWithRegExpr(StringBuffer buffer,
			Component component, String tableID, String sub_ob_pred)
	{
		//@todo chek if it is joined already
		if (regExpByVars == null) {
			return;
		}
		if (alreadyJoined == null) {
			alreadyJoined = new HashMap();
		}
		component = (Component)regExpByVars.get(component.value());
		if (component == null) {
			return;
		}
		if (component.isRegExp() && null == alreadyJoined.get(component)) {
			String varName = component.value();
			Component regExpr = (Component)regExpByVars.get(varName);
			if (regExpr != null) {
				// inner join resources (r+id) on (r+id).id == (t+id).subject
				// inner join namespaces (n+id) on (r+id).namespace == (n+id).id
				// AND concat((n+id).name, (r+id).localname) like
				buffer.append(" inner join ");
				buffer.append(RESOURCES_TABLE);
				buffer.append(" r" + varName);
				buffer.append(" on r" + varName);
				buffer.append(".id = " + tableID);
				buffer.append("." + sub_ob_pred);

				buffer.append(" inner join ");
				buffer.append(NAMESPACES_TABLE);
				buffer.append(" n" + varName);
				buffer.append(" on r" + varName);
				buffer.append(".namespace = n" + varName + ".id AND ");

				buffer.append("concat( n" + varName + ".name, ");
				buffer.append(" r" + varName + ".localname ) LIKE '");
				// 1 . add escChar ahead of all ',%,_
				// 2 . replace all ?,* with (not preceeded by esc) with _,%
				// 3 . move all exc before ?,*
				String esc = regExpr.getRegExpEsc();
				if (esc == null) {
					esc = "\\";
				}
				String likeText = escapeRegExpression(regExpr.getRegExpTemplate(), esc);
				buffer.append(likeText);
				buffer.append("' ESCAPE '" + esc + esc + "' ");
			}
			// mark that it is already joined
			alreadyJoined.put(component, "");
		}
	}

	public int getPremiseCount() {
		return premiseList.size();
	}

	public TripleTemplate getConsequent() {
		return consequent;
	}

	public ArrayList getPremiseCollection() {
		return premiseList;
	}

	public ArrayList getTriggersRule() {
		return triggersRule;
	}

	public static void main(String op[]) {
		/*    String val = "http://www.w3.org/1999/02/22-rdf-syn\\tax-ns#_t%*yp?'e";
		 String esc = "\\";
		 if (0==esc.compareTo("\\"))
		 esc = esc+esc;
		 String valOne = val.replaceAll("[',_,%,"+esc+"]", esc+"$0");
		 String valTwo = valOne.replaceAll("([^"+esc+"])([?])", "$1_");
		 String valTre = valTwo.replaceAll("([^"+esc+"])([\\*])", "$1%");
		 System.out.println(val);
		 System.out.println(valOne);
		 System.out.println(valTwo);
		 System.out.println(valTre);
		 */
		Rule rule = new Rule("test");

		Component subjectComp = new Component("xxx", Component.VAR);
		Component predicateComp = new Component("id", Component.REGEXP);
		predicateComp.setRegExpTemplate("http://www.w3.org/1999/02/22-rdf-syntax-ns#_*", "\\");
		Component objectComp = new Component("yyy", Component.VAR);
		TripleTemplate t = new TripleTemplate(subjectComp, predicateComp, objectComp);
		/*    t.subject = new Component("xxx", Component.VAR);
		 t.predicate = new Component("prop", Component.VAR);
		 t.object = new Component("yyy", Component.VAR);
		 rule.addPremise(t);
		 t = new TripleTemplate();
		 t.subject = new Component("yyy", Component.VAR);
		 t.predicate = new Component("prop", Component.VAR);
		 t.object = new Component("zzz", Component.VAR);
		 */
		rule.addPremise(t);

		subjectComp = new Component("id", Component.VAR);
		predicateComp = new Component("http://www.w3.org/1999/02/22-rdf-syntax-ns#type", Component.URI);
		objectComp = new Component("http://www.w3.org/2000/01/rdf-schema#ContainerMembershipProperty", Component.URI);
		t = new TripleTemplate(subjectComp, predicateComp, objectComp);
		/*    t.subject = new Component("xxx", Component.VAR);
		 t.predicate = new Component("prop", Component.REGEXP);
		 t.predicate.setRegExpTemplate("http://www.w3.org/2000/01/rdf-schema#sub*","\\");
		 t.object = new Component("zzz", Component.VAR);
		 */
		rule.setConsequent(t);

		ArrayList list = rule.getAllSQLs();
		//    ArrayList list = rule.getDependSQL();
		Iterator iter = list.iterator();
		while (iter.hasNext()) {
			System.out.println(iter.next());
		}
	}
}
