/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2003 OntoText Lab, Sirma AI OOD
 *
 *  Contact:
 *	Sirma AI OOD, OntoText Lab.
 *	38A, Christo Botev Blvd.
 *  1000 Sofia, Bulgaria
 *	tel. +359(2)981 00 18
 *	fax. +359(2)981 90 58
 *	info@ontotext.com
 *
 * 	http://www.ontotext.com/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sailimpl.rdbms.rules;

/**
 * <p>Description: Represents a component of a triple template used in a rule.
 * It can be a resource URI, variable name or a regular expression.</p>
 *
 * @author Damyan Ognyanoff
 * @version 1.0
 */
public class Component implements Comparable {

/*------------+
| Constants   |
+------------*/

	// The possible types of components
	public static final int VAR = 1;
	public static final int URI = 2;
	public static final int REGEXP = 3;

/*------------+
| Variables   |
+------------*/

	private int type;
	private String value;
	private String regexpTemplate;

	/**
	 * the escape character to be used in regular expressions
	 */
	private String escapeChar;

/*-------------+
| Constructors |
+-------------*/

	/**
	 * The constructor
	 * @param value of the component
	 * @param type - one of the URI, VAR or REGEXP constants
	 */
	public Component(String value, int type) {
		this.type = type;
		this.value = value;
	}

/*-------------+
| Methods      |
+-------------*/

	/**
	 * Returns <tt>true</tt> when this component has type VAR or REGEXP.
	 **/
	public boolean isVar() {
		return (type == VAR) || (type == REGEXP);
	}

	/**
	 * Returns <tt>true</tt> when this component has type URI.
	 **/
	public boolean isUri() {
		return type == URI;
	}

	/**
	 * Returns <tt>true</tt> when this component has type REGEXP.
	 **/
	public boolean isRegExp() {
		return type == REGEXP;
	}

	/**
	 * Returns the type of this component.
	 *
	 * @return One of the values <tt>VAR</tt>, <tt>URI</tt> or <tt>REGEXP</tt>.
	 **/
	public int getType() {
		return type;
	}

	/**
	 * Returns the value of this component.
	 **/
	public String value() {
		return value;
	}

	public int compareTo(Object obj) {
		if (obj instanceof Component) {
			if (value.length() == 0) {
				return -1;
			}

			return value.compareTo(((Component)obj).value());
		}
		else {
			return -1;
		}
	}

	public boolean equals(Object o) {
		return this.compareTo(o) == 0;
	}

	public int hashCode() {
		return value.hashCode();
	}

	public void setRegExpTemplate(String template, String escape) {
		type = REGEXP;
		regexpTemplate = template;
		if (escape != null) {
			this.escapeChar = escape;
		}
	}

	String getRegExpTemplate() {
		return regexpTemplate;
	}

	String getRegExpEsc() {
		return escapeChar;
	}
}
