/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sailimpl.rdbms.iterators;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.NoSuchElementException;

import org.openrdf.model.Value;

import org.openrdf.sesame.sail.SailInternalException;
import org.openrdf.sesame.sail.ValueIterator;
import org.openrdf.sesame.sailimpl.rdbms.RdfSource;

/**
 * A ResourceIterator that executes an SQL query. RdbmsValueIterator is an
 * abstract class, no instances of RdbmsValueIterator can be created. A subclass
 * of RdbmsValueIterator is defined that either iterates rdfs:Resources or
 * rdfs:Literals. This subclass must implement getResult(), the implementation of
 * getResult() must call the proper contructor of either rdfs:Resource or
 * rdfs:Literal.
 *
 * @author Peter van 't Hof
 * @author Arjohn Kampman
 * @version $Revision: 1.6.4.2 $
 */

public abstract class RdbmsValueIterator implements ValueIterator {

/*----------+
| Variables |
+----------*/

	protected RdfSource _source;

	protected String[] _namespaceNames;

	/**
	 * ResultSet for the query. */
	protected ResultSet _resultSet;

	/**
	 * SQL statement that was used to generate the ResultSet. It cannot be closed
	 * until all results have been read from the ResultSet.
	 */
	protected java.sql.Statement _statement;
	/**
	 * Connection to the database. */
	protected Connection _databaseCon;
	/**
	 * Query to execute. */
	protected String _query;
	/**
	 * Flag indicating whether there are any more results. */
	protected boolean _hasNext;

/*-------------+
| Constructors |
+-------------*/

	/**
	 * Constructor.
	 *
	 * @param databaseCon connection to the repository
	 * @param query to execute
	 */
	public RdbmsValueIterator(RdfSource source, String[] namespaceNames,
					Connection databaseCon, String query)
	{
		_source = source;
		_namespaceNames = namespaceNames;
		_databaseCon = databaseCon;
		_query = query;
		_execQuery();

		if (!_hasNext) {
			// No result for this query, close Connection, the Statement
			// and its ResultSet.
			close();
		}
	}

/*--------+
| Methods |
+--------*/

	private void _execQuery() {
		try {
 			_statement = _databaseCon.createStatement();
			_resultSet = _statement.executeQuery(_query);

			// Check if the new ResultSet contains any values.
			_hasNext = _resultSet.next();
		}
		catch (SQLException e) {
			throw new SailInternalException(e);
		}
	}

	public boolean hasNext() {
		return _hasNext;
	}

	public Value next() {
		if (_hasNext) {
			Value result = null;
			try {
				result = getResult();

				_hasNext = _resultSet.next();

				if (!_hasNext) {	// No more results for this query.
					close();
				}
			}
			catch (SQLException e) {
				throw new SailInternalException(e);
			}
			return result;
		}
		else {
			throw new NoSuchElementException("No more values...");
		}
	}

	public void close() {
		if (_databaseCon != null) {
			try {
				if (_resultSet != null) {
					_resultSet.close();
					_resultSet = null;
				}
				if (_statement != null) {
					_statement.close();
					_statement = null;
				}
				_databaseCon.close();

				/* Prevent connection being closed second time through the
				 * finalize method by garbage collector. The connection can be
				 * reused by another object. 
				 */
				_databaseCon = null;

				_hasNext = false;	// No more results are returned.
			}
			catch (SQLException e) {
				throw new SailInternalException(e);
			}
		}
	}

	/**
	 * Called by the garbage collector on RdbmsValueIterator when garbage
	 * collection determines that there are no more references to the
	 * RdbmsValueIterator.
	 */
	protected void finalize() {
		close();
	}

	/**
	 * Gets the result from the ResultSet. Must be implemented by
	 * RdbmsResourceIterator or RdbmsLiteralIterator in order to call the proper
	 * constructor of either Resource or Literal.
	 *
	 * @return result from ResultSet
	 */
	protected abstract Value getResult();
}
