/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sailimpl.rdbms.iterators;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.NoSuchElementException;

import org.openrdf.model.Resource;
import org.openrdf.model.Statement;
import org.openrdf.model.URI;
import org.openrdf.model.Value;
import org.openrdf.model.impl.StatementImpl;

import org.openrdf.sesame.sail.SailInternalException;
import org.openrdf.sesame.sail.StatementIterator;
import org.openrdf.sesame.sailimpl.rdbms.RdfSource;
import org.openrdf.sesame.sailimpl.rdbms.model.IdBNode;
import org.openrdf.sesame.sailimpl.rdbms.model.IdLiteral;
import org.openrdf.sesame.sailimpl.rdbms.model.IdURI;

/**
 * A StatementIterator that executes an SQL query. The ResultSet should
 * contain 3 columns for each resource or literal that is not fixed. For
 * a resource this is: id, namespace and localname. For a literal this is:
 * id, language and value. The total number of columns expected depends on
 * the number of values/positions that have been specified by the caller.
 * E.g. if the caller asks for all statements with a specific subject, the
 * subject is 'fixed', and doesn't have to be queried from the database.
 * In this case, the query is expected to have six columns: three for the
 * predicate and three for the object.
 *
 * @author Peter van 't Hof
 * @author Arjohn Kampman
 */
public class RdbmsStatementIterator implements StatementIterator {

/*----------+
| Variables |
+----------*/

/*
 * With getObject() a StatementIterator returns the object of the statement
 * it is pointing at. An object can be a resource or a literal, therefore
 * it should know whether to construct a Resource or a Literal out of
 * the ResultSet. A way to establish this is to execute 2 queries, one where
 * the object of the triples table is joined with the resources table and
 * one where it is joined with literals table. The boolean
 * _literalsHasBeenQueried identifies which query is executed, so whether
 * to construct a Resource or a Literal from the ResultSet.
 */

	private RdfSource _source;

	private String[] _namespaceNames;

	/** ResultSet for the query. **/
	private ResultSet _resultSet;

	/**
	 * SQL statement that was used to generate the ResultSet. It cannot be
	 * closed until all results have been read from the ResultSet.
	 **/
	private java.sql.Statement _statement;

	/** Connection to the database. **/
	private Connection _databaseCon;

	/** The query to execute to get the resource objects. **/
	private String _queryResources;

	/** The query to execute to get the literal objects. **/
	private String _queryLiterals;

	/** Flag indicating whether there are any more results. **/
	private boolean _hasNext;

	/** Subject of statement. **/
	private Resource _subject;

	/** Predicate of statement. **/
	private URI _predicate;

	/** Object of statement. **/
	private Value _object;

	/** subject is fixed? **/
	private boolean _subjectFixed;

	/** predicate is fixed? **/
	private boolean _predicateFixed;

	/** object is fixed? **/
	private boolean _objectFixed;

	/** Indicates which query is executed. **/
	private boolean _literalsHasBeenQueried;

/*----------------------------------------------+
| Constructors                                  |
+----------------------------------------------*/

	/**
	 * Constructor.
	 *
	 * @param source the object to supply as 'creator' to created IdValues.
	 * @param namespaceNames an array of namespace names that are index using
	 * their IDs.
	 * @param databaseCon connection to the repository
	 * @param queryResources query specified for the resources table, or
	 * <tt>null</tt> if statements with resources as object do not need to be
	 * queried.
	 * @param queryLiterals query specified for the literals table, or
	 * <tt>null</tt> if statements with literals as object do not need to be
	 * queried.
	 * @param subject A (fixed) subject, or <tt>null</tt> if the subject
	 * should be queried from the database.
	 * @param predicate A (fixed) predicate, or <tt>null</tt> if the predicate
	 * should be queried from the database.
	 * @param object A (fixed) object, or <tt>null</tt> if the object
	 * should be queried from the database.
	 */
	public RdbmsStatementIterator(
			RdfSource source, String[] namespaceNames, Connection databaseCon,
			String queryResources, String queryLiterals,
			Resource subject, URI predicate, Value object)
	{
		_source = source;
		_namespaceNames = namespaceNames;
		_databaseCon = databaseCon;

		_queryResources = queryResources;
		_queryLiterals = queryLiterals;

		_subject = subject;
		_predicate = predicate;
		_object = object;

		_literalsHasBeenQueried = false;

		_subjectFixed = (subject != null);
		_predicateFixed = (predicate != null);
		_objectFixed = (object != null);

		_execQuery(_queryResources);
		_proceed();
	}

/*-------------------------------------------------+
| Methods                                          |
+-------------------------------------------------*/

	private void _proceed() {
		if (!_hasNext) {
			// Are both queries executed?
			if (!_literalsHasBeenQueried) {
				_literalsHasBeenQueried = true;
				_execQuery(_queryLiterals);
				
				_proceed();
			}
			else {
    			close();
			}
		}
	}

	private void _execQuery(String query) {
		try {
			if (query != null) {
				// Query specified.

				if (_resultSet != null) {
					// Close previous result set
					_resultSet.close();
					_resultSet = null;
				}

				if (_statement == null) {
					// Create SQL statement object
					_statement = _databaseCon.createStatement();
				}

				// Execute query
				_resultSet = _statement.executeQuery(query);

				_hasNext = _resultSet.next();
			}
			else {
				_hasNext = false;
			}
		}
		catch (SQLException e) {
			throw new SailInternalException("Unable to execute query: " + query, e);
		}
	}
	
	public boolean hasNext() {
		return _hasNext;
	}

	public Statement next() {
		if (!_hasNext) {
			throw new NoSuchElementException("No more statements...");
		}

		try {
			int idx = 1;

			if (!_subjectFixed) {
				int id = _resultSet.getInt(idx++);
				int nsId = _resultSet.getInt(idx++);
				String localname = _resultSet.getString(idx++);

				if (localname == null) {
					// Some database convert empty strings to NULL
					localname = "";
				}

				if (nsId == 0) {
					_subject = new IdBNode(_source, localname, id);
				}
				else {
					_subject = new IdURI(_source, _namespaceNames[nsId], localname, id);
				}
			}

			if (!_predicateFixed) {
				int id = _resultSet.getInt(idx++);
				int nsId = _resultSet.getInt(idx++);
				String localname = _resultSet.getString(idx++);

				if (localname == null) {
					// Some database convert empty strings to NULL
					localname = "";
				}

				_predicate = new IdURI(_source, _namespaceNames[nsId], localname, id);
			}

			if (!_objectFixed) {
				if (!_literalsHasBeenQueried) {
					// Object is a resource
					int id = _resultSet.getInt(idx++);
					int nsId = _resultSet.getInt(idx++);
					String localname = _resultSet.getString(idx++);

					if (localname == null) {
						// Some database convert empty strings to NULL
						localname = "";
					}

					if (nsId == 0) {
						_object = new IdBNode(_source, localname, id);
					}
					else {
						_object = new IdURI(_source, _namespaceNames[nsId], localname, id);
					}
				}
				else {
					// Object is a literal
					int id = _resultSet.getInt(idx++);
					int dtId = _resultSet.getInt(idx++); // datatype id
					int dtNsId = _resultSet.getInt(idx++); // datatype namespace id
					String dtLname = _resultSet.getString(idx++); // datatype localname
					String lang = _resultSet.getString(idx++);
					String label = _resultSet.getString(idx++);

					if (label == null) {
						// Some database convert empty strings to NULL
						label = "";
					}

					if (dtId != 0) {
						// Datatyped literal
						if (dtLname == null) {
							// Some database convert empty strings to NULL
							dtLname = "";
						}

						URI datatype = new IdURI(_source, _namespaceNames[dtNsId], dtLname, dtId);
						_object = new IdLiteral(_source, label, datatype, id);
					}
					else if (lang != null) {
						// Literal with language attribute
						_object = new IdLiteral(_source, label, lang, id);
					}
					else {
						// Plain literal
						_object = new IdLiteral(_source, label, id);
					}
				}
			}

			_hasNext = _resultSet.next();
			_proceed();

			return new StatementImpl(_subject, _predicate, _object);
		}
		catch (SQLException e) {
			throw new SailInternalException(e);
		}
	}
	
	public void close() {
		if (_databaseCon != null) {
			try {
				if (_resultSet != null) {
					_resultSet.close();
					_resultSet = null;
				}
				if (_statement != null) {
					_statement.close();
					_statement = null;
				}
				_databaseCon.close();

				// Prevent connection being closed second time through the
				// finalize method by garbage collector. The connection can be
				// reused by another object. 
				_databaseCon = null;

				_hasNext = false;	// No more results are returned.
			}
			catch (SQLException e) {
				throw new SailInternalException(e);
			}
		}
	}

	protected void finalize() {
        close();
    }
}
