/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2007 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sailimpl.rdbms;

import java.io.UnsupportedEncodingException;
import java.math.BigInteger;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.openrdf.util.log.ThreadLog;

import org.openrdf.model.BNode;
import org.openrdf.model.Literal;
import org.openrdf.model.Resource;
import org.openrdf.model.Statement;
import org.openrdf.model.URI;
import org.openrdf.model.Value;
import org.openrdf.model.ValueFactory;
import org.openrdf.model.impl.StatementImpl;

import org.openrdf.sesame.sail.NamespaceIterator;
import org.openrdf.sesame.sail.SailInitializationException;
import org.openrdf.sesame.sail.SailInternalException;
import org.openrdf.sesame.sail.StatementIterator;
import org.openrdf.sesame.sail.query.And;
import org.openrdf.sesame.sail.query.BooleanExpr;
import org.openrdf.sesame.sail.query.ConstructQuery;
import org.openrdf.sesame.sail.query.GraphPattern;
import org.openrdf.sesame.sail.query.GraphPatternQuery;
import org.openrdf.sesame.sail.query.Like;
import org.openrdf.sesame.sail.query.LiteralExpr;
import org.openrdf.sesame.sail.query.Not;
import org.openrdf.sesame.sail.query.Null;
import org.openrdf.sesame.sail.query.Or;
import org.openrdf.sesame.sail.query.PathExpression;
import org.openrdf.sesame.sail.query.ProjectionElem;
import org.openrdf.sesame.sail.query.Query;
import org.openrdf.sesame.sail.query.QueryOptimizer;
import org.openrdf.sesame.sail.query.ResourceExpr;
import org.openrdf.sesame.sail.query.SelectQuery;
import org.openrdf.sesame.sail.query.SetOperator;
import org.openrdf.sesame.sail.query.StringExpr;
import org.openrdf.sesame.sail.query.TriplePattern;
import org.openrdf.sesame.sail.query.ValueCompare;
import org.openrdf.sesame.sail.query.ValueExpr;
import org.openrdf.sesame.sail.query.Var;
import org.openrdf.sesame.sail.util.EmptyStatementIterator;
import org.openrdf.sesame.sailimpl.rdbms.iterators.RdbmsNamespaceIterator;
import org.openrdf.sesame.sailimpl.rdbms.iterators.RdbmsStatementIterator;
import org.openrdf.sesame.sailimpl.rdbms.model.IdBNode;
import org.openrdf.sesame.sailimpl.rdbms.model.IdLiteral;
import org.openrdf.sesame.sailimpl.rdbms.model.IdResource;
import org.openrdf.sesame.sailimpl.rdbms.model.IdURI;
import org.openrdf.sesame.sailimpl.rdbms.model.IdValue;

/**
 * A portable implementation of the RdfSource interface for relational
 * databases, based on (a subset of) the SQL2/SQL92 standard. This class defines
 * the read-only methods, the subclass RdfRepository defines the update methods.
 * 
 * @author Arjohn Kampman
 * @author Adam Skutt
 * @version $Revision: 1.46.2.15 $
 */
public class RdfSource implements org.openrdf.sesame.sail.RdfSource, TableNames, ValueFactory {

	/*----------+
	 | Constants |
	 +----------*/

	/** Key used to store the schema version in the metadata table. * */
	public static final String KEY_SCHEMAVERSION = "schemaversion";

	/** Key used to store the schema creator in the metadata table. * */
	public static final String KEY_SCHEMACREATOR = "schemacreator";

	/** Key used to store the status of the export flags in the metadata table. * */
	public static final String KEY_EXPORT_FLAGS = "exportflags";

	/** Value used to indiate an "up-to-date" status in the metadata table. * */
	public static final String VALUE_UP_TO_DATE = "up-to-date";

	/** Value used to indiate an "dirty" status in the metadata table. * */
	public static final String VALUE_DIRTY = "dirty";

	/** Key used to specify a JDBC driver in the initialization parameters. * */
	public static final String JDBC_DRIVER_KEY = "jdbcDriver";

	/** Key used to specify a JDBC URL in the initialization parameters. * */
	public static final String JDBC_URL_KEY = "jdbcUrl";

	/** Key used to specify a username in the initialization parameters. * */
	public static final String USER_KEY = "user";

	/** Key used to specify a password in the initialization parameters. * */
	public static final String PASSWORD_KEY = "password";

	/**
	 * Version of the database schema implemented by this RdfSource. 0 = reserved
	 * number, indicating schemas before this versioning system was implemented.
	 * The value of this 'constant' can be changed by subclasses of this class.
	 */
	// 1: first schema version
	// 2: 'labelKey' column in literals table replaced with 'labelHash' column
	// based on MD5 hash.
	// 3: new table 'expiredvalues' used to removed potentially expired values
	// 4: Renamed column of triples tables: subject --> subj, predicate --> pred,
	// object --> obj
	// 5: Replaced 'expiredValues' table with 'expiredResources' and
	// 'expiredLiterals' table
	// 6: Optimized indexes on triples and allnewtriples tables
	protected int CURRENT_SCHEMA_VERSION;

	/**
	 * Creator of the database schema. The value of this 'constant' can be
	 * changed by subclasses of this class.
	 */
	protected String SCHEMA_CREATOR;

	/**
	 * Equal to the string " NOT NULL" if the database leaves empty string as
	 * they are, or equal to an empty string if the database converts empty
	 * strings to NULL. This constant can be used for creating tables containing
	 * text values that can be empty strings.
	 */
	protected String NN_ON_TEXT;

	/*----------+
	 | Variables |
	 +----------*/

	/**
	 * The version of the schema currently in the database. The version will be
	 * equal after the schema has possibly been updated with a call to
	 * _initDatabase(). The value '-1' indicates that the database doesn't have a
	 * schema yet.
	 */
	protected int _schemaVersion;

	/** An object offering a collection of dbms-specific methods. * */
	protected RDBMS _rdbms;

	/** MD5 digest object for calculating literal label keys. * */
	protected MessageDigest _md5Digest;

	/** The prefix for any new bnode IDs. * */
	private String _bnodePrefix;

	/** The ID for the next bnode that is created. * */
	private int _nextBNodeID;

	/** The ID for the next namespace that is created. * */
	protected int _nextNamespaceId;

	/* local cache of used namespaces. */
	protected Map _namespaceTable;

	protected List _namespaceList;

	/**
	 * Array storing namespace names using their id as the index for position in
	 * the array. This array is for rapid retrieval of a namespace name using a
	 * namespace id.
	 */
	protected String[] _namespaceNames;

	/**
	 * Flag indicating whether the 'export' status of namespaces needs to be
	 * updated before reporting them. An update of the export status normally
	 * takes place the first time getNamespaces() is called after a transaction
	 * has been committed.
	 */
	private boolean _exportStatusUpToDate;

	/*-------------+
	 | Constructors |
	 +-------------*/

	/**
	 * Creates a new RdfSource object that will be able to read triples from an
	 * RDF database.
	 */
	public RdfSource() {
		CURRENT_SCHEMA_VERSION = 6;
		SCHEMA_CREATOR = "org.openrdf.sesame.sailimpl.rdbms.RdfSource";

		_updateBNodePrefix();

		_nextNamespaceId = 1;
		_namespaceTable = new HashMap();
		_namespaceList = new ArrayList();

		try {
			_md5Digest = MessageDigest.getInstance("MD5");
		}
		catch (NoSuchAlgorithmException e) {
			// Should never happen as MD5 should be available on every Java
			// platform
			throw new RuntimeException("MD5 digest algorithm not available on this platform", e);
		}
	}

	/**
	 * Generates a new bnode prefix based on <tt>currentTimeMillis()</tt> and
	 * resets <tt>_nextBNodeID</tt> to <tt>1</tt>.
	 */
	protected void _updateBNodePrefix() {
		// BNode prefix is based on currentTimeMillis(). Combined with a
		// sequential number per session, this gives a unique identifier.
		_bnodePrefix = "node" + Long.toString(System.currentTimeMillis(), 32) + "x";
		_nextBNodeID = 1;
	}

	/*------------------+
	 | Methods from Sail |
	 +------------------*/

	/**
	 * Initializes the RdfSource.
	 * 
	 * @param jdbcDriver
	 *        The String representing the JDBC-driver class, e.g.
	 *        <tt>org.gjt.mm.mysql.Driver</tt>.
	 * @param jdbcUrl
	 *        The String representing the JDBC-url of the database to connect to,
	 *        e.g. <tt>jdbc:mysql://localhost/sesame</tt>.
	 * @exception SailInitializationException
	 *            If the initialization failed.
	 */
	public void initialize(String jdbcDriver, String jdbcUrl)
		throws SailInitializationException
	{
		Map configParams = new HashMap(2);
		configParams.put(JDBC_DRIVER_KEY, jdbcDriver);
		configParams.put(JDBC_URL_KEY, jdbcUrl);

		this.initialize(configParams);
	}

	/**
	 * Initializes the RdfSource.
	 * 
	 * @param jdbcDriver
	 *        The String representing the JDBC-driver class, e.g.
	 *        <tt>org.gjt.mm.mysql.Driver</tt>.
	 * @param jdbcUrl
	 *        The String representing the JDBC-url of the database to connect to,
	 *        e.g. <tt>jdbc:mysql://localhost/sesame</tt>.
	 * @param user
	 *        The username that can be used to connect to the RDBMS.
	 * @param password
	 *        The password of the user in the RDBMS.
	 * @exception SailInitializationException
	 *            If the initialization failed.
	 */
	public void initialize(String jdbcDriver, String jdbcUrl, String user, String password)
		throws SailInitializationException
	{
		Map configParams = new HashMap(4);
		configParams.put(JDBC_DRIVER_KEY, jdbcDriver);
		configParams.put(JDBC_URL_KEY, jdbcUrl);
		configParams.put(USER_KEY, user);
		configParams.put(PASSWORD_KEY, password);

		this.initialize(configParams);
	}

	/**
	 * Initializes the RdfSource. Parameters can be specified using the keys
	 * defined by constants in this class: <tt>JDBC_DRIVER_KEY</tt>,
	 * <tt>JDBC_URL_KEY</tt>,<tt>USER_KEY</tt> and <tt>PASSWORD_KEY</tt>.
	 * See the other <tt>initialize()</tt> methods of this class for a
	 * description of the parameters and their values.
	 * 
	 * @param configParams
	 *        configuration parameters
	 * @exception SailInitializationException
	 *            If the RdfSource could not be initialized using the supplied
	 *            parameters.
	 * @see #JDBC_DRIVER_KEY
	 * @see #JDBC_URL_KEY
	 * @see #USER_KEY
	 * @see #PASSWORD_KEY
	 * @see #initialize(java.lang.String,java.lang.String,java.lang.String,java.lang.String)
	 */
	public void initialize(Map configParams)
		throws SailInitializationException
	{
		// Get initialization parameters
		String jdbcDriver = (String)configParams.get(JDBC_DRIVER_KEY);
		String jdbcUrl = (String)configParams.get(JDBC_URL_KEY);
		String user = (String)configParams.get(USER_KEY);
		String password = (String)configParams.get(PASSWORD_KEY);

		if (jdbcDriver == null) {
			throw new SailInitializationException("parameter 'jdbcDriver' missing");
		}
		if (jdbcUrl == null) {
			throw new SailInitializationException("parameter 'jdbcUrl' missing");
		}

		// Load jdbc driver
		try {
			Class.forName(jdbcDriver);
		}
		catch (ClassNotFoundException e) {
			throw new SailInitializationException("Unable to load JDBC-driver '" + jdbcDriver + "'", e);
		}

		try {
			_rdbms = RDBMS.createInstance(jdbcUrl, user, password);
			NN_ON_TEXT = _rdbms.emptyStringIsNull() ? "" : " NOT NULL";
		}
		catch (SQLException e) {
			throw new SailInitializationException(e);
		}

		// Initialize database schema, if necessary
		_initDatabase();
	}

	/**
	 * Disconnects from the database.
	 */
	public void shutDown() {
		_rdbms.disconnect();
	}

	/*------------------------+
	 | Database initialization |
	 +------------------------*/

	/**
	 * Initializes the database. _initDatabase() creates tables, indexes and
	 * inserts default values into the database.
	 */
	protected void _initDatabase()
		throws SailInitializationException
	{
		try {
			_schemaVersion = _checkDbSchema();

			if (_schemaVersion > CURRENT_SCHEMA_VERSION) {
				throw new SailInitializationException(
						"The database schema was created by a newer version of this Sail!");
			}
			else if (_schemaVersion < CURRENT_SCHEMA_VERSION) {
				// There's no schema yet, or it is out-of-date.
				if (_schemaVersion == -1) {
					ThreadLog.trace("No DB schema yet, creating it...");
				}
				else if (_schemaVersion < CURRENT_SCHEMA_VERSION) {
					ThreadLog.trace("Old DB schema, updating it...");
				}

				_createDbSchema();

				_schemaVersion = CURRENT_SCHEMA_VERSION;
				_exportStatusUpToDate = true;
			}
			else {
				// Restore value of _exportStatusUpToDate flag
				_exportStatusUpToDate = !VALUE_DIRTY.equals(_getRepInfo(KEY_EXPORT_FLAGS));

				ThreadLog.trace("DB schema is up to date");
			}

			// Initialize local caches
			_initNamespaceCache();
		}
		catch (SQLException e) {
			throw new SailInitializationException(e);
		}
	}

	/**
	 * Checks and returns the DB schema version that is recorded in the
	 * REP_INFO_TABLE. If this table does not yet exist, -1 will be returned.
	 */
	protected int _checkDbSchema()
		throws SQLException, SailInitializationException
	{
		ThreadLog.trace("Checking DB schema...");

		int result;

		if (_rdbms.tableExists(REP_INFO_TABLE)) {
			String creator = _getRepInfo(KEY_SCHEMACREATOR);
			String version = _getRepInfo(KEY_SCHEMAVERSION);

			if (!SCHEMA_CREATOR.equals(creator)) {
				// FIXME temporary check to allow backward compatibility
				// with 0.9 class names
				if (!creator.startsWith("nl.aidministrator.rdf.sail.rdbms.")) {
					throw new SailInitializationException("Database schema was not created by this Sail");
				}
				else {
					_setRepInfo(KEY_SCHEMACREATOR, SCHEMA_CREATOR);
				}
			}

			try {
				result = Integer.parseInt(version);
			}
			catch (NumberFormatException nfe) {
				throw new SailInitializationException("schema version is not an integer");
			}
		}
		else {
			// No schema found.
			result = -1;
		}

		return result;
	}

	/*---------------+
	 | Table creation |
	 +---------------*/

	/**
	 * Creates the database schema (the tables and indexes).
	 */
	protected void _createDbSchema()
		throws SQLException
	{
		// First create the repository info table.
		_createRepInfoTable();

		// Namespaces table
		_createNamespacesTable();

		// Resources table
		_createResourcesTable();

		// Literals table
		_createLiteralsTable();

		// Triples table
		_createTriplesTable();
	}

	/**
	 * Creates the REP_INFO_TABLE for storing repository meta info.
	 */
	protected void _createRepInfoTable()
		throws SQLException
	{
		if (_schemaVersion == -1) {
			// No schema yet
			_rdbms.executeUpdate("CREATE TABLE " + REP_INFO_TABLE + " (infokey " + _rdbms.INFOFIELD
					+ " NOT NULL PRIMARY KEY," + " infovalue " + _rdbms.INFOFIELD + " NOT NULL)");

			_setRepInfo(KEY_SCHEMAVERSION, String.valueOf(CURRENT_SCHEMA_VERSION));
			_setRepInfo(KEY_SCHEMACREATOR, SCHEMA_CREATOR);
		}
		else if (_schemaVersion < CURRENT_SCHEMA_VERSION) {
			// Update the version number
			_setRepInfo(KEY_SCHEMAVERSION, String.valueOf(CURRENT_SCHEMA_VERSION));
		}
	}

	/**
	 * Creates the NAMESPACES_TABLE.
	 */
	protected void _createNamespacesTable()
		throws SQLException
	{
		if (_schemaVersion == -1) {
			// No schema yet

			_rdbms.executeUpdate("CREATE TABLE " + NAMESPACES_TABLE + " (id " + _rdbms.ID_INT
					+ " NOT NULL PRIMARY KEY," + " prefix " + _rdbms.PREFIX + NN_ON_TEXT + "," + " name "
					+ _rdbms.NAME + "," + " userDefined " + _rdbms.BOOLEAN + " NOT NULL," + " export "
					+ _rdbms.BOOLEAN + " NOT NULL," + " UNIQUE(prefix))");

			// zero value for denoting anonymous resources. Notice
			// that this is not an actual namespace!
			_rdbms.executeUpdate("INSERT INTO " + NAMESPACES_TABLE + " VALUES(0, '', NULL, " + _rdbms.FALSE
					+ ", " + _rdbms.FALSE + ")");
		}
	}

	/**
	 * Creates the RESOURCES_TABLE.
	 */
	protected void _createResourcesTable()
		throws SQLException
	{
		if (_schemaVersion == -1) {
			// No schema yet
			_rdbms.executeUpdate("CREATE TABLE " + RESOURCES_TABLE + " (id " + _rdbms.ID_INT
					+ " NOT NULL PRIMARY KEY," + " namespace " + _rdbms.ID_INT + " NOT NULL, " + " localname "
					+ _rdbms.LOCALNAME + NN_ON_TEXT + "," + " UNIQUE(namespace, localname))");

			// zero value for 'null' resources. This resource is used to
			// indicate that a literal does not have a datatype.
			_rdbms.executeUpdate("INSERT INTO " + RESOURCES_TABLE + " VALUES(0, 0, '')");
		}
	}

	/**
	 * Creates the LITERALS_TABLE.
	 */
	protected void _createLiteralsTable()
		throws SQLException
	{
		if (_schemaVersion < 2) {
			// Old or non-existent table
			String OLD_LITERALS_TABLE = "old" + LITERALS_TABLE;

			if (_schemaVersion != -1) {
				// Old schema, rename current table
				_rdbms.dropIndex(LITERALS_TABLE, "labelKey");
				_rdbms.renameTable(LITERALS_TABLE, OLD_LITERALS_TABLE);
			}

			// Create literals table
			_rdbms.executeUpdate("CREATE TABLE " + LITERALS_TABLE + " (id " + _rdbms.ID_INT
					+ " NOT NULL PRIMARY KEY," + " datatype " + _rdbms.ID_INT + " NOT NULL," + " labelHash "
					+ _rdbms.LABEL_HASH + " NOT NULL," + " language " + _rdbms.LANGUAGE + "," + " label "
					+ _rdbms.LABEL + NN_ON_TEXT + ")");

			_rdbms.createIndex(LITERALS_TABLE, "labelHash");

			if (_schemaVersion != -1) {
				// Copy all literals from the old to the new table,
				// replacing the 'labelKey' with a 'labelHash'
				Connection addLiteralCon = _rdbms.getConnection();
				addLiteralCon.setAutoCommit(false);
				PreparedStatement addLiteralSt = addLiteralCon.prepareStatement("INSERT INTO " + LITERALS_TABLE
						+ " VALUES(?, ?, ?, ?, ?)");

				Connection con = _rdbms.getConnection();
				java.sql.Statement st = con.createStatement();
				ResultSet rs = st.executeQuery("SELECT id, datatype, language, label FROM " + OLD_LITERALS_TABLE);

				int count = 0;

				while (rs.next()) {
					addLiteralSt.setInt(1, rs.getInt(1)); // id
					addLiteralSt.setInt(2, rs.getInt(2)); // datatype

					String lang = rs.getString(3); // language
					String label = rs.getString(4); // label

					if (label == null) {
						label = "";
					}

					addLiteralSt.setLong(3, _getLabelHash(label));

					if (lang != null) {
						addLiteralSt.setString(4, lang);
					}
					else {
						addLiteralSt.setNull(4, _rdbms.LANGUAGE_TYPE);
					}

					addLiteralSt.setString(5, label);

					addLiteralSt.executeUpdate();
					count++;
				}
				rs.close();
				st.close();
				con.close();

				addLiteralCon.commit();

				addLiteralSt.close();
				addLiteralCon.close();

				_rdbms.optimizeTable(LITERALS_TABLE, count);

				_rdbms.dropTable(OLD_LITERALS_TABLE);
			}
		}
	}

	/**
	 * Creates the TRIPLES_TABLE.
	 */
	protected void _createTriplesTable()
		throws SQLException
	{
		if (_schemaVersion == -1) {
			// No schema yet
			_createTriplesTable(TRIPLES_TABLE, true);

			_rdbms.createIndex(TRIPLES_TABLE, new String[] { "pred", "obj" }, false);
			_rdbms.createIndex(TRIPLES_TABLE, new String[] { "obj", "subj" }, false);
			_rdbms.createUniqueIndex(TRIPLES_TABLE, "id");
		}
		else {
			if (_schemaVersion < 4) {
				_rdbms.renameTableColumn(TRIPLES_TABLE, "subject", "subj", _rdbms.ID_INT + " NOT NULL");
				_rdbms.renameTableColumn(TRIPLES_TABLE, "predicate", "pred", _rdbms.ID_INT + " NOT NULL");
				_rdbms.renameTableColumn(TRIPLES_TABLE, "object", "obj", _rdbms.ID_INT + " NOT NULL");
			}

			if (_schemaVersion < 6) {
				_rdbms.dropIndex(TRIPLES_TABLE, "subj");
				_rdbms.dropIndex(TRIPLES_TABLE, "pred");
				_rdbms.dropIndex(TRIPLES_TABLE, "obj");
				_rdbms.dropIndex(TRIPLES_TABLE, new String[] { "subj", "pred" });
				_rdbms.dropIndex(TRIPLES_TABLE, new String[] { "subj", "obj" });

				_rdbms.createIndex(TRIPLES_TABLE, new String[] { "obj", "subj" }, false);
				_rdbms.createUniqueIndex(TRIPLES_TABLE, "id");
			}
		}
	}

	/**
	 * Creates a table for storing triples. The table will get the specified name
	 * and will have four integer columns (id, subject, predicate and object),
	 * and one boolean column (explicit). If 'uniqueRows' is true, a unique index
	 * will be created on the columns (subject, predicate, object).
	 */
	protected void _createTriplesTable(String tableName, boolean uniqueRows)
		throws SQLException
	{
		StringBuffer update = new StringBuffer(200);

		update.append("CREATE TABLE " + tableName + " ");
		update.append("(id " + _rdbms.ID_INT + " NOT NULL,");
		update.append(" subj " + _rdbms.ID_INT + " NOT NULL,");
		update.append(" pred " + _rdbms.ID_INT + " NOT NULL,");
		update.append(" obj " + _rdbms.ID_INT + " NOT NULL,");
		update.append(" explicit " + _rdbms.BOOLEAN + " NOT NULL");
		if (uniqueRows) {
			update.append(", UNIQUE(subj, pred, obj)");
		}
		update.append(")");

		_rdbms.executeUpdate(update.toString());
	}

	/*--------------------------+
	 | Getting/setting meta info |
	 +--------------------------*/

	/**
	 * Sets or changes a value for a specific key in the REP_INFO_TABLE.
	 */
	protected void _setRepInfo(String key, String value)
		throws SQLException
	{
		// Remove any previous value for this key:
		_rdbms.executeUpdate("DELETE FROM " + REP_INFO_TABLE + " WHERE infokey = '" + _rdbms.escapeString(key)
				+ "'");

		// Insert the new value
		_rdbms.executeUpdate("INSERT INTO " + REP_INFO_TABLE + " VALUES ('" + _rdbms.escapeString(key) + "', '"
				+ _rdbms.escapeString(value) + "')");
	}

	/**
	 * Gets the value for a specific key from the REP_INFO_TABLE.
	 */
	protected String _getRepInfo(String key)
		throws SQLException
	{
		String result = null;

		Connection con = _rdbms.getConnection();
		java.sql.Statement st = con.createStatement();
		ResultSet rs = st.executeQuery("SELECT infovalue FROM " + REP_INFO_TABLE + " WHERE infokey = '"
				+ _rdbms.escapeString(key) + "'");

		if (rs.next()) {
			result = rs.getString(1);
		}

		rs.close();
		st.close();
		con.close();

		return result;
	}

	/**
	 * Sets the status of the namespaces table's <tt>export</tt> column.
	 * 
	 * @param upToDate
	 *        Flag indicating whether the export status is up-to-date.
	 */
	protected void _setExportStatusUpToDate(boolean upToDate)
		throws SQLException
	{
		if (_exportStatusUpToDate != upToDate) {
			_exportStatusUpToDate = upToDate;

			if (upToDate) {
				_setRepInfo(KEY_EXPORT_FLAGS, VALUE_UP_TO_DATE);
			}
			else {
				_setRepInfo(KEY_EXPORT_FLAGS, VALUE_DIRTY);
			}
		}
	}

	/**
	 * Gets the status of the namespaces table's <tt>export</tt> column.
	 * 
	 * @return <tt>true</tt> if the values in the <tt>export</tt> column are
	 *         up-to-date, <tt>false</tt> otherwise.
	 */
	protected boolean _exportStatusUpToDate() {
		return _exportStatusUpToDate;
	}

	/*----------------------------------+
	 | Initialization of local variables |
	 +----------------------------------*/

	/**
	 * Initializes the internal namespace cache and fills it with the contents of
	 * the NAMESPACES_TABLE.
	 */
	protected void _initNamespaceCache()
		throws SQLException
	{
		_namespaceTable.clear();
		_namespaceList.clear();

		Connection con = _rdbms.getConnection();
		java.sql.Statement st = con.createStatement();

		ResultSet rs = st.executeQuery("SELECT id, prefix, name, export " + "FROM " + NAMESPACES_TABLE
				+ " WHERE id <> 0");

		int maxId = 0;

		while (rs.next()) {
			int id = rs.getInt(1);
			String prefix = rs.getString(2);
			String name = rs.getString(3);
			boolean export = rs.getBoolean(4);

			if (prefix == null) {
				prefix = "";
			}

			RdbmsNamespace ns = new RdbmsNamespace(id, prefix, name, export);
			_namespaceTable.put(name, ns);
			_namespaceList.add(ns);

			if (id > maxId) {
				maxId = id;
			}
		}

		rs.close();
		st.close();
		con.close();

		// Next namespace id to be assigned is max + 1
		_nextNamespaceId = maxId + 1;

		// Initialize _namespaceNames array
		_namespaceNames = new String[maxId + 1];
		for (int i = 0; i < _namespaceList.size(); i++) {
			RdbmsNamespace ns = (RdbmsNamespace)_namespaceList.get(i);
			_namespaceNames[ns.getId()] = ns.getName();
		}
	}

	protected int _getNextNamespaceId() {
		return _nextNamespaceId++;
	}

	/*-----------------------+
	 | Methods from RdfSource |
	 +-----------------------*/

	public ValueFactory getValueFactory() {
		return this;
	}

	public StatementIterator getStatements(Resource subj, URI pred, Value obj) {
		return getStatements(subj, pred, obj, false);
	}

	public StatementIterator getStatements(Resource subj, URI pred, Value obj, boolean explicitOnly) {
		int subjId = 0;
		int predId = 0;
		int objId = 0;

		if (subj != null) {
			// Subject specified.
			subjId = _getResourceId(subj);

			if (subjId == 0) {
				// Subject not found, so no matching statements.
				return new EmptyStatementIterator();
			}
		}

		if (pred != null) {
			// Predicate specified.
			predId = _getURIId(pred);

			if (predId == 0) {
				// Predicate not found, so no matching statements.
				return new EmptyStatementIterator();
			}
		}

		if (obj != null) {
			// Object specified.
			objId = _getValueId(obj);

			if (objId == 0) {
				// Object not found, so no matching statements.
				return new EmptyStatementIterator();
			}
		}

		// The StatementIterator requires 2 queries, one where triples.obj
		// is joined with resources.id and one where triples.obj is joined
		// with literals.id.
		String queryResources = null;
		String queryLiterals = null;

		if (obj instanceof Resource) {
			// Object is a resource, thus there is no need to build the
			// query where triples.obj is joined with literals.id,
			// because it will always fail.
			queryResources = _buildGetStatementsQuery(subjId, predId, objId, true, explicitOnly);
		}
		else if (obj instanceof Literal) {
			// Object is a literal, thus there is no need to build the
			// query where triples.obj is joined with resources.id,
			// because it will always fail.
			queryLiterals = _buildGetStatementsQuery(subjId, predId, objId, false, explicitOnly);
		}
		else {
			// Both queries are needed.
			queryResources = _buildGetStatementsQuery(subjId, predId, objId, true, explicitOnly);

			queryLiterals = _buildGetStatementsQuery(subjId, predId, objId, false, explicitOnly);
		}

		try {
			Connection con = _rdbms.getConnection();

			return new RdbmsStatementIterator(this, _namespaceNames, con, queryResources, queryLiterals, subj,
					pred, obj);
		}
		catch (SQLException e) {
			throw new SailInternalException(e);
		}
	}

	protected String _buildGetStatementsQuery(int subjId, int predId, int objId, boolean queryResources,
			boolean explicitOnly)
	{
		// Build select, from and where clause seperately
		StringBuffer select = new StringBuffer(100);
		StringBuffer from = new StringBuffer(200);
		StringBuffer where = new StringBuffer(200);

		// Commas need to be appended after the first parts have been written.
		boolean writeCommas = false;

		// subject
		if (subjId != 0) {
			// subject is specified
			where.append("t.subj = " + subjId + " AND ");
		}
		else {
			// subject is a wildcard
			select.append("r1.id, r1.namespace, r1.localname");
			from.append(RESOURCES_TABLE + " r1");
			where.append("t.subj = r1.id AND ");

			writeCommas = true;
		}

		// predicate
		if (predId != 0) {
			// predicate is specified
			where.append("t.pred = " + predId + " AND ");
		}
		else {
			// predicate is a wildcard
			if (writeCommas) {
				select.append(", ");
				from.append(", ");
			}

			select.append("r2.id, r2.namespace, r2.localname");
			from.append(RESOURCES_TABLE + " r2");
			where.append("t.pred = r2.id AND ");

			writeCommas = true;
		}

		// object
		if (objId != 0) {
			// object is specified
			where.append("t.obj = " + objId);
		}
		else {
			// object is a wildcard
			if (writeCommas) {
				select.append(", ");
				from.append(", ");
			}

			if (queryResources) {
				// object is a resource
				select.append("r3.id, r3.namespace, r3.localname");
				from.append(RESOURCES_TABLE + " r3");
				where.append("t.obj > 0 AND t.obj = r3.id");
			}
			else {
				// object is a literal
				select.append("l.id, dt.id, dt.namespace, dt.localname, l.language, l.label");
				from.append(LITERALS_TABLE + " l, " + RESOURCES_TABLE + " dt");
				where.append("t.obj < 0 AND t.obj = l.id AND l.datatype = dt.id");
			}
		}

		// Build the complete query
		StringBuffer query = new StringBuffer(400);

		query.append("SELECT ");
		if (select.length() == 0) {
			// Select clause is still empty because all parameters are specified
			query.append("t.id");
		}
		else {
			query.append(select.toString());
		}

		query.append(" FROM " + TRIPLES_TABLE + " t");
		if (from.length() > 0) {
			// from clause is not empty
			query.append(", " + from.toString());
		}

		query.append(" WHERE " + where.toString());

		if (explicitOnly) {
			query.append(" AND t.explicit = " + _rdbms.TRUE);
		}

		return query.toString();
	}

	public boolean hasStatement(Resource subj, URI pred, Value obj) {
		return hasStatement(subj, pred, obj, false);
	}

	/**
	 * Merges all TriplePatterns to one large SQL-join and includes as much of
	 * the boolean constraints as possible.
	 */
	public Query optimizeQuery(Query query) {
		if (query instanceof GraphPatternQuery) {
			_optimizeGraphPatternQuery((GraphPatternQuery)query);
		}
		else if (query instanceof SetOperator) {
			SetOperator setOp = (SetOperator)query;
			optimizeQuery(setOp.getLeftArg());
			optimizeQuery(setOp.getRightArg());
		}

		return query;
	}

	private void _optimizeGraphPatternQuery(GraphPatternQuery graphPatternQuery) {
		GraphPattern graphPattern = graphPatternQuery.getGraphPattern();
		List pathExpressions = graphPattern.getPathExpressions();

		if (pathExpressions.size() == 0
				|| (pathExpressions.size() == 1 && graphPattern.getConjunctiveConstraints().isEmpty()))
		{
			// Not much to optimize over the generic optimizations
			QueryOptimizer.optimizeQuery(graphPatternQuery);
			return;
		}

		// Variables used to build the from- and where parts of the SQL query:
		StringBuffer from = new StringBuffer(512);
		StringBuffer where = new StringBuffer(256);

		// Maps Var objects to their ID in SQL, e.g. "t2.subj"
		Map varToSqlIdMapping = new HashMap();

		// List for storing unrecognized path expressions
		List unknownPathExpressions = new ArrayList();

		for (int i = 0; i < pathExpressions.size(); i++) {
			PathExpression pe = (PathExpression)pathExpressions.get(i);

			// Assign an ID to this path expression
			String peId = "t" + i;

			pe = _convertPathExpressionToSQL(pe, peId, from, where, varToSqlIdMapping);

			if (pe != null) {
				// unknown path expression
				unknownPathExpressions.add(pe);
			}
		}

		if (pathExpressions.size() - unknownPathExpressions.size() <= 1
				&& graphPattern.getConjunctiveConstraints().isEmpty())
		{
			// Not much to optimize over the generic optimizations
			QueryOptimizer.optimizeQuery(graphPatternQuery);
			return;
		}

		// inline constraints
		_inlineConstraints(graphPattern, from, where, varToSqlIdMapping);

		// Gather variables that need to be exported. This includes:
		// - variables used in the projection,
		// - variables used in the remaining boolean constraints,
		// - variables used in optional child graph expressions, and
		// - variables used in unkown path expressions
		//
		// Variables that are only used in the set of optimized path expressions
		// are excluded, hence graphPattern.getVariables(Collection) is not used

		Set varSet = new HashSet();

		// - variables used in the projection,
		graphPatternQuery.getProjectionVariables(varSet);

		// - variables used in the remaining boolean constraints,
		BooleanExpr rootConstraint = graphPattern.getRootConstraint();
		if (rootConstraint != null) {
			rootConstraint.getVariables(varSet);
		}

		// - variables used in optional child graph expressions, and
		List optionals = graphPattern.getOptionals();
		for (int i = 0; i < optionals.size(); i++) {
			GraphPattern optionalGP = (GraphPattern)optionals.get(i);
			optionalGP.getVariables(varSet);
		}

		// - variables used in unkown path expressions
		for (int i = 0; i < unknownPathExpressions.size(); i++) {
			PathExpression pe = (PathExpression)unknownPathExpressions.get(i);
			pe.getVariables(varSet);
		}

		// Remove variables not present in the optimized path expressions
		varSet.retainAll(varToSqlIdMapping.keySet());

		// Create an ordered list from the variable set
		List varList = new ArrayList(varSet);

		// Create the SQL query:
		StringBuffer sqlQuery = new StringBuffer(1024);

		// SELECT clause
		sqlQuery.append("SELECT ");

		if (graphPatternQuery.isDistinct()) {
			sqlQuery.append("DISTINCT ");

			// We can remove the distinct flag from the GraphPatternQuery if the
			// entire projection is handled by the SQL query, the pattern contains
			// no optionals and if the
			// projection does not contain any value-transforming functions.
			Set projVarSet = new HashSet();
			graphPatternQuery.getProjectionVariables(projVarSet);

			if (varSet.equals(projVarSet)) {
				// Entire projection is handled by the SQL query

				if (optionals.size() == 0) {
					// The pattern contains no optionals

					boolean hasTransFunc = false;

					if (graphPatternQuery instanceof SelectQuery) {
						ProjectionElem[] projection = ((SelectQuery)graphPatternQuery).getProjection();

						for (int i = 0; i < projection.length; i++) {
							ValueExpr valueExpr = projection[i].getValueExpr();
							if (valueExpr instanceof Var || valueExpr instanceof ResourceExpr
									|| valueExpr instanceof LiteralExpr || valueExpr instanceof Null)
							{
								// these are OK
							}
							else {
								hasTransFunc = true;
								break;
							}
						}
					}
					else if (graphPatternQuery instanceof ConstructQuery) {
						// All known PathExpression objects only use variables, and no
						// value-transforming functions.
					}

					if (!hasTransFunc) {
						graphPatternQuery.setDistinct(false);
					}
				}
			}
		} // end if distinct

		for (int i = 0; i < varList.size(); i++) {
			Var var = (Var)varList.get(i);

			if (i > 0) {
				sqlQuery.append(", ");
			}

			sqlQuery.append((String)varToSqlIdMapping.get(var));
		}

		// FROM clause; string start with ", ", remove these first 2 characters
		from.delete(0, 2);

		sqlQuery.append(" FROM ");
		sqlQuery.append(from);

		// WHERE clause
		if (where.length() > 0) {
			// where clause starts with " AND ", remove these first 5 characters
			where.delete(0, 5);

			sqlQuery.append(" WHERE ");
			sqlQuery.append(where);
		}

		ThreadLog.trace(sqlQuery.toString());

		// Create RdbmsPathExpression
		RdbmsPathExpression rpe = new RdbmsPathExpression(this, _rdbms, varList, sqlQuery.toString());

		// Create new list of PathExpressions
		List newPathExpressions = new ArrayList(1 + unknownPathExpressions.size());
		newPathExpressions.add(rpe);
		newPathExpressions.addAll(unknownPathExpressions);

		graphPattern.setPathExpressions(newPathExpressions);
	}

	/**
	 * Converts a PathExpression to SQL. Table names should be append to
	 * <tt>from</tt>, prefixing the name with the string <tt>", "</tt> and
	 * should be given the name <tt>peId</tt>. Boolean constraints should be
	 * added to <tt>where</tt> and always be prepended with the string
	 * <tt>" AND "</tt> (including spaces).
	 * 
	 * @param pathExpr
	 *        The PathExpression to convert to SQL.
	 * @param peId
	 *        The ID to use for the path expression in the SQL query.
	 * @param from
	 *        The from clause.
	 * @param where
	 *        The where clause.
	 * @param varToSqlIdMapping
	 *        Mapping of Var objects to their SQL IDs.
	 * @return <tt>pathExpr</tt> if the path expression could not be converted,
	 *         or <tt>null</tt> if it could.
	 */
	protected PathExpression _convertPathExpressionToSQL(PathExpression pathExpr, String peId,
			StringBuffer from, StringBuffer where, Map varToSqlIdMapping)
	{
		if (pathExpr instanceof TriplePattern) {
			TriplePattern tp = (TriplePattern)pathExpr;

			from.append(", " + TRIPLES_TABLE + " " + peId);

			_processPathExpressionVar(tp.getSubjectVar(), peId + ".subj", where, varToSqlIdMapping);
			_processPathExpressionVar(tp.getPredicateVar(), peId + ".pred", where, varToSqlIdMapping);
			_processPathExpressionVar(tp.getObjectVar(), peId + ".obj", where, varToSqlIdMapping);

			return null;
		}
		else {
			return pathExpr;
		}
	}

	/**
	 * Processes a path expression variable. A variable with a fixed value will
	 * be assigned this value by adding an equality constraint to the where
	 * clause. A variable that has already been mapped to some other SQL id will
	 * be made equal to this other SQL id.
	 */
	protected void _processPathExpressionVar(Var var, String sqlId, StringBuffer where, Map varToSqlIdMapping)
	{
		if (var.hasValue()) {
			int valueId = _getValueId(var.getValue());

			// FIXME: what if _getValueId() returned 0?
			where.append(" AND " + sqlId + " = " + valueId);
		}
		else {
			// Check if variable has been encountered earlier
			String otherSqlId = (String)varToSqlIdMapping.get(var);

			if (otherSqlId != null) {
				where.append(" AND " + sqlId + " = " + otherSqlId);
			}
			else {
				varToSqlIdMapping.put(var, sqlId);
			}
		}
	}

	/**
	 * Inlines as much of the boolean constraints from <tt>graphPattern</tt> as
	 * possible.
	 */
	protected void _inlineConstraints(GraphPattern graphPattern, StringBuffer from, StringBuffer where,
			Map varToSqlIdMapping)
	{
		boolean constraintsModified = false;

		List conjunctiveConstraints = new LinkedList(graphPattern.getConjunctiveConstraints());

		Iterator iter = conjunctiveConstraints.iterator();

		while (iter.hasNext()) {
			BooleanExpr conjunctiveExpr = (BooleanExpr)iter.next();

			String sqlConstraint = _constraintToSql(conjunctiveExpr, from, varToSqlIdMapping);

			if (sqlConstraint != null) {
				where.append(" AND " + sqlConstraint);
				iter.remove();
				constraintsModified = true;
			}
		}

		if (constraintsModified) {
			graphPattern.setConstraints(conjunctiveConstraints);
		}
	}

	// returns null when the constraint could not be translated to SQL
	protected String _constraintToSql(BooleanExpr booleanExpr, StringBuffer from, Map varToSqlIdMapping) {
		if (booleanExpr instanceof Not) {
			Not not = (Not)booleanExpr;

			String argConstraint = _constraintToSql(not.getArg(), from, varToSqlIdMapping);

			if (argConstraint != null) {
				return "NOT(" + argConstraint + ")";
			}
		}
		else if (booleanExpr instanceof Or) {
			Or or = (Or)booleanExpr;

			String leftConstraint = _constraintToSql(or.getLeftArg(), from, varToSqlIdMapping);
			String rightConstraint = _constraintToSql(or.getRightArg(), from, varToSqlIdMapping);

			if (leftConstraint != null && rightConstraint != null) {
				return "(" + leftConstraint + " OR " + rightConstraint + ")";
			}
		}
		else if (booleanExpr instanceof And) {
			And and = (And)booleanExpr;

			String leftConstraint = _constraintToSql(and.getLeftArg(), from, varToSqlIdMapping);
			String rightConstraint = _constraintToSql(and.getRightArg(), from, varToSqlIdMapping);

			if (leftConstraint != null && rightConstraint != null) {
				return "(" + leftConstraint + " AND " + rightConstraint + ")";
			}
		}
		else if (booleanExpr instanceof Like) {
			Like likeExpr = (Like)booleanExpr;

			if (_rdbms.supportsPatternMatches(likeExpr.caseSensitive())) {
				// Try to translate the Like's expression to SQL
				String exprSqlId = null;
				StringExpr expr = likeExpr.getStringExpr();

				if (expr instanceof ValueExpr) {
					Value exprValue = ((ValueExpr)expr).getValue();

					if (exprValue != null) {
						// Expression is a constant value
						exprSqlId = String.valueOf(_getValueId(exprValue));
					}
					else if (expr instanceof Var) {
						exprSqlId = (String)varToSqlIdMapping.get(expr);
					}
					// FIXME: if the leftArg is a function like label() exprSqlID
					// will
					// still be null and the like operator will not be optimized
				}

				if (exprSqlId != null) {
					// Translate the pattern to SQL
					String pattern = _rdbms.getPatternMatchExpr(likeExpr.getPattern(), likeExpr.caseSensitive());

					// Add join with the literals table
					int j = 0;
					while (from.indexOf(LITERALS_TABLE + " l" + j) >= 0) {
						j++;
					}

					String literalsId = "l" + j;

					from.append(", " + LITERALS_TABLE + " " + literalsId);

					String sqlLike = _rdbms.getPatternMatchOperator(likeExpr.caseSensitive());

					return literalsId + ".id = " + exprSqlId + " AND " + literalsId + ".label " + sqlLike + " '"
							+ pattern + "'";
				}
			}
		}
		else if (booleanExpr instanceof ValueCompare) {
			ValueCompare valueCompare = (ValueCompare)booleanExpr;

			// Only inline = and != comparisons
			if (valueCompare.getOperator() == ValueCompare.EQ || valueCompare.getOperator() == ValueCompare.NE) {
				ValueExpr leftArg = ((ValueCompare)booleanExpr).getLeftArg();
				ValueExpr rightArg = ((ValueCompare)booleanExpr).getRightArg();

				// check if one of the arguments is a datatyped literal.
				// these should not be inlined since type casting may need
				// to take place.
				if (leftArg instanceof Literal) {
					URI leftDt = ((Literal)leftArg).getDatatype();

					if (leftDt != null) {
						return null;
					}
				}
				else if (rightArg instanceof Literal) {
					URI rightDt = ((Literal)rightArg).getDatatype();

					if (rightDt != null) {
						return null;
					}
				}

				String leftSqlId = null;
				if (leftArg.getValue() != null) {
					leftSqlId = String.valueOf(_getValueId(leftArg.getValue()));
				}
				else if (leftArg instanceof Var) {
					leftSqlId = (String)varToSqlIdMapping.get(leftArg);
				}

				String rightSqlId = null;
				if (rightArg.getValue() != null) {
					rightSqlId = String.valueOf(_getValueId(rightArg.getValue()));
				}
				else if (rightArg instanceof Var) {
					rightSqlId = (String)varToSqlIdMapping.get(rightArg);
				}

				if (leftSqlId != null && rightSqlId != null) {
					if (valueCompare.getOperator() == ValueCompare.EQ) {
						return leftSqlId + " = " + rightSqlId;
					}
					else {
						return leftSqlId + " <> " + rightSqlId;
					}
				}
				else {
					return null;
				}
			}
		}

		return null;
	}

	public boolean hasStatement(Resource subj, URI pred, Value obj, boolean explicitOnly) {
		StringBuffer query = new StringBuffer(200);
		query.append("SELECT id FROM " + TRIPLES_TABLE);

		// WHERE clause is not needed if subject, predicate and object are
		// wildcards and explicitOnly is false.
		if (subj != null || pred != null || obj != null || explicitOnly) {
			query.append(" WHERE ");

			boolean writeAND = false;

			if (subj != null) {
				// Subject is specified.
				int subjId = _getResourceId(subj);

				if (subjId == 0) {
					// Resource not found, so statement not present.
					return false;
				}

				query.append("subj = " + subjId);
				writeAND = true;
			}

			if (pred != null) {
				// Predicate is specified.
				int predId = _getURIId(pred);

				if (predId == 0) {
					// URI not found, so statement not present.
					return false;
				}

				if (writeAND) {
					query.append(" AND ");
				}

				query.append("pred = " + predId);
				writeAND = true;
			}

			if (obj != null) {
				// Object is specified.
				int objId = _getValueId(obj);

				if (objId == 0) {
					// Resource not found, so statement not present.
					return false;
				}

				if (writeAND) {
					query.append(" AND ");
				}

				query.append("obj = " + objId);
				writeAND = true;
			}

			if (explicitOnly) {
				if (writeAND) {
					query.append(" AND ");
				}

				query.append("explicit = " + _rdbms.TRUE);
			}
		}

		try {
			return _rdbms.queryHasResults(query.toString());
		}
		catch (SQLException e) {
			throw new SailInternalException(e);
		}
	}

	public NamespaceIterator getNamespaces() {
		if (!_exportStatusUpToDate()) {
			_updateExportedNamespaces();
		}
		return new RdbmsNamespaceIterator(_namespaceList);
	}

	/*-----------------------------------------------------------------------------+
	 | Methods for getting and creating id's for resources, literals and namespaces |
	 +-----------------------------------------------------------------------------*/

	/**
	 * Gets the ID of the supplied value.
	 * 
	 * @return the ID of the supplied value, or 0 if the value was not found.
	 */
	protected int _getValueId(Value value) {
		if (value instanceof Resource) {
			return _getResourceId((Resource)value);
		}
		else if (value instanceof Literal) {
			return _getLiteralId((Literal)value);
		}
		else {
			throw new IllegalArgumentException("parameter 'value' should be of type Resource or Literal");
		}
	}

	/**
	 * Gets the ID of the supplied resource.
	 * 
	 * @return the ID of the supplied resource, or 0 if the resource was not
	 *         found.
	 */
	protected int _getResourceId(Resource res) {
		if (res instanceof URI) {
			return _getURIId((URI)res);
		}
		else if (res instanceof BNode) {
			return _getBNodeId((BNode)res);
		}
		else {
			throw new IllegalArgumentException("parameter 'res' should be of type URI or BNode");
		}
	}

	/**
	 * Gets the ID of the supplied URI.
	 * 
	 * @return the ID of the supplied URI, or 0 if the URI was not found.
	 */
	protected int _getURIId(URI uri) {
		IdURI idURI = null;

		if (_isOwnValue(uri)) {
			// ID is possibly stored in the URI object
			idURI = (IdURI)uri;

			int id = idURI.getInternalId();
			if (id != 0) {
				return id;
			}
		}

		// Retrieve the ID from the database

		int result = 0;

		int nsId = _getNamespaceId(uri.getNamespace());
		if (nsId == 0) {
			// nsId is 0 so uri does not exist.
			return 0;
		}

		String lname = uri.getLocalName();

		StringBuffer query = new StringBuffer(150);
		query.append("SELECT id FROM " + RESOURCES_TABLE);
		query.append(" WHERE namespace = " + nsId);
		query.append(" AND localname ");
		if (_rdbms.emptyStringIsNull() && lname.length() == 0) {
			query.append("IS NULL");
		}
		else {
			query.append("= '");
			query.append(_rdbms.escapeString(lname));
			query.append("'");
		}

		try {
			Connection connection = _rdbms.getConnection();
			java.sql.Statement statement = connection.createStatement();
			ResultSet resultSet = statement.executeQuery(query.toString());

			if (resultSet.next()) {
				// URI was found
				result = resultSet.getInt(1);
			}

			resultSet.close();
			statement.close();
			connection.close();
		}
		catch (SQLException e) {
			throw new SailInternalException(e);
		}

		if (idURI != null) {
			// Store the fetched ID in the URI object for use in any subsequent
			// requests
			idURI.setInternalId(result);
		}

		return result;

	}

	/**
	 * Gets the ID of the supplied BNode.
	 * 
	 * @return the ID of the supplied BNode, or 0 if the BNode was not found.
	 */
	protected int _getBNodeId(BNode bNode) {
		IdBNode idBNode = null;

		if (_isOwnValue(bNode)) {
			// ID is possibly stored in the BNode object
			idBNode = (IdBNode)bNode;

			int id = idBNode.getInternalId();
			if (id != 0) {
				return id;
			}
		}

		// Retrieve the ID from the database

		int result = 0;
		String nodeId = bNode.getID();

		StringBuffer query = new StringBuffer(150);
		query.append("SELECT id FROM " + RESOURCES_TABLE);
		query.append(" WHERE namespace = 0 AND localname = '");
		query.append(_rdbms.escapeString(nodeId));
		query.append("'");

		try {
			Connection connection = _rdbms.getConnection();
			java.sql.Statement statement = connection.createStatement();
			ResultSet resultSet = statement.executeQuery(query.toString());

			if (resultSet.next()) {
				// bnode was found
				result = resultSet.getInt(1);
			}

			resultSet.close();
			statement.close();
			connection.close();
		}
		catch (SQLException e) {
			throw new SailInternalException(e);
		}

		if (idBNode != null) {
			// Store the fetched ID in the BNode object for use in any subsequent
			// requests
			idBNode.setInternalId(result);
		}

		return result;
	}

	/**
	 * Gets the ID of the supplied literal.
	 * 
	 * @return the ID of the supplied literal, or 0 if the literal was not found.
	 */
	protected int _getLiteralId(Literal literal) {
		IdLiteral idLiteral = null;

		if (_isOwnValue(literal)) {
			// ID is possibly stored in the Literal object
			idLiteral = (IdLiteral)literal;

			int id = idLiteral.getInternalId();
			if (id != 0) {
				return id;
			}
		}

		// Build query to retrieve the ID from the database

		// Note: the retrieved labels are compared to the label of the specified
		// literal because CLOB/BLOB-comparisons are not supported by Oracle and
		// SQL Server (and possibly others). The label's hash code is used to
		// make a first selection of possible matching literals.

		int result = 0;
		int datatypeId = 0;

		String label = literal.getLabel();
		String language = literal.getLanguage();
		URI datatype = literal.getDatatype();

		if (datatype != null) {
			datatypeId = _getURIId(datatype);

			if (datatypeId == 0) {
				// datatype not found so literal does not exist
				return 0;
			}
		}

		StringBuffer query = new StringBuffer(200);
		query.append("SELECT id, label FROM ");
		query.append(LITERALS_TABLE);
		query.append(" WHERE ");

		// labelHash
		query.append("labelHash = ");
		query.append(_getLabelHash(label));

		// datatype
		query.append(" AND datatype = " + datatypeId);

		// language
		query.append(" AND language");
		if (language == null) {
			query.append(" IS NULL");
		}
		else {
			query.append(" = '");
			query.append(_rdbms.escapeString(language));
			query.append("'");
		}

		try {
			Connection connection = _rdbms.getConnection();
			java.sql.Statement statement = connection.createStatement();
			ResultSet resultSet = statement.executeQuery(query.toString());

			while (resultSet.next()) {
				// a possibly matching literal was found, check its label
				String resultLabel = resultSet.getString(2);

				if (resultLabel == null) {
					resultLabel = "";
				}

				if (resultLabel.equals(label)) {
					// Label matches
					result = resultSet.getInt(1);
					break;
				}
			}

			resultSet.close();
			statement.close();
			connection.close();
		}
		catch (SQLException e) {
			throw new SailInternalException(e);
		}

		if (idLiteral != null) {
			// Store the fetched ID in the Literal object for use in any subsequent
			// requests
			idLiteral.setInternalId(result);
		}

		return result;
	}

	protected boolean _isOwnValue(Value value) {
		return value instanceof IdValue && ((IdValue)value).getRdfSource() == this;
	}

	/**
	 * Gets the ID of the supplied namespace.
	 * 
	 * @return the ID of the supplied namespace, or 0 if the namespace was not
	 *         found.
	 */
	protected int _getNamespaceId(String namespace) {
		RdbmsNamespace n = (RdbmsNamespace)_namespaceTable.get(namespace);

		if (n != null) {
			return n.getId();
		}

		return 0;
	}

	/**
	 * Calculates a hash code for the supplied label.
	 */
	protected long _getLabelHash(String label) {
		// Calculate a 128 bit (16 byte) MD5 hash and use
		// the last 8 bytes of this hash for the result.
		try {
			byte[] labelData = label.getBytes("UTF-8");
			byte[] md5hash = null;

			synchronized (_md5Digest) {
				_md5Digest.reset();
				md5hash = _md5Digest.digest(labelData);
			}

			return (new BigInteger(md5hash)).longValue();
		}
		catch (UnsupportedEncodingException e) {
			// Should never happen as UTF-8 should be available on every Java
			// platform
			throw new SailInternalException("UTF-8 encoding not available on this platorm", e);
		}
	}

	/**
	 * Chops a list of IDs (integers) contained in the supplied ResultSet into
	 * one or more chunks. The chunks are returned as string of the form
	 * "(n1,n2,...)", including parentheses.
	 * 
	 * @param ResultSet
	 *        A one-column ResultSet containing integers.
	 * @param maxStringLength
	 *        The maximum length (in characters) of a chunk.
	 * @return an array of strings. Each string in the array is a chunk of the
	 *         list of integers, surrounded by brackets and seperated by commas.
	 */
	protected String[] _chunkIdSet(ResultSet rs, int maxStringLength)
		throws SQLException
	{
		ArrayList chunkList = new ArrayList();

		StringBuffer chunk = new StringBuffer(maxStringLength);
		chunk.append("(");

		int count = 0;

		while (rs.next()) {
			String nextId = String.valueOf(rs.getInt(1));

			if (chunk.length() + nextId.length() + 1 > maxStringLength) {
				// Start with a new chunk
				chunk.append(")");
				chunkList.add(chunk.toString());

				chunk.setLength(0);
				chunk.append("(");

				count = 0;
			}

			if (count > 0) {
				chunk.append(",");
			}
			chunk.append(rs.getInt(1));

			count++;
		}

		if (count > 0) {
			chunk.append(")");
			chunkList.add(chunk.toString());
		}

		String[] result = new String[chunkList.size()];
		for (int i = 0; i < chunkList.size(); i++) {
			result[i] = (String)chunkList.get(i);
		}

		return result;
	}

	protected void _updateExportedNamespaces() {
		ThreadLog.trace("Updating exported namespaces information.");

		// First 'reset' all namespaces that are not user-defined to
		// non-exported. This eliminates exporting of namespaces belonging
		// to removed predicates.
		try {
			_rdbms.executeUpdate("UPDATE " + NAMESPACES_TABLE + " SET export = " + _rdbms.FALSE
					+ " WHERE userDefined = " + _rdbms.FALSE);

			// Query for all namespaces that need to be exported: those that
			// are the in predicates of statements.
			Connection con = _rdbms.getConnection();
			java.sql.Statement st = con.createStatement();
			ResultSet rs = st.executeQuery("SELECT DISTINCT r.namespace " + "FROM " + RESOURCES_TABLE + " r, "
					+ TRIPLES_TABLE + " t " + "WHERE t.pred = r.id ");

			String[] idChunks = _chunkIdSet(rs, 3500);
			rs.close();
			st.close();

			con.setAutoCommit(false);
			st = con.createStatement();

			for (int i = 0; i < idChunks.length; i++) {
				st.executeUpdate("UPDATE " + NAMESPACES_TABLE + " SET export = " + _rdbms.TRUE + " WHERE id IN "
						+ idChunks[i]);
			}

			con.commit();
			st.close();
			con.close();

			_setExportStatusUpToDate(true);

			// re-initialize the namespaces to update the cache.
			_initNamespaceCache();
		}
		catch (SQLException e) {
			throw new SailInternalException(e);
		}
	}

	/*-----------------------------------------------------------------------+
	 | Methods for retrieving Resource and Literal objects with a specific ID |
	 +-----------------------------------------------------------------------*/

	/**
	 * Gets the Value for the supplied ID.
	 * 
	 * @return An IdValue object.
	 */
	public IdValue getValue(int id) {
		if (id > 0) {
			return getResource(id);
		}
		else if (id < 0) {
			return getLiteral(id);
		}
		else { // id == 0
			return null;
		}
	}

	/**
	 * Gets the Resource for the supplied ID.
	 * 
	 * @return Either an IdURI or an IdBNode object
	 */
	public IdResource getResource(int id) {
		try {
			IdResource result = null;

			Connection con = _rdbms.getConnection();
			java.sql.Statement st = con.createStatement();
			ResultSet rs = st.executeQuery("SELECT r.namespace, r.localname " + "FROM " + RESOURCES_TABLE
					+ " r " + "WHERE r.id = " + id);

			if (rs.next()) {
				int nsId = rs.getInt(1);
				String localName = rs.getString(2);

				if (localName == null) {
					localName = "";
				}

				if (nsId == 0) {
					// bnode
					result = new IdBNode(this, localName, id);
				}
				else {
					// URI
					result = new IdURI(this, _namespaceNames[nsId], localName, id);
				}
			}

			rs.close();
			st.close();
			con.close();

			return result;
		}
		catch (SQLException e) {
			throw new SailInternalException(e);
		}
	}

	/**
	 * Gets the Literal for the supplied ID.
	 * 
	 * @return An IdLiteral object.
	 */
	public IdLiteral getLiteral(int id) {
		try {
			IdLiteral result = null;

			Connection con = _rdbms.getConnection();
			java.sql.Statement st = con.createStatement();
			ResultSet rs = st.executeQuery("SELECT r.id, r.namespace, r.localname, l.language, l.label "
					+ "FROM " + LITERALS_TABLE + " l, " + RESOURCES_TABLE + " r "
					+ "WHERE l.datatype = r.id AND l.id = " + id);

			if (rs.next()) {
				int dtId = rs.getInt(1); // datatype id
				String label = rs.getString(5);

				if (label == null) {
					label = "";
				}

				if (dtId != 0) {
					// Datatyped literal
					int dtNsId = rs.getInt(2); // datatype namespace id
					String dtLname = rs.getString(3); // datatype localname

					if (dtLname == null) {
						dtLname = "";
					}

					URI datatype = new IdURI(this, _namespaceNames[dtNsId], dtLname, dtId);
					result = new IdLiteral(this, label, datatype, id);
				}
				else {
					String lang = rs.getString(4);

					if (lang != null) {
						// Literal with language attribute
						result = new IdLiteral(this, label, lang, id);
					}
					else {
						// Plain literal
						result = new IdLiteral(this, label, id);
					}
				}
			}

			rs.close();
			st.close();
			con.close();

			return result;
		}
		catch (SQLException e) {
			throw new SailInternalException(e);
		}
	}

	// implements ValueFactory.createURI(String)
	public URI createURI(String uri) {
		return new IdURI(this, uri, 0);
	}

	public URI createURI(String namespace, String localName) {
		return new IdURI(this, namespace, localName, 0);
	}

	// implements ValueFactory.createBNode()
	public BNode createBNode() {
		if (_nextBNodeID == Integer.MAX_VALUE) {
			// Start with a new bnode prefix
			_updateBNodePrefix();
		}

		return createBNode(_bnodePrefix + _nextBNodeID++);
	}

	// implements ValueFactory.createBNode(String)
	public BNode createBNode(String nodeId) {
		return new IdBNode(this, nodeId, 0);
	}

	// implements ValueFactory.createLiteral(String)
	public Literal createLiteral(String value) {
		return new IdLiteral(this, value, 0);
	}

	// implements ValueFactory.createLiteral(String, String)
	public Literal createLiteral(String value, String language) {
		return new IdLiteral(this, value, language, 0);
	}

	// implements ValueFactory.createLiteral(String, URI)
	public Literal createLiteral(String value, URI datatype) {
		return new IdLiteral(this, value, datatype, 0);
	}

	// implements ValueFactory.createStatement(Resource, URI, Value)
	public Statement createStatement(Resource subject, URI predicate, Value object) {
		return new StatementImpl(subject, predicate, object);
	}
}
