/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sailimpl.rdbms;

/**
 * Extension of SailConfig offering specific utility methods for configuring an
 * <tt>org.openrdf.sesame.sailimpl.rdbms.RdfSchemaRepository</tt> Sail. This class
 * offers methods for specifying the JDBC-driver class, the JDBC URL, the
 * username/password combination for the RDBMS, and the inferencer class.
 **/
public class RdfSchemaRepositoryConfig extends RdfRepositoryConfig {

/*--------------+
| Constructors  |
+--------------*/

	/**
	 * Creates a new and unconfigured RdfSchemaRepositoryConfig object for the
	 * specified Sail class.
	 *
	 * @param className A fully qualified Sail class name.
	 **/
	protected RdfSchemaRepositoryConfig(String className) {
		super(className);
	}

	/**
	 * Creates a new and unconfigured RdfSchemaRepositoryConfig object for an
	 * <tt>org.openrdf.sesame.sailimpl.rdbms.RdfSchemaRepository</tt> Sail.
	 **/
	public RdfSchemaRepositoryConfig() {
		this("org.openrdf.sesame.sailimpl.rdbms.RdfSchemaRepository");
	}

	/**
	 * Creates a new RdfSchemaRepositoryConfig object for an
	 * <tt>org.openrdf.sesame.sailimpl.rdbms.RdfSchemaRepository</tt> Sail that uses the
	 * specified JDBC Driver to connect to the specified database.
	 *
	 * @param jdbcDriver The fully qualified name of the Driver class, e.g.
	 * <tt>com.mysql.jdbc.Driver</tt> or <tt>org.postgresql.Driver</tt>.
	 * @param jdbcUrl A JDBC URL specifying the database to connect to, e.g.
	 * <tt>jdbc:mysql://localhost:3306/testdb</tt> or
	 * <tt>jdbc:postgresql://localhost:5432/testdb</tt>.
	 **/
	public RdfSchemaRepositoryConfig(String jdbcDriver, String jdbcUrl) {
		this();

		setJdbcDriver(jdbcDriver);
		setJdbcUrl(jdbcUrl);
	}

	/**
	 * Creates a new RdfSchemaRepositoryConfig object for an
	 * <tt>org.openrdf.sesame.sailimpl.rdbms.RdfSchemaRepository</tt> Sail that uses the
	 * specified JDBC Driver to connect to the specified database. The specified
	 * username/password combination is used to get access to the database.
	 *
	 * @param jdbcDriver The fully qualified name of the Driver class, e.g.
	 * <tt>com.mysql.jdbc.Driver</tt> or <tt>org.postgresql.Driver</tt>.
	 * @param jdbcUrl A JDBC URL specifying the database to connect to, e.g.
	 * <tt>jdbc:mysql://localhost:3306/testdb</tt> or
	 * <tt>jdbc:postgresql://localhost:5432/testdb</tt>.
	 * @param user The name of a user that has access to the database.
	 * @param password The user's password for the database.
	 **/
	public RdfSchemaRepositoryConfig(String jdbcDriver, String jdbcUrl, String user, String password) {
		this(jdbcDriver, jdbcUrl);

		setUser(user);
		setPassword(password);
	}

/*--------------+
| Methods       |
+--------------*/

	/**
	 * Sets the inferencer class that should be used for inferencing.
	 *
	 * @param inferencerClass The fully qualified name of the InferenceServices
	 * class that should be used for inferencing, e.g.
	 * <tt>org.openrdf.sesame.sailimpl.rdbms.RdbmsInferenceServices</tt>.
	 **/
	public void setInferencerClass(String inferencerClass) {
		setParameter(RdfSchemaRepository.INFERENCER_KEY, inferencerClass);
	}

	/**
	 * Gets the inferencer class.
	 **/
	public String getInferencerClass() {
		return getParameter(RdfSchemaRepository.INFERENCER_KEY);
	}
}
