/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sailimpl.rdbms;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Map;

import org.openrdf.util.log.ThreadLog;
import org.openrdf.vocabulary.RDF;
import org.openrdf.vocabulary.RDFS;

import org.openrdf.model.Resource;
import org.openrdf.model.URI;
import org.openrdf.model.Value;
import org.openrdf.model.impl.URIImpl;

import org.openrdf.sesame.sail.LiteralIterator;
import org.openrdf.sesame.sail.SailInitializationException;
import org.openrdf.sesame.sail.SailInternalException;
import org.openrdf.sesame.sail.SailUpdateException;
import org.openrdf.sesame.sail.StatementIterator;
import org.openrdf.sesame.sail.query.DirectSubClassOf;
import org.openrdf.sesame.sail.query.DirectSubPropertyOf;
import org.openrdf.sesame.sail.query.DirectType;
import org.openrdf.sesame.sail.query.PathExpression;
import org.openrdf.sesame.sail.query.TriplePattern;
import org.openrdf.sesame.sail.util.EmptyLiteralIterator;
import org.openrdf.sesame.sail.util.EmptyStatementIterator;
import org.openrdf.sesame.sailimpl.rdbms.iterators.RdbmsLiteralIterator;
import org.openrdf.sesame.sailimpl.rdbms.iterators.RdbmsStatementIterator;

/**
 * A portable implementation of the RdfSchemaRepository interface for
 * relational databases. This class defines the RDF Schema specific
 * methods and takes care of RDF Model Theory inferencing. The superclasses
 * define the basic RDF functionality.
 *
 * @author Arjohn Kampman
 * @version $Revision: 1.15.4.5 $
 */
public class RdfSchemaRepository extends RdfRepository
	implements org.openrdf.sesame.sail.RdfSchemaRepository, TableNames
{

/*-----------+
| Variables  |
+-----------*/

	/**
	 * Key used to store a fully qualified inferencer class name in the
	 * initialization parameters. **/
	public static final String INFERENCER_KEY = "use-inferencer";

	public static final String USE_DEPENDENCY_INFERENCER_KEY = "dependency-inferencing";

	protected InferenceServices _inferencer = null;

	protected boolean _useDependencyInferencer = true;

	// Id 's of standard RDF primitives.
	public int rdfTypeId;
	public int rdfPropertyId;
	public int rdfXMLLiteralId;
	public int rdfStatementId;
	public int rdfSubjectId;
	public int rdfPredicateId;
	public int rdfObjectId;
	public int rdfAltId;
	public int rdfBagId;
	public int rdfSeqId;
	public int rdfListId;
	public int rdfFirstId;
	public int rdfRestId;
	public int rdfNilId;
	public int rdfValueId;

	// Id 's of standard RDF Schema primitives.
	public int rdfsResourceId;
	public int rdfsLiteralId;
	public int rdfsClassId;
	public int rdfsSubClassOfId;
	public int rdfsSubPropertyOfId;
	public int rdfsDomainId;
	public int rdfsRangeId;
	public int rdfsCommentId;
	public int rdfsLabelId;
	public int rdfsDatatypeId;
	public int rdfsContainerId;
	public int rdfsMemberId;
	public int rdfsContainerMembershipPropertyId;
	public int rdfsIsDefinedById;
	public int rdfsSeeAlsoId;

/*----------------+
| Constructors    |
+----------------*/

	public RdfSchemaRepository() {
		super();
	}

/*--------------------------+
| Database initialization   |
+--------------------------*/

	// overrides RdfSource.initialize(Map)
	public void initialize(Map configParams)
		throws SailInitializationException
	{
		ThreadLog.trace("Initializing RdfSchemaRepository...");

		String inferencerClass = (String)configParams.get(INFERENCER_KEY);

		// check if the dependency inferencer is supposed to be used. Default
		// is that it _is_ used.
		String useDependencyInferencer = (String)configParams.get(USE_DEPENDENCY_INFERENCER_KEY);
		if ("no".equalsIgnoreCase(useDependencyInferencer) ||
			"false".equalsIgnoreCase(useDependencyInferencer) ||
			"off".equalsIgnoreCase(useDependencyInferencer))
		{
			_useDependencyInferencer = false;
		}
		else {
			_useDependencyInferencer = true;
		}
		ThreadLog.trace("Dependency inferencer used: " + _useDependencyInferencer);

		if (inferencerClass != null) {
			try {
				_inferencer = (InferenceServices)Class.forName(inferencerClass).newInstance();
				_inferencer.setDependencyInferencer(_useDependencyInferencer);
				ThreadLog.trace("Using inferencer: " + inferencerClass);
			}
			catch (InstantiationException e) {
				throw new SailInternalException(e);
			}
			catch (IllegalAccessException e) {
				throw new SailInternalException(e);
			}
			catch (ClassNotFoundException e) {
				throw new SailInternalException(e);
			}
			catch (ClassCastException e) {
				throw new SailInternalException(
						"The provided class '" + inferencerClass + "' does not implement the '"
						+ InferenceServices.class.getName()+ "' interface", e);
			}
		}

		if (_inferencer == null) {
			ThreadLog.trace("No inferencer specified, using default RDF MT inferencer");
			_inferencer = new RdbmsInferenceServices(_useDependencyInferencer);
		}


		// initialize the inferencer
		_inferencer.initialize(this, configParams);

		super.initialize(configParams);

		_inferencer.afterInitialize();

		try {
			_initRdfSchema();
		}
		catch (SQLException e) {
			throw new SailInitializationException(e);
		}

		ThreadLog.trace("RdfSchemaRepository initialized.");
	}

	protected void _initRdfSchema()
		throws SQLException
	{
		_initRdfPrimitives();

		_inferencer.initRdfSchema();

		// ALL_NEW_TRIPLES_TABLE now contains all axioms and statements
		// that can be derived from the axioms.
		_rdbms.optimizeTable(ALL_NEW_TRIPLES_TABLE);

		// Mark these triples as axioms
		_inferencer.markAxioms();

		// Update the auxiliary tables
		_updateAuxiliaryTables();

		_rdbms.clearTable(ALL_NEW_TRIPLES_TABLE);

		// Set the export status to 'dirty'
		_setExportStatusUpToDate(false);
	}

	protected void _initRdfPrimitives() {
		// Get the IDs of RDF primitives
		rdfTypeId           = _insertURI(URIImpl.RDF_TYPE);
		rdfPropertyId       = _insertURI(URIImpl.RDF_PROPERTY);
		rdfXMLLiteralId     = _insertURI(URIImpl.RDF_XMLLITERAL);
		rdfSubjectId        = _insertURI(URIImpl.RDF_SUBJECT);
		rdfPredicateId      = _insertURI(URIImpl.RDF_PREDICATE);
		rdfObjectId         = _insertURI(URIImpl.RDF_OBJECT);
		rdfStatementId      = _insertURI(URIImpl.RDF_STATEMENT);
		rdfAltId            = _insertURI(URIImpl.RDF_ALT);
		rdfBagId            = _insertURI(URIImpl.RDF_BAG);
		rdfSeqId            = _insertURI(URIImpl.RDF_SEQ);
		rdfListId           = _insertURI(URIImpl.RDF_LIST);
		rdfFirstId          = _insertURI(URIImpl.RDF_FIRST);
		rdfRestId           = _insertURI(URIImpl.RDF_REST);
		rdfNilId            = _insertURI(URIImpl.RDF_NIL);
		rdfValueId		    = _insertURI(URIImpl.RDF_VALUE);

		// Get the IDs of RDF Schema primitives
		rdfsResourceId      = _insertURI(URIImpl.RDFS_RESOURCE);
		rdfsClassId         = _insertURI(URIImpl.RDFS_CLASS);
		rdfsLiteralId       = _insertURI(URIImpl.RDFS_LITERAL);
		rdfsSubClassOfId    = _insertURI(URIImpl.RDFS_SUBCLASSOF);
		rdfsSubPropertyOfId = _insertURI(URIImpl.RDFS_SUBPROPERTYOF);
		rdfsDomainId        = _insertURI(URIImpl.RDFS_DOMAIN);
		rdfsRangeId         = _insertURI(URIImpl.RDFS_RANGE);
		rdfsCommentId       = _insertURI(URIImpl.RDFS_COMMENT);
		rdfsLabelId         = _insertURI(URIImpl.RDFS_LABEL);
		rdfsIsDefinedById   = _insertURI(URIImpl.RDFS_ISDEFINEDBY);
		rdfsSeeAlsoId       = _insertURI(URIImpl.RDFS_SEEALSO);
		rdfsDatatypeId      = _insertURI(URIImpl.RDFS_DATATYPE);
		rdfsContainerId     = _insertURI(URIImpl.RDFS_CONTAINER);
		rdfsMemberId        = _insertURI(URIImpl.RDFS_MEMBER);
		rdfsContainerMembershipPropertyId =
				_insertURI(URIImpl.RDFS_CONTAINERMEMBERSHIPPROPERTY);

		// Try setting the namespace prefixes
		try {
			changeNamespacePrefix(RDF.NAMESPACE, "rdf");
		}
		catch (SailUpdateException e) { // FIXME /* ignore */
		}

		try {
			changeNamespacePrefix(RDFS.NAMESPACE, "rdfs");
		}
		catch (SailUpdateException e) { // FIXME /* ignore */
		}
	}

	protected int _insertURI(URI r)
	{
		// Check if the resource is already in RESOURCES_TABLE
		int id = _getURIId(r);

		if (id == 0) {
			// resource not yet in the table.
			id = _getNextResourceId();
			int nsId = _createIdForNamespace(r.getNamespace());
			String lname = _rdbms.escapeString(r.getLocalName());

			try {
				_rdbms.executeUpdate(
						"INSERT INTO " + RESOURCES_TABLE + " VALUES(" +
						id + ", " + nsId + ", '" + lname + "')");
			}
			catch (SQLException e) {
				throw new SailInternalException(e);
			}
		}

		return id;
	}

	protected void _createDbSchema()
		throws SQLException
	{
		super._createDbSchema();

		// Tables for specific RDF primitives
		if (_schemaVersion == -1) {
			_createOneIdColumnTable(CLASS_TABLE, "id");
			_createOneIdColumnTable(PROPERTY_TABLE, "id");

			_createTwoIdColumnsTable(SUBCLASSOF_TABLE, "sub", "super");
			_createTwoIdColumnsTable(DIRECT_SUBCLASSOF_TABLE, "sub", "super");

			_createTwoIdColumnsTable(SUBPROPERTYOF_TABLE, "sub", "super");
			_createTwoIdColumnsTable(DIRECT_SUBPROPERTYOF_TABLE, "sub", "super");

			_createTwoIdColumnsTable(INSTANCEOF_TABLE, "inst", "class");
			_createTwoIdColumnsTable(PROPER_INSTANCEOF_TABLE, "inst", "class");

			_createTwoIdColumnsTable(DOMAIN_TABLE, "property", "class");
			_createTwoIdColumnsTable(RANGE_TABLE, "property", "class");
		}

		// Tables used during inferencing
		_createAllNewTriplesTable();
		_createInferredTriplesTable();
		_createAllInferredTriplesTable();

		if (_useDependencyInferencer) {
			_createDependenciesTable();

			// Tables used for the Truth Maintenance System
			_createGroundedTriplesTable();
			_createNewGroundedTriplesTable();
		}
	}

	/**
	 * Creates a table with the specified name and having one column of type
	 * ID_INT with name <tt>colName</tt>. The column is part of a primary key.
	 * This method is used to create the CLASS_TABLE and PROPERTY_TABLE.
	 *
	 * @param tableName the name of the table.
	 * @param colName the name of the column.
	 **/
	protected void _createOneIdColumnTable(String tableName, String colName)
		throws SQLException
	{
		_rdbms.executeUpdate(
				"CREATE TABLE " + tableName + " (" +
				colName + " " + _rdbms.ID_INT + " NOT NULL PRIMARY KEY)");
	}

	/**
	 * Creates a table with the specified name and having two columns of type
	 * ID_INT: <tt>col1</tt> and <tt>col2</tt>. Both columns are part of the
	 * primary key, and an additional index is created for the second column.
	 * This method is used to create the SUBCLASSOF_TABLE, DIRECT_SUBCLASSOF_TABLE,
	 * SUBPROPERTYOF_TABLE, DIRECT_SUBPROPERTYOF_TABLE, INSTANCEOF_TABLE,
	 * PROPER_INSTANCEOF_TABLE, DOMAIN_TABLE and RANGE_TABLE.
	 *
	 * @param tableName the name of the table.
	 * @param col1 the name of the first column.
	 * @param col2 the name of the second column.
	 **/
	protected void _createTwoIdColumnsTable(
			String tableName, String col1, String col2)
		throws SQLException
	{
		_rdbms.executeUpdate(
				"CREATE TABLE " + tableName + " (" +
				col1 + " " + _rdbms.ID_INT + " NOT NULL, " +
				col2 + " " + _rdbms.ID_INT + " NOT NULL, " +
				"PRIMARY KEY(" + col1 + ", " + col2 + "))");

		// Create an index over col2
		_rdbms.createIndex(tableName, col2);
	}

	protected void _createAllNewTriplesTable()
		throws SQLException
	{
		if (_schemaVersion == -1) {
			_createTriplesTable(ALL_NEW_TRIPLES_TABLE, true);

			_rdbms.createIndex(ALL_NEW_TRIPLES_TABLE, new String[]{"pred", "obj"}, false);
			_rdbms.createIndex(ALL_NEW_TRIPLES_TABLE, new String[]{"obj", "subj"}, false);
		}
		else {
			if (_schemaVersion < 4) {
				_rdbms.renameTableColumn(ALL_NEW_TRIPLES_TABLE, "subject"  , "subj", _rdbms.ID_INT+" NOT NULL");
				_rdbms.renameTableColumn(ALL_NEW_TRIPLES_TABLE, "predicate", "pred", _rdbms.ID_INT+" NOT NULL");
				_rdbms.renameTableColumn(ALL_NEW_TRIPLES_TABLE, "object"   , "obj" , _rdbms.ID_INT+" NOT NULL");
			}
			
			if (_schemaVersion < 6) {
				_rdbms.dropIndex(ALL_NEW_TRIPLES_TABLE, "subj");
				_rdbms.dropIndex(ALL_NEW_TRIPLES_TABLE, "pred");
				_rdbms.dropIndex(ALL_NEW_TRIPLES_TABLE, "obj");
				
				_rdbms.createIndex(ALL_NEW_TRIPLES_TABLE, new String[]{"pred", "obj"}, false);
				_rdbms.createIndex(ALL_NEW_TRIPLES_TABLE, new String[]{"obj", "subj"}, false);
			}
		}
	}

	protected void _createInferredTriplesTable()
		throws SQLException
	{
		if (_schemaVersion == -1) {
			_createTriplesTable(INFERRED_TABLE, false);
		}
		else if (_schemaVersion < 4) {
			_rdbms.renameTableColumn(INFERRED_TABLE, "subject"  , "subj", _rdbms.ID_INT+" NOT NULL");
			_rdbms.renameTableColumn(INFERRED_TABLE, "predicate", "pred", _rdbms.ID_INT+" NOT NULL");
			_rdbms.renameTableColumn(INFERRED_TABLE, "object"   , "obj" , _rdbms.ID_INT+" NOT NULL");
		}
	}

	protected void _createAllInferredTriplesTable()
		throws SQLException
	{
		if (_schemaVersion == -1) {
			_createTriplesTable(ALL_INFERRED_TABLE, false);
		}
		else if (_schemaVersion < 4) {
			_rdbms.renameTableColumn(ALL_INFERRED_TABLE, "subject"  , "subj", _rdbms.ID_INT+" NOT NULL");
			_rdbms.renameTableColumn(ALL_INFERRED_TABLE, "predicate", "pred", _rdbms.ID_INT+" NOT NULL");
			_rdbms.renameTableColumn(ALL_INFERRED_TABLE, "object"   , "obj" , _rdbms.ID_INT+" NOT NULL");
		}
	}

	protected void _createDependenciesTable()
		throws SQLException
	{
		if (_schemaVersion == -1) {
			_inferencer.createDependenciesTable();
		}
	}

	protected void _createGroundedTriplesTable()
		throws SQLException
	{
		if (_schemaVersion == -1) {
			_rdbms.executeUpdate(
					"CREATE TABLE " + GROUNDED_TRIPLES_TABLE +
					" (id " + _rdbms.ID_INT + " NOT NULL PRIMARY KEY)");
		}
	}

	protected void _createNewGroundedTriplesTable()
		throws SQLException
	{
		if (_schemaVersion == -1) {
			_rdbms.executeUpdate(
					"CREATE TABLE " + NEW_GROUNDED_TRIPLES_TABLE +
					" (id " + _rdbms.ID_INT + " NOT NULL PRIMARY KEY)");
		}
	}

	/*-----------------------------------------+
	| Overridden methods from RdfRepository    |
	+-----------------------------------------*/

	public void clearRepository()
		throws SailUpdateException
	{
		super.clearRepository();
		try {
			_rdbms.clearTable(CLASS_TABLE);
			_rdbms.clearTable(PROPERTY_TABLE);

			_rdbms.clearTable(SUBCLASSOF_TABLE);
			_rdbms.clearTable(DIRECT_SUBCLASSOF_TABLE);

			_rdbms.clearTable(SUBPROPERTYOF_TABLE);
			_rdbms.clearTable(DIRECT_SUBPROPERTYOF_TABLE);

			_rdbms.clearTable(INSTANCEOF_TABLE);
			_rdbms.clearTable(PROPER_INSTANCEOF_TABLE);

			_rdbms.clearTable(DOMAIN_TABLE);
			_rdbms.clearTable(RANGE_TABLE);

			_rdbms.clearTable(ALL_NEW_TRIPLES_TABLE);
			_rdbms.clearTable(INFERRED_TABLE);
			_rdbms.clearTable(ALL_INFERRED_TABLE);

			if (_useDependencyInferencer) {
				_rdbms.clearTable(DEPEND_TABLE);
				_rdbms.clearTable(GROUNDED_TRIPLES_TABLE);
				_rdbms.clearTable(NEW_GROUNDED_TRIPLES_TABLE);
			}
		}
		catch (SQLException e) {
			throw new SailInternalException(e);
		}

		try {
			_initRdfSchema();
		}
		catch (SQLException e) {
			throw new SailInternalException(e);
		}
	}

	/**
	 * Processes the triples from ADDED_TRIPLES_TABLE. This method first
	 * makes any implicit statements from TRIPLES_TABLE that occur in
	 * ADDED_TRIPLES_TABLE explicit. It then copies all triples from
	 * ADDED_TRIPLES_TABLE that are not yet in TRIPLES_TABLE to
	 * NEW_TRIPLES_TABLE.
	 **/
	protected void _processAddedTriples()
		throws SQLException
	{
		// Find any implicit statements that should be made explicit now.
		Connection con = _rdbms.getConnection();
		java.sql.Statement st = con.createStatement();
		ResultSet rs = st.executeQuery(
				"SELECT t.id FROM " +
				TRIPLES_TABLE + " t, " +
				ADDED_TRIPLES_TABLE + " at " +
				"WHERE " +
				"t.subj = at.subj AND " +
				"t.pred = at.pred AND " +
				"t.obj = at.obj AND " +
				"t.explicit = " + _rdbms.FALSE);

		String[] idChunks = _chunkIdSet(rs, 3500);
		rs.close();
		st.close();

		con.setAutoCommit(false);
		st = con.createStatement();

		for (int i = 0; i < idChunks.length; i++) {
			st.executeUpdate(
					"UPDATE " + TRIPLES_TABLE +
					" SET explicit = " + _rdbms.TRUE +
					" WHERE id IN " + idChunks[i]);
		}

		con.commit();
		st.close();
		con.close();

		super._processAddedTriples();
	}

	/**
	 * Updates the inferred statements.
	 **/
	protected void _processChangedTriples()
		throws SQLException
	{
		// Collect all new triples in ALL_NEW_TRIPLES_TABLE
		int count = _rdbms.copyRows(NEW_TRIPLES_TABLE, ALL_NEW_TRIPLES_TABLE);

		if (count > 0) {
			// Infer new statements from the added ones.
			_inferencer.doInferencing();

			// Register new dependencies
			_inferencer.processNewStatements();
		}

		// ALL_NEW_TRIPLES_TABLE now contains all statements, including the
		// ones that were added by the inferencer.
		_rdbms.optimizeTable(ALL_NEW_TRIPLES_TABLE);

		_updateAuxiliaryTables();

		_rdbms.clearTable(ALL_NEW_TRIPLES_TABLE);
	}

	// Overrides RdfRepository._removeExpiredStatements()
	protected void _removeExpiredStatements()
		throws SQLException
	{
		_inferencer.removeExpiredStatements();
	}


	protected void _updateAuxiliaryTables()
		throws SQLException
	{
		ThreadLog.trace("Updating auxiliary tables");

		_updateClasses();
		_updateProperties();

		_updateDomains();
		_updateRanges();

		int addedSubClassesCount = _updateSubClasses();

		_updateSubProperties();

		_updateInstanceOf( (addedSubClassesCount > 0) );
	}

	// Updates classes.
	private void _updateClasses()
		throws SQLException
	{
		_updateClassOrProperties(CLASS_TABLE, rdfsClassId);
	}

	private void _updateProperties()
		throws SQLException
	{
		_updateClassOrProperties(PROPERTY_TABLE, rdfPropertyId);
	}

	private void _updateClassOrProperties(String tableName, int objectId)
		throws SQLException
	{
		// Use ALL_NEW_TRIPLES_TABLE for incremental updates and TRIPLES_TABLE
		// for complete updates when statements were removed.
		String sourceTable = ALL_NEW_TRIPLES_TABLE;
		if (_statementsRemoved) {
			sourceTable = TRIPLES_TABLE;
			_rdbms.clearTable(tableName);
		}

		int count = _rdbms.executeUpdate(
				"INSERT INTO " + tableName +
				" SELECT subj FROM " + sourceTable +
				" WHERE pred = " + rdfTypeId + " AND obj = " + objectId);

		if (count > 0 || _statementsRemoved) {
			_rdbms.optimizeTable(tableName);
		}
	}

	private void _updateDomains()
		throws SQLException
	{
		_updateDomainsOrRanges(DOMAIN_TABLE, rdfsDomainId);
	}

	private void _updateRanges()
		throws SQLException
	{
		_updateDomainsOrRanges(RANGE_TABLE, rdfsRangeId);
	}

	private void _updateDomainsOrRanges(String tableName, int predicateId)
		throws SQLException
	{
		// Use ALL_NEW_TRIPLES_TABLE for incremental updates and TRIPLES_TABLE
		// for complete updates when statements were removed.
		String sourceTable = ALL_NEW_TRIPLES_TABLE;
		if (_statementsRemoved) {
			sourceTable = TRIPLES_TABLE;
			_rdbms.clearTable(tableName);
		}

		int count = _rdbms.executeUpdate(
				"INSERT INTO " + tableName +
				" SELECT subj, obj FROM " + sourceTable +
				" WHERE pred = " + predicateId);

		if (count > 0 || _statementsRemoved) {
			_rdbms.optimizeTable(tableName);
		}
	}

	private int _updateSubClasses()
		throws SQLException
	{
		return _updateSubClassesOrProperties(
				SUBCLASSOF_TABLE, DIRECT_SUBCLASSOF_TABLE, rdfsSubClassOfId);
	}

	private int _updateSubProperties()
		throws SQLException
	{
		return _updateSubClassesOrProperties(
				SUBPROPERTYOF_TABLE, DIRECT_SUBPROPERTYOF_TABLE, rdfsSubPropertyOfId);
	}

	/* Updates subClassOf or subPropertyOf relations. Strings allTable and
	 * directTable indicates which tables are updated. Decides whether a
	 * subClassOf or subPropertyOf relation is a direct or indirect one.
	 */
	private int _updateSubClassesOrProperties(
			String allTable, String directTable, int propId)
		throws SQLException
	{
		// Use ALL_NEW_TRIPLES_TABLE for incremental updates and TRIPLES_TABLE
		// for complete updates when statements were removed.
		String sourceTable = ALL_NEW_TRIPLES_TABLE;
		if (_statementsRemoved) {
			sourceTable = TRIPLES_TABLE;
			_rdbms.clearTable(allTable);
		}

		// Add any new relations to allTable.
		int count = _rdbms.executeUpdate(
				"INSERT INTO " + allTable +
				" SELECT subj, obj FROM " + sourceTable +
				" WHERE pred = " + propId);

		// The directTable only needs to be updated if new rows were added
		// to allTale.
		if (count > 0 || _statementsRemoved) {
			_rdbms.optimizeTable(allTable);

			_rdbms.clearTable(directTable);

			// Create extra, temporary table for storing non-cycles.
			_rdbms.executeUpdate(
					"CREATE TABLE noncycles (" +
					"sub " + _rdbms.ID_INT + " NOT NULL, " +
					"super " + _rdbms.ID_INT + " NOT NULL, " +
					"UNIQUE(sub, super))");

			// Copy all relations that are not self-references or part of cycles
			_rdbms.executeUpdate(
					"INSERT INTO noncycles " +
					"SELECT s1.* " +
					"FROM " + allTable + " s1 " +
					"LEFT JOIN " + allTable + " s2 " +
					"ON s1.sub = s2.super AND s1.super = s2.sub " +
					"WHERE s2.sub IS NULL");

			_rdbms.optimizeTable("noncycles");


			// Copy all direct relations from noncycles to directTable. All
			// direct relations are all relations A --> C for which no
			// relations can be found such that A --> B --> C.
			_rdbms.executeUpdate(
					"INSERT INTO " + directTable +
					" SELECT nc1.*" +
					" FROM noncycles nc1" +
					" LEFT JOIN noncycles nc2" +
					" ON nc1.sub = nc2.sub" +
					" LEFT JOIN noncycles nc3" +
					" ON nc3.sub = nc2.super AND nc3.super = nc1.super" +
					" GROUP BY nc1.sub, nc1.super" +
					" HAVING count(nc3.sub) = 0"); // NULL values don't add to count

			_rdbms.optimizeTable(directTable);

			// Drop temporary table.
			_rdbms.dropTable("noncycles");
		}

		return count;
	}

	protected void _updateInstanceOf(boolean newSubClassesAdded)
		throws SQLException
	{
		// Use ALL_NEW_TRIPLES_TABLE for incremental updates and TRIPLES_TABLE
		// for complete updates when statements were removed.
		String sourceTable = ALL_NEW_TRIPLES_TABLE;
		if (_statementsRemoved) {
			sourceTable = TRIPLES_TABLE;
			_rdbms.clearTable(INSTANCEOF_TABLE);
		}

		// Copy all new rdf:type relations to instanceOf table.
		int count = _rdbms.executeUpdate(
				"INSERT INTO " + INSTANCEOF_TABLE +
				" SELECT subj, obj FROM " + sourceTable +
				" WHERE pred = " + rdfTypeId);

		// The proper_instanceof table only needs to be updated if new rows
		// were added to the instanceof table, or if new subclasses have been
		// added.
		if (count > 0 || newSubClassesAdded ||  _statementsRemoved) {
			_rdbms.optimizeTable(INSTANCEOF_TABLE);

			// Clear proper_instanceof table.
			_rdbms.clearTable(PROPER_INSTANCEOF_TABLE);

			// Drop index to speed up the proces.
			// FIXME: do we need to do this?
			//			_rdbms.dropIndex(DIRECT_SUBCLASSOF_TABLE, "super");

			// Copy all proper instanceOf relations to proper instanceof table.
			// Proper instanceOf relations are those relations (a type b) for
			// which no proper subclass of b can be found of which a is also an
			// instance.
			_rdbms.executeUpdate(
					"INSERT INTO " + PROPER_INSTANCEOF_TABLE +
					" SELECT type.inst, type.class" +
					" FROM " + INSTANCEOF_TABLE + " type" +
					" LEFT JOIN " + INSTANCEOF_TABLE + " subtype" +
					" ON subtype.inst = type.inst" +
					" LEFT JOIN " + DIRECT_SUBCLASSOF_TABLE + " dsco" +
					" ON type.class = dsco.super AND subtype.class = dsco.sub" +
					" GROUP BY type.inst, type.class" +
					" HAVING count(dsco.sub) = 0");

			_rdbms.optimizeTable(PROPER_INSTANCEOF_TABLE);

			// Re-create index.
			//			_rdbms.createIndex(DIRECT_SUBCLASSOF_TABLE, "super");
		}
	}

	/**
	 * A more efficient implementation is possible now that the full RDF
	 * Schema is stored.
	 **/
	protected void _updateExportedNamespaces() {
		ThreadLog.trace("Updating exported namespaces information.");

		try {
			// First 'reset' all namespaces that are not user-defined to
			// non-exported. This eliminates exporting of namespaces
			// belonging to removed predicates.
			_rdbms.executeUpdate(
					"UPDATE " + NAMESPACES_TABLE +
					" SET export = " + _rdbms.FALSE +
					" WHERE userDefined = " + _rdbms.FALSE);
			// Query for all namespaces that need to be exported: those that
			// are the in subject of statements where the predicate is
			// rdf:type and the object is rdf:Property.
			Connection con = _rdbms.getConnection();
			java.sql.Statement st = con.createStatement();
			ResultSet rs = st.executeQuery(
					"SELECT DISTINCT r.namespace " +
					"FROM " + RESOURCES_TABLE  + " r, " + TRIPLES_TABLE + " t "+
					"WHERE t.subj = r.id " +
					"AND t.pred = " + rdfTypeId + " " +
					"AND t.obj = " + rdfPropertyId);

			String[] idChunks = _chunkIdSet(rs, 3500);
			rs.close();
			st.close();

			con.setAutoCommit(false);
			st = con.createStatement();

			for (int i = 0; i < idChunks.length; i++) {
				st.executeUpdate(
						"UPDATE " + NAMESPACES_TABLE +
						" SET export = " + _rdbms.TRUE +
						" WHERE id IN " + idChunks[i]);
			}

			con.commit();
			st.close();
			con.close();

			// Set the export status to 'up-to-date'
			_setExportStatusUpToDate(true);

			// re-initialize the namespaces to update the cache.
			_initNamespaceCache();
		}
		catch (SQLException e) {
			throw new SailInternalException(e);
		}
	}

/*-----------------------------+
| Methods from RdfSchemaSource |
+-----------------------------*/

	public StatementIterator getExplicitStatements(Resource subj, URI pred, Value obj) {
		return super.getStatements(subj, pred, obj, true);
	}

	public boolean hasExplicitStatement(Resource subj, URI pred, Value obj) {
		return super.hasStatement(subj, pred, obj, true);
	}

	public StatementIterator getClasses() {
		return _queryOneColumnTable(CLASS_TABLE, "id", URIImpl.RDF_TYPE, URIImpl.RDFS_CLASS);
	}

	public boolean isClass(Resource resource) {
		return _isRowInOneColumnTable(CLASS_TABLE, "id", resource);
	}

	public StatementIterator getProperties() {
		return _queryOneColumnTable(PROPERTY_TABLE, "id", URIImpl.RDF_TYPE, URIImpl.RDF_PROPERTY);
	}

	public boolean isProperty(Resource resource) {
		return _isRowInOneColumnTable(PROPERTY_TABLE, "id", resource);
	}

	public StatementIterator getSubClassOf(Resource subClass, Resource superClass) {
		return _queryTwoColumnTable(
				SUBCLASSOF_TABLE, "sub", "super",
				subClass, URIImpl.RDFS_SUBCLASSOF, superClass);
	}

	public StatementIterator getDirectSubClassOf(Resource subClass, Resource superClass) {
		return _queryTwoColumnTable(
				DIRECT_SUBCLASSOF_TABLE, "sub", "super",
				subClass, URIImpl.RDFS_SUBCLASSOF, superClass);
	}

	public boolean isSubClassOf(Resource subClass, Resource superClass) {
		return _isRowInTwoColumnTable(
				SUBCLASSOF_TABLE, "sub", "super",
				subClass, superClass);
	}

	public boolean isDirectSubClassOf(Resource subClass, Resource superClass) {
		return _isRowInTwoColumnTable(
				DIRECT_SUBCLASSOF_TABLE, "sub", "super",
				subClass, superClass);
	}

	public StatementIterator getSubPropertyOf(Resource subProperty, Resource superProperty) {
		return _queryTwoColumnTable(
				SUBPROPERTYOF_TABLE, "sub", "super",
				subProperty, URIImpl.RDFS_SUBPROPERTYOF, superProperty);
	}

	public StatementIterator getDirectSubPropertyOf(Resource subProperty, Resource superProperty) {
		return _queryTwoColumnTable(
				DIRECT_SUBPROPERTYOF_TABLE, "sub", "super",
				subProperty, URIImpl.RDFS_SUBPROPERTYOF, superProperty);
	}

	public boolean isSubPropertyOf(Resource subProperty, Resource superProperty) {
		return _isRowInTwoColumnTable(
				SUBPROPERTYOF_TABLE, "sub", "super",
				subProperty, superProperty);
	}

	public boolean isDirectSubPropertyOf(Resource subProperty, Resource superProperty) {
		return _isRowInTwoColumnTable(
				DIRECT_SUBPROPERTYOF_TABLE, "sub", "super",
				subProperty, superProperty);
	}

	public StatementIterator getDomain(Resource prop, Resource domain) {
		return _queryTwoColumnTable(
				DOMAIN_TABLE, "property", "class",
				prop, URIImpl.RDFS_DOMAIN, domain);
	}

	public StatementIterator getRange(Resource prop, Resource domain) {
		return _queryTwoColumnTable(
				RANGE_TABLE, "property", "class",
				prop, URIImpl.RDFS_DOMAIN, domain);
	}

	public StatementIterator getType(Resource anInstance, Resource aClass) {
		return _queryTwoColumnTable(
				INSTANCEOF_TABLE, "inst", "class",
				anInstance, URIImpl.RDF_TYPE, aClass);
	}

	public StatementIterator getDirectType(Resource anInstance, Resource aClass) {
		return _queryTwoColumnTable(
				PROPER_INSTANCEOF_TABLE, "inst", "class",
				anInstance, URIImpl.RDF_TYPE, aClass);
	}

	public boolean isType(Resource anInstance, Resource aClass) {
		return _isRowInTwoColumnTable(
				INSTANCEOF_TABLE, "inst", "class",
				anInstance, aClass);
	}

	public boolean isDirectType(Resource anInstance, Resource aClass) {
		return _isRowInTwoColumnTable(
				PROPER_INSTANCEOF_TABLE, "inst", "class",
				anInstance, aClass);
	}

	public LiteralIterator getLiterals(String label, String language, URI datatype) {
		int datatypeId = 0;
		if (datatype != null) {
			datatypeId = _getURIId(datatype);

			if (datatypeId == 0) {
				// datatype URI not found
				return new EmptyLiteralIterator();
			}
		}

		String query =
				"SELECT l.id, dt.id, dt.namespace, dt.localname, l.language, l.label " +
				"FROM " + LITERALS_TABLE + " l, " + RESOURCES_TABLE + " dt " +
				"WHERE l.datatype = dt.id";

		boolean firstPartWritten = false;

		// label
		if (label != null) {
			// labelHash
			long labelHash = _getLabelHash(label);
			query += " l.labelHash = " + labelHash;

			// label
			query += " AND l.label ";
			if (_rdbms.emptyStringIsNull() && label.length() == 0) {
				query += "IS NULL";
			}
			else {
				query += "= '" + _rdbms.escapeString(label) + "'";
			}

			firstPartWritten = true;
		}

		//  language
		if (language != null && datatype == null) {
			if (firstPartWritten) {
				query += " AND";
			}
			query += " l.language = '" + _rdbms.escapeString(language) + "'";

			firstPartWritten = true;
		}

		// datatype
		if (datatype != null) {
			if (firstPartWritten) {
				query += " AND";
			}
			query += "dt.id = " + datatypeId;
		}

		try {
			Connection con = _rdbms.getConnection();
			return new RdbmsLiteralIterator(this, _namespaceNames, con, query);
		}
		catch (SQLException e) {
			throw new SailInternalException(e);
		}
	}

	/**
	 * Queries a one-column table like <tt>CLASS_TABLE</tt> or <tt>PROPER_INSTANCEOF_TABLE</tt>.
	 *
	 * @param tableName The name of the table to query.
	 * @param colName The name of the column in the table.
	 * @param prop The property that should be returned by the resulting StatementIterator.
	 * @param obj The object that should be returned by the resulting StatementIterator.
	 * @return A StatementIterator containing the requested statements.
	 **/
	protected StatementIterator _queryOneColumnTable(String tableName, String colName, URI prop, Resource obj) {
		String query =
				"SELECT r.id, r.namespace, r.localname FROM " +
					tableName + " t, " + 
					RESOURCES_TABLE + " r " +
				"WHERE t."+colName + " = r.id";
		try {
			Connection con = _rdbms.getConnection();
			return new RdbmsStatementIterator(
					this, _namespaceNames, con,
					query, null, null, prop, obj);
		}
		catch (SQLException e) {
			throw new SailInternalException(e);
		}
	}

	/**
	 * Checks whether a resource is present in a one-column table like
	 * <tt>CLASS_TABLE</tt> or <tt>PROPERTY_TABLE</tt>.
	 *
	 * @param tableName The name of the table to query.
	 * @param colName The name of the column in the table.
	 * @param val The value that the column should have.
	 **/
	protected boolean _isRowInOneColumnTable(String tableName, String colName, Resource val) {
		int id = _getResourceId(val);

		if (id == 0) {
			// unknown resource
			return false;
		}

		String query =
				"SELECT * FROM " + tableName +
				" WHERE " + colName + " = " + id;

		try {
			return _rdbms.queryHasResults(query);
		}
		catch (SQLException e) {
			throw new SailInternalException(e);
		}
	}

	/**
	 * Queries a two-column table like
	 * <tt>SUBCLASSOF_TABLE</tt>, <tt>DIRECT_SUBCLASSOF_TABLE</tt>,
	 * <tt>SUBPROPERTYOF_TABLE</tt>, <tt>DIRECT_SUBPROPERTYOF_TABLE</tt>,
	 * <tt>INSTANCEOF_TABLE</tt>, <tt>PROPER_INSTANCEOF_TABLE</tt>,
	 * <tt>DOMAIN_TABLE</tt> or <tt>RANGE_TABLE</tt>.
	 *
	 * @param tableName The name of the table to query.
	 * @param col1Name The name of the first column in the table.
	 * @param col2Name The name of the second column in the table.
	 * @param val1 The value that the first column should have, or
	 * <tt>null</tt> for a wildcard.
	 * @param prop The property that should be returned by the
	 * resulting StatementIterator.
	 * @param val2 The value that the second column should have, or
	 * <tt>null</tt> for a wildcard.
	 * @return A StatementIterator containing the requested statements.
	 **/
	protected StatementIterator _queryTwoColumnTable(
			String tableName, String col1Name, String col2Name,
			Resource val1, URI prop, Resource val2)
	{
		int id1 = 0, id2 = 0;

		if (val1 != null) {
			id1 = _getResourceId(val1);
			if (id1 == 0) {
				// unknown resource
				return new EmptyStatementIterator();
			}
		}

		if (val2 != null) {
			id2 = _getResourceId(val2);
			if (id2 == 0) {
				// unknown resource
				return new EmptyStatementIterator();
			}
		}

		String select = "SELECT ";
		String from = " FROM " + tableName + " t";
		String where = " WHERE ";

		// val1
		if (id1 != 0) {
			// val1 is specified
			where += "t."+col1Name + " = " + id1;
		}
		else {
			// val1 is a wildcard
			select += "r1.id, r1.namespace, r1.localname";
			from += ", " + RESOURCES_TABLE + " r1";
			where += "t."+col1Name + " = r1.id";
		}

		// val2
		if (id2 != 0) {
			// val2 is specified
			where += " AND t."+col2Name + " = " + id2;
		}
		else {
			// val2 is a wildcard
			if (id1 == 0) {
				select += ", ";
			}

			select += "r2.id, r2.namespace, r2.localname";
			from += ", " + RESOURCES_TABLE + " r2";
			where += " AND t."+col2Name + " = r2.id";
		}

		if (id1 != 0 && id2 != 0) {
			select += "*";
		}

		String query = select + from + where;

		try {
			Connection con = _rdbms.getConnection();
			return new RdbmsStatementIterator(
					this, _namespaceNames, con,
					query, null, val1, prop, val2);
		}
		catch (SQLException e) {
			throw new SailInternalException(e);
		}
	}

	/**
	 * Checks whether a combination of two resources is present in
	 * a two-column table like
	 * <tt>SUBCLASSOF_TABLE</tt>, <tt>DIRECT_SUBCLASSOF_TABLE</tt>,
	 * <tt>SUBPROPERTYOF_TABLE</tt>, <tt>DIRECT_SUBPROPERTYOF_TABLE</tt>,
	 * <tt>INSTANCEOF_TABLE</tt>, <tt>PROPER_INSTANCEOF_TABLE</tt>,
	 * <tt>DOMAIN_TABLE</tt> or <tt>RANGE_TABLE</tt>
	 *
	 * @param tableName The name of the table to query.
	 * @param col1Name The name of the first column in the table.
	 * @param col2Name The name of the second column in the table.
	 * @param val1 The value that the first column should have.
	 * @param val2 The value that the second column should have.
	 **/
	protected boolean _isRowInTwoColumnTable(
			String tableName, String col1Name, String col2Name,
			Resource val1, Resource val2)
	{
		int id1 = _getResourceId(val1);
		if (id1 == 0) {
			return false;
		}

		int id2 = _getResourceId(val2);
		if (id2 == 0) {
			return false;
		}

		String query = 
				"SELECT * FROM " + tableName +
				" WHERE " + col1Name + " = " + id1 +
				" AND " + col2Name + " = " + id2;

		try {
			return _rdbms.queryHasResults(query);
		}
		catch (SQLException e) {
			throw new SailInternalException(e);
		}
	}

	// Overrides RdfSource._convertPathExpressionToSQL(...)
	protected PathExpression _convertPathExpressionToSQL(
			PathExpression pathExpr, String peId,
			StringBuffer from, StringBuffer where, Map varToSqlIdMapping)
	{
		if (pathExpr instanceof DirectType) {
			DirectType dt = (DirectType)pathExpr;

			from.append(", " + PROPER_INSTANCEOF_TABLE + " " + peId);

			_processPathExpressionVar(dt.getSubjectVar(), peId+".inst", where, varToSqlIdMapping);
			_processPathExpressionVar(dt.getObjectVar(), peId+".class", where, varToSqlIdMapping);

			return null;
		}
		else if (pathExpr instanceof DirectSubClassOf) {
			DirectSubClassOf dsco = (DirectSubClassOf)pathExpr;

			from.append(", " + DIRECT_SUBCLASSOF_TABLE + " " + peId);

			_processPathExpressionVar(dsco.getSubjectVar(), peId+".sub", where, varToSqlIdMapping);
			_processPathExpressionVar(dsco.getObjectVar(), peId+".super", where, varToSqlIdMapping);

			return null;
		}
		else if (pathExpr instanceof DirectSubPropertyOf) {
			DirectSubPropertyOf dspo = (DirectSubPropertyOf)pathExpr;

			from.append(", " + DIRECT_SUBPROPERTYOF_TABLE + " " + peId);

			_processPathExpressionVar(dspo.getSubjectVar(), peId+".sub", where, varToSqlIdMapping);
			_processPathExpressionVar(dspo.getObjectVar(), peId+".super", where, varToSqlIdMapping);

			return null;
		}
		else {
			if (pathExpr instanceof TriplePattern) {
				TriplePattern tp = (TriplePattern)pathExpr;
				Value predicate = tp.getPredicateVar().getValue();

				if (URIImpl.RDF_TYPE.equals(predicate)) {
					// Use INSTANCEOF_TABLE
					from.append(", " + INSTANCEOF_TABLE + " " + peId);

					_processPathExpressionVar(tp.getSubjectVar(), peId+".inst", where, varToSqlIdMapping);
					_processPathExpressionVar(tp.getObjectVar(), peId+".class", where, varToSqlIdMapping);

					return null;
				}
				else if (URIImpl.RDFS_SUBCLASSOF.equals(predicate)) {
					// Use SUBCLASSOF_TABLE
					from.append(", " + SUBCLASSOF_TABLE + " " + peId);

					_processPathExpressionVar(tp.getSubjectVar(), peId+".sub", where, varToSqlIdMapping);
					_processPathExpressionVar(tp.getObjectVar(), peId+".super", where, varToSqlIdMapping);

					return null;
				}
				else if (URIImpl.RDFS_SUBPROPERTYOF.equals(predicate)) {
					// Use SUBPROPERTYOF_TABLE
					from.append(", " + SUBPROPERTYOF_TABLE + " " + peId);

					_processPathExpressionVar(tp.getSubjectVar(), peId+".sub", where, varToSqlIdMapping);
					_processPathExpressionVar(tp.getObjectVar(), peId+".super", where, varToSqlIdMapping);

					return null;
				}
			}

			return super._convertPathExpressionToSQL(pathExpr, peId, from, where, varToSqlIdMapping);
		}
	}

	public RDBMS getRDBMS() {
		return _rdbms;
	}
}
