/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sailimpl.rdbms;

import org.openrdf.sesame.config.SailConfig;

/**
 * Extension of SailConfig offering specific utility methods for configuring an
 * <tt>org.openrdf.sesame.sailimpl.rdbms.RdfRepository</tt> Sail. This class
 * offers methods for specifying the JDBC-driver class, the JDBC URL and the
 * username/password combination for the RDBMS.
 **/
public class RdfRepositoryConfig extends SailConfig {

/*--------------+
| Constructors  |
+--------------*/

	/**
	 * Creates a new and unconfigured RdfRepositoryConfig object for the
	 * specified Sail class. This is a protected constructor that can be used
	 * by subclasses to specified a different Sail class.
	 *
	 * @param className A fully qualified Sail class name.
	 **/
	protected RdfRepositoryConfig(String className) {
		super(className);
	}

	/**
	 * Creates a new and unconfigured RdfRepositoryConfig object for an
	 * <tt>org.openrdf.sesame.sailimpl.rdbms.RdfRepository</tt> Sail.
	 **/
	public RdfRepositoryConfig() {
		this("org.openrdf.sesame.sailimpl.rdbms.RdfRepository");
	}

	/**
	 * Creates a new RdfRepositoryConfig object for an
	 * <tt>org.openrdf.sesame.sailimpl.rdbms.RdfRepository</tt> Sail that uses the
	 * specified JDBC Driver to connect to the specified database.
	 *
	 * @param jdbcDriver The fully qualified name of the Driver class, e.g.
	 * <tt>com.mysql.jdbc.Driver</tt> or <tt>org.postgresql.Driver</tt>.
	 * @param jdbcUrl A JDBC URL specifying the database to connect to, e.g.
	 * <tt>jdbc:mysql://localhost:3306/testdb</tt> or
	 * <tt>jdbc:postgresql://localhost:5432/testdb</tt>.
	 **/
	public RdfRepositoryConfig(String jdbcDriver, String jdbcUrl) {
		this();

		setJdbcDriver(jdbcDriver);
		setJdbcUrl(jdbcUrl);
	}

	/**
	 * Creates a new RdfRepositoryConfig object for an
	 * <tt>org.openrdf.sesame.sailimpl.rdbms.RdfRepository</tt> Sail that uses the
	 * specified JDBC Driver to connect to the specified database. The specified
	 * username/password combination is used to get access to the database.
	 *
	 * @param jdbcDriver The fully qualified name of the Driver class, e.g.
	 * <tt>com.mysql.jdbc.Driver</tt> or <tt>org.postgresql.Driver</tt>.
	 * @param jdbcUrl A JDBC URL specifying the database to connect to, e.g.
	 * <tt>jdbc:mysql://localhost:3306/testdb</tt> or
	 * <tt>jdbc:postgresql://localhost:5432/testdb</tt>.
	 * @param user The name of a user that has access to the database.
	 * @param password The user's password for the database.
	 **/
	public RdfRepositoryConfig(String jdbcDriver, String jdbcUrl, String user, String password) {
		this(jdbcDriver, jdbcUrl);

		setUser(user);
		setPassword(password);
	}

/*--------------+
| Methods       |
+--------------*/

	/**
	 * Sets the JDBC Driver class that should be used to connect to the
	 * database.
	 *
	 * @param jdbcDriver The fully qualified name of the Driver class, e.g.
	 * <tt>com.mysql.jdbc.Driver</tt> or <tt>org.postgresql.Driver</tt>.
	 **/
	public void setJdbcDriver(String jdbcDriver) {
		setParameter(RdfSource.JDBC_DRIVER_KEY, jdbcDriver);
	}

	/**
	 * Gets the fully qualified class name of the used JDBC driver.
	 **/
	public String getJdbcDriver() {
		return getParameter(RdfSource.JDBC_DRIVER_KEY);
	}

	/**
	 * Sets the JDBC URL that identifies the database to connect to.
	 *
	 * @param jdbcUrl A JDBC URL specifying the database to connect to, e.g.
	 * <tt>jdbc:mysql://localhost:3306/testdb</tt> or
	 * <tt>jdbc:postgresql://localhost:5432/testdb</tt>.
	 **/
	public void setJdbcUrl(String jdbcUrl) {
		setParameter(RdfSource.JDBC_URL_KEY, jdbcUrl);
	}

	/**
	 * Gets the JDBC URL that identifies the database to connect to.
	 **/
	public String getJdbcUrl() {
		return getParameter(RdfSource.JDBC_URL_KEY);
	}

	/**
	 * Sets the username that should be used to log in on the database.
	 *
	 * @param user The name of a user that has access to the database.
	 **/
	public void setUser(String user) {
		setParameter(RdfSource.USER_KEY, user);
	}

	/**
	 * Gets the username that is used to log in on the database.
	 **/
	public String getUser() {
		return getParameter(RdfSource.USER_KEY);
	}

	/**
	 * Sets the password that should be used to log in on the database.
	 *
	 * @param password The user's password for the database.
	 **/
	public void setPassword(String password) {
		setParameter(RdfSource.PASSWORD_KEY, password);
	}

	/**
	 * Gets the password that is used to log in on the database.
	 **/
	public String getPassword() {
		return getParameter(RdfSource.PASSWORD_KEY);
	}
}
