/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sailimpl.rdbms;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.openrdf.util.log.ThreadLog;

import org.openrdf.model.BNode;
import org.openrdf.model.Literal;
import org.openrdf.model.Resource;
import org.openrdf.model.URI;
import org.openrdf.model.Value;

import org.openrdf.sesame.sail.Namespace;
import org.openrdf.sesame.sail.SailChangedListener;
import org.openrdf.sesame.sail.SailInitializationException;
import org.openrdf.sesame.sail.SailInternalException;
import org.openrdf.sesame.sail.SailUpdateException;
import org.openrdf.sesame.sail.util.SailChangedEventImpl;

/**
 * A portable implementation of the RdfRepository interface for relational
 * databases, based on (a subset of) the SQL2/SQL92 standard. This class
 * defines the update methods. The superclass RdfSource defines the
 * read-only methods.
 *
 * @author Arjohn Kampman
 **/
public class RdfRepository extends RdfSource implements
		org.openrdf.sesame.sail.RdfRepository, TableNames
{

	/*----------+
	 | Variables |
	 +----------*/

	protected List _sailChangedListeners;

	protected SailChangedEventImpl _sailChangedEvent;

	/** 
	 * Key used to specify how often to commit during transaction, in number of triples.
	 */
	protected static final String COMMIT_INTERVAL_KEY = "commitInterval";

	/**
	 * The ID for the next resource that will be added. This variable starts with
	 * value 1 and is increased everytime a resource is added.
	 **/
	protected int _nextResourceId;

	/**
	 * The ID for the next literal that will be added. This variable starts with
	 * value -1 and is decreased everytime a literal is added.
	 **/
	protected int _nextLiteralId;

	/**
	 * The ID for the next statement that will be added. This variable starts with
	 * value 1 and is increased everytime a statement is added.
	 **/
	protected int _nextStatementId;

	/** Flag indicating whether a transaction is currently being performed. **/
	protected boolean _transactionStarted;

	/** Flag indicating whether statements were added during a transaction. **/
	protected boolean _statementsAdded;

	/** Flag indicating whether statements were removed during a transaction. **/
	protected boolean _statementsRemoved;

	/**
	 * The number of triples added via {@link#addStatement(Resource, URI, Value}
	 * at which the database transaction will be commited. The user can specify 
	 * this value by setting the <code>commitInterval</code> parameter in the 
	 * repository configuration.  Defaults to 1000 if unspecified.
	 * @see #COMMIT_INTERVAL_KEY
	 * @see #addStatement(Resource,URI,Value)
	 */
	protected int _triplesCommitInterval = 1000;

	// Connection and PreparedStatement used during a transaction.
	protected Connection _addRawTriplesConn;

	protected PreparedStatement _addRawTriplesSt;

	protected int _rawTriplesCount;

	/*----------------------------------------+
	 | Constructors                            |
	 +----------------------------------------*/

	public RdfRepository() {
		super();

		_nextResourceId = 1;
		_nextLiteralId = -1;
		_nextStatementId = 1;

		_sailChangedListeners = new ArrayList(0);

		_transactionStarted = false;
	}

	/*----------------------------------------+
	 | overriden methods from RdfSource        |
	 +----------------------------------------*/

	/**
	 * @see org.openrdf.sesame.sail.Sail#initialize(java.util.Map)
	 */
	public void initialize(Map configParams)
		throws SailInitializationException
	{
		String commitIntervalStr = (String)configParams.get(COMMIT_INTERVAL_KEY);

		if (commitIntervalStr != null) {
			try {
				_triplesCommitInterval = Integer.parseInt(commitIntervalStr);
			}
			catch (NumberFormatException e) {
				throw new SailInitializationException(e);
			}
		}

		super.initialize(configParams);
	}

	/*------------------------+
	 | Database initialization |
	 +------------------------*/

	/**
	 * Initializes the database. _initDatabase() creates tables, indexes and
	 * inserts default values into the database.
	 **/
	protected void _initDatabase()
		throws SailInitializationException
	{
		super._initDatabase();

		try {
			// Initialize _nextResourceId, _nextLiteralId and _nextStatementId;
			_setNextResourceId();
			_setNextLiteralId();
			_setNextStatementId();
		}
		catch (SQLException e) {
			throw new SailInitializationException(e);
		}
	}

	protected void _createDbSchema()
		throws SQLException
	{
		super._createDbSchema();

		// Tables used to add new statements:
		_createAddedTriplesTable();
		_createNewTriplesTable();
		_createRawTriplesTable();
		_createExpiredTriplesTable();
		_createExpiredResourcesTable();
		_createExpiredLiteralsTable();
	}

	protected void _createAddedTriplesTable()
		throws SQLException
	{
		if (_schemaVersion == -1) {
			_createTriplesTable(ADDED_TRIPLES_TABLE, false);

			// Add an index over (subj, pred, obj)
			_rdbms.createIndex(ADDED_TRIPLES_TABLE,
				new String[] {"subj", "pred", "obj" }, false);
		}
		else if (_schemaVersion < 4) {
			_rdbms.renameTableColumn(ADDED_TRIPLES_TABLE,
				"subject", "subj", _rdbms.ID_INT + " NOT NULL");
			_rdbms.renameTableColumn(ADDED_TRIPLES_TABLE,
				"predicate", "pred", _rdbms.ID_INT + " NOT NULL");
			_rdbms.renameTableColumn(ADDED_TRIPLES_TABLE,
				"object", "obj", _rdbms.ID_INT + " NOT NULL");
		}
	}

	protected void _createNewTriplesTable()
		throws SQLException
	{
		if (_schemaVersion == -1) {
			_createTriplesTable(NEW_TRIPLES_TABLE, false);
		}
		else if (_schemaVersion < 4) {
			_rdbms.renameTableColumn(NEW_TRIPLES_TABLE,
				"subject", "subj", _rdbms.ID_INT + " NOT NULL");
			_rdbms.renameTableColumn(NEW_TRIPLES_TABLE,
				"predicate", "pred", _rdbms.ID_INT + " NOT NULL");
			_rdbms.renameTableColumn(NEW_TRIPLES_TABLE,
				"object", "obj", _rdbms.ID_INT + " NOT NULL");
		}
	}

	protected void _createRawTriplesTable()
		throws SQLException
	{
		if (_schemaVersion < 2) {
			// Old or non-existent table

			if (_schemaVersion != -1) {
				// Drop old table
				_rdbms.dropTable(RAW_TRIPLES_TABLE);
			}

			_rdbms.executeUpdate(
				"CREATE TABLE " + RAW_TRIPLES_TABLE + " (" +
					// statement ID
					"id " + _rdbms.ID_INT + " NOT NULL, " +
					// subject
					"subjNs " + _rdbms.ID_INT + " NOT NULL, " +
					"subjLname " + _rdbms.LOCALNAME + NN_ON_TEXT + ", " +
					// predicate
					"predNs " + _rdbms.ID_INT + " NOT NULL, " +
					"predLname " + _rdbms.LOCALNAME + NN_ON_TEXT + ", " +
					// object (resource, or datatype of literal)
					"objNs " + _rdbms.ID_INT + " NOT NULL, " +
					"objLname " + _rdbms.LOCALNAME + NN_ON_TEXT + ", " +
					// object (literal)
					"objLabelHash " + _rdbms.LABEL_HASH + ", " +
					"objLang " + _rdbms.LANGUAGE + ", " +
					"objLabel " + _rdbms.LABEL + ", " +
					"objIsLiteral " + _rdbms.BOOLEAN + ")");
		}
	}

	protected void _createExpiredTriplesTable()
		throws SQLException
	{
		if (_schemaVersion == -1) {
			_rdbms.executeUpdate(
				"CREATE TABLE " + EXPIRED_TRIPLES_TABLE +
				" (id " + _rdbms.ID_INT + " NOT NULL)");
		}
		else if (_schemaVersion < 5) {
			// EXPIRED_VALUES_TABLE has been replaced by two new tables:
			// EXPIRED_RESOURCES_TABLE and EXPIRED_LITERALS_TABLE
			_rdbms.dropTable(EXPIRED_VALUES_TABLE);
		}
	}

	protected void _createExpiredResourcesTable()
		throws SQLException
	{
		if (_schemaVersion < 5) {
			_rdbms.executeUpdate(
				"CREATE TABLE " + EXPIRED_RESOURCES_TABLE +
				" (id " + _rdbms.ID_INT + " NOT NULL)");
		}
	}

	protected void _createExpiredLiteralsTable()
		throws SQLException
	{
		if (_schemaVersion < 5) {
			_rdbms.executeUpdate(
				"CREATE TABLE " + EXPIRED_LITERALS_TABLE +
				" (id " + _rdbms.ID_INT + " NOT NULL)");
		}
	}

	/*----------------------------------------+
	 | Initialization of local variables       |
	 +----------------------------------------*/

	protected void _setNextResourceId()
		throws SQLException
	{
		Connection con = _rdbms.getConnection();
		java.sql.Statement st = con.createStatement();
		ResultSet rs = st.executeQuery("SELECT MAX(id) FROM " + RESOURCES_TABLE);

		if (rs.next()) {
			// next resource ID is maximum + 1
			_nextResourceId = rs.getInt(1) + 1;
		}
		else {
			// start with ID 1
			_nextResourceId = 1;
		}

		rs.close();
		st.close();
		con.close();
	}

	protected int _getNextResourceId() {
		return _nextResourceId++;
	}

	protected void _setNextLiteralId()
		throws SQLException
	{
		Connection con = _rdbms.getConnection();
		java.sql.Statement st = con.createStatement();
		ResultSet rs = st.executeQuery("SELECT MIN(id) FROM " + LITERALS_TABLE);

		if (rs.next()) {
			// next literal ID is minimum - 1
			_nextLiteralId = rs.getInt(1) - 1;
		}
		else {
			// start with ID -1
			_nextLiteralId = -1;
		}

		rs.close();
		st.close();
		con.close();
	}

	protected int _getNextLiteralId() {
		return _nextLiteralId--;
	}

	protected void _setNextStatementId()
		throws SQLException
	{
		Connection con = _rdbms.getConnection();
		java.sql.Statement st = con.createStatement();
		ResultSet rs = st.executeQuery("SELECT MAX(id) FROM " + TRIPLES_TABLE);

		if (rs.next()) {
			// next statement ID is maximum + 1
			_nextStatementId = rs.getInt(1) + 1;
		}
		else {
			// start with ID 1
			_nextStatementId = 1;
		}

		rs.close();
		st.close();
		con.close();
	}

	protected int _getNextStatementId() {
		return _nextStatementId++;
	}

	/*----------------------------------------+
	 | methods from RdfRepository              |
	 +----------------------------------------*/

	protected void _prepareUploadConnection()
		throws SQLException
	{
		_addRawTriplesConn = _rdbms.getConnection();
		_addRawTriplesConn.setAutoCommit(false);
		_addRawTriplesSt = _addRawTriplesConn.prepareStatement(
			"INSERT INTO " + RAW_TRIPLES_TABLE +
			" VALUES(?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
	}

	protected void _closeUploadConnection()
		throws SQLException
	{
		if (_rawTriplesCount > 0) {
			_addRawTriplesConn.commit();
			_rawTriplesCount = 0;
		}
		_addRawTriplesSt.close();
		_addRawTriplesConn.close();
	}

	public void startTransaction() {
		if (_transactionStarted) {
			throw new SailInternalException(
					"A transaction has already been started");
		}

		try {
			_prepareUploadConnection();

			_rawTriplesCount = 0;

			_transactionStarted = true;
			_sailChangedEvent = new SailChangedEventImpl();
		}
		catch (SQLException e) {
			throw new SailInternalException(e);
		}
	}

	public boolean transactionStarted() {
		return _transactionStarted;
	}

	public void addStatement(Resource subj, URI pred, Value obj)
		throws SailUpdateException
	{
		if (!_transactionStarted) {
			throw new SailUpdateException("No transaction was started");
		}

		try {
			// statement ID
			_addRawTriplesSt.setInt(1, _getNextStatementId());

			int nsId;

			// subject
			if (subj instanceof BNode) {
				BNode bNode = (BNode)subj;
				String nodeId = bNode.getID();

				_addRawTriplesSt.setInt(2, 0); // subject namespace
				_addRawTriplesSt.setString(3, nodeId);
			}
			else {
				URI uri = (URI)subj;
				nsId = _createIdForNamespace(uri.getNamespace());
				_addRawTriplesSt.setInt(2, nsId);
				_addRawTriplesSt.setString(3, uri.getLocalName());
			}

			// predicate
			nsId = _createIdForNamespace(pred.getNamespace());
			_addRawTriplesSt.setInt(4, nsId);
			_addRawTriplesSt.setString(5, pred.getLocalName());

			// object
			if (obj instanceof Resource) {
				// Resource columns
				if (obj instanceof BNode) {
					BNode bNode = (BNode)obj;
					_addRawTriplesSt.setInt(6, 0); // object namespace
					_addRawTriplesSt.setString(7, bNode.getID());
				}
				else {
					URI uri = (URI)obj;
					nsId = _createIdForNamespace(uri.getNamespace());
					_addRawTriplesSt.setInt(6, nsId);
					_addRawTriplesSt.setString(7, uri.getLocalName());
				}

				// Literal columns
				_addRawTriplesSt.setLong(8, 0L); // labelHash
				_addRawTriplesSt.setNull(9, _rdbms.LANGUAGE_TYPE); // lang
				_addRawTriplesSt.setNull(10, _rdbms.LABEL_TYPE); // label

				_addRawTriplesSt.setBoolean(11, false); // obj is not a literal
			}
			else if (obj instanceof Literal) {
				// Literal columns
				Literal objLit = (Literal)obj;

				String label = objLit.getLabel();
				String lang = objLit.getLanguage();
				URI datatype = objLit.getDatatype();

				// datatype
				if (datatype != null) {
					// Datatype should always be a URI
					nsId = _createIdForNamespace(datatype.getNamespace());
					_addRawTriplesSt.setInt(6, nsId);
					_addRawTriplesSt.setString(7, datatype.getLocalName());
				}
				else {
					_addRawTriplesSt.setInt(6, 0); // nsId
					_addRawTriplesSt.setString(7, ""); // lname
				}

				// labelHash
				_addRawTriplesSt.setLong(8, _getLabelHash(label));

				// lang. Language does not apply to literals with a datatype.
				if (lang != null && datatype == null) {
					_addRawTriplesSt.setString(9, lang);
				}
				else {
					_addRawTriplesSt.setNull(9, _rdbms.LANGUAGE_TYPE);
				}

				// Label
				_addRawTriplesSt.setString(10, label);

				_addRawTriplesSt.setBoolean(11, true); // obj is a literal
			}

			_addRawTriplesSt.executeUpdate();
		}
		catch (SQLException e) {
			throw new SailInternalException(e);
		}

		_statementsAdded = true;
		_rawTriplesCount++;

		if (_rawTriplesCount >= _triplesCommitInterval) {
			try {
				// Commit the transaction to the rawTriples table
				_closeUploadConnection();

				// Process the raw triples
				_processRawTriples();

				// Start a new transaction to the rawTriples table
				_prepareUploadConnection();
			}
			catch (SQLException e) {
				throw new SailInternalException(e);
			}
		}
	}

	public int removeStatements(Resource subj, URI pred, Value obj)
		throws SailUpdateException
	{
		if (!_transactionStarted) {
			throw new SailUpdateException("No transaction started");
		}

		// Register statements that should be removed in the expired triples
		// table. Note: only explicit statements can be removed.
		String query =
			"INSERT INTO " + EXPIRED_TRIPLES_TABLE +
			" SELECT id FROM " + TRIPLES_TABLE +
			" WHERE explicit = " + _rdbms.TRUE;

		// Subject
		if (subj != null) {
			int subjId = _getResourceId(subj);
			if (subjId == 0) {
				// subject not found, so no matching statements
				return 0;
			}

			query += " AND subj = " + subjId;
		}

		// Predicate
		if (pred != null) {
			int predId = _getURIId(pred);
			if (predId == 0) {
				// predicate not found, so no matching statements
				return 0;
			}

			query += " AND pred = " + predId;
		}

		// Object
		if (obj != null) {
			int objId = _getValueId(obj);
			if (objId == 0) {
				// object not found, so no matching statements
				return 0;
			}

			query += " AND obj = " + objId;
		}

		try {
			int count = _rdbms.executeUpdate(query);
			if (count > 0) {
				_rdbms.optimizeTable(EXPIRED_TRIPLES_TABLE, count);
				_statementsRemoved = true;
			}
			return count;
		}
		catch (SQLException e) {
			throw new SailInternalException(e);
		}
	}

	public void commitTransaction() {
		try {
			// Commit the transaction to the rawTriples table
			_closeUploadConnection();

			if (_statementsRemoved) {
				_sailChangedEvent.setStatementsRemoved(true);
				_recordExpiredValues();
				_removeExpiredStatements();
				_removeExpiredValues();
			}

			if (_statementsAdded) {
				_sailChangedEvent.setStatementsAdded(true);
				// Process the raw triples and add them to ADDED_TRIPLES_TABLE
				_processRawTriples();

				_rdbms.optimizeTable(ADDED_TRIPLES_TABLE);

				_removeDuplicatesFromAddedTriples();

				_processAddedTriples();

				// Add these new triples to the triples table
				int newTriplesCount = _rdbms.copyRows(NEW_TRIPLES_TABLE,
						TRIPLES_TABLE);
				if (newTriplesCount == 0) {
					_rdbms.optimizeTable(TRIPLES_TABLE, newTriplesCount);
					_statementsAdded = false;
				}
			}

			if (_statementsAdded || _statementsRemoved) {

				// Hook for subclasses to do 'stuff' as a result of the added or
				// removed statements.
				_processChangedTriples();

				_rdbms.clearTable(NEW_TRIPLES_TABLE);
				_rdbms.clearTable(EXPIRED_TRIPLES_TABLE);

				// Set the export status to 'dirty'
				_setExportStatusUpToDate(false);

				_statementsAdded = false;
				_statementsRemoved = false;

			}
			_notifySailChanged(_sailChangedEvent);
		}
		catch (SQLException e) {
			throw new SailInternalException(e);
		}
		finally {
			_sailChangedEvent = null;
			_transactionStarted = false;
		}
	}

	/**
	 * Records which values (resources as well as literals) can potentially be
	 * removed because they are referenced by expired statements. This method
	 * uses the <tt>expiredTriples</tt> table to determine which statements are
	 * expired and stores any referenced resources and literals in the
	 * <tt>expiredResources</tt> and <tt>expiredLiterals</tt> tables,
	 * respectively. This method should be called before the expired triples
	 * have actually been removed.
	 *
	 * @see #_removeExpiredValues
	 **/
	protected void _recordExpiredValues()
		throws SQLException
	{
		// Resources
		_rdbms.executeUpdate(
			"INSERT INTO " + EXPIRED_RESOURCES_TABLE +
			" SELECT t.subj FROM " +
				TRIPLES_TABLE + " t, " +
				EXPIRED_TRIPLES_TABLE + " expt" +
			" WHERE t.id = expt.id");
		_rdbms.executeUpdate(
			"INSERT INTO " + EXPIRED_RESOURCES_TABLE +
			" SELECT t.pred FROM " +
				TRIPLES_TABLE + " t, " +
				EXPIRED_TRIPLES_TABLE + " expt" +
			" WHERE t.id = expt.id");
		_rdbms.executeUpdate(
			"INSERT INTO " + EXPIRED_RESOURCES_TABLE +
			" SELECT t.obj FROM " +
				TRIPLES_TABLE + " t, " +
				EXPIRED_TRIPLES_TABLE + " expt" +
			" WHERE t.id = expt.id AND t.obj > 0");

		// Literals
		_rdbms.executeUpdate(
			"INSERT INTO " + EXPIRED_LITERALS_TABLE +
			" SELECT t.obj FROM " +
				TRIPLES_TABLE + " t, " +
				EXPIRED_TRIPLES_TABLE + " expt" +
			" WHERE t.id = expt.id AND t.obj < 0");
	}

	/**
	 * Removes any values that are no longer referenced by triples. This method
	 * uses the value IDs recorded in the tables <tt>expiredResources</tt> and
	 * <tt>expiredLiterals</tt> to determine which values are candidates for
	 * removal.
	 **/
	protected void _removeExpiredValues()
		throws SQLException
	{
		// Delete the expired resources
		Connection con = _rdbms.getConnection();
		java.sql.Statement st = con.createStatement();
		ResultSet rs = st.executeQuery(
			"SELECT DISTINCT expr.id" +
			" FROM " + EXPIRED_RESOURCES_TABLE + " expr " +
			" LEFT JOIN " + TRIPLES_TABLE + " t " +
			" ON expr.id = t.subj OR expr.id = t.pred OR expr.id = t.obj " + // used in a triple
			" LEFT JOIN " + LITERALS_TABLE + " l " +
			" ON expr.id = l.datatype " + // used as datatype
			" WHERE t.id IS NULL AND l.id IS NULL");

		String[] idChunks = _chunkIdSet(rs, 3500);
		rs.close();
		st.close();
		con.close();

		_rdbms.clearTable(EXPIRED_RESOURCES_TABLE);

		if (idChunks.length > 0) {
			// Remove the selected resources
			con = _rdbms.getConnection();
			con.setAutoCommit(false);
			st = con.createStatement();

			for (int i = 0; i < idChunks.length; i++) {
				st.executeUpdate(
					"DELETE FROM " + RESOURCES_TABLE +
					" WHERE id IN " + idChunks[i]);
			}
			con.commit();
			st.close();
			con.close();

			_rdbms.optimizeTable(RESOURCES_TABLE);
		}

		// Delete the expired literals
		con = _rdbms.getConnection();
		st = con.createStatement();
		rs = st.executeQuery(
			"SELECT expl.id FROM " + EXPIRED_LITERALS_TABLE + " expl" +
			" LEFT JOIN " + TRIPLES_TABLE + " t" +
			" ON expl.id = t.obj" +
			" WHERE t.id IS NULL");

		idChunks = _chunkIdSet(rs, 3500);
		rs.close();
		st.close();
		con.close();

		_rdbms.clearTable(EXPIRED_LITERALS_TABLE);

		if (idChunks.length > 0) {
			// Remove the selected literals
			con = _rdbms.getConnection();
			con.setAutoCommit(false);
			st = con.createStatement();

			for (int i = 0; i < idChunks.length; i++) {
				st.executeUpdate("DELETE FROM " + LITERALS_TABLE + " WHERE id IN "
						+ idChunks[i]);
			}
			con.commit();
			st.close();
			con.close();

			_rdbms.optimizeTable(LITERALS_TABLE);
		}
	}

	/**
	 * Removes the statements that are registered in EXPIRED_TRIPLES_TABLE and
	 * any associated values that are no longer referenced as a result of the
	 * removal.
	 **/
	protected void _removeExpiredStatements()
		throws SQLException
	{
		// Delete the expired statements
		Connection con = _rdbms.getConnection();
		java.sql.Statement st = con.createStatement();
		ResultSet rs = st.executeQuery(
			"SELECT DISTINCT id FROM " + EXPIRED_TRIPLES_TABLE);

		String[] idChunks = _chunkIdSet(rs, 3500);
		rs.close();
		st.close();

		con.setAutoCommit(false);
		st = con.createStatement();

		int count = 0;
		for (int i = 0; i < idChunks.length; i++) {
			count += st.executeUpdate(
				"DELETE FROM " + TRIPLES_TABLE +
				" WHERE id IN " + idChunks[i]);
			_processChunkFromRemoveExpiredStatements(idChunks[i]);
		}

		con.commit();
		st.close();
		con.close();

		_rdbms.optimizeTable(TRIPLES_TABLE, count);
	}

	protected void _processChunkFromRemoveExpiredStatements(String idList)
		throws SQLException
	{
		// do nothing, method is overriden by subclasses
	}

	protected void _processRawTriples()
		throws SQLException
	{
		_rdbms.optimizeTable(RAW_TRIPLES_TABLE);

		// RAW_TRIPLES_TABLE can contain new resources and literals. These
		// should be added to their respective tables first.
		_addNewResources();
		_addNewLiterals();

		// Now convert the resources and literals to IDs and copy the
		// rows to ADDED_TRIPLES_TABLE.
		_copyRawTriplesToAddedTriples();

		_rdbms.clearTable(RAW_TRIPLES_TABLE);
	}

	/**
	 * Adds resources from the RAW_TRIPLES_TABLE that are not yet in
	 * RESOURCES_TABLE, to this last table.
	 **/
	protected void _addNewResources()
		throws SQLException
	{
		Connection addResourceCon = _rdbms.getConnection();
		addResourceCon.setAutoCommit(false);
		PreparedStatement addResourceSt = addResourceCon.prepareStatement(
			"INSERT INTO " + RESOURCES_TABLE + " VALUES(?, ?, ?)");

		Connection con = _rdbms.getConnection();
		java.sql.Statement st = con.createStatement();

		// SUBJECT
		String localnamesAreNullOr = _rdbms.emptyStringIsNull() ?
			"rt.subjLname IS NULL AND r.localname IS NULL OR " : "";
		String query =
			"SELECT DISTINCT rt.subjNs, rt.subjLname " +
			"FROM " + RAW_TRIPLES_TABLE + " rt " +
			"LEFT JOIN " + RESOURCES_TABLE + " r " +
			"ON rt.subjNs = r.namespace AND (" +
				localnamesAreNullOr	+ "rt.subjLname = r.localname) " +
			"WHERE r.namespace IS NULL";

		_processNewResources(st.executeQuery(query),
			addResourceCon, addResourceSt);

		// PREDICATE
		localnamesAreNullOr = _rdbms.emptyStringIsNull() ?
			"rt.predLname IS NULL AND r.localname IS NULL OR " : "";
		query =
			"SELECT DISTINCT rt.predNs, rt.predLname " +
			"FROM " + RAW_TRIPLES_TABLE + " rt " +
			"LEFT JOIN " + RESOURCES_TABLE + " r " +
			"ON rt.predNs = r.namespace AND (" +
				localnamesAreNullOr + "rt.predLname = r.localname) " +
			"WHERE r.namespace IS NULL";

		_processNewResources(st.executeQuery(query), addResourceCon,
				addResourceSt);

		// OBJECT
		localnamesAreNullOr = _rdbms.emptyStringIsNull() ?
			"rt.objLname IS NULL AND r.localname IS NULL OR " : "";
		query =
			"SELECT DISTINCT rt.objNs, rt.objLname " +
			"FROM " + RAW_TRIPLES_TABLE + " rt " +
			"LEFT JOIN " + RESOURCES_TABLE + " r " +
			"ON rt.objNs = r.namespace AND (" +
				localnamesAreNullOr + "rt.objLname = r.localname) " +
			"WHERE r.namespace IS NULL";

		_processNewResources(st.executeQuery(query),
				addResourceCon, addResourceSt);

		st.close();
		con.close();

		addResourceSt.close();
		addResourceCon.close();
	}

	protected void _processNewResources(ResultSet rs, Connection addResourceCon,
			PreparedStatement addResourceSt)
		throws SQLException
	{
		int newCount = 0;

		while (rs.next()) {
			int ns = rs.getInt(1);
			String lname = rs.getString(2);

			if (lname == null) {
				lname = "";
			}

			int resId = _getNextResourceId();
			addResourceSt.setInt(1, resId);
			addResourceSt.setInt(2, ns);
			addResourceSt.setString(3, lname);

			addResourceSt.executeUpdate();

			newCount++;
		}
		rs.close();

		if (newCount > 0) {
			addResourceCon.commit();
			_rdbms.optimizeTable(RESOURCES_TABLE, newCount);
		}
	}

	protected void _addNewLiterals()
		throws SQLException
	{
		Connection addLiteralCon = _rdbms.getConnection();
		addLiteralCon.setAutoCommit(false);
		PreparedStatement addLiteralSt = addLiteralCon.prepareStatement(
			"INSERT INTO " + LITERALS_TABLE +
			" VALUES(?, ?, ?, ?, ?)");

		String localnamesAreNullOr = _rdbms.emptyStringIsNull() ?
			"rt.objLname IS NULL AND dt.localname IS NULL OR " : "";
		String labelsAreNullOr = _rdbms.emptyStringIsNull() ?
			"rt.objLabel IS NULL AND l.label IS NULL OR " : "";

		// Get all new literals
		String query =
			"SELECT DISTINCT dt.id, rt.objLang, rt.objLabelHash, rt.objLabel " +
			"FROM " + RAW_TRIPLES_TABLE + " rt " +
			"LEFT JOIN " + RESOURCES_TABLE + " dt ON " + // join with resources table for datatype
				"rt.objNs = dt.namespace AND ("	+
					localnamesAreNullOr + "rt.objLname = dt.localname) " +
			"LEFT JOIN " + LITERALS_TABLE + " l ON " +
				"rt.objLabelHash = l.labelHash " + // equal label hash
				"AND dt.id = l.datatype " + // equal datatype
				"AND (rt.objLang IS NULL AND l.language IS NULL OR rt.objLang = l.language) " + // equal language
				"AND (" + labelsAreNullOr + "rt.objLabel = l.label) " + // equal label
				"WHERE " + "rt.objIsLiteral = " + _rdbms.TRUE + " AND " + // row contains a literal
				"l.id IS NULL"; // no such literal exists yet

		Connection con = _rdbms.getConnection();
		java.sql.Statement st = con.createStatement();
		ResultSet rs = st.executeQuery(query);

		int newCount = 0;

		while (rs.next()) {
			// will return 0 in case dt.id is NULL, which is fine for us:
			int datatypeId = rs.getInt(1);
			String lang = rs.getString(2);
			long labelHash = rs.getLong(3);
			String label = rs.getString(4);

			if (label == null) {
				label = "";
			}

			addLiteralSt.setInt(1, _getNextLiteralId());
			addLiteralSt.setInt(2, datatypeId);
			addLiteralSt.setLong(3, labelHash);
			if (lang != null) {
				addLiteralSt.setString(4, lang);
			}
			else {
				addLiteralSt.setNull(4, _rdbms.LANGUAGE_TYPE);
			}
			addLiteralSt.setString(5, label);

			addLiteralSt.executeUpdate();

			newCount++;
		}
		rs.close();
		st.close();
		con.close();

		if (newCount > 0) {
			addLiteralCon.commit();
			_rdbms.optimizeTable(LITERALS_TABLE, newCount);
		}

		addLiteralSt.close();
		addLiteralCon.close();
	}

	/**
	 * Converts the resources and literals from the RAW_TRIPLES_TABLE to
	 * IDs and copies these rows to the ADDED_TRIPLES_TABLE.
	 **/
	protected void _copyRawTriplesToAddedTriples()
		throws SQLException
	{
		String subjLnamesAreNullOr = _rdbms.emptyStringIsNull() ?
			"rt.subjLname IS NULL AND r1.localname IS NULL OR " : "";
		String predLnamesAreNullOr = _rdbms.emptyStringIsNull() ?
			"rt.predLname IS NULL AND r2.localname IS NULL OR " : "";
		String objLnamesAreNullOr = _rdbms.emptyStringIsNull() ?
			"rt.objLname IS NULL AND r3.localname IS NULL OR " : "";
		String labelsAreNullOr = _rdbms.emptyStringIsNull() ?
			"rt.objLabel IS NULL AND l.label IS NULL OR " : "";

		// Case where object is a resource:
		_rdbms.executeUpdate(
			"INSERT INTO " + ADDED_TRIPLES_TABLE + " " +
			"SELECT rt.id, r1.id, r2.id, r3.id, " + _rdbms.TRUE + " " +
			"FROM " +
				RAW_TRIPLES_TABLE + " rt, " +
				RESOURCES_TABLE + " r1, " +
				RESOURCES_TABLE + " r2, " +
				RESOURCES_TABLE + " r3 " +
			"WHERE " +
				"rt.objIsLiteral = " + _rdbms.FALSE + " AND " + // obj must be a resource

				"rt.subjNs = r1.namespace AND " +
				"(" + subjLnamesAreNullOr	+ "rt.subjLname = r1.localname) AND " +

				"rt.predNs = r2.namespace AND " +
				"(" + predLnamesAreNullOr + "rt.predLname = r2.localname) AND " +

				"rt.objNs = r3.namespace AND " +
				"(" + objLnamesAreNullOr + "rt.objLname = r3.localname)");

		// Case where object is a literal:
		_rdbms.executeUpdate(
			"INSERT INTO " + ADDED_TRIPLES_TABLE + " " +
			"SELECT rt.id, r1.id, r2.id, l.id, " + _rdbms.TRUE + " " +
			"FROM " +
				RAW_TRIPLES_TABLE + " rt, " +
				RESOURCES_TABLE + " r1, " +
				RESOURCES_TABLE + " r2, " +
				LITERALS_TABLE + " l, " +
				RESOURCES_TABLE + " r3 " +
			"WHERE " +
				"rt.objIsLiteral = " + _rdbms.TRUE + " AND " + // obj must be a literal

				"rt.subjNs = r1.namespace AND " +
				"(" + subjLnamesAreNullOr + "rt.subjLname = r1.localname) AND " +

				"rt.predNs = r2.namespace AND " +
				"(" + predLnamesAreNullOr + "rt.predLname = r2.localname) AND " +

				"rt.objLabelHash = l.labelHash AND " +
				"(rt.objLang IS NULL AND l.language IS NULL OR rt.objLang = l.language) AND " +
				"(" + labelsAreNullOr + "rt.objLabel = l.label) AND " +
				"l.datatype = r3.id AND " + "r3.namespace = rt.objNs AND " +
				"(" + objLnamesAreNullOr + "rt.objLname = r3.localname)");
	}

	/**
	 * Removes duplicate rows from the ADDED_TRIPLES_TABLE. Rows are duplicates
	 * if their subject, predicate and object are equal. The value of the
	 * <tt>id</tt> is ignored.
	 **/
	protected void _removeDuplicatesFromAddedTriples()
		throws SQLException
	{
		// Query for all IDs of 'duplicate' rows for which the 'original'
		// row's ID is smaller. This returns all IDs of duplicate rows,
		// except for the row with the lowest ID.
		Connection con = _rdbms.getConnection();
		java.sql.Statement st = con.createStatement();
		ResultSet rs = st.executeQuery(
			"SELECT DISTINCT at2.id FROM " +
				ADDED_TRIPLES_TABLE + " at1, " +
				ADDED_TRIPLES_TABLE + " at2 " +
			"WHERE at1.subj = at2.subj " +
			"AND at1.pred = at2.pred " +
			"AND at1.obj = at2.obj " +
			"AND at1.id < at2.id");

		String[] idChunks = _chunkIdSet(rs, 3500);
		rs.close();
		st.close();

		con.setAutoCommit(false);
		st = con.createStatement();

		for (int i = 0; i < idChunks.length; i++) {
			st.executeUpdate(
				"DELETE FROM " + ADDED_TRIPLES_TABLE +
				" WHERE id IN " + idChunks[i]);
		}

		con.commit();
		st.close();
		con.close();
	}

	/**
	 * Processes the triples from ADDED_TRIPLES_TABLE. This method copies
	 * all triples from ADDED_TRIPLES_TABLE that are not yet in
	 * TRIPLES_TABLE to NEW_TRIPLES_TABLE.
	 **/
	protected void _processAddedTriples()
		throws SQLException
	{
		_rdbms.executeUpdate(
			"INSERT INTO " + NEW_TRIPLES_TABLE + " " +
			"SELECT at.* " +
			"FROM " + ADDED_TRIPLES_TABLE + " at " +
			"LEFT JOIN " + TRIPLES_TABLE + " t " +
			"ON at.subj = t.subj AND at.pred = t.pred AND at.obj = t.obj " +
			"WHERE t.subj IS NULL");

		_rdbms.clearTable(ADDED_TRIPLES_TABLE);

		_rdbms.optimizeTable(NEW_TRIPLES_TABLE);
	}

	/**
	 * This method is called when new statements have been added to, or some
	 * statements have been removed from this repository. The new statements
	 * are available for further processing by subclasses of this class in
	 * table NEW_TRIPLES_TABLE. This table will be cleared directly after the
	 * call to this method has returned.
	 **/
	protected void _processChangedTriples()
		throws SQLException
	{
		// Do nothing
	}

	public void clearRepository()
		throws SailUpdateException
	{
		if (!transactionStarted()) {
			throw new SailUpdateException("No transaction was started");
		}
		try {
			_rdbms.clearTable(ADDED_TRIPLES_TABLE);
			_rdbms.clearTable(NEW_TRIPLES_TABLE);
			_rdbms.clearTable(RAW_TRIPLES_TABLE);
			_rdbms.clearTable(EXPIRED_TRIPLES_TABLE);
			_rdbms.clearTable(EXPIRED_RESOURCES_TABLE);
			_rdbms.clearTable(EXPIRED_LITERALS_TABLE);

			_rdbms.clearTable(TRIPLES_TABLE);
			_rdbms.clearTable(LITERALS_TABLE);
			_rdbms.clearTable(RESOURCES_TABLE);
			_rdbms.clearTable(NAMESPACES_TABLE);

			// Reinsert the rows with ID 0 in the namespaces and resources table
			_rdbms.executeUpdate(
					"INSERT INTO " + NAMESPACES_TABLE +
					" VALUES(0, '', NULL, " + _rdbms.FALSE + ", " + _rdbms.FALSE + ")");

			_rdbms.executeUpdate(
					"INSERT INTO " + RESOURCES_TABLE +
					" VALUES(0, 0, '')");

			// Set the export status to 'up-to-date'
			_setExportStatusUpToDate(true);

			_initNamespaceCache();
			_setNextResourceId();
			_setNextLiteralId();
			_setNextStatementId();

			_updateBNodePrefix();

			_sailChangedEvent.setStatementsRemoved(true);
		}
		catch (SQLException e) {
			throw new SailInternalException(e);
		}
	}

	public void changeNamespacePrefix(String namespace, String prefix)
		throws SailUpdateException
	{
		if (!transactionStarted()) {
			throw new SailUpdateException("No transaction was started");
		}

		// Truncate prefix if it is too long
		if (prefix.length() > _rdbms.MAX_PREFIX_LENGTH) {
			ThreadLog.log("Prefix '" + prefix + "' too long, truncating it to "
					+ _rdbms.MAX_PREFIX_LENGTH + " characters");
			prefix = prefix.substring(0, _rdbms.MAX_PREFIX_LENGTH);
		}

		try {
			// Check if prefix is already used:
			RdbmsNamespace ns = _getNamespaceForPrefix(prefix);

			if (ns != null) {
				// prefix already in use

				if (ns.getName().equals(namespace)) {
					// identical mapping already exists
					return;
				}
				else {
					throw new SailUpdateException("Prefix '" + prefix
							+ "'is already used for another namespace");
				}
			}

			// Update the RDBMS.
			_rdbms.executeUpdate(
				"UPDATE " + NAMESPACES_TABLE +
				" SET prefix = '" + _rdbms.escapeString(prefix) + "'," +
				" userDefined = " + _rdbms.TRUE + "," +
				" export = " + _rdbms.TRUE +
				" WHERE name = '" + _rdbms.escapeString(namespace) + "'");

			// Update the cache
			ns = (RdbmsNamespace)_namespaceTable.get(namespace);
			if (ns != null) {
				ns.setPrefix(prefix);
				ns.setExported(true);
			}
		}
		catch (SQLException e) {
			throw new SailInternalException(e);
		}
	}

	/*-----------------------------------------+
	 | Methods for creating id's for namespaces |
	 +-----------------------------------------*/

	protected int _createIdForNamespace(String namespace) {
		int namespaceId = 0;

		if (namespace != null) {
			namespaceId = _getNamespaceId(namespace);

			if (namespaceId == 0) {
				// unknown namespace, create it.
				namespaceId = _addNamespace(namespace);
			}
		}

		return namespaceId;
	}

	private int _addNamespace(String namespace) {
		// Find an unused namespace prefix
		String prefix = null;
		int nsId = _getNextNamespaceId();

		while (true) {
			prefix = "ns" + nsId;
			Namespace ns = _getNamespaceForPrefix(prefix);

			if (ns == null) {
				// prefix not used yet
				break;
			}
			else {
				// prefix already used, try again
				nsId = _getNextNamespaceId();
			}
		}

		return _addNamespace(nsId, prefix, namespace);
	}

	private int _addNamespace(int nsId, String prefix, String namespace) {
		// Insert into namespaces table
		try {
			_rdbms.executeUpdate(
				"INSERT INTO " + NAMESPACES_TABLE + " VALUES (" +
					nsId + ", '" +
					_rdbms.escapeString(prefix) + "', '" +
					_rdbms.escapeString(namespace) + "', " +
					_rdbms.FALSE + ", " + // userDefined
					_rdbms.FALSE + ")"); // export
		}
		catch (SQLException e) {
			throw new SailInternalException(e);
		}

		// Insert in local cache
		RdbmsNamespace ns = new RdbmsNamespace(nsId, prefix, namespace, false);
		_namespaceTable.put(namespace, ns);
		_namespaceList.add(ns);

		if (nsId >= _namespaceNames.length) {
			// Increase size of array
			int newLength = _namespaceNames.length * 2;
			newLength = Math.max(newLength, nsId + 1);

			String[] newArray = new String[newLength];
			System.arraycopy(_namespaceNames, 0, newArray, 0, _namespaceNames.length);
			_namespaceNames = newArray;
		}
		_namespaceNames[nsId] = namespace;

		return nsId;
	}

	/**
	 * Gets the RdbmsNamespace with the specified prefix.
	 * @return The requested RdbmsNamespace, or null if no such namespace could
	 * be found.
	 **/
	protected RdbmsNamespace _getNamespaceForPrefix(String prefix) {
		Iterator iter = _namespaceList.iterator();
		while (iter.hasNext()) {
			RdbmsNamespace ns = (RdbmsNamespace)iter.next();

			if (ns.getPrefix().equals(prefix)) {
				return ns;
			}
		}

		return null;
	}


	/* (non-Javadoc)
	 * @see org.openrdf.sesame.sail.RdfRepository#addListener(org.openrdf.sesame.sail.SailChangedListener)
	 */
	public void addListener(SailChangedListener listener) {
		synchronized(_sailChangedListeners) {
			_sailChangedListeners.add(listener);
		} // end synchronized block
	}

	/* (non-Javadoc)
	 * @see org.openrdf.sesame.sail.RdfRepository#removeListener(org.openrdf.sesame.sail.SailChangedListener)
	 */
	public void removeListener(SailChangedListener listener) {
		synchronized(_sailChangedListeners) {
			_sailChangedListeners.remove(listener);
		} // end synchronized block
	}

	protected void _notifySailChanged(SailChangedEventImpl event) {
		synchronized(_sailChangedListeners) {
			if (_sailChangedListeners.size() > 0) {
				if (event.sailChanged()) { // only notify if something actually changed
					Iterator listeners = _sailChangedListeners.iterator();

					while (listeners.hasNext()) {
						SailChangedListener listener = (SailChangedListener)listeners.next();
						listener.sailChanged(event);
					}
				}
			}
		} // end synchronized block
	}
}
