/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sailimpl.rdbms;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import org.openrdf.util.log.ThreadLog;
import org.openrdf.vocabulary.RDF;

import org.openrdf.sesame.sail.SailInternalException;

/**
 * Inferencer using the rules from the RDF Semantics Recommendation (10
 * February 2004). See http://www.w3.org/TR/2004/REC-rdf-mt-20040210/
 * This inferencer operates directly on the database used by the
 * RdfSchemaRepository.
 **/
public class RdfMTInferencer implements TableNames {

/*-------------------------------------+
| Constants                            |
+-------------------------------------*/

	// Some constants used to map rule numbers to indexes in arrays
	private static final int Rdf1     = 0;
	private static final int Rdfs2_1  = 1;
	private static final int Rdfs2_2  = 2;
	private static final int Rdfs3_1  = 3;
	private static final int Rdfs3_2  = 4;
	private static final int Rdfs4a   = 5;
	private static final int Rdfs4b   = 6;
	private static final int Rdfs5_1  = 7;
	private static final int Rdfs5_2  = 8;
	private static final int Rdfs6    = 9;
	private static final int Rdfs7_1  = 10;
	private static final int Rdfs7_2  = 11;
	private static final int Rdfs8    = 12;
	private static final int Rdfs9_1  = 13;
	private static final int Rdfs9_2  = 14;
	private static final int Rdfs10   = 15;
	private static final int Rdfs11_1 = 16;
	private static final int Rdfs11_2 = 17;
	private static final int Rdfs12   = 18;
	private static final int Rdfs13   = 19;
	private static final int RX1      = 20;

	private static final int RULECOUNT = 21;

	private static final String[] RULENAMES = {
			"Rdf1",
			"Rdfs2_1",
			"Rdfs2_2",
			"Rdfs3_1",
			"Rdfs3_2",
			"Rdfs4a",
			"Rdfs4b",
			"Rdfs5_1",
			"Rdfs5_2",
			"Rdfs6",
			"Rdfs7_1",
			"Rdfs7_2",
			"Rdfs8",
			"Rdfs9_1",
			"Rdfs9_2",
			"Rdfs10",
			"Rdfs11_1",
			"Rdfs11_2",
			"Rdfs12",
			"Rdfs13",
			"RX1"};

/*
This inferencer makes use of the fact that results of a rule don't trigger
all other inference rules. I.e: if one rule infers a new subClassOf
statement, this will only trigger rules whose precendents can match such
a statement.

The following table shows the dependencies between the rules. An 'x' in this
table means that the rule on this row triggers the rule in that column.

     | triggers rule:
rule:| 1 2_1 2_2 3_1 3_2 4a 4b 5_1  5_2   6 7_1 7_2  8  9_1 9_2 10 11_1 11_2 12 13 X1
-----+--------------------------------------------------------------------------------
1    | -  x   -   x   -  x   -   -    -   x  x   -   -  -   x   -  -    -     -  -  -
2_1  | -  x   -   x   -  -   -   -    -   x  x   -   x  -   x   x  -    -     x  x  -
2_2  | -  x   -   x   -  -   -   -    -   x  x   -   x  -   x   x  -    -     x  x  -
3_1  | -  x   -   x   -  -   -   -    -   x  x   -   x  -   x   x  -    -     x  x  -
3_2  | -  x   -   x   -  -   -   -    -   x  x   -   x  -   x   x  -    -     x  x  -
4a   | -  x   -   x   -  -   -   -    -   -  x   -   -  -   x   -  -    -     -  -  -
4b   | -  x   -   x   -  -   -   -    -   -  x   -   -  -   x   -  -    -     -  -  -
5_1  | -  -   -   -   -  -   -   x    x   -  x   x   -  -   -   -  -    -     -  -  -
5_2  | -  -   -   -   -  -   -   x    x   -  x   x   -  -   -   -  -    -     -  -  -
6    | -  x   -   x   -  -   -   -    -   -  x   -   -  -   -   -  -    -     -  -  -
7_1  | -  x   x   x   x  -   -   x    x   x  x   x   x  x   x   x  x    x     x  x  x
7_2  | -  x   x   x   x  -   -   x    x   x  x   x   x  x   x   x  x    x     x  x  x
8    | -  x   -   x   -  -   -   -    -   -  x   -   -  x   -   -  x    x     -  -  -
9_1  | -  -   -   x   -  -   -   -    -   x  x   -   x  -   x   x  -    -     x  x  -
9_2  | -  -   -   x   -  -   -   -    -   x  x   -   x  -   x   x  -    -     x  x  -
10   | -  x   -   x   -  -   -   -    -   -  x   -   -  -   -   -  -    -     -  -  -
11_1 | -  -   -   -   -  -   -   -    -   -  x   -   -  x   -   -  x    x     -  -  -
11_2 | -  -   -   -   -  -   -   -    -   -  x   -   -  x   -   -  x    x     -  -  -
12   | -  x   -   x   -  -   x   x    x   -  x   x   -  -   -   -  -    -     -  -  -
13   | -  x   -   -   -  -   -   -    -   -  x   -   -  x   -   -  x    x     -  -  -
X1   | -  x   -   x   -  -   -   -    -   -  x   -   -  -   x   -  -    -     x  -  -
*/
	private static final boolean _ = false;
	private static final boolean X = true;

	private static final boolean[][] TRIGGERS = {
//    1    2_2   3_2    4b   5_2   7_1    8    9_2   11_1   12   X1
//      2_1   3_1    4a   5_1    6    7_2   9_1   10   11_2   13
	{ _, X, _, X, _, X, _, _, _, X, X, _, _, _, X, _, _, _, _, _, _},// 1
	{ _, X, _, X, _, _, _, _, _, X, X, _, X, _, X, X, _, _, X, X, _},// 2_1
	{ _, X, _, X, _, _, _, _, _, X, X, _, X, _, X, X, _, _, X, X, _},// 2_2
	{ _, X, _, X, _, _, _, _, _, X, X, _, X, _, X, X, _, _, X, X, _},// 3_1
	{ _, X, _, X, _, _, _, _, _, X, X, _, X, _, X, X, _, _, X, X, _},// 3_2
	{ _, X, _, X, _, _, _, _, _, _, X, _, _, _, X, _, _, _, _, _, _},// 4a
	{ _, X, _, X, _, _, _, _, _, _, X, _, _, _, X, _, _, _, _, _, _},// 4b
	{ _, _, _, _, _, _, _, X, X, _, X, X, _, _, _, _, _, _, _, _, _},// 51
	{ _, _, _, _, _, _, _, X, X, _, X, X, _, _, _, _, _, _, _, _, _},// 52
	{ _, X, _, X, _, _, _, _, _, _, X, _, _, _, _, _, _, _, _, _, _},// 6
	{ _, X, X, X, X, _, _, X, X, X, X, X, X, X, X, X, X, X, X, X, X},// 7_1
	{ _, X, X, X, X, _, _, X, X, X, X, X, X, X, X, X, X, X, X, X, X},// 7_2
	{ _, X, _, X, _, _, _, _, _, _, X, _, _, X, _, _, X, X, _, _, _},// 8
	{ _, _, _, X, _, _, _, _, _, X, X, _, X, _, X, X, _, _, X, X, _},// 9_1
	{ _, _, _, X, _, _, _, _, _, X, X, _, X, _, X, X, _, _, X, X, _},// 9_2
	{ _, X, _, X, _, _, _, _, _, _, X, _, _, _, _, _, _, _, _, _, _},// 10
	{ _, _, _, _, _, _, _, _, _, _, X, _, _, X, _, _, X, X, _, _, _},// 11_1
	{ _, _, _, _, _, _, _, _, _, _, X, _, _, X, _, _, X, X, _, _, _},// 11_2
	{ _, X, _, X, _, _, X, X, X, _, X, X, _, _, _, _, _, _, _, _, _},// 12
	{ _, X, _, _, _, _, _, _, _, _, X, _, _, X, _, _, X, X, _, _, _},// 13
	{ _, X, _, X, _, _, _, _, _, _, X, _, _, _, X, _, _, _, X, _, _},// X1
	};

/*-------------------------------------+
| Variables                            |
+-------------------------------------*/

	private RdfSchemaRepository _repository;
	private RDBMS _rdbms;

	private Connection _insertCon;
	private PreparedStatement _insertSt;

	/** Flags indicating which rules should be evaluated. **/
	private boolean[] _checkRule = new boolean[RULECOUNT];

	/** Flags indicating which rules should be evaluated next iteration. **/
	private boolean[] _checkRuleNextIter = new boolean[RULECOUNT];

	// Variables for optimization statistics
	private int _totalInferred = 0;

	// number of inferred statements per rule.
	private int[] _ruleCount = new int[RULECOUNT];

	// total processing time per rule.
	private long[] _ruleTime = new long[RULECOUNT];

/*-------------------------------------+
| Constructors                         |
+-------------------------------------*/

	public RdfMTInferencer(RdfSchemaRepository repository, RDBMS rdbms) {
		_repository = repository;
		_rdbms = rdbms;
	}

/*-------------------------------------+
| Methods                              |
+-------------------------------------*/

	/**
	 * Prepares a connection and a PreparedStatement for adding
	 * rows to the INFERRED_TABLE.
	 **/
	private void _prepareInsertConnection()
		throws SQLException
	{
		_insertCon = _rdbms.getConnection();
		_insertCon.setAutoCommit(false);
		_insertSt = _insertCon.prepareStatement(
				"INSERT INTO " + INFERRED_TABLE +
				" VALUES(?, ?, ?, ?, ?)");

		// All statements are implicit here:
		_insertSt.setBoolean(5, false);
	}

	/**
	 * Closes the connection that was prepared in _prepareInsertConnection().
	 * If <tt>mustCommit</tt> is true, the started transaction is committed
	 * first.
	 **/
	private void _closeInsertConnection(boolean mustCommit)
		throws SQLException
	{
		if (mustCommit) {
			/*
			int[] results = _insertSt.executeBatch();
			for (int i = 0; i  < results.length; i++) {
				if (results[i] == Statement.EXECUTE_FAILED) {
					throw new SQLException("Failed to add one or more rows to table " + INFERRED_TABLE);
				}
			}
			*/
			_insertCon.commit();
		}

		_insertSt.close();
		_insertCon.close();
	}

	public void initialize() {
		try {
			// As defined in the RDF MT: add a number of default statements
			// concerning the definition of RDF and RDF Schema primitives
			_addAxioms();

			// ...and apply the inference rules to them.
			doInferencing();
		}
		catch (SQLException e) {
			throw new SailInternalException(e);
		}
	}

	private void _addAxioms()
		throws SQLException
	{
		ThreadLog.trace("adding axioms");

		// Insert all axioms into INFERRED_TABLE

		_prepareInsertConnection();

		// RDF axiomatic triples (from RDF Semantics, section 3.1):
		
		_insertAxiom(_repository.rdfTypeId, _repository.rdfTypeId, _repository.rdfPropertyId);
		_insertAxiom(_repository.rdfSubjectId, _repository.rdfTypeId, _repository.rdfPropertyId);
		_insertAxiom(_repository.rdfPredicateId, _repository.rdfTypeId, _repository.rdfPropertyId);
		_insertAxiom(_repository.rdfObjectId, _repository.rdfTypeId, _repository.rdfPropertyId);
		
		_insertAxiom(_repository.rdfFirstId, _repository.rdfTypeId, _repository.rdfPropertyId);
		_insertAxiom(_repository.rdfRestId, _repository.rdfTypeId, _repository.rdfPropertyId);
		_insertAxiom(_repository.rdfValueId, _repository.rdfTypeId, _repository.rdfPropertyId);
		
		_insertAxiom(_repository.rdfNilId, _repository.rdfTypeId, _repository.rdfListId);
		
		// RDFS axiomatic triples (from RDF Semantics, section 4.1):
		
		_insertAxiom(_repository.rdfTypeId, _repository.rdfsDomainId, _repository.rdfsResourceId);
		_insertAxiom(_repository.rdfsDomainId, _repository.rdfsDomainId, _repository.rdfPropertyId);
		_insertAxiom(_repository.rdfsRangeId, _repository.rdfsDomainId, _repository.rdfPropertyId);
		_insertAxiom(_repository.rdfsSubPropertyOfId, _repository.rdfsDomainId, _repository.rdfPropertyId);
		_insertAxiom(_repository.rdfsSubClassOfId, _repository.rdfsDomainId, _repository.rdfsClassId);
		_insertAxiom(_repository.rdfSubjectId, _repository.rdfsDomainId, _repository.rdfStatementId);
		_insertAxiom(_repository.rdfPredicateId, _repository.rdfsDomainId, _repository.rdfStatementId);
		_insertAxiom(_repository.rdfObjectId, _repository.rdfsDomainId, _repository.rdfStatementId);
		_insertAxiom(_repository.rdfsMemberId, _repository.rdfsDomainId, _repository.rdfsResourceId);
		_insertAxiom(_repository.rdfFirstId, _repository.rdfsDomainId, _repository.rdfListId);
		_insertAxiom(_repository.rdfRestId, _repository.rdfsDomainId, _repository.rdfListId);
		_insertAxiom(_repository.rdfsSeeAlsoId, _repository.rdfsDomainId, _repository.rdfsResourceId);
		_insertAxiom(_repository.rdfsIsDefinedById, _repository.rdfsDomainId, _repository.rdfsResourceId);
		_insertAxiom(_repository.rdfsCommentId, _repository.rdfsDomainId, _repository.rdfsResourceId);
		_insertAxiom(_repository.rdfsLabelId, _repository.rdfsDomainId, _repository.rdfsResourceId);
		_insertAxiom(_repository.rdfValueId, _repository.rdfsDomainId, _repository.rdfsResourceId);
		
		_insertAxiom(_repository.rdfTypeId, _repository.rdfsRangeId, _repository.rdfsClassId);
		_insertAxiom(_repository.rdfsDomainId, _repository.rdfsRangeId, _repository.rdfsClassId);
		_insertAxiom(_repository.rdfsRangeId, _repository.rdfsRangeId, _repository.rdfsClassId);
		_insertAxiom(_repository.rdfsSubPropertyOfId, _repository.rdfsRangeId, _repository.rdfPropertyId);
		_insertAxiom(_repository.rdfsSubClassOfId, _repository.rdfsRangeId, _repository.rdfsClassId);
		_insertAxiom(_repository.rdfSubjectId, _repository.rdfsRangeId, _repository.rdfsResourceId);
		_insertAxiom(_repository.rdfPredicateId, _repository.rdfsRangeId, _repository.rdfsResourceId);
		_insertAxiom(_repository.rdfObjectId, _repository.rdfsRangeId, _repository.rdfsResourceId);
		_insertAxiom(_repository.rdfsMemberId, _repository.rdfsRangeId, _repository.rdfsResourceId);
		_insertAxiom(_repository.rdfFirstId, _repository.rdfsRangeId, _repository.rdfsResourceId);
		_insertAxiom(_repository.rdfRestId, _repository.rdfsRangeId, _repository.rdfListId);
		_insertAxiom(_repository.rdfsSeeAlsoId, _repository.rdfsRangeId, _repository.rdfsResourceId);
		_insertAxiom(_repository.rdfsIsDefinedById, _repository.rdfsRangeId, _repository.rdfsResourceId);
		_insertAxiom(_repository.rdfsCommentId, _repository.rdfsRangeId, _repository.rdfsLiteralId);
		_insertAxiom(_repository.rdfsLabelId, _repository.rdfsRangeId, _repository.rdfsLiteralId);
		_insertAxiom(_repository.rdfValueId, _repository.rdfsRangeId, _repository.rdfsResourceId);
		
		_insertAxiom(_repository.rdfAltId, _repository.rdfsSubClassOfId, _repository.rdfsContainerId);
		_insertAxiom(_repository.rdfBagId, _repository.rdfsSubClassOfId, _repository.rdfsContainerId);
		_insertAxiom(_repository.rdfSeqId, _repository.rdfsSubClassOfId, _repository.rdfsContainerId);
		_insertAxiom(_repository.rdfsContainerMembershipPropertyId, _repository.rdfsSubClassOfId, _repository.rdfPropertyId);
		
		_insertAxiom(_repository.rdfsIsDefinedById, _repository.rdfsSubPropertyOfId, _repository.rdfsSeeAlsoId);

		_insertAxiom(_repository.rdfXMLLiteralId, _repository.rdfTypeId, _repository.rdfsDatatypeId);
		_insertAxiom(_repository.rdfXMLLiteralId, _repository.rdfsSubClassOfId, _repository.rdfsLiteralId);
		_insertAxiom(_repository.rdfsDatatypeId, _repository.rdfsSubClassOfId, _repository.rdfsClassId);
		
		_closeInsertConnection(true);

		// Copy all axioms that are not yet in the triples table
		int newAxioms = _rdbms.executeUpdate(
				"INSERT INTO " + NEW_TRIPLES_TABLE + " " +
				"SELECT inf.* " +
				"FROM " + INFERRED_TABLE + " inf " +
				"LEFT JOIN " + TRIPLES_TABLE + " t ON " +
					"inf.subj = t.subj AND " +
					"inf.pred = t.pred AND " +
					"inf.obj = t.obj " +
				"WHERE " +
					"t.subj IS NULL");

		if (newAxioms > 0) {
			_rdbms.copyRows(NEW_TRIPLES_TABLE, TRIPLES_TABLE);
			_rdbms.copyRows(NEW_TRIPLES_TABLE, ALL_NEW_TRIPLES_TABLE);
		}

		// Clear INFERRED_TABLE
		_rdbms.clearTable(INFERRED_TABLE);
	}

	private void _insertAxiom(int subjId, int predId, int objId)
		throws SQLException
	{
		_insertSt.setInt(1, _repository._getNextStatementId());
		_insertSt.setInt(2, subjId);
		_insertSt.setInt(3, predId);
		_insertSt.setInt(4, objId);
		_insertSt.executeUpdate();
	}

	public void doInferencing() {
		// initialize some vars
		_totalInferred = 0;
		int iteration = 0;

		// All rules needs to be checked:
		for (int i = 0; i < RULECOUNT; i++) {
			_ruleCount[i] = 0;
			_ruleTime[i] = 0;
			_checkRuleNextIter[i] = true;
		}

		int nofInferred = 1;
		ThreadLog.trace("starting inferencing");

		try {
			while (nofInferred > 0) {
				iteration++;
				// check which rules need applying this iteration:
				_prepareNextIteration();
	
				// Apply inferencing to all statements in newTriples.
				// Each rule will insert any newly inferred statements into
				// INFERRED_TABLE and ALL_INFERRED_TABLE.
	
				nofInferred = 0;
	
				// Batch-oriented processing:
				nofInferred += _applyRuleRdf1();
				nofInferred += _applyRuleRdfs2_1();
				nofInferred += _applyRuleRdfs2_2();
				nofInferred += _applyRuleRdfs3_1();
				nofInferred += _applyRuleRdfs3_2();
				nofInferred += _applyRuleRdfs4a();
				nofInferred += _applyRuleRdfs4b();
				nofInferred += _applyRuleRdfs5_1();
				nofInferred += _applyRuleRdfs5_2();
				nofInferred += _applyRuleRdfs6();
				nofInferred += _applyRuleRdfs6();
				nofInferred += _applyRuleRdfs7_1();
				nofInferred += _applyRuleRdfs7_2();
				nofInferred += _applyRuleRdfs8();
				nofInferred += _applyRuleRdfs9_1();
				nofInferred += _applyRuleRdfs9_2();
				nofInferred += _applyRuleRdfs10();
				nofInferred += _applyRuleRdfs11_1();
				nofInferred += _applyRuleRdfs11_2();
				nofInferred += _applyRuleRdfs12();
				nofInferred += _applyRuleRdfs13();
				nofInferred += _applyRuleX1();
	
				_totalInferred += nofInferred;
	
				// The triples from NEW_TRIPLES_TABLE have been processed,
				// continue with the newly inferred triples from ALL_INFERRED_TABLE
	
				_rdbms.clearTable(NEW_TRIPLES_TABLE);
	
				if (nofInferred > 0) {
					_rdbms.copyRows(ALL_INFERRED_TABLE, NEW_TRIPLES_TABLE);
					_rdbms.copyRows(ALL_INFERRED_TABLE, ALL_NEW_TRIPLES_TABLE);
	
					_rdbms.clearTable(ALL_INFERRED_TABLE);
	
					_rdbms.optimizeTable(NEW_TRIPLES_TABLE);
					_rdbms.optimizeTable(TRIPLES_TABLE);
				}
	
				ThreadLog.trace("iteration " + iteration + " done; " +
					"inferred " + nofInferred + " new statements");
			}
	
			_rdbms.optimizeTable(ALL_NEW_TRIPLES_TABLE);
		}
		catch (SQLException e) {
			throw new SailInternalException(e);
		}

		// Print some statistics
		ThreadLog.trace("---RdfMTInferencer statistics:---");
		ThreadLog.trace("total statements inferred = " + _totalInferred);
		for (int i = 0; i < RULECOUNT; i++) {
			ThreadLog.trace(
					"rule " + RULENAMES[i] + ": time=" + _ruleTime[i] +
					";\t#inferred=" + _ruleCount[i]);
		}
		ThreadLog.trace("---end of statistics:---");
	}

	private void _prepareNextIteration() {
		for (int i = 0; i < RULECOUNT; i++) {
			_checkRule[i] = _checkRuleNextIter[i];

			// reset for next iteration:
			_checkRuleNextIter[i] = false;
		}
	}

	/* rdf1. 
	 * xxx aaa yyy --> aaa rdf:type rdf:Property
	 */
	private int _applyRuleRdf1()
		throws SQLException
	{
		String query =
			"SELECT DISTINCT " +
				"nt.pred, " + _repository.rdfTypeId + ", " + _repository.rdfPropertyId +
			" FROM " + NEW_TRIPLES_TABLE + " nt " +
			"LEFT JOIN " + TRIPLES_TABLE + " t ON " +
			"   nt.pred = t.subj AND " +
			"   t.pred = " + _repository.rdfTypeId + " AND " +
			"   t.obj = " + _repository.rdfPropertyId + " " +
			"WHERE " +
			"   t.subj IS NULL";

		return _applyRule(Rdf1, query);
	}
	
	/* rdf2.
	 * uuu aaa lll --> _:nnn rdf:type rdf:XMLLiteral
	 * (lll is a wellformed XML literal)
	 * Rule not implemented.  
	 */

	/* rdfs1.
	 * uuu aaa lll --> _:nnn rdf:type rdfs:Literal
	 * (lll is a plain literal)
	 * Rule not implemented.
	 */
	
	/* rdfs2.
	 * 2_1. xxx aaa yyy &&           (nt)
	 *      aaa rdfs:domain zzz -->  (t1)
	 *      xxx rdf:type zzz         (t2)
	 */
	private int _applyRuleRdfs2_1()
		throws SQLException
	{
		String query =
			"SELECT DISTINCT " +
				"nt.subj, " + _repository.rdfTypeId + ", " + "t1.obj " +
			"FROM " + NEW_TRIPLES_TABLE + " nt " +
			"LEFT JOIN " + TRIPLES_TABLE + " t1 ON " +
			"   nt.pred = t1.subj AND " +
			"   t1.pred = " + _repository.rdfsDomainId + " " +
			"LEFT JOIN " + TRIPLES_TABLE + " t2 ON " +
			"   nt.subj = t2.subj AND " +
			"   t1.obj = t2.obj AND " +
			"   t2.pred = " + _repository.rdfTypeId + " " +
			"WHERE " +
			"   t1.obj > 0 AND " + // domain class needs to be a resource
			"   t2.subj IS NULL AND " +
			"   t1.subj IS NOT NULL";

		return _applyRule(Rdfs2_1, query);
	}

	/* rdfs2.
	 * 2_2. aaa rdfs:domain zzz &&  (nt)
	 *     xxx aaa yyy -->         (t1)
	 *     xxx rdf:type zzz        (t2)
	 */
	private int _applyRuleRdfs2_2()
		throws SQLException
	{
		String query =
			"SELECT DISTINCT " +
				"t1.subj, " + _repository.rdfTypeId + ", " + "nt.obj " +
			"FROM " + NEW_TRIPLES_TABLE + " nt " +
			"LEFT JOIN " + TRIPLES_TABLE + " t1 ON " +
			"   nt.subj = t1.pred " +
			"LEFT JOIN " + TRIPLES_TABLE + " t2 ON " +
			"   nt.obj = t2.obj AND " +
			"   t1.subj = t2.subj AND " +
			"   t2.pred = " + _repository.rdfTypeId + " " +
			"WHERE " +
			"   nt.pred = " + _repository.rdfsDomainId + " AND " +
			"   nt.obj > 0 AND " + // domain class needs to be a resource
			"   t2.subj IS NULL AND " +
			"   t1.subj IS NOT NULL";

		return _applyRule(Rdfs2_2, query);
	}

	/* rdfs3.
	 * 3_1. xxx aaa uuu &&          (nt)
	 *     aaa rdfs:range zzz -->  (t1)
	 *     uuu rdf:type zzz        (t2)
	 */
	private int _applyRuleRdfs3_1()
		throws SQLException
	{
		String query =
			"SELECT DISTINCT " +
				"nt.obj, " + _repository.rdfTypeId + ", " + "t1.obj " +
			"FROM " + NEW_TRIPLES_TABLE + " nt " +
			"LEFT JOIN " + TRIPLES_TABLE + " t1 ON " +
			"   nt.pred = t1.subj AND " +
			"   t1.pred = " + _repository.rdfsRangeId + " " +
			"LEFT JOIN " + TRIPLES_TABLE + " t2 ON " +
			"   nt.obj = t2.subj AND " +
			"   t1.obj = t2.obj AND " +
			"   t2.pred = " + _repository.rdfTypeId + " " +
			"WHERE " +
			"   nt.obj > 0 AND " + // object needs to be a resource
			"   t1.obj > 0 AND " + // range class needs to be a resource
			"   t2.subj IS NULL AND " +
			"   t1.subj IS NOT NULL";

		return _applyRule(Rdfs3_1, query);
	}

	/* rdfs3.
	 * 3_2. aaa rdfs:range zzz &&  (nt)
	 *     xxx aaa uuu -->        (t1)
	 *     uuu rdf:type zzz       (t2)
	 */
	private int _applyRuleRdfs3_2()
		throws SQLException
	{
		String query =
			"SELECT DISTINCT " +
				"t1.obj, " + _repository.rdfTypeId + ", " + "nt.obj " +
			"FROM " + NEW_TRIPLES_TABLE + " nt " +
			"LEFT JOIN " + TRIPLES_TABLE + " t1 ON " +
			"   nt.subj = t1.pred " +
			"LEFT JOIN " + TRIPLES_TABLE + " t2 ON " +
			"   t1.obj = t2.subj AND " +
			"   nt.obj = t2.obj AND " +
			"   t2.pred = " + _repository.rdfTypeId + " " +
			"WHERE " +
			"   t1.obj > 0 AND " + // object needs to be a resource
			"   nt.obj > 0 AND " + // range class needs to be a resource
			"   nt.pred = " + _repository.rdfsRangeId + " AND " +
			"   t2.subj IS NULL AND " +
			"   t1.subj IS NOT NULL";

		return _applyRule(Rdfs3_2, query);
	}

	/* rdfs4a.
	 * xxx aaa yyy --> xxx rdf:type rdfs:Resource
	 */
	private int _applyRuleRdfs4a()
		throws SQLException
	{
		String query =
			"SELECT DISTINCT " +
				"nt.subj, " + _repository.rdfTypeId + ", " + _repository.rdfsResourceId +
			" FROM " + NEW_TRIPLES_TABLE + " nt " +
			"LEFT JOIN " + TRIPLES_TABLE + " t ON " +
			"   nt.subj = t.subj AND " +
			"   t.pred = " + _repository.rdfTypeId + " AND " +
			"   t.obj = " + _repository.rdfsResourceId + " " +
			"WHERE " +
			"   t.subj IS NULL";

		return _applyRule(Rdfs4a, query);
	}

	/* rdfs4b. 
	 * xxx aaa uuu --> uuu rdf:type rdfs:Resource
	 */
	private int _applyRuleRdfs4b()
		throws SQLException
	{
		String query =
			"SELECT DISTINCT " +
				"nt.obj, " + _repository.rdfTypeId + ", " + _repository.rdfsResourceId +
			" FROM " + NEW_TRIPLES_TABLE + " nt " +
			"LEFT JOIN " + TRIPLES_TABLE + " t ON " +
			"   nt.obj = t.subj AND " +
			"   t.pred = " + _repository.rdfTypeId + " AND " +
			"   t.obj = " + _repository.rdfsResourceId + " " +
			"WHERE " +
			"   nt.obj > 0 AND " + // object needs to be a resource
			"   t.subj IS NULL";

		return _applyRule(Rdfs4b, query);
	}

	/* rdfs5.
	 * 5_1. aaa rdfs:subPropertyOf bbb &&   (nt)
	 *     bbb rdfs:subPropertyOf ccc -->  (t1)
	 *     aaa rdfs:subPropertyOf ccc      (t2)
	 */
	private int _applyRuleRdfs5_1()
		throws SQLException
	{
		String query =
			"SELECT DISTINCT " +
				"nt.subj, " + _repository.rdfsSubPropertyOfId + ", " + "t1.obj " +
			"FROM " + NEW_TRIPLES_TABLE + " nt " +
			"LEFT JOIN " + TRIPLES_TABLE + " t1 ON " +
			"   nt.obj = t1.subj AND " +
			"   t1.pred = " + _repository.rdfsSubPropertyOfId + " " +
			"LEFT JOIN " + TRIPLES_TABLE + " t2 ON " +
			"   nt.subj = t2.subj AND " +
			"   t1.obj = t2.obj AND " +
			"   t2.pred = " + _repository.rdfsSubPropertyOfId + " " +
			"WHERE " +
			"   nt.pred = " + _repository.rdfsSubPropertyOfId + " AND " +
			"   t1.obj > 0 AND " + // ccc needs to be a resource
			"   t2.subj IS NULL AND " +
			"   t1.subj IS NOT NULL";

		return _applyRule(Rdfs5_1, query);
	}

	/* rdfs5.
	 * 5_2. bbb rdfs:subPropertyOf ccc && (nt)
	 *     aaa rdfs:subPropertyOf bbb -->  (t1)
	 *     aaa rdfs:subPropertyOf ccc      (t2)
	 */
	private int _applyRuleRdfs5_2()
		throws SQLException
	{
		String query =
			"SELECT DISTINCT " +
				"t1.subj, " + _repository.rdfsSubPropertyOfId + ", " + "nt.obj " +
			"FROM " + NEW_TRIPLES_TABLE + " nt " +
			"LEFT JOIN " + TRIPLES_TABLE + " t1 ON " +
			"   nt.subj = t1.obj AND " +
			"   t1.pred = " + _repository.rdfsSubPropertyOfId + " " +
			"LEFT JOIN " + TRIPLES_TABLE + " t2 ON " +
			"   nt.obj = t2.obj AND " +
			"   t1.subj = t2.subj AND " +
			"   t2.pred = " + _repository.rdfsSubPropertyOfId + " " +
			"WHERE " +
			"   nt.pred = " + _repository.rdfsSubPropertyOfId + " AND " +
			"   nt.obj > 0 AND " + // ccc needs to be a resource
			"   t2.subj IS NULL AND " +
			"   t1.subj IS NOT NULL";

		return _applyRule(Rdfs5_2, query);
	}

	/* rdfs6. 
	 * xxx rdf:type rdf:Property --> xxx rdfs:subPropertyOf xxx
	 * reflexivity of rdfs:subPropertyOf
	 */
	private int _applyRuleRdfs6()
		throws SQLException
	{
		String query =
			"SELECT DISTINCT " +
				"nt.subj, " + _repository.rdfsSubPropertyOfId + ", " + "nt.subj " +
			"FROM " + NEW_TRIPLES_TABLE + " nt " +
			"LEFT JOIN " + TRIPLES_TABLE + " t ON " +
			"   nt.subj = t.subj AND " +
			"   t.subj = t.obj AND " +
			"   t.pred = " + _repository.rdfsSubPropertyOfId + " " +
			"WHERE " +
			"   nt.pred = " + _repository.rdfTypeId + " AND " +
			"   nt.obj = " + _repository.rdfPropertyId + " AND " +
			"   t.subj IS NULL";

		return _applyRule(Rdfs6, query);
	}

	/* rdfs7.
	 * 7_1. xxx aaa yyy &&                  (nt)
	 *     aaa rdfs:subPropertyOf bbb -->  (t1)
	 *     xxx bbb yyy                     (t2)
	 */
	private int _applyRuleRdfs7_1()
		throws SQLException
	{
		String query =
			"SELECT DISTINCT " +
				"nt.subj, t1.obj, nt.obj " +
			"FROM " + NEW_TRIPLES_TABLE + " nt " +
			"LEFT JOIN " + TRIPLES_TABLE + " t1 ON " +
			"   nt.pred = t1.subj AND " +
			"   t1.pred = " + _repository.rdfsSubPropertyOfId + " " +
			"LEFT JOIN " + RESOURCES_TABLE + " r ON " +
			"   r.id = t1.obj " +
			"LEFT JOIN " + TRIPLES_TABLE + " t2 ON " +
			"   nt.subj = t2.subj AND " +
			"   nt.obj = t2.obj AND " +
			"   t1.obj = t2.pred " +
			"WHERE " +
			"   t1.obj > 0 AND r.namespace > 0 AND " + 
			"   t2.subj IS NULL AND " +
			"   t1.subj IS NOT NULL";

		return _applyRule(Rdfs7_1, query);
	}

	/* rdfs7.
	 * 7_2. aaa rdfs:subPropertyOf bbb &&  (nt)
	 *     xxx aaa yyy -->                (t1)
	 *     xxx bbb yyy                    (t2)
	 */
	private int _applyRuleRdfs7_2()
		throws SQLException
	{
		String query =
			"SELECT DISTINCT " +
				"t1.subj, nt.obj, t1.obj " +
			"FROM " + NEW_TRIPLES_TABLE + " nt " +
			"LEFT JOIN " + TRIPLES_TABLE + " t1 ON " +
			"   nt.subj = t1.pred " +
			"LEFT JOIN " + RESOURCES_TABLE + " r ON " +
			"   r.id = nt.obj " +
			"LEFT JOIN " + TRIPLES_TABLE + " t2 ON " +
			"   t1.subj = t2.subj AND " +
			"   t1.obj = t2.obj AND " +
			"   nt.obj = t2.pred " +
			"WHERE " +
			"   nt.pred = " + _repository.rdfsSubPropertyOfId + " AND " +
			"   nt.obj > 0 AND r.namespace > 0 AND " + 
			"   t2.subj IS NULL AND " +
			"   t1.subj IS NOT NULL";

		return _applyRule(Rdfs7_2, query);
	}

	/* rdfs8. 
	 * xxx rdf:type rdfs:Class --> xxx rdfs:subClassOf rdfs:Resource 
	 * */
	private int _applyRuleRdfs8()
		throws SQLException
	{
		String query =
			"SELECT DISTINCT " +
				"nt.subj, " + _repository.rdfsSubClassOfId + ", " + _repository.rdfsResourceId +
			" FROM " + NEW_TRIPLES_TABLE + " nt " +
			"LEFT JOIN " + TRIPLES_TABLE + " t ON " +
			"   nt.subj = t.subj AND " +
			"   t.pred = " + _repository.rdfsSubClassOfId + " AND " +
			"   t.obj = " + _repository.rdfsResourceId + " " +
			"WHERE " +
			"   nt.pred = " + _repository.rdfTypeId + " AND " +
			"   nt.obj = " + _repository.rdfsClassId + " AND " +
			"   t.subj IS NULL";

		return _applyRule(Rdfs8, query);
	}
	
	/* rdfs9.
	 * 9_1. xxx rdfs:subClassOf yyy &&  (nt)
	 *     aaa rdf:type xxx -->        (t1)
	 *     aaa rdf:type yyy            (t2)
	 */
	private int _applyRuleRdfs9_1()
		throws SQLException
	{
		String query =
			"SELECT DISTINCT " +
				"t1.subj, " + _repository.rdfTypeId + ", " + "nt.obj " +
			"FROM " + NEW_TRIPLES_TABLE + " nt " +
			"LEFT JOIN " + TRIPLES_TABLE + " t1 ON " +
			"   nt.subj = t1.obj AND " +
			"   t1.pred = " + _repository.rdfTypeId + " " +
			"LEFT JOIN " + TRIPLES_TABLE + " t2 ON " +
			"   t1.subj = t2.subj AND " +
			"   nt.obj = t2.obj AND " +
			"   t2.pred = " + _repository.rdfTypeId + " " +
			"WHERE " +
			"   nt.pred = " + _repository.rdfsSubClassOfId + " AND " +
			"   nt.obj > 0 AND " + // yyy needs to be a resource
			"   t2.subj IS NULL AND " +
			"   t1.subj IS NOT NULL";

		return _applyRule(Rdfs9_1, query);
	}

	/* rdfs9.
	 * 9_2. aaa rdf:type xxx &&          (nt)
	 *     xxx rdfs:subClassOf yyy -->  (t1)
	 *     aaa rdf:type yyy             (t2)
	 */
	private int _applyRuleRdfs9_2()
		throws SQLException
	{
		String query =
			"SELECT DISTINCT " +
				"nt.subj, " + _repository.rdfTypeId + ", " + "t1.obj " +
			"FROM " + NEW_TRIPLES_TABLE + " nt " +
			"LEFT JOIN " + TRIPLES_TABLE + " t1 ON " +
			"   nt.obj = t1.subj AND " +
			"   t1.pred = " + _repository.rdfsSubClassOfId + " " +
			"LEFT JOIN " + TRIPLES_TABLE + " t2 ON " +
			"   nt.subj = t2.subj AND " +
			"   t1.obj = t2.obj AND " +
			"   t2.pred = " + _repository.rdfTypeId + " " +
			"WHERE " +
			"   nt.pred = " + _repository.rdfTypeId + " AND " +
			"   t1.obj > 0 AND " + // yyy needs to be a resource
			"   t2.subj IS NULL AND " +
			"   t1.subj IS NOT NULL";

		return _applyRule(Rdfs9_2, query);
	}


	/* rdfs10. 
	 * xxx rdf:type rdfs:Class --> xxx rdfs:subClassOf xxx
	 * reflexivity of rdfs:subClassOf
	 */
	private int _applyRuleRdfs10()
		throws SQLException
	{
		String query =
			"SELECT DISTINCT " +
				"nt.subj, " + _repository.rdfsSubClassOfId + ", " + "nt.subj " +
			"FROM " + NEW_TRIPLES_TABLE + " nt " +
			"LEFT JOIN " + TRIPLES_TABLE + " t ON " +
			"   nt.subj = t.subj AND " +
			"   t.subj = t.obj AND " +
			"   t.pred = " + _repository.rdfsSubClassOfId + " " +
			"WHERE " +
			"   nt.pred = " + _repository.rdfTypeId + " AND " +
			"   nt.obj = " + _repository.rdfsClassId + " AND " +
			"   t.subj IS NULL";

		return _applyRule(Rdfs10, query);
	}

	/* rdfs11.
	 * 11_1. xxx rdfs:subClassOf yyy &&  (nt)
	 *     yyy rdfs:subClassOf zzz -->  (t1)
	 *     xxx rdfs:subClassOf zzz      (t2)
	 */
	private int _applyRuleRdfs11_1()
		throws SQLException
	{
		String query =
			"SELECT DISTINCT " +
				"nt.subj, " + _repository.rdfsSubClassOfId + ", " + "t1.obj " +
			"FROM " + NEW_TRIPLES_TABLE + " nt " +
			"LEFT JOIN " + TRIPLES_TABLE + " t1 ON " +
			"   nt.obj = t1.subj AND " +
			"   t1.pred = " + _repository.rdfsSubClassOfId + " " +
			"LEFT JOIN " + TRIPLES_TABLE + " t2 ON " +
			"   nt.subj = t2.subj AND " +
			"   t1.obj = t2.obj AND " +
			"   t2.pred = " + _repository.rdfsSubClassOfId + " " +
			"WHERE " +
			"   nt.pred = " + _repository.rdfsSubClassOfId + " AND " +
			"   t1.obj > 0 AND " + // zzz needs to be a resource
			"   t2.subj IS NULL AND " +
			"   t1.subj IS NOT NULL";

		return _applyRule(Rdfs11_1, query);
	}

	/* rdfs11.
	 * 11_2. yyy rdfs:subClassOf zzz &&  (nt)
	 *     xxx rdfs:subClassOf yyy -->  (t1)
	 *     xxx rdfs:subClassOf zzz      (t2)
	 */
	private int _applyRuleRdfs11_2()
		throws SQLException
	{
		String query =
			"SELECT DISTINCT " +
				"t1.subj, " + _repository.rdfsSubClassOfId + ", " + "nt.obj " +
			"FROM " + NEW_TRIPLES_TABLE + " nt " +
			"LEFT JOIN " + TRIPLES_TABLE + " t1 ON " +
			"   nt.subj = t1.obj AND " +
			"   t1.pred = " + _repository.rdfsSubClassOfId + " " +
			"LEFT JOIN " + TRIPLES_TABLE + " t2 ON " +
			"   t1.subj = t2.subj AND " +
			"   nt.obj = t2.obj AND " +
			"   t2.pred = " + _repository.rdfsSubClassOfId + " " +
			"WHERE " +
			"   nt.pred = " + _repository.rdfsSubClassOfId + " AND " +
			"   nt.obj > 0 AND " + // zzz needs to be a resource
			"   t2.subj IS NULL AND " +
			"   t1.subj IS NOT NULL";

		return _applyRule(Rdfs11_2, query);
	}

	
	/* rdfs12. 
	 * xxx rdf:type rdfs:ContainerMembershipProperty -->
	 *     xxx rdfs:subPropertyOf rdfs:member
	 */
	private int _applyRuleRdfs12()
		throws SQLException
	{
		String query =
			"SELECT DISTINCT " +
				"nt.subj, " + _repository.rdfsSubPropertyOfId + ", " + _repository.rdfsMemberId +
			" FROM " + NEW_TRIPLES_TABLE + " nt " +
			"LEFT JOIN " + TRIPLES_TABLE + " t ON " +
			"   nt.subj = t.subj AND " +
			"   t.obj = " + _repository.rdfsMemberId + " AND " +
			"   t.pred = " + _repository.rdfsSubPropertyOfId + " " +
			"WHERE " +
			"   nt.pred = " + _repository.rdfTypeId + " AND " +
			"   nt.obj = " + _repository.rdfsContainerMembershipPropertyId +
			" AND " +
			"   t.subj IS NULL";

		return _applyRule(Rdfs12, query);

	}

	/* rdfs13.  xxx rdf:type rdfs:Datatype --> xxx rdfs:subClassOf rdfs:Literal
	 */
	private int _applyRuleRdfs13()
		throws SQLException
	{
		String query =
			"SELECT DISTINCT " +
				"nt.subj, " + _repository.rdfsSubClassOfId + ", " + _repository.rdfsLiteralId +
			" FROM " + NEW_TRIPLES_TABLE + " nt " +
			"LEFT JOIN " + TRIPLES_TABLE + " t ON " +
			"   nt.subj = t.subj AND " +
			"   t.obj = " + _repository.rdfsLiteralId + " AND " +
			"   t.pred = " + _repository.rdfsSubClassOfId + " " +
			"WHERE " +
			"   nt.pred = " + _repository.rdfTypeId + " AND " +
			"   nt.obj = " + _repository.rdfsDatatypeId +
			" AND " +
			"   t.subj IS NULL";

		return _applyRule(Rdfs13, query);
	}

	/* X1. xxx rdf:_* yyy -->
	 *     rdf:_* rdf:type rdfs:ContainerMembershipProperty
	 *
	 * This is an extra rule for list membership properties (_1, _2, _3,
	 * ...). The RDF MT does not specificy a production for this, but specificies this as a non-closed
	 * set of axioms (section 3.1).
	 */
	private int _applyRuleX1()
		throws SQLException
	{
		int rdfNsId = _repository._getNamespaceId(RDF.NAMESPACE);

		if (rdfNsId == 0) {
			// Namespace not used by any resource.
			return 0;
		}

		// Create like pattern matching localnames starting with an underscore
		String likePattern;
		if (_rdbms.getSearchStringEscape() != null) {
			// Use default rdbms escape character
			likePattern = "'" + _rdbms.getSearchStringEscape() + "_%' ";
		}
		else if (_rdbms.supportsLikeEscapeClause()) {
			// Specify backslash as escape character
			likePattern = "'\\_%' ESCAPE '\\' ";
		}
		else {
			throw new SQLException("Unable to create like pattern; no escape character available");
		}

		// Create query
		String query =
			"SELECT DISTINCT " +
				"nt.pred, " + _repository.rdfTypeId + ", " +
				_repository.rdfsContainerMembershipPropertyId + " " +
			"FROM " + NEW_TRIPLES_TABLE + " nt " +
			"LEFT JOIN " + RESOURCES_TABLE + " r ON " +
				"nt.pred = r.id AND " +
				"r.namespace = " + rdfNsId + " AND " +
				"r.localname LIKE " + likePattern +
			"LEFT JOIN " + TRIPLES_TABLE + " t ON " +
				"nt.pred = t.subj AND " +
				"t.pred = " + _repository.rdfTypeId + " AND " +
				"t.obj = " + _repository.rdfsContainerMembershipPropertyId + " " +
			"WHERE " +
				"r.id IS NOT NULL AND " +
				"t.subj IS NULL";

		return _applyRule(RX1, query);
	}

	private int _applyRule(int ruleNo, String query)
		throws SQLException
	{
		if (!_checkRule[ruleNo]) {
			return 0;
		}

		//ThreadLog.trace("Evaluating rule " + RULENAMES[ruleNo]);

		long startTime = System.currentTimeMillis();

		int nofInferred = 0;

		_prepareInsertConnection();

		Connection queryCon = _rdbms.getConnection();
		java.sql.Statement querySt = queryCon.createStatement();
		ResultSet rs = querySt.executeQuery(query);

		while (rs.next()) {
			_insertSt.setInt(1, _repository._getNextStatementId()); // id
			_insertSt.setInt(2, rs.getInt(1)); // subj
			_insertSt.setInt(3, rs.getInt(2)); // pred
			_insertSt.setInt(4, rs.getInt(3)); // obj
			//_insertSt.addBatch();
			_insertSt.executeUpdate();

			nofInferred++;
		}

		rs.close();
		querySt.close();
		queryCon.close();

		_closeInsertConnection(nofInferred > 0);

		//ThreadLog.trace(nofInferred + " stats inferred");

		if (nofInferred > 0) {
			_ruleCount[ruleNo] += nofInferred;

			// Copy the newly inferred statements from INFERRED_TABLE to
			// ALL_INFERRED_TABLE...
			_rdbms.copyRows(INFERRED_TABLE, ALL_INFERRED_TABLE);

			// ...and to TRIPLES_TABLE.
			_rdbms.copyRows(INFERRED_TABLE, TRIPLES_TABLE);

			// Clear INFERRED_TABLE
			_rdbms.clearTable(INFERRED_TABLE);
		}

		long endTime = System.currentTimeMillis();
		_ruleTime[ruleNo] += (endTime - startTime);

		if (nofInferred > 0) {
			// Check which rules are triggered by this one.
			boolean[] triggers = TRIGGERS[ruleNo];

			for (int i = 0; i < RULECOUNT; i++) {
				if (triggers[i] == true) {
					_checkRuleNextIter[i] = true;
				}
			}
		}

		return nofInferred;
	}
}
