/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sailimpl.rdbms;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;
import java.util.List;

import org.openrdf.model.Value;

import org.openrdf.sesame.sail.SailInternalException;
import org.openrdf.sesame.sail.query.PathExpression;
import org.openrdf.sesame.sail.query.Var;

/**
 * @author Arjohn Kampman
 * @version $Revision: 1.5.4.2 $
 */
public class RdbmsPathExpression implements PathExpression {

/*---------------------------------------+
| Variables                              |
+---------------------------------------*/

	protected RdfSource _creator;

	protected RDBMS _rdbms;

	/**
	 * Current connection to the database (if any).
	 **/
	protected Connection _databaseCon;

	/**
	 * SQL statement that was used to generate the current ResultSet (if any).
	 **/
	protected java.sql.Statement _statement;

	/**
	 * Current ResultSet for the query (if any).
	 **/
	protected ResultSet _resultSet;

	protected List _varList;

	protected String _sqlQuery;

	protected int[] _previousValues;

/*----------------------------------------------+
| Constructors                                  |
+----------------------------------------------*/

	/**
	 * Constructor.
	 *
	 * @param creator The object to supply as 'creator' to created IdValues.
	 * @param rdbms An object representing the database to query.
	 * @param varList List of variables that need to be instantiated.
	 * @param sqlQuery SQL query to use to instantiate the variables.
	 */
	public RdbmsPathExpression(
			RdfSource creator, RDBMS rdbms, List varList, String sqlQuery)
	{
		_creator = creator;
		_rdbms = rdbms;
		_varList = varList;
		_sqlQuery = sqlQuery;

		_previousValues = new int[_varList.size()];
	}

/*-------------------------------------------------+
| Methods                                          |
+-------------------------------------------------*/

	public void initialize(org.openrdf.sesame.sail.RdfSource rdfSource) {
		try {
			_databaseCon = _rdbms.getConnection();
			_statement = _databaseCon.createStatement();
			_resultSet = _statement.executeQuery(_sqlQuery);
		}
		catch (SQLException e) {
			throw new SailInternalException(e);
		}
	}

	public boolean selectNext(org.openrdf.sesame.sail.RdfSource rdfSource) {
		try {
			boolean moreResults = _resultSet.next();

			if (moreResults) {
				for (int i = 0; i < _varList.size(); i++) {
					Var var = (Var)_varList.get(i);

					int valueId = _resultSet.getInt(i + 1);

					if (valueId != _previousValues[i]) {
						// New value is different from previous one, retrieve
						// new value from the database
						Value value = _creator.getValue(valueId);
						var.setValue(value);
						_previousValues[i] = valueId;
					}
				}
			}
			else { // no more results
				clear();

				// Unset all variables
				for (int i = 0; i < _varList.size(); i++) {
					Var var = (Var)_varList.get(i);
					var.setValue(null);
				}
			}

			return moreResults;
		}
		catch (SQLException e) {
			throw new SailInternalException(e);
		}
	}

	public void clear() {
		if (_databaseCon != null) {
			try {
				if (_resultSet != null) {
					_resultSet.close();
					_resultSet = null;
				}

				if (_statement != null) {
					_statement.close();
					_statement = null;
				}

				_databaseCon.close();
				_databaseCon = null;
			}
			catch (SQLException e) {
				throw new SailInternalException(e);
			}
		}
	}

	public void getVariables(Collection variables) {
		variables.addAll(_varList);
	}

	protected void finalize() {
		try {
        	clear();
		}
		catch (SailInternalException e) {
			// SQLException wrapped in a SailInternalException, ignore it
		}
    }
}
