/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sailimpl.rdbms;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Types;
import java.util.HashMap;
import java.util.Map;

import org.openrdf.util.StringUtil;
import org.openrdf.util.jdbc.ConnectionPool;
import org.openrdf.util.log.ThreadLog;

/**
 * A representation of an RDBMS. This class defines DBMS-specific constants
 * and methods that are needed by the Sail classes in this package. Subclasses
 * of this class will override specific values and methods if this is needed
 * for that specific database.
 *
 * @author Arjohn Kampman
 * @author Adam Skutt
 **/
public class RDBMS {

/*-----------------------------------------------+
| Static creation methods                        |
+-----------------------------------------------*/

	public static RDBMS createInstance(String jdbcUrl, String user, String passwd)
		throws SQLException
	{
		// Try to detect which database we're going to talk to.
		String dbName = null;
		String dbVersion = null;

		try {
			Connection con = (user == null) ?
					DriverManager.getConnection(jdbcUrl) :
					DriverManager.getConnection(jdbcUrl, user, passwd);

			try {
				DatabaseMetaData metaData = con.getMetaData();

				dbName = metaData.getDatabaseProductName();
				dbVersion = metaData.getDatabaseProductVersion();
			}
			finally {
				con.close();
			}
		}
		catch (SQLException e) {
			ThreadLog.warning("Failed to get database parameters", e);
		}

		if (dbName != null) {
			ThreadLog.log("Detected database: " + dbName);

			if (dbVersion != null) {
				ThreadLog.log("Database version: " + dbVersion);
			}
		}

		RDBMS result;

		if ("MySQL".equalsIgnoreCase(dbName)) {
			result = new MySQL();
		}
		else if ("PostgreSQL".equalsIgnoreCase(dbName)) {
			result = new PostgreSQL();
		}
		else if ("Oracle".equalsIgnoreCase(dbName)) {
			result = new Oracle();
		}
		else if ("Microsoft SQL Server".equalsIgnoreCase(dbName)) {
			result = new SQLServer();
		}
		else {
			result = new RDBMS();
		}

		result.setConnectionInfo(jdbcUrl, user, passwd);

		return result;
	}

/*-----------------------------------------------+
| Database dependent "constants"                 |
+-----------------------------------------------*/

	/** Flag indicating whether the database supports LIKE "..." ESCAPE '\' constructions. **/
	protected boolean _supportsLikeEscapeClause = false;

	/** The string that can be used to escape wildcard characters in patterns. **/
	protected String _searchStringEscape = null;

/*-----------------------------------------------+
| Database dependent datatypes (SQL92 defaults). |
+-----------------------------------------------*/

	/** Datatype for IDs (integer). **/
	public String ID_INT = "integer";
	public int ID_INT_TYPE = Types.INTEGER;

	/** Datatype of localname of resource. **/
	public String LOCALNAME = "character varying(255)";
	public int LOCALNAME_TYPE = Types.VARCHAR;

	/** Datatype of language of literal. **/
	public String LANGUAGE = "character varying(16)";
	public int LANGUAGE_TYPE = Types.VARCHAR;

	/** Datatype of label of literal (arbitrary length unicode string). **/
	public String LABEL = "clob";
	public int LABEL_TYPE = Types.CLOB;

	/** Datatype of labelHash of literal (a signed 64-bit long). **/
	public String LABEL_HASH = "bigint";
	public int LABEL_HASH_TYPE = Types.BIGINT;

	/** Datatype of prefix of namespace. **/
	public String PREFIX = "character varying(16)";
	public int PREFIX_TYPE = Types.VARCHAR;
	public int MAX_PREFIX_LENGTH = 16;

	/** Datatype of name of namespace. **/
	public String NAME = "clob";
	public int NAME_TYPE = Types.CLOB;

	/** Datatype of boolean. **/
	public String BOOLEAN = "boolean";
	public int BOOLEAN_TYPE = Types.BOOLEAN;

	/** Boolean value 'true'. **/
	public String TRUE = "TRUE";

	/** Boolean value 'false'. **/
	public String FALSE = "FALSE";

	/** Datatype of fields (both key and value) in the repository info table (unicode strings of max 255 characters). **/
	public String INFOFIELD = "character varying(255)";
	public int INFOFIELD_TYPE = Types.VARCHAR;

/*----------------------------------------+
| Variables                               |
+----------------------------------------*/

	/** A connection pool that keeps track of reusable connections. **/
	protected ConnectionPool _conPool;

	/**
	 * Stores information on the number of rows that a specific table contains.
	 * The key is the table's name (String), the value is an Integer.
	 **/
	protected Map _tableRowCounts = new HashMap(32);

	/**
	 * Stores information on the number of rows that have been modified in a
	 * specific table contains. The key is the table's name (String), the value
	 * is an Integer.
	 **/
	protected Map _tableModRowCounts = new HashMap(32);

/*----------------------------------------+
| Constructors                            |
+----------------------------------------*/

	/**
	 * Creates a nw RDBMS.
	 **/
	public RDBMS() {
	}

/*----------------------------------------+
| Methods related to connections          |
+----------------------------------------*/

	/**
	 * Sets the information that is required for connecting to the RDBMS.
	 **/
	public void setConnectionInfo(String jdbcUrl, String user, String password)
		throws SQLException
	{
		disconnect();

		// Create ConnectionPool
		_createConnectionPool(jdbcUrl, user, password);

		// Initialize database-dependent "constants"
		Connection con = getConnection();
		DatabaseMetaData metaData = con.getMetaData();
		_initConstants(metaData);
		con.close();
	}

	/**
	 * Creates a connection pool for this RDBMS.
	 **/
	protected void _createConnectionPool(String jdbcUrl, String user, String passwd) {
		_conPool = new ConnectionPool(jdbcUrl, user, passwd);

		// Don't check status of connections, this takes too long:
		_conPool.setCheckConnections(false);

		// Queries should not run for more than 1 hour:
		_conPool.setMaxUseTime(60 * 60 * 1000);
	}

	/**
	 * Initializes database-dependent "constants" using the metadata supplied by
	 * the JDBC-driver.
	 **/
	protected void _initConstants(DatabaseMetaData metaData)
		throws SQLException
	{
		_supportsLikeEscapeClause = metaData.supportsLikeEscapeClause();
		ThreadLog.trace("RDBMS supports like escape-clause: " + _supportsLikeEscapeClause);

		_searchStringEscape = metaData.getSearchStringEscape();
		if (_searchStringEscape != null && _searchStringEscape.length() == 0) {
			_searchStringEscape = null;
		}
		ThreadLog.trace("RDBMS search string escape: " + _searchStringEscape);		
	}

	/**
	 * Closes any open connections to the RDBMS.
	 **/
	public void disconnect() {
		if (_conPool != null) {
			_conPool.drain();
		}
	}

	/**
	 * Gets a connection to the RDBMS.
	 **/
	public Connection getConnection()
		throws SQLException
	{
		return _conPool.getConnection();
	}

/*----------------------------------------+
| Utility methods for querying            |
+----------------------------------------*/

	/**
	 * Executes an update query.
	 *
	 * @return The return value of the update (e.g. number of affected rows).
	 **/
	public int executeUpdate(String query)
		throws SQLException
	{
		Connection con = getConnection();

		try {
			Statement st = con.createStatement();

			try {
				return st.executeUpdate(query);
			}
			finally {
				st.close();
			}
		}
		finally {
			con.close();
		}
	}

	/**
	 * Evaluates the specific query and check whether it has any results.
	 **/
	public boolean queryHasResults(String query)
		throws SQLException
	{
		Connection con = getConnection();

		try {
			Statement st = con.createStatement();

			try {
				ResultSet rs = st.executeQuery(query);
				boolean result = rs.next();
				rs.close();

				return result;
			}
			finally {
				st.close();
			}
		}
		finally {
			con.close();
		}
	}

/*----------------------------------------+
| Methods for indexes                     |
+----------------------------------------*/

	/**
	 * Creates an index on the specific column in the specific table.
	 *
	 * @param table The table name.
	 * @param column The column name.
	 **/
	public void createIndex(String table, String column) 
	 	throws SQLException
	{
		createIndex(table, new String[] {column}, false);
	}

	/**
	 * Creates a unique index on the specified column in the specified table.
	 *
	 * @param table The table name.
	 * @param column The column name.
	 **/
	public void createUniqueIndex(String table, String column) 
		throws SQLException
	{
		createIndex(table, new String[] {column}, true);
	}

	/**
	 * Creates an index on the specified columns in the specified table.
	 *
	 * @param table The table name.
	 * @param columns The column names
	 * @param unique Flag indicating whether the index should be a unique index.
	 **/
	public void createIndex(String table, String[] columns, boolean unique)
		throws SQLException
	{
		if (columns.length == 0) {
			throw new IllegalArgumentException(
					"columns must contain at least one column name");
		}

		StringBuffer query = new StringBuffer(64);
		query.append("CREATE ");
		if (unique) {
			query.append("UNIQUE ");
		}
		query.append("INDEX ").append(getIndexName(table, columns));
		query.append(" ON ").append(table);

		query.append(" (").append(columns[0]);

		for (int i = 1; i < columns.length; i++) {
			query.append(", ").append(columns[i]);
		}
		query.append(")");

		executeUpdate(query.toString());
	}

	/**
	 * Drops the index on the specified column name from the specified table.
	 *
	 * @param table The table name.
	 * @param column The column name.
	 **/
	public void dropIndex(String table, String column)
	 	throws SQLException
	{
		dropIndex(table, new String[] {column});
	}

	/**
	 * Drops the index on the specified columns from the specified table.
	 *
	 * @param table The table name.
	 * @param columns The column names.
	 **/
	public void dropIndex(String table, String[] columns)
	 	throws SQLException
	{
		executeUpdate("DROP INDEX " + getIndexName(table, columns));
	}

	/**
	 * Creates an index name based on the name of the column and table that
	 * it's supposed to index.
	 **/
	public String getIndexName(String table, String column) {
		return getIndexName(table, new String[] {column});
	}

	/**
	 * Creates an index name based on the name of the columns and table that
	 * it's supposed to index.
	 **/
	public String getIndexName(String table, String[] columns) {
		StringBuffer name = new StringBuffer(32);

		name.append(table).append("_").append(columns[0]);

		for (int i = 1; i < columns.length; i++) {
			name.append("_").append(columns[i]);
		}

		name.append("_idx");

		return name.toString();
	}

/*----------------------------------------+
| Methods for tables                      |
+----------------------------------------*/

	/**
	 * Checks whether a table with the specified name exists.
	 **/
	public boolean tableExists(String tableName)
		throws SQLException
	{
		boolean tableExists = false;

		Connection con = getConnection();

		try {
			Statement st = con.createStatement();

			try {
				ResultSet rs = st.executeQuery("select count(*) from " + tableName);
				tableExists = rs.next();
				rs.close();
			}
			catch (SQLException e) {
				// ignore, assume this means that the table doesn't exist
			}
			finally {
				st.close();
			}
		}
		finally {
			con.close();
		}

        return tableExists;
	}

	/**
	 * Optimizes a table. The actual action taken depends on the database.
	 **/
	public void optimizeTable(String tableName, int modifiedRowsCount)
	 	throws SQLException
	{
		//ThreadLog.trace("optimizeTable(\"" + tableName + "\", " + modifiedRowsCount + "\")");
		if (modifiedRowsCount <= 0) {
			return;
		}

		// Get row count
		Integer i = (Integer)_tableRowCounts.get(tableName);
		int rowCount = (i == null) ? _getRowCount(tableName) : i.intValue();

		// Get modified row count
		i = (Integer)_tableModRowCounts.get(tableName);
		int modRowCount = (i == null) ? 0 : i.intValue();

		// Update modified row count
		modRowCount += modifiedRowsCount;

		//ThreadLog.trace("rowCount=" + rowCount + "; modRowCount=" + modRowCount);

		if (modRowCount > 10000 || // More than 10000 rows changed
			rowCount / modRowCount <= 2) // More than 50% of the rows changed
		{
			optimizeTable(tableName);
			rowCount = _getRowCount(tableName);
			modRowCount = 0;
		}

		// Update statistics
		_tableRowCounts.put(tableName, new Integer(rowCount));
		_tableModRowCounts.put(tableName, new Integer(modRowCount));
		//ThreadLog.trace("optimizeTable() done");
	}

	protected int _getRowCount(String tableName)
		throws SQLException
	{
		Connection con = getConnection();
		try {
			Statement st = con.createStatement();

			try {
				ResultSet rs = st.executeQuery("SELECT COUNT(*) FROM " + tableName);
				rs.next();
				int result = rs.getInt(1);
				rs.close();

				return result;
			}
			finally {
				st.close();
			}
		}
		finally {
			con.close();
		}
	}

	/**
	 * Optimizes a table. The actual action taken depends on the database.
	 **/
	public void optimizeTable(String tableName)
	 	throws SQLException
	{
		// There is no default for this in SQL92.
	}

	/**
	 * Clears a table.
	 **/
	public final void clearTable(String tableName)
		throws SQLException
	{
		_clearTable(tableName);

		_tableRowCounts.put(tableName, new Integer(0));
		_tableModRowCounts.put(tableName, new Integer(0));
	}

	protected void _clearTable(String tableName)
		throws SQLException
	{
		executeUpdate("DELETE FROM " + tableName);
	}

	/**
	 * Drops a table.
	 **/
	public final void dropTable(String tableName)
		throws SQLException
	{
		_dropTable(tableName);

		_tableRowCounts.remove(tableName);
		_tableModRowCounts.remove(tableName);
	}

	protected void _dropTable(String tableName)
		throws SQLException
	{
		executeUpdate("DROP TABLE " + tableName);
	}

	/**
	 * Renames a table.
	 **/
	public final void renameTable(String currentTableName, String newTableName)
		throws SQLException
	{
		_renameTable(currentTableName, newTableName);

		Integer rowCount = (Integer)_tableRowCounts.remove(currentTableName);
		if (rowCount != null) {
			_tableRowCounts.put(newTableName, rowCount);
		}

		Integer modRowCount = (Integer)_tableModRowCounts.remove(currentTableName);
		if (modRowCount != null) {
			_tableModRowCounts.put(newTableName, modRowCount);
		}
	}

	protected void _renameTable(String currentTableName, String newTableName)
		throws SQLException
	{
		executeUpdate("ALTER TABLE " + currentTableName + " RENAME TO " + newTableName);
	}

	public void renameTableColumn(String tableName, String currentColumnName, String newColumnName, String columnSignature)
		throws SQLException
	{
		executeUpdate("ALTER TABLE " + tableName + " RENAME COLUMN " + currentColumnName + " TO " + newColumnName);
	}

	/**
	 * Copies rows from one table to another.
	 * @return the number of rows that were copied.
	 **/
	public int copyRows(String sourceTable, String targetTable)
		throws SQLException
	{
		return executeUpdate(
				"INSERT INTO " + targetTable +
				" SELECT * FROM " + sourceTable);
	}

	/**
	 * Copies distinct rows from one table to another (duplicates are suppressed).
	 * @return the number of rows that were copied.
	 **/
	public int copyDistinctRows(String sourceTable, String targetTable)
		throws SQLException
	{
		return executeUpdate(
				"INSERT INTO " + targetTable +
				" SELECT DISTINCT * FROM " + sourceTable);
	}

/*----------------------------------------+
| Other methods                           |
+----------------------------------------*/

	/**
	 * Converts a boolean value to a string representation that can be used
	 * in a query for this RDBMS.
	 **/
	public String convertBoolean(boolean b) {
		return b ? TRUE : FALSE;
	}

	/**
	 * Escapes any special characters in the specifed string such that it can
	 * be used in a query for this RDBMS.
	 *
	 * @return The original string with escape codes for any special characters.
	 **/
	public String escapeString(String s) {
		String result = StringUtil.gsub("\\", "\\\\", s);
		result = StringUtil.gsub("'", "\\'", result);
		return result;
	}

	/**
	 * Should return <tt>true</tt> if the database converts empty string to
	 * NULL. Default return value is <tt>false</tt>.
	 **/
	public boolean emptyStringIsNull() {
		return false;
	}
	
	public boolean supportsPatternMatches(boolean caseSensitive) {
		return false;
	}
	
	public String getPatternMatchOperator(boolean caseSensitive) {
		throw new UnsupportedOperationException();
	}

	public String getPatternMatchExpr(String pattern, boolean caseSensitive) {
		throw new UnsupportedOperationException();
	}

	/**
	 * Indicates whether the database supports LIKE "..." ESCAPE '\' constructions.
	 **/
	public boolean supportsLikeEscapeClause() {
		return _supportsLikeEscapeClause;
	}

	/**
	 * Returns the character string that can be used to escape special
	 * characters in string patterns.
	 **/
	public String getSearchStringEscape() {
		return _searchStringEscape;
	}
}
