/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sailimpl.rdbms;

import java.sql.SQLException;
import java.sql.Types;

import org.openrdf.util.StringUtil;

/**
 * Defines PostgreSQL specific SQL syntax.
 */
public class PostgreSQL extends RDBMS {

	/**
	 * Initializes PostgreSQL specific SQL syntax.
	 */
	public PostgreSQL() {
		super();

		// override type for text fields; PostgreSQL's 'text' datatype allows
		// long string values to be stored without having a negative effect on
		// performance (when compared to char or varchar fields). Further, CLOBs
		// cannot be used as they are mapped to oid instead of text.
		LOCALNAME = LANGUAGE = LABEL = NAME = INFOFIELD = "text";
		LOCALNAME_TYPE = LANGUAGE_TYPE = LABEL_TYPE = NAME_TYPE = INFOFIELD_TYPE = Types.LONGVARCHAR;
	}

	public void optimizeTable(String tableName)
		throws SQLException
	{
		executeUpdate("VACUUM ANALYZE " + tableName);
	}

	// Overrides RDBMS._clearTable()
	protected void _clearTable(String tableName)
		throws SQLException
	{
		executeUpdate("TRUNCATE TABLE " + tableName);
	}

	public boolean supportsPatternMatches(boolean caseSensitive) {
		return true;
	}

	public String getPatternMatchOperator(boolean caseSensitive) {
		return caseSensitive ? "LIKE" : "ILIKE";
	}

	public String getPatternMatchExpr(String pattern, boolean caseSensitive) {
		// Note that the backslash already has a special meaning in string
		// literals, so to write a pattern constant that contains a backslash you
		// must write two backslashes in an SQL statement. Thus, writing a
		// pattern that actually matches a literal backslash means writing four
		// backslashes in the statement.
		// Source: http://www.postgresql.org/docs/8.1/interactive/functions-matching.html

		// Note: the distinction between case-sensitive and case-insensitive i
		// made by the operator being used (see method getPatternMatchOperator).

		// escape any escape characters
		pattern = StringUtil.gsub("\\", "\\\\\\\\", pattern);

		// Escape any SQL-wildcard characters
		pattern = StringUtil.gsub("%", "\\\\%", pattern);
		pattern = StringUtil.gsub("_", "\\\\_", pattern);

		// Replace pattern wildcards with SQL wildcards
		pattern = StringUtil.gsub("*", "%", pattern);

		return pattern;
	}
}
