/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sailimpl.rdbms;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Types;

import org.openrdf.util.StringUtil;

/**
 * Defines Oracle specific SQL syntax.
 * tested with oracle 9.2.0.1.0
 *
 * @author Holger Lausen
 * @version $Revision: 1.9.4.2 $
 */
public class Oracle extends RDBMS {

	/**
	 * Initializes Oracle specific SQL syntax.
	 */
	public Oracle() {
		super();

		BOOLEAN = "number(1)";
		BOOLEAN_TYPE = Types.TINYINT;
		TRUE = "1";
		FALSE = "0";
		LABEL_HASH = "number(21,0)"; // 64-bit signed integer
		LABEL = "varchar(4000)"; // CLOB is problematic with comparison to strings in where clause
		LABEL_TYPE = Types.VARCHAR;
		NAME = "varchar(4000)";
		NAME_TYPE = Types.VARCHAR;
	}

	/**
	 * within Oracle '' gets converted to null
	 */
	public boolean emptyStringIsNull() {
		return true;
	}
	
	/**
	 * Creates an index name based on the name of the columns and table that
	 * it's supposed to index. This method overrides the implementation in the
	 * superclass because some index names get to long for Oracle.
	 **/
	public String getIndexName(String table, String[] columns) {
		StringBuffer name = new StringBuffer(32);

		name.append(table).append("_");

		for (int i = 0; i < columns.length; i++) {
			name.append( columns[i].substring(0, Math.min(4,columns[i].length())) );
		}

		return name.toString();
	}

	// Overrides RDBMS.optimizeTable()
	public void optimizeTable(String tableName)
		throws SQLException
	{
		Connection con = getConnection();
		Statement st = con.createStatement();

		ResultSet rs = st.executeQuery("ANALYZE TABLE " + tableName + " ESTIMATE STATISTICS");

		rs.close();
		st.close();
		con.close();
	}

	// Overrides RDBMS._clearTable()
	protected void _clearTable(String tableName)
		throws SQLException
	{
		executeUpdate("TRUNCATE TABLE " + tableName);
	}

	/**
	 * Drops a table, cascading to any constraints. This method overrides the
	 * default implementation, which only drops the table.
	 **/
	protected void _dropTable(String tableName)
		throws SQLException
	{
		executeUpdate("DROP TABLE " + tableName + " CASCADE CONSTRAINTS");
	}

	/**
	 * Replaces any occurences of <tt>'</tt> with <tt>''</tt>.
	 *
	 * @return The original string with quotes escaped to two quotes.
	 **/
	public String escapeString(String s) {
		String result = StringUtil.gsub("'", "''", s);
		return result;
	}
}

