/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sailimpl.rdbms;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Types;

import org.openrdf.util.StringUtil;
import org.openrdf.util.log.ThreadLog;

/**
 * Defines MySQL specific SQL syntax.
 * 
 * @author Arjohn Kampman
 */
public class MySQL extends RDBMS {

	private int _majorVersion;

	private int _minorVersion;

	/**
	 * Initializes MySQL specific SQL syntax.
	 */
	public MySQL() {
		super();
	}

	public int getMajorVersion() {
		return _majorVersion;
	}

	public int getMinorVersion() {
		return _minorVersion;
	}

	protected void _initConstants(DatabaseMetaData metaData)
		throws SQLException
	{
		super._initConstants(metaData);

		_majorVersion = metaData.getDatabaseMajorVersion();
		_minorVersion = metaData.getDatabaseMinorVersion();
		
		// FIXME: MySQL doesn't allow the VARCHAR fields larger than 255 to be
		// indexed. This is problematic for some data with large local names in
		// URIs

		// The default character types are case-insensitive in
		// MySQL, define case-sensitive alternatives:
		if (_majorVersion == 4 && _minorVersion >= 1 || _majorVersion >= 5) {
			// MySQL 4.1 and higher support character set and -collation
			// specifications
			ThreadLog.trace("Detected MySQL 4.1 or newer, using case-sensitive utf-8 character columns");

			LOCALNAME = "VARCHAR(255) CHARACTER SET utf8 COLLATE utf8_bin";
			LANGUAGE = "VARCHAR(16) CHARACTER SET utf8 COLLATE utf8_bin";
			LABEL = "TEXT CHARACTER SET utf8 COLLATE utf8_bin";
			LABEL_TYPE = Types.LONGVARCHAR;
			PREFIX = "VARCHAR(16) CHARACTER SET utf8 COLLATE utf8_bin";
			NAME = "TEXT CHARACTER SET utf8 COLLATE utf8_bin";
			NAME_TYPE = Types.LONGVARCHAR;
			INFOFIELD = "VARCHAR(255) CHARACTER SET utf8 COLLATE utf8_bin";
		}
		else {
			ThreadLog.trace("Detected MySQL 4.0 or older, using binary varchar columns");

			LOCALNAME = "VARCHAR(255) BINARY";
			LANGUAGE = "VARCHAR(16) BINARY";
			LABEL = "BLOB";
			LABEL_TYPE = Types.BLOB;
			PREFIX = "VARCHAR(16) BINARY";
			NAME = "BLOB";
			NAME_TYPE = Types.BLOB;
			INFOFIELD = "VARCHAR(255) BINARY";
		}

		// MySQL doesn't support the 'boolean' datatype
		BOOLEAN = "BOOL";
		TRUE = "1";
		FALSE = "0";
	}

	/**
	 * Overrides dropIndex in RDBMS. In MySQL, an index is identified by both its
	 * index and table name.
	 */
	public void dropIndex(String table, String[] columns)
		throws SQLException
	{
		executeUpdate("DROP INDEX " + getIndexName(table, columns) + " ON " + table);
	}

	// Overrides RDBMS.optimizeTable()
	public void optimizeTable(String tableName)
		throws SQLException
	{
		// ThreadLog.trace("optimizing table " + tableName);
		Connection con = getConnection();
		Statement st = con.createStatement();

		ResultSet rs = st.executeQuery("OPTIMIZE TABLE " + tableName);

		rs.close();
		st.close();
		con.close();
		// ThreadLog.trace("done optimizing table");
	}

	// Overrides RDBMS._clearTable()
	protected void _clearTable(String tableName)
		throws SQLException
	{
		executeUpdate("TRUNCATE TABLE " + tableName);
	}

	public void renameTableColumn(String tableName,
		String currentColumnName, String newColumnName, String columnSignature)
		throws SQLException
	{
		executeUpdate(
				"ALTER TABLE " + tableName +
				" CHANGE " + currentColumnName + " " + newColumnName + " " + columnSignature);
	}

	public boolean supportsPatternMatches(boolean caseSensitive) {
		return true;
	}

	public String getPatternMatchOperator(boolean caseSensitive) {
		return caseSensitive ? "LIKE" : "REGEXP";
	}

	public String getPatternMatchExpr(String pattern, boolean caseSensitive) {
		// To use a literal instance of a special character in a regular
		// expression, precede it by two backslash (\) characters. The MySQL
		// parser interprets one of the backslashes, and the regular expression
		// library interprets the other.
		// Source: http://mysql.com/doc/refman/5.0/en/regexp.html

		if (caseSensitive) {
			// escape any escape characters
			pattern = StringUtil.gsub("\\", "\\\\\\\\", pattern);

			// Escape any SQL-wildcard characters
			pattern = StringUtil.gsub("%", "\\\\%", pattern);
			pattern = StringUtil.gsub("_", "\\\\_", pattern);

			// Replace pattern wildcards with SQL wildcards
			pattern = StringUtil.gsub("*", "%", pattern);

			return pattern;
		}
		else {
			// case insensitive matches in mysql are done using regular expressions
			// FIXME: see if there are more efficiently alternatives

			// REGEXP by default do substring matches. Therefore, wildcards at the
			// start and end of the pattern can be ignored
			boolean startsWithWildcard = pattern.startsWith("*");
			boolean endsWithWildcard = pattern.endsWith("*") && pattern.length() >= 2;

			if (startsWithWildcard) {
				pattern = pattern.substring(1);
			}
			if (endsWithWildcard) {
				pattern = pattern.substring(0, pattern.length() - 1);
			}

			StringBuffer regexp = new StringBuffer(pattern.length() * 4);

			if (!startsWithWildcard) {
				// expression needs to match the start of the string
				regexp.append('^');
			}

			for (int i = 0; i < pattern.length(); i++) {
				char c = pattern.charAt(i);

				if (c == '\\') {
					// Escape the escape character
					regexp.append("\\\\\\\\");
				}
				else if (c == '%' || c == '_' || c == '^' || c == '$') {
					// Escape special characters
					regexp.append("\\\\").append(c);
				}
				else if (c == '*') {
					// Replace with SQL-wildcard
					regexp.append(".*");
				}
				else {
					char ucChar = Character.toUpperCase(c);
					char lcChar = Character.toLowerCase(c);

					if (ucChar == lcChar) {
						// No upper- and lowercase variants for this character
						regexp.append(c);
					}
					else {
						regexp.append('[');
						regexp.append(ucChar);
						regexp.append(lcChar);
						regexp.append(']');
					}
				}
			}

			if (!endsWithWildcard) {
				// expression needs to match the end of the string
				regexp.append('$');
			}

			return regexp.toString();
		}
	}
}
