/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2003 OntoText Lab, Sirma AI OOD
 *
 *  Contact:
 *	Sirma AI OOD, OntoText Lab.
 *	38A, Christo Botev Blvd.
 *  1000 Sofia, Bulgaria
 *	tel. +359(2)981 00 18
 *	fax. +359(2)981 90 58
 *	info@ontotext.com
 *
 * 	http://www.ontotext.com/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sailimpl.rdbms;

import java.io.IOException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Map;

import org.xml.sax.SAXException;

import org.openrdf.util.log.ThreadLog;

import org.openrdf.model.impl.URIImpl;

import org.openrdf.sesame.sail.SailInternalException;
import org.openrdf.sesame.sail.SailUpdateException;
import org.openrdf.sesame.sailimpl.rdbms.rules.Rule;
import org.openrdf.sesame.sailimpl.rdbms.rules.RuleParser;
import org.openrdf.sesame.sailimpl.rdbms.rules.TripleTemplate;

/**
 * <p>Title: Custom Inference Rules</p>
 * <p>Description: </p>
 * <p>Copyright: Copyright (c) 2003</p>
 * <p>Company: Ontotext Lab. Sirma AI</p>
 * @author Damyan Ognyanoff
 * @version 1.0
 */
public class CustomInferenceServices implements InferenceServices {

	public static final String PARAM_RULE_FILE = "rule-file";

	static final String DEFAULT_RULE_FILE = "org/openrdf/sesame/sailimpl/rdbms/entailment-rdfmt-REC.xml";

	private static int maxTemplateCountsPerRule = 2;

	private boolean _useDependencyInferencer = true;

	public static int getMaxTemplateCountsPerRule() {
		return maxTemplateCountsPerRule;
	}

	ArrayList rules = null;

	ArrayList axioms = null;

	RdfSchemaRepository _sail = null;

	protected Connection _insertCon;

	protected PreparedStatement _insertSt;

	protected int iteration;

	protected int _nofInferred = 0;

	// Variables for optimization statistics
	protected int _totalInferred = 0;

	// number of inferred statements per rule.
	protected int[] _ruleCount = null;

	// total processing time per rule.
	protected long[] _ruleTime = null;

	ArrayList inferenceSQLs = null;

	ArrayList dependenceSQLs = null;

	protected boolean[][] _ruleTrigers = null;

	protected boolean[] _ruleIsApplied = null;

	public CustomInferenceServices() {
	}

	public void setDependencyInferencer(boolean useDependencyInferencer) {
		_useDependencyInferencer = useDependencyInferencer;
	}

	public void initialize(RdfSchemaRepository sail, Map configParams) {
		_sail = sail;
		
		String ruleFileName = (String)configParams.get(PARAM_RULE_FILE);
		if (ruleFileName == null) {
			ThreadLog.warning("No rule file defined. using default.");
			ruleFileName = DEFAULT_RULE_FILE;
		}
		ThreadLog.trace("Using rule file: " + ruleFileName);

		RuleParser parser = new RuleParser();

		try {
			parser.load(ruleFileName);
		}
		catch (IOException e) {
			ThreadLog.error("Rule parser i/o error: " + e.getMessage());
			throw new SailInternalException(e);
		}
		catch (SAXException e) {
			ThreadLog.error("Error parsing entailment rules: " + e.getMessage());
			throw new SailInternalException(e);
		}

		rules = parser.getRules();
		axioms = parser.getAxioms();
		Iterator iter = rules.iterator();
		int max = 0;
		while (iter.hasNext()) {
			Rule r = (Rule)iter.next();
			if (max < r.getPremiseCount()) {
				max = r.getPremiseCount();
			}
		}
		maxTemplateCountsPerRule = max;
	}

	/**
	 * invoked to initilize the basic RDFS schema for the repository
	 */
	public void initRdfSchema() {
		// preSet the resource IDs used by axioms and rules
		// they are collected and stored in Rule.idMap;
		Iterator iter = Rule.constantsIter();
		while (iter.hasNext()) {
			String uri = (String)iter.next();
			int id = _sail._insertURI(new URIImpl(uri));
			Rule.setId(uri, id);
		}
		_initialize();
	}

	protected void _buildSQLs() {
		inferenceSQLs = new ArrayList();
		dependenceSQLs = new ArrayList();
		Iterator iter = rules.iterator();
		while (iter.hasNext()) {
			Rule r = (Rule)iter.next();
			ArrayList list = r.getAllSQLs();
			inferenceSQLs.addAll(list);
			list = r.getDependSQL();
			dependenceSQLs.addAll(list);
		}
		_ruleCount = new int[inferenceSQLs.size()];
		_ruleTime = new long[inferenceSQLs.size()];
		_ruleIsApplied = new boolean[inferenceSQLs.size()];
		_ruleTrigers = buildTriggers(rules);
	}

	void _initialize() {
		try {
			// As defined in the RDF MT: add a number of default statements
			// concerning the definition of RDF and RDF Schema primitives
			_addAxioms();
			_buildSQLs();
			// ...and apply the inference rules to them.
			doInferencing();
		}
		catch (SailUpdateException e) {
			throw new SailInternalException(e);
		}
	}

	/**
	 * the entry point for statement inferencing
	 */
	public void doInferencing() {
		iteration = 1;

		// initialize some vars
		_totalInferred = 0;

		for (int i = 0; i < inferenceSQLs.size(); i++) {
			_ruleCount[i] = 0;
			_ruleTime[i] = 0;
		}

		try {
			_nofInferred = 1;
			ThreadLog.trace("starting inferencing");

			// at first iteration we should apply all the rules(rule variants)
			boolean rulesToApply[] = new boolean[_ruleIsApplied.length];
			java.util.Arrays.fill(rulesToApply, true);
			while (_nofInferred > 0) {

				// Apply inferencing to all statements in newTriples.
				// Each rule will insert any newly inferred statements into
				// INFERRED_TABLE and ALL_INFERRED_TABLE.

				_nofInferred = 0;

				// the line bellow means that there is no rule that infers new statements
				java.util.Arrays.fill(_ruleIsApplied, false);
				// Batch-oriented processing:
				for (int ruleNo = 0; ruleNo < inferenceSQLs.size(); ruleNo++) {
					// should we apply the rule?
					if (false == rulesToApply[ruleNo]) {
						continue;
					}
					// how much new statements are infered by the rule
					int newTriplesInferred = _applyRule(ruleNo, (String)inferenceSQLs.get(ruleNo));
					// mark that this rule triggers some other rules at next iteration
					_ruleIsApplied[ruleNo] = (newTriplesInferred > 0);
					_nofInferred += newTriplesInferred;
				}
				_totalInferred += _nofInferred;
				// let clear all the rule activation flags
				java.util.Arrays.fill(rulesToApply, false);
				if (_nofInferred > 0) {
					// iterate over the flags thet indicate thet some rule has inferred new statements
					for (int riRow = 0; riRow < rulesToApply.length; ++riRow) {
						if (false == _ruleIsApplied[riRow]) {
							continue;
						}

						// get the list of triggered rules by thet rule
						boolean[] ruleRowFromTrigers = _ruleTrigers[riRow];
						for (int riColumn = 0; riColumn < rulesToApply.length; ++riColumn) {
							// merge it with the current state
							rulesToApply[riColumn] = (rulesToApply[riColumn] || ruleRowFromTrigers[riColumn]);
						}
					} // computing the list of active rules during the next iteration
				} // if there are new statements inferred in the last iteration

				// The triples from NEW_TRIPLES_TABLE have been processed,
				// continue with the newly inferred triples from ALL_INFERRED_TABLE

				_sail._rdbms.clearTable(NEW_TRIPLES_TABLE);

				if (_nofInferred > 0) {
					_sail._rdbms.copyRows(ALL_INFERRED_TABLE, NEW_TRIPLES_TABLE);
					_sail._rdbms.copyRows(ALL_INFERRED_TABLE, ALL_NEW_TRIPLES_TABLE);

					_sail._rdbms.clearTable(ALL_INFERRED_TABLE);

					_sail._rdbms.optimizeTable(NEW_TRIPLES_TABLE);
					_sail._rdbms.optimizeTable(TRIPLES_TABLE);
				}

				ThreadLog.trace("iteration " + iteration + " done; " + "inferred " + _nofInferred + " new statements");

				iteration++;
			}

			_sail._rdbms.optimizeTable(ALL_NEW_TRIPLES_TABLE);
		}
		catch (SQLException e) {
			throw new SailInternalException(e);
		}
		catch (SailUpdateException e) {
			throw new SailInternalException(e);
		}

		// Print some statistics
		ThreadLog.trace("---RdfMTInferencer statistics:---");
		ThreadLog.trace("total statements inferred = " + _totalInferred);
		for (int i = 0; i < inferenceSQLs.size(); i++) {
			ThreadLog.trace("rule " + i + ": time=" + _ruleTime[i] + ";\t#inferred=" + _ruleCount[i]);
		}
		ThreadLog.trace("---end of statistics:---");
	}

	/**
	 * invoked when some statements are being removed
	 * @throws SQLException
	 */
	public void removeExpiredStatements()
		throws SQLException
	{
		if (_useDependencyInferencer) {
			_makeExpiredStatementsInferred();
			_determineGroundedStatements();

			// Get the IDs of statements that are no longer grounded.
			Connection con = _sail._rdbms.getConnection();
			java.sql.Statement st = con.createStatement();
			ResultSet rs = st.executeQuery(
					"SELECT t.id FROM " + TRIPLES_TABLE + " t " +
					"LEFT JOIN " + GROUNDED_TRIPLES_TABLE + " g " +
					"ON t.id = g.id " +
					"WHERE g.id IS NULL");

			String[] idChunks = _sail._chunkIdSet(rs, 3500);
			rs.close();
			st.close();

			// Delete these statements from TRIPLES_TABLE and DEPEND_TABLE.
			con.setAutoCommit(false);
			st = con.createStatement();

			for (int i = 0; i < idChunks.length; i++) {
				st.executeUpdate(
						"DELETE FROM " + TRIPLES_TABLE +
						" WHERE id IN " + idChunks[i]);
				st.executeUpdate(
						"DELETE FROM " + DEPEND_TABLE +
						" WHERE id IN " + idChunks[i]);

				for (int j = 1; j <= maxTemplateCountsPerRule; j++) {
					st.executeUpdate(
							"DELETE FROM " + DEPEND_TABLE +
							" WHERE dep" + j + " IN " + idChunks[i]);
				}

				_sail._processChunkFromRemoveExpiredStatements(idChunks[i]);

			}

			con.commit();
			st.close();
			con.close();

			_sail._rdbms.clearTable(GROUNDED_TRIPLES_TABLE);
			_sail._rdbms.clearTable(NEW_GROUNDED_TRIPLES_TABLE);
		}
		else { // don't use dependency inferencer
			// mark all expired statements as inferred
			_makeExpiredStatementsInferred();

			// remove all inferred statements
			_removeAllInferred();

			// copy all TRIPLES to the NEW_TRIPLES table
			_sail._rdbms.copyRows(TRIPLES_TABLE, NEW_TRIPLES_TABLE);

			// clean the TRIPLES table.
			_sail._rdbms.clearTable(TRIPLES_TABLE);

			// re-compute the closure
			_initialize();
		}
	}

	/**
	 * Set the 'explicit' flag to 'false' for all statements whose ID
	 * is in the EXPIRED_TRIPLES_TABLE.
	 **/
	protected void _makeExpiredStatementsInferred()
		throws SQLException
	{
		Connection con = _sail._rdbms.getConnection();
		java.sql.Statement st = con.createStatement();
		ResultSet rs = st.executeQuery("SELECT DISTINCT id FROM " + EXPIRED_TRIPLES_TABLE);

		String[] idChunks = _sail._chunkIdSet(rs, 3500);

		rs.close();
		st.close();

		// Mark all expired statements to be inferred. The statements
		// cannot just be removed, as it is possible that they can also be
		// inferred from other statements.
		con.setAutoCommit(false);
		st = con.createStatement();

		for (int i = 0; i < idChunks.length; i++) {
			st.executeUpdate(
					"UPDATE " + TRIPLES_TABLE +
					" SET explicit = " + _sail._rdbms.FALSE +
					" WHERE id IN " + idChunks[i]);
		}

		con.commit();
		st.close();
		con.close();
	}

	protected void _determineGroundedStatements()
		throws SQLException
	{
		// Statement with ID '0' is grounded. '0' is used in dependencies
		// for axioms (two '0' values), and statements that are dependent
		// of only one statement (dep2 is '0').
		_sail._rdbms.executeUpdate(
				"INSERT INTO " + GROUNDED_TRIPLES_TABLE + " VALUES(0)");

		// All explicit statements are grounded:
		_sail._rdbms.executeUpdate(
				"INSERT INTO " + GROUNDED_TRIPLES_TABLE +
				" SELECT id FROM " + TRIPLES_TABLE +
				" WHERE explicit = " + _sail._rdbms.TRUE);

		int count;
		do {
			String query =
				"INSERT INTO " + NEW_GROUNDED_TRIPLES_TABLE +
				" SELECT DISTINCT d.id FROM " + DEPEND_TABLE + " d" +
				" LEFT JOIN " + GROUNDED_TRIPLES_TABLE + " g ON d.id = g.id";

			for (int i = 1; i <= maxTemplateCountsPerRule; i++) {
				query += 
					" LEFT JOIN " + GROUNDED_TRIPLES_TABLE + " g" + i +
					" ON d.dep" + i + " = g" + i + ".id";
			}

			query += " WHERE g.id IS NULL";

			for (int i = 1; i <= maxTemplateCountsPerRule; i++) {
				query += " AND g" + i + ".id IS NOT NULL";
			}

			count = _sail._rdbms.executeUpdate(query);
			if (count > 0) {
				// Some more grounded statements were found
				_sail._rdbms.copyRows(NEW_GROUNDED_TRIPLES_TABLE, GROUNDED_TRIPLES_TABLE);
				_sail._rdbms.clearTable(NEW_GROUNDED_TRIPLES_TABLE);
				_sail._rdbms.optimizeTable(GROUNDED_TRIPLES_TABLE);
			}
		}
		while (count > 0);
	}

	/**
	 * invoked when some staements are being added so to infer the dependancy information for them
	 * @throws SQLException
	 */
	public void processNewStatements() {
		if (_useDependencyInferencer) {
			Iterator iter = dependenceSQLs.iterator();
			while (iter.hasNext()) {
				String s = (String)iter.next();

				//System.out.println("inferencer query = " +s);
				try {
					_sail._rdbms.executeUpdate(s);
				}
				catch (SQLException e) {
					e.printStackTrace();
				}
			}
		}
	}

	/**
	 * invoked to mark the currently added statements as axioms
	 * @throws SQLException
	 */
	public void markAxioms() {
		if (_useDependencyInferencer) {
			ThreadLog.trace("adding dependencies for axioms");
			try {
				String query =
						"INSERT INTO " + DEPEND_TABLE +
						" SELECT id";

				for (int i = 1; i <= maxTemplateCountsPerRule; i++) {
					query += ", 0";
				}

				query += " FROM " + ALL_NEW_TRIPLES_TABLE;

				_sail._rdbms.executeUpdate(query);
			}
			catch (SQLException e) {
				e.printStackTrace();
			}
			ThreadLog.trace("dependencies for axioms added");
		}
	}

	/**
	 * Prepares a connection and a PreparedStatement for adding
	 * rows to the INFERRED_TABLE.
	 **/
	protected void _prepareInsertConnection()
		throws SQLException
	{
		_insertCon = _sail._rdbms.getConnection();
		_insertCon.setAutoCommit(false);
		_insertSt = _insertCon.prepareStatement(
				"INSERT INTO " + INFERRED_TABLE +
				" VALUES(?, ?, ?, ?, ?)");

		// All statements are implicit here:
		_insertSt.setBoolean(5, false);
	}

	/**
	 * Closes the connection that was prepared in _prepareInsertConnection().
	 * If <tt>mustCommit</tt> is true, the started transaction is committed
	 * first.
	 **/
	protected void _closeInsertConnection(boolean mustCommit)
		throws SQLException
	{
		if (mustCommit) {
			_insertCon.commit();
		}

		_insertSt.close();
		_insertCon.close();
	}

	protected void _addAxioms()
		throws SailUpdateException
	{
		ThreadLog.trace("adding axioms");

		try {
			// Insert all axioms into INFERRED_TABLE

			_prepareInsertConnection();

			// Axiomatic triples from RDF Semantics, section 3.3:
			Iterator iter = axioms.iterator();
			while (iter.hasNext()) {
				TripleTemplate t = (TripleTemplate)iter.next();
				_insertSt.setInt(1, _sail._getNextStatementId());
				_insertSt.setInt(2, Rule.getIntId(t.subject));
				_insertSt.setInt(3, Rule.getIntId(t.predicate));
				_insertSt.setInt(4, Rule.getIntId(t.object));
				_insertSt.executeUpdate();
			}

			_closeInsertConnection(true);

			// Copy all axioms that are not yet in the triples table
			int newAxioms = _sail._rdbms.executeUpdate(
					"INSERT INTO " + NEW_TRIPLES_TABLE +
					" SELECT inf.*" +
					" FROM " + INFERRED_TABLE + " inf" +
					" LEFT JOIN " + TRIPLES_TABLE + " t ON" +
						" inf.subj = t.subj AND" +
						" inf.pred = t.pred AND" +
						" inf.obj = t.obj" +
					" WHERE t.subj IS NULL");

			if (newAxioms > 0) {
				_sail._rdbms.copyRows(NEW_TRIPLES_TABLE, TRIPLES_TABLE);
				_sail._rdbms.copyRows(NEW_TRIPLES_TABLE, ALL_NEW_TRIPLES_TABLE);
			}

			// Clear INFERRED_TABLE
			_sail._rdbms.clearTable(INFERRED_TABLE);
		}
		catch (SQLException e) {
			throw new SailUpdateException(e);
		}
	}

	protected int _applyRule(int ruleNo, String query)
		throws SailUpdateException
	{
		long startTime = System.currentTimeMillis();

		int nofInferred = 0;
		try {
			_prepareInsertConnection();

			Connection queryCon = _sail._rdbms.getConnection();
			java.sql.Statement querySt = queryCon.createStatement();
			ResultSet rs = querySt.executeQuery(query);

			while (rs.next()) {
				_insertSt.setInt(1, _sail._getNextStatementId()); // id
				_insertSt.setInt(2, rs.getInt(1)); // subj
				_insertSt.setInt(3, rs.getInt(2)); // pred
				_insertSt.setInt(4, rs.getInt(3)); // obj
				_insertSt.executeUpdate();

				nofInferred++;
			}

			rs.close();
			querySt.close();
			queryCon.close();

			_closeInsertConnection(nofInferred > 0);

			if (nofInferred > 0) {
				_ruleCount[ruleNo] += nofInferred;

				// Copy the newly inferred statements from INFERRED_TABLE to
				// ALL_INFERRED_TABLE...
				_sail._rdbms.copyRows(INFERRED_TABLE, ALL_INFERRED_TABLE);

				// ...and to TRIPLES_TABLE.
				_sail._rdbms.copyRows(INFERRED_TABLE, TRIPLES_TABLE);

				// Clear INFERRED_TABLE
				_sail._rdbms.clearTable(INFERRED_TABLE);
			}
		}
		catch (SQLException e) {
			ThreadLog.error("SQL error on rule(" + ruleNo + "): " + query);
			throw new SailUpdateException(e);
		}

		long endTime = System.currentTimeMillis();
		_ruleTime[ruleNo] += (endTime - startTime);

		return nofInferred;
	}

	/**
	 * need to handle creation of DEPEND table because some of the rules may need more
	 * than 2 columns to store the dependancy (rules with t=hree or more triple templates in CustomInferencer)
	 * An example of such rule:
	 *	 a rdf:type owl:TransitiveProperty
	 *	 x a b
	 *	 b a c
	 * infer:
	 *	 x a c
	 * so the (x a c) triple is dependant from all those
	 * @throws SQLException
	 */
	public void createDependenciesTable()
		throws SQLException
	{
		if (_useDependencyInferencer) {
			String[] index1Opts = new String[maxTemplateCountsPerRule + 1];
			String[] index2Opts = new String[maxTemplateCountsPerRule];
			index1Opts[0] = "id";
			index1Opts[1] = "dep1";
			index1Opts[2] = "dep2";
			index2Opts[0] = "dep1";
			index2Opts[1] = "dep2";
			String query =
					"CREATE TABLE " + DEPEND_TABLE +
					" (id " + _sail._rdbms.ID_INT + " NOT NULL ";

			for (int i = 1; i <= maxTemplateCountsPerRule; i++) {
				query += ", dep" + i + " " + _sail._rdbms.ID_INT + " NOT NULL";
				index1Opts[i] = "dep" + i;
				index2Opts[i - 1] = "dep" + i;
			}
			query += ")";
			_sail._rdbms.executeUpdate(query);

			_sail._rdbms.createIndex(DEPEND_TABLE, index1Opts, false);
			_sail._rdbms.createIndex(DEPEND_TABLE, index2Opts, false);
		}
	}

	/**
	 * invoked to ensure that thre RDBMS instance is already created within Sail
	 */
	public void afterInitialize() {
		//System.out.println("* AfterInit custom inference");
		if (_sail._rdbms instanceof PostgreSQL) {
			//System.out.println("* Running postgres");
			try {
				_sail._rdbms.executeUpdate(
						"CREATE FUNCTION \"concat\" (text,text)" +
						" RETURNS text AS 'SELECT $1 || $2;' LANGUAGE 'sql';");
			}
			catch (SQLException e) {
				String msg = e.getMessage();
				if (msg.indexOf("already exists") > -1) {
					// FIXME there has to be a better way to handle this
					// concat function already specified, ignore error
				}
				else {
					throw new SailInternalException(e);
				}
			}
		}
	}

	/**
	 * matchTriples - given a consequent and a premise it simply mathces them
	 * reporting that the consequent can be a premise for a rule
	 * @param t1
	 * @param t2
	 * @return true if it si applicable. the arguments are compared by-component.
	 * In case of a varible both componets of the triple can be unified.
	 * In case of an explicit URIs they should be exactly the same.
	 */
	protected boolean matchTriples(TripleTemplate t1, TripleTemplate t2) {
		// special case on transitivity rules
		if (t1.subject.isVar() && t1.object.isVar() &&
			t2.subject.isVar() && t2.object.isVar() &&
			t1.subject.value().equalsIgnoreCase(t1.object.value()) &&
			!t2.subject.value().equalsIgnoreCase(t2.object.value()))
		{
			return false;
		}

		if (!t1.subject.isVar() && !t2.subject.isVar() &&
			!t1.subject.value().equalsIgnoreCase(t2.subject.value()))
		{
			return false;
		}

		if (!t1.predicate.isVar() && !t2.predicate.isVar() &&
			!t1.predicate.value().equalsIgnoreCase(t2.predicate.value()))
		{
			return false;
		}

		if (!t1.object.isVar() && !t2.object.isVar() &&
			!t1.object.value().equalsIgnoreCase(t2.object.value()))
		{
			return false;
		}

		return true;
	} //matchTriples()

	/**
	 * method buildTriggers - given a list of rules it builds the inter-rule dependancy table.
	 * e.g. If a rule inferrs some new statemnents which rules should be applied on next iteration.
	 * This method is implementation specific because it deals with several variants of a rule.
	 * @param rules ArrayList with the Rule instances
	 * @return square array of boolean values. each row holds the flags that indicate which rules(
	 * rule variants) are trigered by this rule(rule variant)
	 */
	protected boolean[][] buildTriggers(ArrayList rules) {
		// Determine the total number of rule variants, which is equal
		// to the amount of premises for the complete set of rules
		int num_rule_variants = 0;
		for (int i = 0; i < rules.size(); i++) {
			Rule r = (Rule)rules.get(i);
			num_rule_variants += r.getPremiseCount();
		}

		// Create a square triggers array
		boolean[][] result = new boolean[num_rule_variants][num_rule_variants];

		int rowIndex = 0;

		for (int i = 0; i < rules.size(); i++) {
			Rule activator = (Rule)rules.get(i);
			int colIndex = 0;

			for (int j = 0; j < rules.size(); j++) {
				Rule activated = (Rule)rules.get(j);

				// Determine if 'activator' triggers 'activated'
				boolean inTrigersList = false;
				if (activator.getTriggersRule() != null) {
					Iterator trIter = activator.getTriggersRule().iterator();

					while (trIter.hasNext()) {
						String toCheck = (String)trIter.next();
						if (toCheck.equalsIgnoreCase(activated.getName())) {
							inTrigersList = true;
							break;
						}
					}
				}

				if (inTrigersList) {
					TripleTemplate consequent = activator.getConsequent();

					for (int indexActivated = 0; indexActivated < activated.getPremiseCount(); indexActivated++) {
						TripleTemplate premiseToCheck =
								(TripleTemplate)activated.getPremiseCollection().get(indexActivated);

						boolean valueToSet = matchTriples(consequent, premiseToCheck);

						for (int activatorSize = 0; activatorSize < activator.getPremiseCount(); activatorSize++) {
							result[rowIndex + activatorSize][colIndex + indexActivated] = valueToSet;
						}
					}
				}

				colIndex += activated.getPremiseCount();
			}

			rowIndex += activator.getPremiseCount();
		}

		return result;
	} // buildTriggers();

	private void _removeAllInferred()
		throws SQLException
	{
		_sail._rdbms.executeUpdate(
				"DELETE FROM " + TRIPLES_TABLE +
				" WHERE explicit = " + _sail._rdbms.FALSE);
	}
}
