/*  OMM - Ontology Middleware Module
*  Copyright (C) 2002 OntoText Lab, Sirma AI OOD
*
*  Contact:
*  Sirma AI OOD, OntoText Lab.
*  38A, Christo Botev Blvd.
*  1000 Sofia, Bulgaria
*  tel. +359(2)981 00 18
*  fax. +359(2)981 90 58
*  info@ontotext.com
*
*  http://www.ontotext.com/
*
*  This library is free software; you can redistribute it and/or
*  modify it under the terms of the GNU Lesser General Public
*  License as published by the Free Software Foundation; either
*  version 2.1 of the License, or (at your option) any later version.
*
*  This library is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
*  Lesser General Public License for more details.
*
*  You should have received a copy of the GNU Lesser General Public
*  License along with this library; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package org.openrdf.sesame.sailimpl.omm.versioning;
/**
 @todo: short notes about the main idea
 * The general idea is to handle only start/commitTransaction and to shiff about
 * changes in Triples table related to explicit statements only - if such occur
 * we should update all _HIST tables among the creation of a new stateID to
 * associate all changes with it.
 *
 **/
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.openrdf.util.log.ThreadLog;

import org.openrdf.model.Literal;
import org.openrdf.model.Resource;
import org.openrdf.model.URI;
import org.openrdf.model.Value;

import org.openrdf.sesame.config.AccessDeniedException;
import org.openrdf.sesame.config.ConfigurationException;
import org.openrdf.sesame.config.RepositoryConfig;
import org.openrdf.sesame.config.SailConfig;
import org.openrdf.sesame.config.SystemConfig;
import org.openrdf.sesame.config.UnknownRepositoryException;
import org.openrdf.sesame.config.UserInfo;
import org.openrdf.sesame.omm.SessionContext;
import org.openrdf.sesame.omm.VersionManagement;
import org.openrdf.sesame.repository.local.LocalRepository;
import org.openrdf.sesame.repository.local.LocalService;
import org.openrdf.sesame.sail.NamespaceIterator;
import org.openrdf.sesame.sail.RdfRepository;
import org.openrdf.sesame.sail.Sail;
import org.openrdf.sesame.sail.SailInitializationException;
import org.openrdf.sesame.sail.SailInternalException;
import org.openrdf.sesame.sail.SailUpdateException;
import org.openrdf.sesame.sailimpl.rdbms.RDBMS;
import org.openrdf.sesame.sailimpl.rdbms.RdfSchemaRepository;
import org.openrdf.sesame.server.SesameServer;

/**
 * <p>Title: </p>
 * <p>Description: </p>
 * <p>Copyright: Copyright (c) 2003</p>
 * <p>Company: </p>
 * @author unascribed
 * @version 1.0
 */
public class VersioningRdbmsSail
    extends RdfSchemaRepository implements VersionManagement {

  public VersioningRdbmsSail() {
  }

  protected ArrayList createdRepositories = new ArrayList();
  protected Connection adminConnection = null;

  /**
   * this methos DROPs all temporary brenches of the repository. invoked either
   * in clearRepository() or whitin shutDown()
   */
  protected void dropBranchedVersions()
  {
    if (adminConnection != null)
    {
      ThreadLog.trace("adminConnection != null");

      Iterator iter = createdRepositories.iterator();
      try {
        java.sql.Statement st = adminConnection.createStatement();
        while (iter.hasNext())
        {
          String branchedDatabaseName = (String)iter.next();
          ThreadLog.trace("branchedDatabaseName ");
          try {
            st.executeQuery("DROP DATABASE "+branchedDatabaseName);
            ThreadLog.trace("ok\n");
          } catch (SQLException ex) {
            ThreadLog.trace("false "+ex.getMessage()+"\n");
          }
        }
      } catch(SQLException e) {
        ThreadLog.trace("SQLException "+e.getMessage()+"\n");
      }
    }
    createdRepositories.clear();
  }

  /**
   * Allow the SAIL to synchronize any stale data. The SAIL can assume that
   * shutDown() is called before an application is stopped.
   *
   * @exception SailInternalException To indicate an internal error.
   */
  public void shutDown()
  {
    ThreadLog.trace("VersioningRdbmsSail ->shutDown(begin)");
    dropBranchedVersions();
    super.shutDown();
    ThreadLog.trace("VersioningRdbmsSail ->shutDown(end)");
  }

  protected int VersionUid;
  protected int baseUrlIndex;

  /**
   * Initializes the TmsSQL92Sail. The supplied Map should contain the
   * following key-value pairs:
   * <table>
   * <tr><td>key</td><td>value</td></tr>
   * <tr><td>jdbcDriver</td><td>The String representing the JDBC-driver
   * class, e.g. "org.gjt.mm.mysql.Driver"</td></tr>
   * <tr><td>jdbcUrl</td><td>The String representing the JDBC-url of the
   * database to connect to, e.g. "jdbc:mysql://localhost/sesame"</td></tr>
   * <tr><td>user</td><td>The username that can be used to connect to the
   * DBMS</td></tr>
   * <tr><td>password</td><td>The password of the user in the DBMS</td></tr>
   * </table>
   *
   * @param configParams configuration parameters
   * @exception SailInitializationException If the TmsSQL92Sail could not
   * be initialized using the supplied parameters.
   */
  public void initialize(Map configParams)
      throws SailInitializationException
  {
    ThreadLog.trace("initializing VersioningRdbmsSail");

    super.initialize(configParams);

    ThreadLog.trace("VersioningRdbmsSail initialized");
  }

 /*----------------------------------------+
   | Database initialization                 |
   +----------------------------------------*/

   /**
    * Initializes the database. _initDatabase() creates tables, indexes and
    * inserts default values into the database.
    **/
  protected void _initDatabase()
      throws SailInitializationException
  {
    try {
      super._initDatabase();
      // EXPECTED_STATEMENTS_TABLE table
      createExpectedSIDSTable();

      // BaseUrlTable table
      createBaseUrlTable();

      // Updates table
      createUpdatesTable();

      // TripleHIST table
      createTripleHIST_Table();

      // versions table
      createVersionTable();

      // active states table
      createActiveStatesTable();
    }
    catch (SQLException e) {
      throw new SailInitializationException(e);
    }
  } //initDatabase

  private static final String SUFFIX = "_HIST";
  protected static final String UPDATES_TABLE = "updates";
  protected static final String VERSION_TABLE = "version";
  protected static final String TRIPLES_HIST_TABLE = TRIPLES_TABLE+SUFFIX;
  protected static final String BASEURL_TABLE = "baseurls";
  protected static final String EXPECTED_STATEMENTS_TABLE = "expected_sids";

  protected static final String ACTIVE_STATES_TABLE = "active";

  //  SQLTables
  /**
   * we need a unique suffix for the branched states of the repository so this help us to
   * reuse the same sufix everywhere
   * @param state is the stateID which suffix to get
   * @return a string suffix
   */
  private String getSuffix(int state)
  {
    String result = "";
    result+="_"+state;
    return result;
  }

  protected RDBMS RDBMS() {
      return _rdbms;
  }

  protected boolean createExpectedSIDSTable()
      throws SQLException
  {
    if (!RDBMS().tableExists(EXPECTED_STATEMENTS_TABLE)) {
      String tableQ = "CREATE TABLE "+EXPECTED_STATEMENTS_TABLE+" ("+
                      "sid " + RDBMS().ID_INT + " NOT NULL,"+
                      "found " + RDBMS().BOOLEAN + ")";
      RDBMS().executeUpdate(tableQ);
      return true;
    }
    return false;
  }


  protected boolean createBaseUrlTable()
      throws SQLException
  {
    if (!RDBMS().tableExists(BASEURL_TABLE)) {
      String tableQ = "CREATE TABLE "+BASEURL_TABLE+" ("+
                      "urlID " + RDBMS().ID_INT + " NOT NULL,"+
                      "url " + RDBMS().LOCALNAME + ")";
      RDBMS().executeUpdate(tableQ);
      return true;
    }
    return false;
  }

  protected boolean createUpdatesTable()
      throws SQLException
  {
    if (!RDBMS().tableExists(UPDATES_TABLE)) {
      String tableQ = "CREATE TABLE "+UPDATES_TABLE+" ("+
                      "uid " + RDBMS().ID_INT + " NOT NULL,"+
                      "time TIMESTAMP ,"+
                      "usid " + RDBMS().ID_INT + " ,"+
                      "kind " + RDBMS().BOOLEAN + " , "+
                      "urlID " + RDBMS().ID_INT + ")";
      RDBMS().executeUpdate(tableQ);

      RDBMS().createIndex(UPDATES_TABLE, new String[] {"uid"}, true);
      return true;
    }
    return false;
  }

  // TripleHIST table
  protected boolean createTripleHIST_Table()
      throws SQLException
  {
    if (!RDBMS().tableExists(TRIPLES_HIST_TABLE)) {
      String tableQ = "CREATE TABLE "+TRIPLES_HIST_TABLE+" ("+
                      "id " + RDBMS().ID_INT + " NOT NULL,"+
                      "subject " + RDBMS().ID_INT + " NOT NULL REFERENCES " +
                      RESOURCES_TABLE + "(id), " +
                      "predicate " + RDBMS().ID_INT + " NOT NULL REFERENCES " +
                      RESOURCES_TABLE + "(id), " +
                      "object " + RDBMS().ID_INT + " NOT NULL, " +
                      "explicit " + RDBMS().BOOLEAN + " NOT NULL, " +
                      "BornAt " + RDBMS().ID_INT + " , "+
                      "DiedAt " + RDBMS().ID_INT + " )";
      RDBMS().executeUpdate(tableQ);

      RDBMS().createIndex(TRIPLES_HIST_TABLE, new String[]{"id"}, false);
      return true;
    }
    return false;
  } // createTripleHIST_Table()

  // Version Table
  protected boolean createVersionTable()
      throws SQLException
  {
    if (!RDBMS().tableExists(VERSION_TABLE)) {
      String tableQ = "CREATE TABLE "+VERSION_TABLE+" ("+
                      "vid " + RDBMS().ID_INT + " NOT NULL,"+
                      "usid " + RDBMS().ID_INT + " , "+
                      "uid " + RDBMS().ID_INT + " , "+
                      "label VARCHAR(255) )";
      RDBMS().executeUpdate(tableQ);

      RDBMS().createIndex(VERSION_TABLE, new String[]{"vid"}, true);
      return true;
    }
    return false;
  } // createVersionTable()

  /**
   * create aa helper table that will track number of active conections to particular
   * update states od the repository. Because for each previous state that is used we create
   * a separate RO repository. we should track when it is not longer needed
   * (by the number of the requests made for its creation) and remove it permanently
   * @throws SQLException if somthing is fron with the DB connection
   * @return true if table was just created
   */
  protected boolean createActiveStatesTable()
      throws SQLException
  {
    if (!RDBMS().tableExists(ACTIVE_STATES_TABLE)) {
      String tableQ = "CREATE TABLE "+ACTIVE_STATES_TABLE+" ("+
                      "uid " + RDBMS().ID_INT + " NOT NULL,"+
                      "refs " + RDBMS().ID_INT + " )";
      RDBMS().executeUpdate(tableQ);
      return true;
    }
    return false;
  }


  /**
   * it is invoked from super.RemoveExpiredStatements to deal with these id chunks
   * @param idList chunk with ids o fexpired statements
   * @throws SQLException
   */
  protected void _processChunkFromRemoveExpiredStatements(String idList) throws SQLException {
    // the actual ID of the update will be added to UPDATE_TABLE later in _processChangedTriples
    RDBMS().executeUpdate(
      "UPDATE "+ TRIPLES_HIST_TABLE +
      " SET DiedAt = "+ (getStateUid()+1) +
      " WHERE id IN " + idList
      );
  }

/**
 * add's a new explicit statement to repository
 * @param subj is either BNode ot URI denoting the statement's subject
 * @param pred is an URI for predicate
 * @param obj is the object value of the statement (Resource or Literal)
 * @throws SailUpdateException
 */
  public void addStatement(Resource subj, URI pred, Value obj)
          throws SailUpdateException
 {

    /**
     * In INCEMENTAL_MODE ONLY
     * 1. if such a statement exist - mark it in the 'expected' table
     */
  if (SessionContext.getContext().updateMode != SessionContext.INCREMENTAL_MODE) {
    int subjId = _getResourceId(subj);
    int predId = _getURIId(pred);
    int objId = _getValueId(obj);

    int statementId = 0;

    boolean hasRow = false;
    boolean bExplicit = false;
    try
    {
      Connection con = RDBMS().getConnection();
      java.sql.Statement st = con.createStatement();

      ResultSet rs = st.executeQuery(
          "SELECT id, explicit "+
          " FROM "+TRIPLES_TABLE+
          " WHERE subject="+subjId+
          " AND predicate="+predId+
          " AND object="+objId
          );
      if (rs.next())
      {
        statementId = rs.getInt(1);
        bExplicit = rs.getString(2).equalsIgnoreCase(RDBMS().convertBoolean(true));
        hasRow = true;
      }
      rs.close();
      st.close();
      con.close();

      if (hasRow == true) {
        // here we mark that there is such a triple in the table
        RDBMS().executeUpdate("UPDATE "+EXPECTED_STATEMENTS_TABLE+
                              " SET found=1 WHERE sid = "+statementId);
      }

    } catch (SQLException e) {
      throw new SailInternalException(e);
    }
  } // ent !INCREMENTAL_MODE
   super.addStatement(subj, pred, obj);
 } // addStatement()

/**
 * Clears the repository. After the repository was cleared, it will
 * be in the same state as after initialization.
 * Our Sail implementation clears all additional tables an all currently available
 * branches that are not permanent.
 * @exception SailInternalException To indicate an internal error.
 *
 */
  public void clearRepository() throws SailUpdateException
  {
	super.clearRepository();
    try {
      RDBMS().executeUpdate("DROP TABLE " + UPDATES_TABLE);

      RDBMS().executeUpdate("DROP TABLE " + TRIPLES_HIST_TABLE);

      RDBMS().executeUpdate("DROP TABLE " + VERSION_TABLE);
      RDBMS().executeUpdate("DROP TABLE " + BASEURL_TABLE);
      RDBMS().executeUpdate("DROP TABLE " + EXPECTED_STATEMENTS_TABLE);
      dropBranchedVersions();
    }
    catch (SQLException e) {
      throw new SailInternalException(e);
    }
}

  /**
   * Starts a transaction. A transaction needs to be started before
   * data can be added to or removed from the repository.
   * @exception SailInternalException To indicate an internal error.
   **/

  public void startTransaction() {
    super.startTransaction();
    baseUrlIndex = 0;
    SessionContext sc = SessionContext.getContext();
    if (sc.updateMode != SessionContext.INCREMENTAL_MODE) {
      String url = sc.baseUrl;
      if (url.length() > 0)  {
        try {
          Connection con = RDBMS().getConnection();
          java.sql.Statement st = con.createStatement();
          // try to find the BASE URL in the cache
          System.out.println("CP="+1);
          ResultSet rs = null;
          try {
            rs = st.executeQuery("SELECT urlID FROM "+BASEURL_TABLE+
                                 " WHERE url = '"+RDBMS().escapeString(url)+"'");
            if (rs.next())
              baseUrlIndex = rs.getInt(1);
            rs.close();
          } catch (SQLException e) {
            throw new SailInternalException(e);
          }
          System.out.println("baseUrlIndex="+baseUrlIndex);
          if (baseUrlIndex == 0)
          {
            // try to add it in the cache
            int max = 0;
            rs = st.executeQuery("SELECT max(urlID) FROM "+BASEURL_TABLE);
            if (rs.next())
              max = rs.getInt(1);
            rs.close();
            baseUrlIndex = max+1;
            RDBMS().executeUpdate("INSERT INTO "+BASEURL_TABLE+
                                  " VALUES ("+baseUrlIndex+", '"+RDBMS().escapeString(url)+"')");
            System.out.println("max="+baseUrlIndex);
          }
          // build expected_resources_table to be able to mark the appeared ones
          // so to remove the 'obsoleted' at commit_transaction
          // how we can build that table
          // 1. first we collect all updateIDS created when that base url was
          //    used on startTransaction(the code above);
          // 2. add the id's of the still 'Live' explicit triples
          //    that were submited to the repository during these the updates (when
          //    the 'BASEURL' was used)
          // 3. afterwards on each addStatement() of particular explicit statement
          //    we mark the valid ones (using the flag from EXPECTED_STATEMENTS_TABLE)
          // 4. on commitTransaction()
          //    a). add the UpdateID with the provided BASEURLIndex
          //    b). remove any unused statements mentioned in the EXPECTED_STATEMENTS_TABLE
          //    without the flag set.
          // 5. Enjoy the ADAPTIVE Upload
          // 6. WE shall implement RE-INIT mode doing remove of the statements with
          //    ids collected on step 2 and then just continuing as usual;

          // step.1
          RDBMS().executeUpdate(" TRUNCATE "+EXPECTED_STATEMENTS_TABLE);

          RDBMS().executeUpdate(" INSERT INTO "+EXPECTED_STATEMENTS_TABLE+
                                " SELECT th.id, 0 FROM "+TRIPLES_HIST_TABLE+" th "+
                                " INNER JOIN "+UPDATES_TABLE+" u on th.BornAt = u.uid "+
                                " WHERE u.urlID = "+baseUrlIndex+" AND th.DiedAt = 0 AND th.explicit = 1");

		  /*
          if (sc.updateMode == SessionContext.REINIT_MODE) {
          }
		  */
          st.close();
          con.close();
        } catch (SQLException e) {
          throw new SailInternalException(e);
        }
      } // an empty url
    }
    VersionUid = -1;
    VersionUid = getStateUid();
  }

  /**
   * Commits a started transaction. A transaction needs to be commited
   * to be guaranteed that data has been added to, or removed from the
   * repository.
   * @exception SailInternalException To indicate an internal error.
   **/
  public void commitTransaction() {
    SessionContext sc = SessionContext.getContext();
    try {
      if (sc.updateMode == SessionContext.ADAPTIVE_MODE)
      {
        // here we remove the 'obsolete' statements from repository
        // these statements do not appear in the data under the baseURL
        Connection con = null;
        java.sql.Statement st = null;
        ResultSet rs = null;
        try {
          con = RDBMS().getConnection();
          st = con.createStatement();
          rs = st.executeQuery("SELECT sid FROM "+EXPECTED_STATEMENTS_TABLE+
                               " WHERE found = 0"
                               );
          while (rs.next())
          {
              int sid = rs.getInt(1);
              if (0 < RDBMS().executeUpdate("UPDATE "+TRIPLES_TABLE+
                  " SET explicit="+RDBMS().FALSE+" WHERE id="+sid))
              {
                _statementsRemoved = true;
              }
          }
          rs.close();
          st.close();
          con.close();
        } catch(SQLException e) {
          throw new SailInternalException(e);
        }
        RDBMS().executeUpdate("TRUNCATE "+EXPECTED_STATEMENTS_TABLE);
      }
    } catch (SQLException ex) {
      throw new SailInternalException(ex);
    }
    super.commitTransaction();
    VersionUid = -1;
  }

  /**
   * Helper to retrive the user ID of the current user (that user that starts the
   * operation). See the SessionContext
   * @return the id of the current user
   */
  protected int getCurrentUserID()
  {
    SessionContext o = SessionContext.getContext();
    if (o == null)
      throw new SailInternalException("null session context in getCurrentUserID()");
    if (o.userID == 0)
      o.userID = 2; // set the user id to the 'anonymous' account
    return o.userID;
  }

  /**
   * Return current value of teh Update counter. If it is not cased we compute it
   * on the fly.
   * @return curent value of the update counter
   */
  protected int getStateUid()
  {
    SessionContext o = SessionContext.getContext();
    if (o == null)
      throw new SailInternalException("null session context in getStateUid()");
    VersionUid = o.VersionState;

    if (VersionUid <= 0)
    {
      try {
        Connection con = RDBMS().getConnection();
        java.sql.Statement st = con.createStatement();

        ResultSet rs = st.executeQuery(
            "SELECT max(uid) FROM " + UPDATES_TABLE);
        if (rs.next())
          VersionUid = rs.getInt(1);
        else
          VersionUid = 0;
        rs.close();
        st.close();
        con.close();
      } catch (SQLException ex) {
        throw new SailInternalException(ex);
      }
    }
    return VersionUid;
  }

  /**
   * Return a list of all Update states of the current repository as Strings.
   * Mainly used in the HTML User interface.
   * @return Iterator throuh the values of the Update Counter
   */
  public Iterator getUpdateIds()
  {
    return new Iterator() {
      // initialize
      Connection con = null;
      java.sql.Statement st = null;
      ResultSet rs = null;
      boolean hn =false;
      {
        try {
          con = RDBMS().getConnection();
          st = con.createStatement();

          rs = st.executeQuery(
              "SELECT uid FROM " + UPDATES_TABLE);
          hn = rs.next();
          if (!hn)
          {
            rs.close();
            st.close();
            con.close();
          }
        } catch (SQLException ex) {
          throw new SailInternalException(ex);
        }
      }
      public boolean hasNext() {
        return hn;
      }
      public Object next()
      {
        Object res = null;
        if (hn) {
          try {
            res = rs.getString(1);
            hn = rs.next();
            if (!hn)
            {
              rs.close();
              st.close();
              con.close();
            }
          } catch (SQLException ex) {
            throw new SailInternalException(ex);
          }
        }
        return res;
      }
      public void remove(){}
    };
  } // getUpdateIds()

  /**
   * Return a list of all Version Ids of the current repository as Strings.
   * @return Iterator over all version ids
   */
  public Iterator getVersionIds()
  {
    return new Iterator() {
      // initialize
      Connection con = null;
      java.sql.Statement st = null;
      ResultSet rs = null;
      boolean hn =false;
      {
        try {
          con = RDBMS().getConnection();
          st = con.createStatement();

          rs = st.executeQuery(
              "SELECT vid FROM " + VERSION_TABLE);
          hn = rs.next();
          if (!hn)
          {
            rs.close();
            st.close();
            con.close();
          }
        } catch (SQLException ex) {
          throw new SailInternalException(ex);
        }
      }
      public boolean hasNext() {
        return hn;
      }
      public Object next()
      {
        Object res = null;
        if (hn) {
          try {
            res = rs.getString(1);
            hn = rs.next();
            if (!hn)
            {
              rs.close();
              st.close();
              con.close();
            }
          } catch (SQLException ex) {
            throw new SailInternalException(ex);
          }
        }
        return res;
      }
      public void remove(){}
    };
  } // getVersionIds()


  /**
   * Create a labeled version for a statte of the repository assigning the
   * necessary mata-information about thet operation.
   * @param stateUID of a particular repository state
   * @param label associated with the state
   *
   * NOTE: This method can throw a security exception if the request is made on
   * behalf of the user with insufficent rights to create versions
   */
  public void labelState(long stateUID, String label)
  {
    int usid = getCurrentUserID();
    try {
      Connection con = RDBMS().getConnection();
      java.sql.Statement st = con.createStatement();
      ResultSet rs = st.executeQuery("select max(vid) from "+VERSION_TABLE);
      int maxvid = 0;
      if (rs.next())
        maxvid = rs.getInt(1);
      rs.close();
      // ad a system default label
      if (label == null || label.length() ==0)
        label = "New label "+(maxvid+1);
      st.executeUpdate("insert into "+VERSION_TABLE+
                       " VALUES ("+(maxvid+1)+" , "+usid+", "+stateUID+" ,"+
                       " '"+label+"')" );
      st.close();
      con.close();
    } catch (SQLException e) {
      throw new SailInternalException(e);
    }
  } //labelState

  /**
   * Create a labeled version of the curent repository state.
   * @param label to be associated with the state
   * NOTE: This method can throw a security exception if the request is made on
   * behalf of the user with insufficent rights to create versions
   */
  public void labelCurrentState(String label)
  {
    int saveState = VersionUid;
    VersionUid = -1;
    int stateUID = getStateUid();
    VersionUid = saveState;
    labelState(stateUID, label);
  }

  /**
   * Restore the repository to previous state removing all statements added
   * after the value of the update counter and revive all removed ones.
   * @param stateUID - the update counter of a particular repository state
   * NOTE: This method can throw a security exception if the request is made on
   * behalf of the user with insufficent rights to revert the repository
   */
  public void revertToState(long stateUID)
  {
    throw new RuntimeException( "revertToState is not implemented" );
  }

  /**
   * Which Sail class should be used to access the intermediate branched versions
   * of the repository - to speed up the work with the RO version we use the most
   * basic sail implementation
   * @return the name of sail class.
   */
  protected String alias()
  {
    return super.getClass().getSuperclass().getName();
  }
  /**
   * Sets the repository to given statete for further read operations.
   * @param stateUID - the update counter of a particular repository state
   */
  public void workWithState(long stateUID)
  {

    //  FIXME: should check if it is possible
    SessionContext sc = SessionContext.getContext();
    int vs = sc.VersionState;
    if (vs > 0)
    {
      try {
        Connection con = RDBMS().getConnection();
        java.sql.Statement st = con.createStatement();
        ResultSet rs = st.executeQuery("SELECT refs FROM "+ACTIVE_STATES_TABLE+
                                       " WHERE uid ="+vs);
        boolean toRemove = false;
        if (rs.next())
        {
          int refs = rs.getInt(1);
          if (refs == 1)
            toRemove = true;
        }
        rs.close();
        st.close();
        con.close();
        if (!toRemove)
          RDBMS().executeUpdate("UPDATE "+ACTIVE_STATES_TABLE+
                                " SET refs=refs-1 WHERE uid="+vs);
        else {
          RDBMS().executeUpdate("DELETE FROM "+ACTIVE_STATES_TABLE+
                                " WHERE uid="+vs);
        } // remove or not

      } catch (SQLException e) {
        throw new SailInternalException(e);
      }
    } // if (vs > 0)
    if (stateUID > 0)
    {
      try {
        Connection con = RDBMS().getConnection();
        java.sql.Statement st = con.createStatement();
        ResultSet rs = st.executeQuery("SELECT refs FROM "+ACTIVE_STATES_TABLE+
                                       " WHERE uid ="+stateUID);
        boolean toInsert = true;
        if (rs.next())
        {
          int refs = rs.getInt(1);
          toInsert = false;
        }
        rs.close();
        if (!toInsert)
          RDBMS().executeUpdate("UPDATE "+ACTIVE_STATES_TABLE+
                                " SET refs=refs+1 WHERE uid="+stateUID);
        else {
          RDBMS().executeUpdate("INSERT INTO "+ACTIVE_STATES_TABLE+
                                " VALUES ( "+stateUID+" , "+1+")");

          branchState(stateUID);
        } // insert or not
        st.close();
        con.close();
      } catch (SQLException e) {
        throw new SailInternalException(e);
      }
    } // if (stateUID > 0)
  }

  public String branchState(long stateUID)
  {
    String result = "";
    SessionContext sc = SessionContext.getContext();

    try {
      SystemConfig sysConfig = SesameServer.getSystemConfig();

      String branchId = sc.repository + "branch_" + getSuffix((int)stateUID);

      RepositoryConfig rc = sysConfig.cloneRepository(sc.repository, branchId);
      rc.setTitle("branch of " + getSuffix((int)stateUID) + " " + rc.getTitle());
      sc.repository = branchId;

      boolean b_do_not_append_data = false;
      boolean b_do_not_create_data = true;

      Iterator iter = rc.getSailList().iterator();
      while (iter.hasNext())
      {
        SailConfig sailConfig = (SailConfig)iter.next();

        String param = sailConfig.getSailClass();
        if (param.equals(getClass().getName())) {
          sailConfig.setSailClass(alias());
        }

        String url = sailConfig.getParameter("jdbcUrl");
        if (url != null)
        {
          int pos = url.lastIndexOf('/');
          String repdb = url.substring(pos+1);
          repdb += getSuffix((int)stateUID)+"branch";

          url = url.substring(0,pos+1);
          sailConfig.setParameter("jdbcUrl",url+repdb);
          String user = sailConfig.getParameter("user");
          String pass = sailConfig.getParameter("password");

          Connection c1 = null;
          try {
            c1 = java.sql.DriverManager.getConnection(url+repdb, user, pass);
          }
          catch (SQLException ignore) {}

          if (c1 != null)
          {
            c1.close();
            b_do_not_append_data = true && b_do_not_create_data;
            break;
          }

          if (b_do_not_create_data)
          {
            if (adminConnection == null) {
              adminConnection = java.sql.DriverManager.getConnection(url, user, pass);
            }

            java.sql.Statement st2 = adminConnection.createStatement();
            try {
              st2.execute("DROP DATABASE "+repdb);
            } catch(SQLException e) { }

            st2.execute("CREATE DATABASE "+repdb);
            createdRepositories.add(repdb);
            st2.close();
            b_do_not_create_data = false;
          }
        } // not have url
      } //while

      // Give admin user access to new repository
      UserInfo uiAdmin = sysConfig.getUserInfo(1);
      uiAdmin.addReadWriteAccess(rc);

      if (b_do_not_append_data == false)
      {
        ThreadLog.trace("branch begin");
        
        LocalService service = SesameServer.getLocalService();
        
        service.login(uiAdmin.getLogin(), uiAdmin.getPassword());
        
        LocalRepository rep = (LocalRepository)service.getRepository(rc.getRepositoryId());
        
        Sail otherSail = rep.getSail();
        
        ThreadLog.trace("branch end");
        Connection con = RDBMS().getConnection();
        java.sql.Statement st = con.createStatement();
        ResultSet rs = null;
        if (otherSail != null)
        {
          int old_id = sc.userID;
          sc.userID = 1; // as admin
          rs = st.executeQuery(
              "SELECT n1.name, r1.localname,n2.name, r2.localname, n3.name, r3.localname FROM "+
                TRIPLES_HIST_TABLE+" th ,"+
                RESOURCES_TABLE+" r1, "+ // was HIST_
                NAMESPACES_TABLE+" n1, "+
                RESOURCES_TABLE+" r2, "+ // was HIST_
                NAMESPACES_TABLE+" n2, "+
                RESOURCES_TABLE+" r3, "+ // was HIST_
                NAMESPACES_TABLE+" n3 "+
              " WHERE th.explicit="+RDBMS().TRUE+" AND th.BornAt <= "+stateUID+
              " AND (th.DiedAt > "+stateUID+" OR th.DiedAt = 0)"+
              " AND r1.id=th.subject"+
              " AND r2.id=th.predicate"+
              " AND r3.id=th.object"+
              " AND r1.namespace =n1.id"+
              " AND r2.namespace =n2.id"+
              " AND r3.namespace =n3.id"
              );
          ((RdfRepository)otherSail).startTransaction();
          while (rs.next())
          {
            /**@todo: instead of resource use the new Impl objects import triples with Resource objects */
            String nsString = rs.getString(1);
            Resource subj = null;
            if (nsString == null || 0==nsString.compareToIgnoreCase("NULL"))
              subj = new org.openrdf.model.impl.BNodeImpl(rs.getString(2));
            else
              subj = new org.openrdf.model.impl.URIImpl(nsString, rs.getString(2));

            URI pred = new org.openrdf.model.impl.URIImpl(rs.getString(3), rs.getString(4));

            Value obj = null;
            String valString = rs.getString(5);
            if (valString == null || 0==valString.compareToIgnoreCase("NULL"))
              obj = new org.openrdf.model.impl.BNodeImpl(rs.getString(6));
            else
              obj = new org.openrdf.model.impl.URIImpl(valString, rs.getString(6));
            try {
              ((RdfRepository)otherSail).addStatement(subj,pred, obj);
            } catch (SailUpdateException e) {
              throw new SailInternalException(e);
            }
          }
          rs.close();

          rs = st.executeQuery(
              "SELECT n1.name, r1.localname,n2.name, r2.localname, l1.label, l1.language FROM "+
              TRIPLES_HIST_TABLE+" th ,"+
              RESOURCES_TABLE+" r1, "+ // was HIST_
              NAMESPACES_TABLE+" n1, "+
              RESOURCES_TABLE+" r2, "+ // was HIST_
              NAMESPACES_TABLE+" n2, "+
              LITERALS_TABLE+" l1 "+ // was HIST_
              " WHERE th.explicit="+RDBMS().TRUE+" AND th.BornAt <= "+stateUID+
              " AND (th.DiedAt > "+stateUID+" OR th.DiedAt = 0)"+
              " AND r1.id = th.subject"+
              " AND r2.id = th.predicate"+
              " AND l1.id = th.object"+
              " AND r1.namespace =n1.id"+
              " AND r2.namespace =n2.id"
              );
          while (rs.next())
          {
            /**@todo: instead of resource use the new Impl objects import triples with Literal objects*/
            String nsString = rs.getString(1);
            Resource subj = null;
            if (nsString == null || 0==nsString.compareToIgnoreCase("NULL"))
              subj = new org.openrdf.model.impl.BNodeImpl(rs.getString(2));
            else
              subj = new org.openrdf.model.impl.URIImpl(nsString, rs.getString(2));
            URI pred = new org.openrdf.model.impl.URIImpl(rs.getString(3), rs.getString(4));
            String val = rs.getString(5);
            String lang = rs.getString(6);
            Value obj = new org.openrdf.model.impl.LiteralImpl(val, lang);
            try {
              ((RdfRepository)otherSail).addStatement(subj,pred, obj);
            } catch (SailUpdateException e) {
              throw new SailInternalException(e);
            }
          }
          rs.close();
          ((RdfRepository)otherSail).commitTransaction();

          // we should update the namespaces
          NamespaceIterator nsIter = getNamespaces();
          while (nsIter.hasNext()) {
            nsIter.next();
            try {
              ((RdfRepository)otherSail).changeNamespacePrefix(nsIter.getName(), nsIter.getPrefix());
            }
            catch (SailUpdateException e) {
              ThreadLog.warning(
                  "Unable to set namespace prefix '" + nsIter.getPrefix() +
                  "' for namespace '" + nsIter.getName() + "': " +
                  e.getMessage());
            }
          }

          sc.userID = old_id; // as current user
        } // have a Sail
        st.close();
        con.close();
      }
      sc.repository = rc.getRepositoryId();
      result = rc.getRepositoryId();
    } catch (ConfigurationException e) {
    	throw new SailInternalException(e);
    } catch (SQLException e) {
      throw new SailInternalException(e);
    } catch (AccessDeniedException e){
      throw new SailInternalException(e);
    } catch (UnknownRepositoryException e){
      throw new SailInternalException(e);
    }
    return result;
  }

  /**
   * Retrive list of all labeled states of the repository.
   * @return a list of Versin interfaces for each labeled state of the repository
   */
  public Iterator getVersions()
  {
    return new Iterator() {
      // initialize
      Connection con = null;
      java.sql.Statement st = null;
      ResultSet rs = null;
      boolean hn =false;
      {
        try {
          con = RDBMS().getConnection();
          st = con.createStatement();

          rs = st.executeQuery(
              "SELECT vid, usid, uid, label FROM " + VERSION_TABLE);
          hn = rs.next();
          if (!hn)
          {
            rs.close();
            st.close();
            con.close();
          }
        } catch (SQLException ex) {
          throw new SailInternalException(ex);
        }
      }
      public boolean hasNext() {
        return hn;
      }
      public Object next()
      {
        Object res = null;
        if (hn) {
          try {
            int vid = rs.getInt(1);
            int usid = rs.getInt(2);
            int uid = rs.getInt(3);
            String label = rs.getString(4);
            res = new VersionOMM(vid, usid, uid, label);
            hn = rs.next();
            if (!hn)
            {
              rs.close();
              st.close();
              con.close();
            }
          } catch (SQLException ex) {
            throw new SailInternalException(ex);
          }
        }
        return res;
      }
      public void remove()
          { }
    };
  }

  /**
   * Perform locking of statements in the repository
   * @param statementsList - list of statemensts to lock
   */
  public void lockStatements(Iterator statementsList)
  {
    throw new RuntimeException( "lockStatements is not implemented" );
  }
  /**
   * Perform unlocking of statements in the repository
   * @param statementsList - list of statemensts to unlock
   */
  public void unlockStatements(Iterator statementsList)
  {
    throw new RuntimeException( "unlockStatements is not implemented" );
  }

  /**
   * Stop the increment of the update counter. Usefull for a kind of batch updates
   * or adding a distinct daml coinstructs at once.
   */
  public void pauseCounterIncrement()
  {
    SessionContext.getContext().bIncrementIsPaused = true;
  }

  /**
   * Coninue with the normal increment of the update counter on each modification
   * made in the repository.
   */
  public void continueCounterIncrement()
  {
    SessionContext.getContext().bIncrementIsPaused = false;
  }

  /**
   * Check if the update couter ss paused
   * @return true if the updateCouter is paused, flase otherwise
   */
  public boolean isPausedCounterIncrement()
  {
    return SessionContext.getContext().bIncrementIsPaused;
  }

  public static final String MADE_BY_URI = "http://www.ontotext.com/otk/2002/03/kcs.rdfs#madeBy";
  public static final String USER_ID_URI = "http://www.ontotext.com/otk/2002/03/kcs.rdfs#userID";
  public static final String VERSION_NAME_URI = "http://www.ontotext.com/otk/2002/03/kcs.rdfs#versionName";
  public static final String VERSION_UPDATE_ID = "http://www.ontotext.com/otk/2002/03/kcs.rdfs#stateLabeled";
  public static final String BORN_AT_URI = "http://www.ontotext.com/otk/2002/03/kcs.rdfs#bornAt";
  public static final String DIED_AT_URI = "http://www.ontotext.com/otk/2002/03/kcs.rdfs#diedAt";
  public static final String MADE_ON_URI = "http://www.ontotext.com/otk/2002/03/kcs.rdfs#madeOn";

  public static final String EXPLICIT_KEY = "EXPLICIT";
  public static final String KIND_KEY = "KIND";

  /**
   * Retrieves the meta info associated with a statement.
   * @param subj the subject of the statement
   * @param pred the predicate of the statement
   * @param obj the object of the statement
   * @return a map of meta info keys vs meta info values
   */
  public Map getMetaInfo(String subj, String pred, String obj) {
    if (subj == null || pred == null || obj == null){
      throw new SailInternalException("Cannot Retrieve MetaInfo for a statement"
                                      +" that is not in the repository:\n"
                                      +"<"+subj+","+pred+","+obj+">");
    }

    Map map = new HashMap();
    Resource subjRes = new org.openrdf.model.impl.URIImpl(subj);
    Resource predRes = new org.openrdf.model.impl.URIImpl(pred);
    Resource objRes = null;
    try {
      objRes = new org.openrdf.model.impl.URIImpl(obj);
    } catch (IllegalArgumentException e) {
    }
    // find subj pred obj ids
    int subjId = _getResourceId(subjRes);
    int predId = _getResourceId(predRes);
    int objId = 0;
    if (objRes != null)
      objId = _getResourceId(objRes);
    if ( objId == 0 ) {
      Literal objLit = new org.openrdf.model.impl.LiteralImpl(obj);
      objId = _getLiteralId(objLit);
    }

    if (subjId == 0 || objId == 0 || predId == 0) {
      throw new SailInternalException("Cannot Retrieve MetaInfo for a statement"
                                      +" that is not in the repository:\n"
                                      +"<"+subj+","+pred+","+obj+">");
    }

    int bornAt;
    int diedAt;
    int statementId;
    boolean explicit;

    {//get all
      Connection con = null;
      java.sql.Statement st = null;
      ResultSet rs = null;
      try {
        con = RDBMS().getConnection();
        st = con.createStatement();

        rs = st.executeQuery(
            "SELECT id, explicit, BornAt, DiedAt FROM "+
            TRIPLES_HIST_TABLE+
            " WHERE subject="+subjId+
            " AND predicate="+predId+
            " AND object="+objId);
        int countLifetime = 0;
        while (rs.next()) {
          String suffix = "";
          if (countLifetime > 0)
            suffix = suffix+countLifetime;
          statementId = rs.getInt(1);
          explicit = rs.getBoolean(2);
          bornAt = rs.getInt(3);
          diedAt = rs.getInt(4);
          map.put(EXPLICIT_KEY+suffix,explicit?"true":"false");
          map.put(BORN_AT_URI+suffix,new Integer(bornAt));
          if (diedAt != 0 ) {
            map.put(DIED_AT_URI+suffix,new Integer(diedAt));
          }
          countLifetime++;
        }
        rs.close();
        st.close();
        con.close();
      } catch (SQLException ex) {
        throw new SailInternalException(ex);
      }
    }

    return map;
  }

  /**
   * Retrieves the meta info associated with an update.
   * @param updateId the id of the update
   * @return a map of meta info keys vs meta info values
   */
  public Map getUpdateMetaInfo(String updateId) {
    Map map = new HashMap();
    Connection con = null;
    java.sql.Statement st = null;
    ResultSet rs = null;
    try {
      con = RDBMS().getConnection();
      st = con.createStatement();

      rs = st.executeQuery(
          "SELECT uid, time, usid, kind FROM "
          + UPDATES_TABLE
          + " WHERE uid = " + updateId);
      if (rs.next()) {
        String time = rs.getString(2);
        int usid = rs.getInt(3);
        int kind = rs.getInt(4);

        UserInfo ui = SesameServer.getSystemConfig().getUserInfo(usid);
        if (ui != null)
          map.put(MADE_BY_URI, ui.getFullName());
        else
          map.put(MADE_BY_URI, "unknown user");
        map.put(USER_ID_URI, new Integer(usid));
        map.put(MADE_ON_URI, time);
        String updateKind = "";
        switch (kind) {
          case 1:
            updateKind = "add";
            break;
          case 2:
            updateKind = "remove";
            break;
          case 3:
            updateKind = "add/remove";
            break;
          default:
            updateKind = "UNKNOWN/BUG";
            break;
        }
        map.put(KIND_KEY, updateKind);
      }
      rs.close();
      st.close();
      con.close();
    } catch (SQLException ex) {
      throw new SailInternalException(ex);
    }

    return map;
  }

  /**
   * Retrieves the meta info associated with a version .
   * @param versionId the id of the update
   * @return a map of meta info keys vs meta info values
   */
  public Map getVersionMetaInfo(String versionId) {
    Map map = new HashMap();
    Connection con = null;
    java.sql.Statement st = null;
    ResultSet rs = null;
    try {
      con = RDBMS().getConnection();
      st = con.createStatement();

      rs = st.executeQuery(
          "SELECT vid, usid, uid, label FROM "
          + VERSION_TABLE
          + " WHERE vid = " + versionId);
      if (rs.next()) {
        int vid = rs.getInt(1);
        int usid = rs.getInt(2);
        int uid = rs.getInt(3);
        String label = rs.getString(4);

        UserInfo ui = SesameServer.getSystemConfig().getUserInfo(usid);
        map.put(USER_ID_URI,new Long(uid));
        if (ui != null)
          map.put(MADE_BY_URI, ui.getFullName());
        else
          map.put(MADE_BY_URI, "unknown user");
        map.put(VERSION_UPDATE_ID, new Integer(uid));
        map.put(VERSION_NAME_URI, label);
      }
      rs.close();
      st.close();
      con.close();
    } catch (SQLException ex) {
      throw new SailInternalException(ex);
    }
    return map;
  } // getVersionMetaInfo()


  /**
   * Updates the inferred statements.
   **/
  protected void _processChangedTriples()
          throws SQLException
  {
    try {

      //_statementsRemoved || was in IF below
      // create new update
        if (((_statementsAdded) || (_statementsRemoved))&& !isPausedCounterIncrement())
        {
          int updateKind = 0;
          if (_statementsAdded)
            updateKind = updateKind+1;
          if (_statementsRemoved)
            updateKind = updateKind+2;
          RDBMS().executeUpdate("INSERT into "+UPDATES_TABLE+
                                " VALUES("+(getStateUid()+1)+", NULL, "+getCurrentUserID()+", "+
                                updateKind+" , "+baseUrlIndex+")");
          if (getStateUid() == 1)
            labelCurrentState("Auto Version after the first Upload");
        }

        RDBMS().executeUpdate(
                              " INSERT INTO " + TRIPLES_HIST_TABLE + " SELECT " +
                              " t.id, t.subject, t.predicate, t.object, "+RDBMS().TRUE+", "+
                              getStateUid() + ", " + 0 +
                              " from "+NEW_TRIPLES_TABLE+" t"
                              );
    } catch (SQLException e) {
      e.printStackTrace();
    }
    super._processChangedTriples();
  }

  //>-- VersionManagement interface
}
