/*	OMM - Ontology Middleware Module
 *	Copyright (C) 2002 OntoText Lab, Sirma AI OOD
 *
 *	Contact:
 *	Sirma AI OOD, OntoText Lab.
 *	38A, Christo Botev Blvd.
 *	1000 Sofia, Bulgaria
 *	tel. +359(2)981 00 18
 *	fax. +359(2)981 90 58
 *	info@ontotext.com
 *
 * 	http://www.ontotext.com/
 *
 *	This library is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU Lesser General Public
 *	License as published by the Free Software Foundation; either
 *	version 2.1 of the License, or (at your option) any later version.
 *
 *	This library is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 *	Lesser General Public License for more details.
 *
 *	You should have received a copy of the GNU Lesser General Public
 *	License along with this library; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	02111-1307	USA
 */
package org.openrdf.sesame.sailimpl.omm.security;

import java.io.FileOutputStream;
import java.io.IOException;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.openrdf.util.jdbc.ConnectionPool;
import org.openrdf.util.log.ThreadLog;
import org.openrdf.vocabulary.RDF;
import org.openrdf.vocabulary.RDFS;

import org.openrdf.model.Graph;
import org.openrdf.model.Literal;
import org.openrdf.model.Resource;
import org.openrdf.model.URI;
import org.openrdf.model.Value;
import org.openrdf.model.ValueFactory;
import org.openrdf.model.impl.LiteralImpl;
import org.openrdf.model.impl.URIImpl;
import org.openrdf.model.impl.ValueFactoryImpl;

import org.openrdf.sesame.Sesame;
import org.openrdf.sesame.admin.XmlAdminMsgWriter;
import org.openrdf.sesame.constants.RDFFormat;
import org.openrdf.sesame.omm.SecurityServices;
import org.openrdf.sesame.omm.SessionContext;
import org.openrdf.sesame.omm.VersionManagement;
import org.openrdf.sesame.query.MalformedQueryException;
import org.openrdf.sesame.query.QueryErrorType;
import org.openrdf.sesame.query.TableQueryResultListener;
import org.openrdf.sesame.repository.local.LocalRepository;
import org.openrdf.sesame.repository.local.LocalService;
import org.openrdf.sesame.sail.LiteralIterator;
import org.openrdf.sesame.sail.NamespaceIterator;
import org.openrdf.sesame.sail.RdfRepository;
import org.openrdf.sesame.sail.RdfSchemaRepository;
import org.openrdf.sesame.sail.RdfSchemaSource;
import org.openrdf.sesame.sail.RdfSource;
import org.openrdf.sesame.sail.Sail;
import org.openrdf.sesame.sail.SailChangedListener;
import org.openrdf.sesame.sail.SailInitializationException;
import org.openrdf.sesame.sail.SailInternalException;
import org.openrdf.sesame.sail.SailUpdateException;
import org.openrdf.sesame.sail.StackedSail;
import org.openrdf.sesame.sail.StatementIterator;
import org.openrdf.sesame.sail.query.Query;
import org.openrdf.sesame.sailimpl.rdbms.RdbmsNamespace;

/* Jena depndency
import com.hp.hpl.mesa.rdf.jena.mem.*;
import com.hp.hpl.mesa.rdf.jena.common.prettywriter.*;
*/

/**
 * SecuritySail.java
 * <p>
 * Title: Knowledge Control System
 * </p>
 * <p>
 * </p>
 * <p>
 * Company: OntoText Lab. Sirma AI.
 * </p>
 * 
 * @author borislav popov
 * @version 1.0 Security Sail's purpose is to provide a fine grained security
 *          filtering layer over an RDF repository (e.g. Sesame) in a stacked
 *          sails architecture.
 *          <p>
 *          TODO:
 *          <ul>
 *          <li>PAUSED : getDomain,getRange :is properly filtered ?
 *          <li>remove the literal set object method from the pattern
 *          restrictions
 *          <li>BUG in the WEB interface: when selecting Extract data from the
 *          repository
 *          <li>isValueAccessible implement fully : now only for resource
 *          <li>QueryRestrictions : is correctly implemented? is defined to
 *          restrict statements but is used to restrict resources
 *          </ul>
 */
public class SecuritySail
		implements RdfSchemaRepository, StackedSail,
		SecurityServices, VersionManagement, TableQueryResultListener
{
	/** the transient triples for the current transaction */
	private ArrayList transients = new ArrayList();

	private static int lastResourceId = 0;
	private static int lastLiteralId = 0;

	/*
	 * the following base references are either null either are casted to the
	 * type specified but all are one and the same object
	 */
	/** base Sail */
	private Sail baseSail = null;
	/** base RdfSchemaSource */
	private RdfSchemaSource baseRdfSchemaSource = null;
	/** base RdfRepository */
	private RdfRepository baseRdfRepository = null;
	/** base RdfSource */
	private RdfSource baseRdfSource = null;
	/** base VersionManagement Sail */
	private VersionManagement versionMngmt = null;

	/** the current query results table consisting of Values */
	private ArrayList queryResult = new ArrayList();

	/** Maps executed queries vs. returned results list */
	Map queryResults = new HashMap();

	/**
	 * flag to indicate the status of the query: whether it is ready or is being
	 * used currently
	 */
	boolean isQueryReady = true;


	/** the class of the VersionManagement interface */
	private final static String VERSION_SAIL = "org.openrdf.sesame.omm.VersionManagement";

	/* > demo data related constants */
	private final static String SKILL = "http://www.ontotext.com/otk/2002/05/skills.rdfs#Skill";
	private final static String SKILL_LEVEL =
		"http://www.ontotext.com/otk/2002/05/skills.rdfs#SkillLevel";


	private final static String BUS_SKILL =
		"http://www.ontotext.com/otk/2002/05/sirma_skills_hier.rdfs#BusinessSkill";
	private final static String ESPIONAGE_SKILL =
		"http://www.ontotext.com/otk/2002/05/sirma_skills_hier.rdfs#BusinessEspionage";
	private final static String TECH_SKILL =
		"http://www.ontotext.com/otk/2002/05/sirma_skills_hier.rdfs#TechnicalSkill";
	private final static String SWING_SKILL =
		"http://www.ontotext.com/otk/2002/05/sirma_skills_hier.rdfs#Swing";
	private final static String DEMO_SWING_SKILL =
		"http://www.ontotext.com/otk/2002/05/sirma_enter_kb.rdf#DemoSwingSkill";
	private final static String RDF_TYPE =
		"http://www.w3.org/1999/02/22-rdf-syntax-ns#type";
	private final static String RESOURCE = "http://www.w3.org/2000/01/rdf-schema#Resource";
	private final static String PERSON ="http://www.ontotext.com/otk/2002/05/enterprise.rdfs#Person";

	private final static String HAS_SKILL = "http://www.ontotext.com/otk/2002/05/skills.rdfs#hasSkill";
	private final static String HAS_POSITION = "http://www.ontotext.com/otk/2002/05/enterprise.rdfs#hasPosition";
	private final static String SUB_CLASS_OF = "http://www.w3.org/2000/01/rdf-schema#subClassOf";

	private final static String AI_SKILL_QUERY = "select * from {X} http://www.w3.org/2000/01/rdf-schema#subClassOf {Y} where X = http://www.ontotext.com/otk/2002/05/sirma_skills_hier.rdfs#AISkill";
	private final static String HAS_POSITION_QUERY = "select * from {X} http://www.ontotext.com/otk/2002/05/enterprise.rdfs#hasPosition {Y}";

	private final static String AI_SKILL ="http://www.ontotext.com/otk/2002/05/sirma_skills_hier.rdfs#AISkill";
	private final static String MITAC ="http://www.ontotext.com/otk/2002/05/sirma_enter_kb.rdf#DimitarManov";
	/* > db related constants */

	private final static String KCS_URI = "http://www.ontotext.com/otk/2002/03/KCS.rdfs";
	private final static String BASE_KCS_URI = "http://www.ontotext.com/otk/2002/03/kcs.rdfs#";


	/** Name of table containing Users. */
	public static final String USERS_TABLE = "users";
	/** Name of table containing Users' roles. */
	public static final String USERS_ROLES_TABLE = "users_roles";
	/** Name of table containing Users' Rules. */
	public static final String USERS_RULES_TABLE = "users_rules";
	/** Name of table containing Roles. */
	public static final String ROLES_TABLE = "roles";
	/** Name of table containing Roles Hierarchy. */
	public static final String ROLES_HIERARCHY_TABLE = "roles_hierarchy";
	/** Name of table containing Roles' Rules. */
	public static final String ROLES_RULES_TABLE = "roles_rules";
	/** Name of table containing Security Rules. */
	public static final String SECURITY_RULES_TABLE = "security_rules";
	/** Name of table containing Restrictions. */
	public static final String RESTRICTIONS_TABLE = "restrictions";
	/** Name of table containing Resource Or Properties Restrictions. */
	public static final String RES_PROP_RESTRS_TABLE = "res_propr_restrs";
	/** Name of table containing Pattern Restrictions. */
	public static final String PATTERN_RESTRS_TABLE = "pattern_restrs";
	/** Name of table containing Query Restrictions. */
	public static final String QUERY_RESTRS_TABLE = "query_restrs";

	/** Name of table containing Resources. */
	public static final String RESOURCES_TABLE = "resources";
	/** Name of table containing Literals. */
	public static final String LITERALS_TABLE = "literals";
	/** Name of table containing Namespaces. */
	public static final String NAMESPACES_TABLE = "namespaces";

	/** drop table statement */
	public static final String DROP_TABLE = "DROP TABLE ";
	/** select * from statement */
	public static final String SELECT = "SELECT * FROM ";
	/** INSERT INTO statement */
	public static final String INSERT = "INSERT INTO ";
	/** VALUES part of statement */
	public static final String VALUES = " VALUES ";

	/** Integer datatype. */
	protected final static String INT = "INTEGER UNSIGNED";
	/** character 20 datatype */
	protected final static String CHR20 = "character varying(20)";
	/** character 255 datatype */
	protected final static String CHR255 = "character varying(255)";

	/* < db related constants */

	/** a map of uris vs resource ids */
	static Map resIdsByUri = new HashMap();

	/** conection pool */
	protected ConnectionPool conPool = null;

	/* > Security Services implemenation related members */
	/** the set of users */
	private Set users = new HashSet();
	/** the map of users' logins vs users objects */
	private Map usersByLogin = new HashMap();
	/** the map of users by id */
	private Map usersById = new HashMap();

	/** the set of security roles */
	private Set roles = new HashSet();

	/** the map of roles' ids vs roles objects */
	private Map rolesById = new HashMap();

	/** the map of roles' names vs roles objects */
	private Map rolesByName = new HashMap();

	/** the map of restriction ids vs. restriction objects */
	private Map restrictionsById = new HashMap();

	/** the map of rules ids vs security rules objects. */
	private Map rulesById = new HashMap();


	/* < Security Services implemenation related members */

	/** Resources by Id map */
	private Map resById = new HashMap();

	/** Ids by Resources map */
	private Map idByRes = new HashMap();

	/** the last resource id that has been used */
	private int lastResId = 0;

	/** Namespaces by Id map */
	private Map namespaces = new HashMap();
	/** Literals by Id map */
	private Map literalById = new HashMap();
	/** Ids by Literals map */
	private Map idByLiteral = new HashMap();


	public SecuritySail() {
	}


	/* > */
		/* > RDFSchemaSource implementation */

	public StatementIterator getExplicitStatements(
		Resource subj, URI pred, Value obj)
	{
		StatementIterator si = baseRdfSchemaSource.getExplicitStatements(subj,pred,obj);
		try {
			si = new StatementFilterIterator(si,this,Right.READ);
		}
		catch (Exception e){
			throw new SailInternalException(e);
		}
		return si;
	}

	public boolean hasExplicitStatement(Resource subj, URI pred, Value obj)
	{
		boolean has = baseRdfSchemaSource.hasExplicitStatement(subj,pred,obj);
		if ( has ) {
			has = isStatementAccessible(subj,pred,obj,Right.READ);
		}
		return has;
	}

	public Query optimizeQuery(Query qc) {
		return qc;
	}

	public StatementIterator getClasses() {
		StatementIterator si = baseRdfSchemaSource.getClasses();
		try {
			si = new StatementFilterIterator(si,this,Right.READ);
		} catch (Exception e){
			throw new SailInternalException(e);
		}
		return si;
	}

	public boolean isClass(Resource resource) {
		boolean is = baseRdfSchemaSource.isClass(resource);
		if ( is ) {
			 is = is && isResourceAccessible(resource);
		}
		return is;
	}

	public StatementIterator getProperties() {
		StatementIterator si = baseRdfSchemaSource.getProperties();

		return si;
	}

	public boolean isProperty(Resource resource) {
		boolean is = baseRdfSchemaSource.isProperty(resource);
		if ( is ) {
			is = is && isResourceAccessible(resource);
		}
		return is;
	}

	public StatementIterator getSubClassOf(Resource subClass, Resource superClass) {
		StatementIterator si = baseRdfSchemaSource.getSubClassOf(subClass, superClass);
		try {
			si = new StatementFilterIterator(si, this, Right.READ);
		}
		catch (Exception e){
			throw new SailInternalException(e);
		}
		return si;
	}

	public StatementIterator getDirectSubClassOf(Resource subClass, Resource superClass) {
		StatementIterator si = baseRdfSchemaSource.getDirectSubClassOf(subClass, superClass);
		try {
			si = new StatementFilterIterator(si, this, Right.READ);
		} catch (Exception e){
			throw new SailInternalException(e);
		}
		return si;
	}

	public boolean isSubClassOf(Resource subClass, Resource superClass) {
		boolean is = baseRdfSchemaSource.isSubClassOf(subClass,superClass);
		if ( is ) {
			is = is && isResourceAccessible(subClass);
			is = is && isResourceAccessible(superClass);
		}
		return is;
	}

	public boolean isDirectSubClassOf(Resource subClass, Resource superClass) {
		boolean is = baseRdfSchemaSource.isDirectSubClassOf(subClass,superClass);
		if ( is ) {
			is = is && isResourceAccessible(subClass);
			is = is && isResourceAccessible(superClass);
		}
		return is;
	}

	public StatementIterator getSubPropertyOf(Resource subProperty, Resource superProperty) {
		StatementIterator si = baseRdfSchemaSource.getSubPropertyOf(subProperty, superProperty);
		try {
			si = new StatementFilterIterator(si, this, Right.READ);
		} catch (Exception e){
			throw new SailInternalException(e);
		}
		return si;
	}

	public StatementIterator getDirectSubPropertyOf(Resource subProperty, Resource superProperty) {
		StatementIterator si = baseRdfSchemaSource.getDirectSubPropertyOf(subProperty, superProperty);
		try {
			si = new StatementFilterIterator(si, this, Right.READ);
		} catch (Exception e){
			throw new SailInternalException(e);
		}
		return si;
	}

	public boolean isSubPropertyOf(Resource subProperty, Resource superProperty) {
		boolean is = baseRdfSchemaSource.isSubPropertyOf(subProperty,superProperty);
		if ( is ) {
			is = is && isResourceAccessible(subProperty);
			is = is && isResourceAccessible(superProperty);
		}
		return is;
	}

	public boolean isDirectSubPropertyOf(Resource subProperty, Resource superProperty) {
		boolean is = baseRdfSchemaSource.isDirectSubPropertyOf(subProperty,superProperty);
		if ( is ) {
			is = is && isResourceAccessible(subProperty);
			is = is && isResourceAccessible(superProperty);
		}
		return is;
	}

	public StatementIterator getDomain(Resource prop, Resource domain) {
		StatementIterator si = baseRdfSchemaSource.getDomain(prop, domain);
		try {
			si = new StatementFilterIterator(si, this, Right.READ);
		}
		catch (Exception e) {
			throw new SailInternalException(e);
		}
		return si;
	}

	public StatementIterator getRange(Resource prop, Resource range) {
		StatementIterator si = baseRdfSchemaSource.getRange(prop, range);
		try {
			si = new StatementFilterIterator(si, this, Right.READ);
		}
		catch (Exception e) {
			throw new SailInternalException(e);
		}
		return si;
	}

	public StatementIterator getType(Resource anInstance, Resource aClass) {
		StatementIterator si = baseRdfSchemaSource.getType(anInstance, aClass);
		try {
			si = new StatementFilterIterator(si, this, Right.READ);
		}
		catch (Exception e) {
			throw new SailInternalException(e);
		}
		return si;
	}

	public StatementIterator getDirectType(Resource anInstance, Resource aClass) {
		StatementIterator si = baseRdfSchemaSource.getDirectType(anInstance, aClass);
		try {
			si = new StatementFilterIterator(si, this, Right.READ);
		}
		catch (Exception e) {
			throw new SailInternalException(e);
		}
		return si;
	}

	public boolean isType(Resource anInstance, Resource aClass) {
		boolean is = false;
		if (isResourceAccessible(anInstance) &&
			isResourceAccessible(aClass))
		{
			is = baseRdfSchemaSource.isType(anInstance,aClass);
		}
		return is;
	}

	public boolean isDirectType(Resource anInstance, Resource aClass) {
		boolean is = false;
		if (isResourceAccessible(anInstance) &&
			isResourceAccessible(aClass))
		{
			is = baseRdfSchemaSource.isDirectType(anInstance,aClass);
		}
		return is;
	}

	public LiteralIterator getLiterals(
			String label, String language, URI datatype)
	{
		// FIXME: do we need to filter this?
		return baseRdfSchemaSource.getLiterals(label, language, datatype);
	}

/* < RDFSchemaSource implementation */

/* > RDFRepository Implementation */
	public void startTransaction() {
		baseRdfRepository.startTransaction();

		// reset the query results since they might be inaccurate after the
		// transaction
		queryResults = new HashMap();
	}

	public void commitTransaction() {
		String status = "" ;
		if (areThereTransients()) {

			// freeze THE UPDATE COUNTER
			if (null != versionMngmt)
				versionMngmt.pauseCounterIncrement();

			baseRdfRepository.commitTransaction();

			//start a new transaction to remove the irrelevant statements
			baseRdfRepository.startTransaction();

			status = removeIrrelevantTransients();

			// unFreeze the COUNTER
			if (null != versionMngmt)
				versionMngmt.continueCounterIncrement();
		} // if there are transinients

		baseRdfRepository.commitTransaction();

		if (status.length()>0) {
			ThreadLog.trace(status);
			throw new SailInternalException(status);
		}
	}

	public boolean transactionStarted() {
		return baseRdfRepository.transactionStarted();
	}

	public void addStatement(Resource subj, URI pred, Value obj) throws SailUpdateException {
		baseRdfRepository.addStatement(subj,pred,obj);
		if (!isStatementAccessible(subj,pred,obj,Right.ADD)) {
			addTransient(subj,pred,obj);
			throw new SailUpdateException(
			"Statement Added temporarily.\n "+
			"The permissions will be checked at the end of the transaction.");
		}
	}

//			public boolean addStatement(Resource subj, URI pred, Value obj, boolean
// bExplicit) throws SailUpdateException {
//				return baseRdfRepository.addStatement(subj,pred,obj,bExplicit);
//			}

	public int removeStatements(Resource subj, URI pred, Value obj) throws SailUpdateException {

		StatementIterator si = baseRdfRepository.getStatements(subj,pred,obj);
		StatementFilterIterator sfi = null;
		try {
			sfi = new StatementFilterIterator(si,this,Right.REMOVE);
		} catch (Exception e) {
			throw new SailInternalException(e);
		}
		int removed = 0;
		while ( sfi.hasNext() ) {
			org.openrdf.model.Statement st =
				(org.openrdf.model.Statement) sfi.next();
			removed += baseRdfRepository.removeStatements(
				st.getSubject(),st.getPredicate(),st.getObject());
		}
		return removed;
	}

	public void clearRepository() 
		throws SailUpdateException 
	{
		if (
			isRepositoryAccessible(Right.REMOVE)
			||
			isRepositoryAccessible(Right.ADMIN)) {

				baseRdfRepository.clearRepository();
		} // if accessible
		else {
			throw new SailInternalException(
			"Cannot Clear the Repository. Access Denied.");
		}
	}

	public void changeNamespacePrefix(String namespace, String prefix) throws SailUpdateException {
		baseRdfRepository.changeNamespacePrefix(namespace, prefix);
	}
/* > RDFRepository Implementation */

/* > RDFSource Implementation */
	public ValueFactory getValueFactory() {
		//FIXME: implement
		return null;
	}

	public StatementIterator getStatements(Resource subj, URI pred, Value obj) {
		StatementIterator si = baseRdfSource.getStatements(subj,pred,obj);
		try {
			si = new StatementFilterIterator(si,this,Right.READ);
		} catch (Exception e){
			throw new SailInternalException(e);
		}
		return si;
	}

	public boolean hasStatement(Resource subj, URI pred, Value obj) {
		boolean has = baseRdfSource.hasStatement(subj,pred,obj);
		if ( has ) {
			has = has && isStatementAccessible(subj,pred,obj,Right.READ);
		}
		return has;
	}

	public NamespaceIterator getNamespaces() {
		return baseRdfSource.getNamespaces();
	}
/* < RDFSource Implementation */


	public void initialize(Map configParams) throws SailInitializationException {
		ThreadLog.trace("initializing SecuritySail>>>");
		// Get initialization parameters
		String jdbcDriver = (String)configParams.get("jdbcDriver");
		String jdbcUrl = (String)configParams.get("jdbcUrl");
		String user = (String)configParams.get("user");
		String password = (String)configParams.get("password");
		String securitySetup = (String)configParams.get("security_setup");


		if (jdbcDriver == null) {
			throw new SailInitializationException("parameter 'jdbcDriver' missing");
		}
		if (jdbcUrl == null) {
			throw new SailInitializationException("parameter 'jdbcUrl' missing");
		}
		if (user == null) {
			throw new SailInitializationException("parameter 'user' missing");
		}
		if (password == null) {
			throw new SailInitializationException("parameter 'password' missing");
		}

		// Load jdbc driver
		try {
			Class.forName(jdbcDriver);
		}
		catch (ClassNotFoundException e) {
			throw new SailInitializationException(
											"Unable to load JDBC-driver '" + jdbcDriver + "'", e);
		}

		// Create ConnectionPool
		conPool = new ConnectionPool(jdbcUrl, user, password);

		// Initialize database schema, if necessary
		initDatabase();

		Connection con = null;
		java.sql.Statement st = null;
		try {
			con = conPool.getConnection();
			st = con.createStatement();
			loadNameSpaceTable(con,st);
			loadResourcesTable(con,st);
			loadLiteralsTable(con,st);


			if ( null != securitySetup) {
				ThreadLog.trace("Loading Security Policy from url == "+securitySetup);

				/*
				 * IMPORT SECURITY SETUP USING THE JENA DEPENDENT IMPORT
				 * java.net.URL surl = new java.net.URL(securitySetup);
				 * 
				 * BufferedReader br = new BufferedReader(new
				 * InputStreamReader(surl.openStream()));
				 * baseRdfRepository.startTransaction();
				 * 
				 * importSecurityPolicy(br);
				 * baseRdfRepository.commitTransaction();
				 */


				/* import security setup using the in-memory sail import */
				java.net.URL surl = new java.net.URL(securitySetup);
				
				LocalService service = Sesame.getService();
				
				LocalRepository repository = service.createRepository("tempSecurityRep", false);
				
				repository.addData(surl, "", RDFFormat.RDFXML, true, 
						new XmlAdminMsgWriter(new FileOutputStream("ImportSecurityPolicy.Log")));
				
				

				//org.openrdf.sesame.sailimpl.memory.RdfRepository memRepos;
				//Map map = new HashMap();
				//map.put("file","TempSecurityPolicy.nt");
				//memRepos= new org.openrdf.sesame.sailimpl.memory.RdfRepository();
				//memRepos.initialize(map);

				//memRepos.clearRepository();
				//RdfAdmin admin = new RdfAdmin(memRepos);
				//admin.addRdfModel(surl.openStream(),securitySetup ,
				//	 new XmlAdminMsgWriter(),
				//	 true);
				baseRdfRepository.startTransaction();
				importPolicy(repository.getGraph());
				baseRdfRepository.commitTransaction();
				
				service.removeRepository(repository.getRepositoryId());
				repository = null;
				
				loadNameSpaceTable(con,st);
				loadResourcesTable(con,st);
				loadLiteralsTable(con,st);

				//Store the setup to the repository
				storeSecuritySetup();
			}
			else {
				ThreadLog.trace("Loading Security Policy from the database");
				loadSecuritySetup();
			}
		}
		catch (Exception x) {
			throw new SailInternalException("While reading the security setup.",x);
		}

		ThreadLog.trace("<<<SecuritySail initialized successfully");
	} // initialize(Map)

	public void shutDown() {
		baseSail.shutDown();
	}

/* < Sail Implementation */

/* < RdfSchemaRepository implementation */

/* < StackedSail implementation */

	public void setBaseSail(Sail baseSail) {
		this.baseSail = baseSail;

		if (baseSail instanceof RdfSchemaSource )
			baseRdfSchemaSource = (RdfSchemaSource)baseSail;

		if (baseSail instanceof RdfSource )
			baseRdfSource = (RdfSource)baseSail;

		if (baseSail instanceof RdfRepository )
			baseRdfRepository = (RdfRepository)baseSail;

		versionMngmt = (VersionManagement)org.openrdf.sesame.sail.util.QuerySailStack.queryInterface( baseSail, VERSION_SAIL);
	}

	public Sail getBaseSail() {
		return this.baseSail;
	}
/* < StackedSail implementation */

/* > Version Management implementation */
	public void labelState(long stateUID, String label) {
		if ( isVersionTrackingAccessible() )
			versionMngmt.labelState(stateUID, label);
	}

	public void labelCurrentState(String label) {
		if ( isVersionTrackingAccessible() )
			versionMngmt.labelCurrentState(label) ;
	}

	public void revertToState(long stateUID) {
		if ( isVersionTrackingAccessible() )
			versionMngmt.revertToState(stateUID);
	}

	public String branchState(long stateUID) {
		String result = "";
		if ( isVersionTrackingAccessible() )
			result = versionMngmt.branchState(stateUID);
		return result;
	}

	public void workWithState(long stateUID) {
		if ( isVersionTrackingAccessible() )
			versionMngmt.workWithState(stateUID);
	}

	public Iterator getVersions() {
		Iterator i = new ArrayList().iterator();
		if ( isVersionTrackingAccessible() )
			i = versionMngmt.getVersions();
		return i;
	}

	public void lockStatements(Iterator statementsList) {
		if ( isVersionTrackingAccessible() )
			versionMngmt.lockStatements(statementsList);
	}

	public void unlockStatements(Iterator statementsList) {
		if ( isVersionTrackingAccessible() )
			versionMngmt.unlockStatements(statementsList);
	}

	public Iterator getUpdateIds() {
		Iterator i = new ArrayList().iterator();
		if ( isVersionTrackingAccessible() )
			i= versionMngmt.getUpdateIds();
		return i;
	}

	public void pauseCounterIncrement() {
		if ( isVersionTrackingAccessible() )
			versionMngmt.pauseCounterIncrement();
	}

	public void continueCounterIncrement() {
		if ( isVersionTrackingAccessible() )
			versionMngmt.continueCounterIncrement();
	}

	public boolean isPausedCounterIncrement() {
		if ( isVersionTrackingAccessible() )
			return versionMngmt.isPausedCounterIncrement();
		return false;
	}

	public Iterator getVersionIds() {
		Iterator i = new ArrayList().iterator();
		if ( isVersionTrackingAccessible() )
			i= versionMngmt.getVersionIds();
		return i;
	}

	/* MetaInfo Retrieval Methods */

	public Map getMetaInfo(String subj, String pred, String obj) {
		Map mi = new HashMap();
		if ( isVersionTrackingAccessible() )
			mi = versionMngmt.getMetaInfo(subj,pred,obj);
		return mi;
	}

	public Map getUpdateMetaInfo(String updateId) {
		Map mi = new HashMap();
		if ( isVersionTrackingAccessible() )
			mi = versionMngmt.getUpdateMetaInfo(updateId);
		return mi;
	}

	public Map getVersionMetaInfo(String versionId) {
		Map mi = new HashMap();
		if ( isVersionTrackingAccessible() )
			mi = versionMngmt.getVersionMetaInfo(versionId);
		return mi;
	}

/* < Version Management implementation */

/* > SecurityServices implementation */
	public int getResourceId(Resource res) throws SecurityException{
		Integer i = (Integer)idByRes.get(res);
		if (null == i) {
			throw new SecurityException("Id not found for resource:\n" +res);
		}
		int ii = i.intValue();
		return ii;
	}

	public Resource getResource(int id) {
		Resource res = (Resource)resById.get(new Integer(id));
		return res;
	}

	public boolean isVersionTrackingAccessible() {
		return isRepositoryAccessible(Right.ADMIN) && versionMngmt != null;
	}// isVersionTrackingAccessible(Right)

	public boolean isSchemaAccessible(Right right) {
		boolean result = false;
		/* authenticate */
		SessionContext context = SessionContext.getContext();
		int userId = context.userID;
		/* check */
		User user = (User)usersById.get(new Integer(userId));
		if (null!=user) {
			Set rulz = user.getRules();
			rulz.addAll(RoleImpl.getRules(user.getRoles(),false));
			ArrayList rulis = new ArrayList(rulz);
			for (int i = 0 ; i < rulis.size() ; i++ ) {
				Rule ru = (Rule)rulis.get(i);
				Restriction rst = ru.getRestriction();
				if (rst.getType() != Restriction.REPOSITORY
					&&	rst.getType() != Restriction.SCHEMA) {
						continue;
				}

				if ( right.equals(Right.READ) ) {
					if (ru.getReadRight()) {
						result = true;
						break;
					}
				}

				if ( right.equals(Right.ADD) ) {
					if (ru.getAddRight()) {
						result = true;
						break;
					}
				}

				if ( right.equals(Right.REMOVE) ) {
					if (ru.getRemoveRight()) {
						result = true;
						break;
					}
				}
			} // for rules
		} // if user
		return result;
	} // isSchemaAccessible(Right)

	public boolean isRepositoryAccessible(Right right) {
		boolean result = false;
		/* authenticate */
		SessionContext context = SessionContext.getContext();
		int userId = context.userID;
		/* check */
		User user = (User)usersById.get(new Integer(userId));
		if (null!=user) {
			Set rulz = user.getRules();
			rulz.addAll(RoleImpl.getRules(user.getRoles(),false));
			ArrayList rulis = new ArrayList(rulz);
			for (int i = 0 ; i < rulis.size() ; i++ ) {
				Rule ru = (Rule)rulis.get(i);
				Restriction rst = ru.getRestriction();
				if (rst.getType() != Restriction.REPOSITORY) {
					continue;
				}
				if ( right.equals(Right.READ) ) {
					if (ru.getReadRight()) {
						result = true;
						break;
					}
				}

				if ( right.equals(Right.REMOVE) ) {
					if (ru.getRemoveRight()) {
						result = true;
						break;
					}
				}

				if ( right.equals(Right.ADMIN) ) {
					if (ru.getAdminRight()) {
						result = true;
						break;
					}
				}

				if ( right.equals(Right.HISTORY) ) {
					if (ru.getHistoryRight()) {
						result = true;
						break;
					}
				}


			} // for rules
		} // if user
		return result;
	} // isRepositoryAccessible(Right)

	public boolean isStatementAccessible(Resource subj, URI pred, Value obj, Right right) {
		boolean result = false;
		/* authenticate */
		SessionContext context = SessionContext.getContext();
		int userId = context.userID;
		/* check */
		User user = (User)usersById.get(new Integer(userId));
		if (null!=user) {
			Set rulz = user.getRules();
			rulz.addAll(RoleImpl.getRules(user.getRoles(),false));
			ArrayList rulis = new ArrayList(rulz);
			for (int i = 0 ; i < rulis.size() ; i++ ) {
				Rule ru = (Rule)rulis.get(i);

				if (!ru.getAddRight() && !ru.getRemoveRight() && !ru.getReadRight()) {
					continue;
				}

				if (right.equals(Right.READ) && !ru.getReadRight() ||
					right.equals(Right.REMOVE) && !ru.getRemoveRight() ||
					right.equals(Right.ADD) && !ru.getAddRight())
				{
					continue;
				}

				Restriction rst = ru.getRestriction();

				if (rst.type == Restriction.REPOSITORY ) {
					// OPTIMIZE
					result = true;
					break;
				}

				if (rst.type == Restriction.SCHEMA) {
					/* check if the subject is part of the schema */
					if (baseRdfSchemaSource.isType(subj,URIImpl.RDFS_CLASS) ||
						baseRdfSchemaSource.isType(subj,URIImpl.RDF_PROPERTY))
					{
						result = true;
						break;
					} // if
				} // schema

				/*
				 * CHECK SUBJECT : IF ACCESSIBLE : THEN THE WHOLE STATEMENT IS
				 * ACCESSIBLE
				 */
				if (isResourceAccessible(subj,right)) {
					result = true;
					break;
				}



				/* PROPERTIES */
				if (rst.type == Restriction.PROPERTIES) {
					PropertiesRestriction propr = (PropertiesRestriction) rst;
					ArrayList props = new ArrayList(propr.getProperties());
					for (int pi = 0 ; pi < props.size() ; pi++ ) {
						Resource prop = (Resource)props.get(pi);
						if ( baseRdfSchemaSource.isSubPropertyOf(pred,prop) ) {
							result = true;
							break;
						}
					} // for properties
					if ( result ) break;
				} // Properties

				/* PATTERN */
				if (rst.type == Restriction.PATTERN ) {

					boolean subjAccessible = true ;
					boolean predAccessible = true ;
					boolean objAccessible = true ;

					PatternRestriction pat = (PatternRestriction) rst;
					/* SUBJECT */
					ArrayList subjs = new ArrayList( pat.getSubjectRestrictions());
					for ( int si = 0 ; si < subjs.size() ; si++ ) {
						subjAccessible = false;
						ResourceRestriction sr = (ResourceRestriction)subjs.get(si);
						ArrayList resList = new ArrayList(sr.getResources());
						for ( int ri=0 ; ri < resList.size() ; ri++) {
							Resource reso = (Resource) resList.get(ri);
							if (sr.getType() == Restriction.CLASSES) {
								if (baseRdfSchemaSource.isType(subj,reso)) {
									subjAccessible = true;
									break;
								}
							} else {
								if (sr.getType() == Restriction.CLASSES_OVER_SCHEMA) {
									if (baseRdfSchemaSource.isSubClassOf(subj,reso)) {
										subjAccessible = true;
										break;
									}
								} else {
									if (sr.getType() == Restriction.INSTANCES) {
										if (subj.equals(reso)) {
											subjAccessible = true;
											break;
										}
									} else {
											throw new SailInternalException(
											"Object Restrictions (part of the Pattern Restriction)\n "+
											"should be of type ClassesRestriction or InstancesRestriction");
									}
								}
							} // else

						} // for reslist
						if (subjAccessible) break;
					}	// for subjectrestrs

					/* PREDICATE */
					ArrayList preds = new ArrayList( pat.getPredicateRestrictions());
					for ( int predi = 0 ; predi < preds.size() ; predi++ ) {
						predAccessible = false;
						PropertiesRestriction pr = (PropertiesRestriction)preds.get(predi);
						ArrayList propList = new ArrayList(pr.getProperties());
						for ( int propi = 0 ; propi < propList.size(); propi++) {
							Resource prop = (Resource)propList.get(propi);
							if ( baseRdfSchemaSource.isSubPropertyOf(pred,prop)){
								 predAccessible = true;
								 break;
							}
						} // for propi
						if ( predAccessible ) break;
					} // for predi




					/* OBJECT */
					ArrayList objs = new ArrayList( pat.getObjectRestrictions());
					for ( int obji = 0 ; obji < objs.size() ; obji++ ) {
						objAccessible = false;

						Object objRestr = objs.get(obji);
						if (objRestr instanceof Literal) {
							Value val = (Value) objRestr;
							if ( val.equals(obj)) {
								objAccessible = true;
								break;
							}
						} else {
							if (objRestr instanceof ResourceRestriction) {
								ResourceRestriction rr = (ResourceRestriction)objRestr;
								ArrayList resList = new ArrayList(rr.getResources());
								for ( int ri=0 ; ri < resList.size() ; ri++) {
									Resource reso = (Resource) resList.get(ri);
									if (rr.getType() == Restriction.CLASSES) {
										if (obj instanceof Resource &&
											baseRdfSchemaSource.isType((Resource)obj,reso))
										{
											objAccessible = true;
											break;
										}
									}
									else if (rr.getType() == Restriction.CLASSES_OVER_SCHEMA) {
										if (obj instanceof Resource &&
											baseRdfSchemaSource.isSubClassOf((Resource)obj,reso))
										{
											objAccessible = true;
											break;
										}
									}
									else if (rr.getType() == Restriction.INSTANCES) {
										if (obj.equals(reso)) {
											objAccessible = true;
											break;
										}
									} // else

								} // for reslist

							} else {
								throw new SailInternalException(
									"The Object's Restrictions (in a Pattern restriction) should be \n"+
									"either a Literal, either a ResourceRestriction.");
							}
						} //else

						if ( objAccessible ) break;
					} // for obji



					if ( subjAccessible && predAccessible && objAccessible ) {
						result = true;
						break;
					}
				} // PATTERN

			} //for all rulez
		}	// user not null
		return result;
	} //isStatementAccessible(Resource,URI,Value,Right)

	public boolean isStatementAccessible(org.openrdf.model.Statement st,Right right) {
		return isStatementAccessible(st.getSubject(),st.getPredicate(),st.getObject(),right);
	} // isStatementAccessible(Statement,Right)

	public boolean isValueAccessible(Value val) {
		boolean is = true;
		if ( val instanceof Resource) {
			is = isResourceAccessible((Resource)val);
		}
		return is;
	}

	public boolean isResourceAccessible(Resource res, Right right) {
		boolean result = false;
		/* authenticate */
		SessionContext context = SessionContext.getContext();
		int userId = context.userID;
		/* check */
		User user = (User)usersById.get(new Integer(userId));
		if (null!=user) {
			Set rulz = user.getRules();
			rulz.addAll(RoleImpl.getRules(user.getRoles(),false));
			ArrayList rulis = new ArrayList(rulz);
			for (int i = 0 ; i < rulis.size() ; i++ ) {
				Rule ru = (Rule)rulis.get(i);

				if (!ru.getAddRight() && !ru.getRemoveRight() && !ru.getReadRight()) {
					continue;
				}

				if (right.equals(Right.READ) && !ru.getReadRight() ||
					right.equals(Right.REMOVE) && !ru.getRemoveRight() ||
					right.equals(Right.ADD) && !ru.getAddRight())
				{
					continue;
				}

				Restriction rst = ru.getRestriction();

				if (rst.type == Restriction.REPOSITORY ) {
					result = true;
					break;
				}

				if (rst.type == Restriction.SCHEMA) {
					/* check if the resource is part of the schema */
					if (baseRdfSchemaSource.isType(res,URIImpl.RDFS_CLASS)
						|| baseRdfSchemaSource.isType(res,URIImpl.RDF_PROPERTY)) {
						result = true;
						break;
					} // if
				} // schema

				if (rst.type == Restriction.CLASSES){
					ClassesRestriction cr = (ClassesRestriction) rst;
					ArrayList classes = new ArrayList(cr.getResources());
					for (int j=0; j < classes.size() ; j++) {
						Resource clas = (Resource)classes.get(j);
						if (baseRdfSchemaSource.isType(res,clas)) {
								result = true;
								break;
						}
					} // for classes
					if (result) break;
				} //if classes

				if (rst.type == Restriction.CLASSES_OVER_SCHEMA){
					ResourceRestriction rr = (ResourceRestriction) rst;
					ArrayList classes = new ArrayList(rr.getResources());
					for (int j=0; j < classes.size() ; j++) {
						Resource clas = (Resource)classes.get(j);
						if (baseRdfSchemaSource.isSubClassOf(res,clas)) {
								result = true;
								break;
						}
					} // for classes
					if (result) break;
				} //if classes over schema

				if (rst.type == Restriction.INSTANCES){
					InstancesRestriction ir = (InstancesRestriction) rst;
					ArrayList instances = new ArrayList(ir.getResources());
					for (int j=0; j < instances.size() ; j++) {
						URI clas = (URI)instances.get(j);
						if (res instanceof URI) {
							if ( ((URI)res).getLocalName().equals(clas.getLocalName()) &&
								 ((URI)res).getNamespace().equals(clas.getNamespace()) )
							{
								result = true;
								break;
							}
						}
						else {
							// FIXME what to do for bNodes?
						}
					} // for instances
					if (result) break;
				} //if instances

				if (rst.type == Restriction.QUERY) {
					result = isQuAccessible((QueryRestriction) rst, res);
					if (result) break;
				}

			} //for all rulez
		}	// user not null
		return result;

	} // isResourceAccessible(Resource,Right)


	public boolean isResourceAccessible(Resource res) {
		return isResourceAccessible(res,Right.READ);
	}


	public void addUser(int id, String login, String password, String name) {
		try {
			User user = new UserImpl(id,login,password,name);
			users.add(user);
			usersByLogin.put(login,user);
			usersById.put(new Integer(id),user);
		} catch (Exception e) {
			throw new SailInternalException(e);
		}
	}

	public void removeUser(String login) {
		User user = (User)usersByLogin.get(login);
		if (null!=user) {
			usersByLogin.remove(login);
			users.remove(user);
		}
	}

	public Set getUsers() {
		return users;
	}

	public User getUser(String login) {
		return (User)usersByLogin.get(login);
	}

	public User getUser(int id) {
		return (User)usersById.get(new Integer(id));
	}

	public Role createRole(int id, String name, String description, Set parents) {
		try {
			Role role = new RoleImpl(id,name,description);
			role.setParentRoles(parents);
			roles.add(role);
			rolesByName.put(name,role);
			return role;
		} catch (Exception e) {
			throw new SailInternalException(e);
		}
	}

	public void removeRole(String name) {
		Role role = (Role)rolesByName.get(name);
		if(null!=role) {
			rolesById.remove(new Integer(role.getId()));
			rolesByName.remove(name);
			roles.remove(role);
		}
	}

	public void removeRole(int id) {
		Role role = (Role)rolesById.get(new Integer(id));
		if(null!=role) {
			rolesById.remove(new Integer(id));
			rolesByName.remove(role.getName());
			roles.remove(role);
		}
	}

	public Set getRoles() {
		return roles;
	}

	public Role getRole(String name) {
		return (Role)rolesByName.get(name);
	}

	public Role getRole(int id) {
		return (Role)rolesById.get(new Integer(id));
	} // main

	public Restriction createRestriction(
		int id, int type, String name, String description) throws NullParameterException{
		Restriction restr = null;
		switch(type){
			case Restriction.REPOSITORY : {
				restr = Restriction.createRepositoryRestriction(id,name,description);
				break;
			}
			case Restriction.SCHEMA : {
				restr = Restriction.createSchemaRestriction(id,name,description);
				break;
			}
			case Restriction.QUERY : {
				restr = Restriction.createQueryRestriction(id,name,description);
				break;
			}
			case Restriction.INSTANCES : {
				restr = Restriction.createInstancesRestriction(id,name,description);
				break;
			}
			case Restriction.PATTERN : {
				restr = Restriction.createPatternRestriction(id,name,description);
				break;
			}
			case Restriction.PROPERTIES : {
				restr = Restriction.createPropertiesRestriction(id,name,description);
				break;
			}
			case Restriction.CLASSES : {
				restr = Restriction.createClassesRestriction(id,name,description);
				break;
			}
			case Restriction.CLASSES_OVER_SCHEMA : {
				restr = Restriction.createClassesOverSchemaRestriction(id,name,description);
				break;
			}
			default: {
				throw new SailInternalException("Unknown restriction type ["+type+"].");
			}
		} // switch

		restrictionsById.put(new Integer(id),restr);

		return restr;

	}// createRestriction(int,int,string,string)

	public Restriction getRestriction(int id) {
		return (Restriction) restrictionsById.get(new Integer(id));
	}

	/**
	 * Exports the Security Policy to a Sail. To be used with an In-Memory sail
	 * for example (org.openrdf.sesame.sail.memory.RdfRepository).
	 * 
	 * @param repos the RdfRepository to export to
	 */
	public void exportPolicy(RdfRepository repos){
		// definitions of resources used to describe the policy
		URI proId = new URIImpl(BASE_KCS_URI+"id");
		URI proName = new URIImpl(BASE_KCS_URI+"name");
		URI proDescr = new URIImpl(BASE_KCS_URI+"descritpion");
		URI proSuperRole = new URIImpl(BASE_KCS_URI+"superRole");
		URI proIncludeRule = new URIImpl(BASE_KCS_URI+"includeRule");
		URI proRuleRestriction = new URIImpl(BASE_KCS_URI+"ruleRestriction");
		URI proRightsGranted = new URIImpl(BASE_KCS_URI+"rightsGranted");
		URI proRestrictionType = new URIImpl(BASE_KCS_URI+"restrictionType");
		URI proIncludeProperty = new URIImpl(BASE_KCS_URI+"includeProperty");

		URI proSubjectRestr = new URIImpl(BASE_KCS_URI+"subjectRestr");
		URI proPredicateRestr = new URIImpl(BASE_KCS_URI+"predicateRestr");
		URI proObjectRestr = new URIImpl(BASE_KCS_URI+"objectRestr");

		URI proRestrOnQuery = new URIImpl(BASE_KCS_URI+"restrOnQuery");

		URI proIncludeResource = new URIImpl(BASE_KCS_URI+"includeResource");

		URI proRdfType = new URIImpl(RDF.TYPE);

		URI proHasRole = new URIImpl(BASE_KCS_URI+"hasRole");

		URI proHasRule = new URIImpl(BASE_KCS_URI+"hasRule");


		URI resRole = new URIImpl(BASE_KCS_URI+"Role");

		URI resRule = new URIImpl(BASE_KCS_URI+"SecurityRule");

		URI resRepositRestr = new URIImpl(BASE_KCS_URI+"RepositoryRestriction");

		URI resSchemaRestr = new URIImpl(BASE_KCS_URI+"SchemaRestriction");

		URI resClassesRestr = new URIImpl(BASE_KCS_URI+"ClassesRestriction");

		URI resInstancesRestr = new URIImpl(BASE_KCS_URI+"InstancesRestriction");

		URI resClassesOverSchemaRestr = new URIImpl(BASE_KCS_URI+"ClassesOverSchemaRestriction");

		URI resPaternRestr = new URIImpl(BASE_KCS_URI+"PatternRestriction");

		URI resPropertiesRestr = new URIImpl(BASE_KCS_URI+"PropertiesRestriction");

		URI resQueryRestr = new URIImpl(BASE_KCS_URI+"QueryRestriction");


		URI resUser = new URIImpl(BASE_KCS_URI+"User");

		URI resRestriction = new URIImpl(BASE_KCS_URI+"Restriction");


		repos.startTransaction();
		try {
			repos.clearRepository();
		}
		catch (SailUpdateException e) {
			throw new RuntimeException(e);
		}
		repos.commitTransaction();

		ArrayList userz = new ArrayList(users);
		ArrayList queList = new ArrayList();
		ArrayList rolz = new ArrayList(roles);
		Set rulez = new HashSet(RoleImpl.getRules(roles,false));

		try {
			for ( int ui = 0; ui<userz.size() ; ui++) {
				User usr = (User) userz.get(ui);
				rulez.addAll(usr.getRules());
				// current user
				URI resCUser = new URIImpl(usr.getUri());
				repos.addStatement(resCUser,proRdfType,resUser);
				repos.addStatement(resCUser,proId,new LiteralImpl(Integer.toString(usr.getId())));
				ArrayList userRulz = new ArrayList(usr.getRules());
				for ( int i = 0 ; i < userRulz.size() ; i++ ) {
					repos.addStatement(resCUser,proHasRule,
					new URIImpl(((Rule)userRulz.get(i)).getUri()));
				}
				ArrayList userRolz = new ArrayList(usr.getRoles());
				for ( int i = 0 ; i < userRolz.size() ; i++ ) {
					repos.addStatement(resCUser,proHasRole,
					new URIImpl(((Role)userRolz.get(i)).getUri()));
				}

			} // for userz
		}
		catch(Exception x){
			throw new SailInternalException(x);
		}

		try {
			for ( int roi = 0 ; roi < rolz.size(); roi++) {
				Role role = (Role)rolz.get(roi);
				rulez.addAll(role.getRules(true));

				// current role
				URI resCRole = new URIImpl(role.getUri());
				repos.addStatement(resCRole,proRdfType,resRole);
				repos.addStatement(resCRole,proId,new LiteralImpl(Integer.toString(role.getId())));
				repos.addStatement(resCRole,proName,new LiteralImpl(role.getName()));
				repos.addStatement(resCRole,proDescr,new LiteralImpl(role.getDescription()));

				ArrayList parents = new ArrayList(role.getParentRoles());

				for (int i = 0 ; i < parents.size() ; i++ ) {
					Role pRole = (Role)parents.get(i);
					repos.addStatement(resCRole, proSuperRole,
														 new URIImpl(pRole.getUri()));
				} // for

				ArrayList rulzList = new ArrayList(role.getRules(true));

				for (int i = 0 ; i < rulzList.size() ; i++ ) {
					Rule rule = (Rule)rulzList.get(i);
					repos.addStatement(resCRole,proIncludeRule,
														 new URIImpl(rule.getUri()));
				} // for
			} // for rolz
		}
		catch(Exception x){
			throw new SailInternalException(x);
		}

		// rules
		try {
			ArrayList rulz = new ArrayList(rulez);
			for (int rui=0; rui<rulz.size(); rui++) {
				Rule rule = (Rule)rulz.get(rui);
				URI resCRule = new URIImpl(rule.getUri());
				repos.addStatement(resCRule,proRdfType,resRule);
				repos.addStatement(resCRule,proId,new LiteralImpl(Integer.toString(rule.getId())));
				repos.addStatement(resCRule,proName,new LiteralImpl(rule.getName()));
				repos.addStatement(resCRule,proRuleRestriction, new URIImpl(rule.getRestriction().getUri()));
				repos.addStatement(resCRule,proDescr,new LiteralImpl(rule.getDescription()));

				if (rule.getReadRight())
					repos.addStatement(resCRule,proRightsGranted,new LiteralImpl(Right.READ.toString()));
				if (rule.getRemoveRight())
					repos.addStatement(resCRule,proRightsGranted,new LiteralImpl(Right.REMOVE.toString()));
		 		if (rule.getAddRight())
					repos.addStatement(resCRule,proRightsGranted,new LiteralImpl(Right.ADD.toString()));
				if (rule.getAdminRight())
					repos.addStatement(resCRule,proRightsGranted,new LiteralImpl(Right.ADMIN.toString()));
				if (rule.getHistoryRight())
					repos.addStatement(resCRule,proRightsGranted,new LiteralImpl(Right.HISTORY.toString()));
			} // for rulz
		}
		catch(Exception x){
			throw new SailInternalException(x);
		}

		try {
			ArrayList restrz = new ArrayList(Restriction.getRestrictions());
			for (int ri = 0; ri < restrz.size(); ri++) {
				Restriction r = (Restriction)restrz.get(ri);
				// current restriction
				URI resCR = new URIImpl (r.getUri());
				repos.addStatement(resCR, proId,new LiteralImpl(Integer.toString(r.getId())));
				repos.addStatement(resCR, proName,new LiteralImpl(r.getName()));
				repos.addStatement(resCR, proDescr,new LiteralImpl(r.getDescription()));
				repos.addStatement(resCR, proRestrictionType,new LiteralImpl(Restriction.type2String(r.getType())));
				int type = r.getType();

				switch (type) {
					case 1 : repos.addStatement(resCR,proRdfType,resRepositRestr);break;
					case 2 : repos.addStatement(resCR,proRdfType,resSchemaRestr);break;
					case 3 : repos.addStatement(resCR,proRdfType,resClassesRestr);break;
					case 4 : repos.addStatement(resCR,proRdfType,resInstancesRestr);break;
					case 5 : repos.addStatement(resCR,proRdfType,resPropertiesRestr);break;
					case 6 : repos.addStatement(resCR,proRdfType,resPaternRestr);break;
					case 7 : repos.addStatement(resCR,proRdfType,resQueryRestr);break;
					case 8 : repos.addStatement(resCR,proRdfType,resClassesOverSchemaRestr);break;
				}

				switch (type) {
					case 5 : //properties restriction
						ArrayList props = new ArrayList(((PropertiesRestriction)r).getProperties());
						for ( int i = 0 ; i < props.size() ; i++ ) {
							repos.addStatement(resCR,proIncludeProperty,
							new URIImpl(((URI)props.get(i)).getURI())) ;
						}
						break;
					case 6 : // PATTERn restriction
						PatternRestriction pr = (PatternRestriction) r;
						ArrayList su = new ArrayList(pr.getSubjectRestrictions());
						for ( int i = 0 ; i < su.size(); i++ ) {
							repos.addStatement(resCR,proSubjectRestr,
							new URIImpl(
							((Restriction)su.get(i)).getUri())
							);
						}
						ArrayList pu = new ArrayList(pr.getPredicateRestrictions());
						for ( int i = 0 ; i < pu.size(); i++ ) {
							repos.addStatement(resCR, proPredicateRestr,
							new URIImpl(
							((Restriction)pu.get(i)).getUri()));
						}
						ArrayList ou = new ArrayList(pr.getObjectRestrictions());
						for ( int i = 0 ; i < ou.size(); i++ ) {
							repos.addStatement(resCR,proObjectRestr,
							new URIImpl(
							((Restriction)ou.get(i)).getUri()));
						}
						break;
					case 7 : //QUERY Restriction
						ArrayList qList = new ArrayList(((QueryRestriction)r).getQueries());
						for ( int i = 0 ; i < qList.size(); i++) {
							repos.addStatement(resCR,proRestrOnQuery,new LiteralImpl((String)qList.get(i)));
						}
						break;
				}

				if ( r instanceof ResourceRestriction ) {
					ArrayList rlist = new ArrayList(((ResourceRestriction)r).getResources());
					for ( int i = 0 ; i < rlist.size() ; i++ ) {
						repos.addStatement(resCR,proIncludeResource,
						new URIImpl(((URI)rlist.get(i)).getURI()));
					}
				}

			} // for restrictions
		}
		catch(Exception x){
			throw new SailInternalException(x);
		}
	} // exportPolicy()

/*
 * using Jena public void exportSecurityPolicy2RDF(Writer out) {
 * 
 * if (!isRepositoryAccessible(Right.ADMIN)) { throw new SailInternalException(
 * "The current user has no permissions to export the Security Setup."); }
 * com.hp.hpl.mesa.rdf.jena.model.Model model = new ModelMem(); try {
 * model.read(KCS_URI);
 * 
 * com.hp.hpl.mesa.rdf.jena.model.Property proId =
 * model.createProperty(BASE_KCS_URI+"id");
 * com.hp.hpl.mesa.rdf.jena.model.Property proName =
 * model.createProperty(BASE_KCS_URI+"name");
 * com.hp.hpl.mesa.rdf.jena.model.Property proDescr =
 * model.createProperty(BASE_KCS_URI+"descritpion");
 * com.hp.hpl.mesa.rdf.jena.model.Property proSuperRole =
 * model.createProperty(BASE_KCS_URI+"superRole");
 * com.hp.hpl.mesa.rdf.jena.model.Property proIncludeRule =
 * model.createProperty(BASE_KCS_URI+"includeRule");
 * com.hp.hpl.mesa.rdf.jena.model.Property proRuleRestriction =
 * model.createProperty(BASE_KCS_URI+"ruleRestriction");
 * com.hp.hpl.mesa.rdf.jena.model.Property proRightsGranted =
 * model.createProperty(BASE_KCS_URI+"rightsGranted");
 * com.hp.hpl.mesa.rdf.jena.model.Property proRestrictionType =
 * model.createProperty(BASE_KCS_URI+"restrictionType");
 * com.hp.hpl.mesa.rdf.jena.model.Property proIncludeProperty =
 * model.createProperty(BASE_KCS_URI+"includeProperty");
 * 
 * com.hp.hpl.mesa.rdf.jena.model.Property proSubjectRestr =
 * model.createProperty(BASE_KCS_URI+"subjectRestr");
 * com.hp.hpl.mesa.rdf.jena.model.Property proPredicateRestr =
 * model.createProperty(BASE_KCS_URI+"predicateRestr");
 * com.hp.hpl.mesa.rdf.jena.model.Property proObjectRestr =
 * model.createProperty(BASE_KCS_URI+"objectRestr");
 * 
 * com.hp.hpl.mesa.rdf.jena.model.Property proRestrOnQuery =
 * model.createProperty(BASE_KCS_URI+"restrOnQuery");
 * 
 * com.hp.hpl.mesa.rdf.jena.model.Property proIncludeResource =
 * model.createProperty(BASE_KCS_URI+"includeResource");
 * 
 * com.hp.hpl.mesa.rdf.jena.model.Property proRdfType =
 * model.createProperty(RDF.TYPE);
 * 
 * com.hp.hpl.mesa.rdf.jena.model.Property proHasRole =
 * model.createProperty(BASE_KCS_URI+"hasRole");
 * 
 * com.hp.hpl.mesa.rdf.jena.model.Property proHasRule =
 * model.createProperty(BASE_KCS_URI+"hasRule");
 * 
 * 
 * com.hp.hpl.mesa.rdf.jena.model.Resource resRole =
 * model.getResource(BASE_KCS_URI+"Role");
 * 
 * com.hp.hpl.mesa.rdf.jena.model.Resource resRule =
 * model.getResource(BASE_KCS_URI+"SecurityRule");
 * 
 * com.hp.hpl.mesa.rdf.jena.model.Resource resRepositRestr =
 * model.getResource(BASE_KCS_URI+"RepositoryRestriction");
 * 
 * com.hp.hpl.mesa.rdf.jena.model.Resource resSchemaRestr =
 * model.getResource(BASE_KCS_URI+"SchemaRestriction");
 * 
 * com.hp.hpl.mesa.rdf.jena.model.Resource resClassesRestr =
 * model.getResource(BASE_KCS_URI+"ClassesRestriction");
 * 
 * com.hp.hpl.mesa.rdf.jena.model.Resource resInstancesRestr =
 * model.getResource(BASE_KCS_URI+"InstancesRestriction");
 * 
 * com.hp.hpl.mesa.rdf.jena.model.Resource resClassesOverSchemaRestr =
 * model.getResource(BASE_KCS_URI+"ClassesOverSchemaRestriction");
 * 
 * com.hp.hpl.mesa.rdf.jena.model.Resource resPaternRestr =
 * model.getResource(BASE_KCS_URI+"PatternRestriction");
 * 
 * com.hp.hpl.mesa.rdf.jena.model.Resource resPropertiesRestr =
 * model.getResource(BASE_KCS_URI+"PropertiesRestriction");
 * 
 * com.hp.hpl.mesa.rdf.jena.model.Resource resQueryRestr =
 * model.getResource(BASE_KCS_URI+"QueryRestriction");
 * 
 * 
 * com.hp.hpl.mesa.rdf.jena.model.Resource resUser =
 * model.getResource(BASE_KCS_URI+"User");
 * 
 * com.hp.hpl.mesa.rdf.jena.model.Resource resRestriction =
 * model.getResource(BASE_KCS_URI+"Restriction");
 * 
 * 
 * 
 * ArrayList userz = new ArrayList(users); ArrayList queList = new ArrayList();
 * ArrayList rolz = new ArrayList(roles); Set rulez = new
 * HashSet(RoleImpl.getRules(roles,false));
 * 
 * for ( int ui = 0; ui <userz.size() ; ui++) { User usr = (User) userz.get(ui);
 * rulez.addAll(usr.getRules()); com.hp.hpl.mesa.rdf.jena.model.Resource res =
 * model.createResource(BASE_KCS_URI+"User_"+usr.getId());
 * res.addProperty(proRdfType,resUser); res.addProperty(proId,usr.getId());
 * ArrayList userRulz = new ArrayList(usr.getRules()); for ( int i = 0 ; i <
 * userRulz.size() ; i++ ) {
 * 
 * res.addProperty(proHasRule,
 * model.createResource(BASE_KCS_URI+"Rule_"+((Rule)userRulz.get(i)).getName()) ); }
 * ArrayList userRolz = new ArrayList(usr.getRoles()); for ( int i = 0 ; i <
 * userRolz.size() ; i++ ) { res.addProperty(proHasRole,
 * model.createResource(BASE_KCS_URI+"Role_"+((Role)userRolz.get(i)).getName())); }
 *  } // for userz
 * 
 * 
 * for ( int roi = 0 ; roi < rolz.size(); roi++) { Role role =
 * (Role)rolz.get(roi); rulez.addAll(role.getRules(true));
 * 
 * com.hp.hpl.mesa.rdf.jena.model.Resource res =
 * model.createResource(BASE_KCS_URI+"Role_"+role.getName())
 * .addProperty(proRdfType,resRole) .addProperty(proId,role.getId())
 * .addProperty(proName,role.getName())
 * .addProperty(proDescr,role.getDescription());
 * 
 * ArrayList parents = new ArrayList(role.getParentRoles());
 * 
 * for (int i = 0 ; i < parents.size() ; i++ ) { Role pRole =
 * (Role)parents.get(i); res.addProperty(proSuperRole,
 * model.createResource(BASE_KCS_URI+"Role_"+pRole.getName())); } // for
 * 
 * ArrayList rulzList = new ArrayList(role.getRules(true));
 * 
 * for (int i = 0 ; i < rulzList.size() ; i++ ) { Rule rule =
 * (Rule)rulzList.get(i); res.addProperty(proIncludeRule,
 * model.createResource(BASE_KCS_URI+"Rule_"+rule.getName())); } // for } // for
 * rolz
 * 
 * 
 * ArrayList rulz = new ArrayList(rulez); for (int rui=0; rui <rulz.size();
 * rui++) { Rule rule = (Rule)rulz.get(rui);
 * com.hp.hpl.mesa.rdf.jena.model.Resource res =
 * model.createResource(BASE_KCS_URI+"Rule_"+rule.getName());
 * res.addProperty(proRdfType,resRule) .addProperty(proId,rule.getId())
 * .addProperty(proName,rule.getName()) .addProperty(proRuleRestriction,
 * model.createResource(BASE_KCS_URI+"Restriction_"+rule.getRestriction().getName()));
 * res.addProperty(proDescr,rule.getDescription());
 * 
 * if (rule.getReadRight())
 * res.addProperty(proRightsGranted,Right.READ.toString()); if
 * (rule.getRemoveRight())
 * res.addProperty(proRightsGranted,Right.REMOVE.toString()); if
 * (rule.getAddRight()) res.addProperty(proRightsGranted,Right.ADD.toString());
 * if (rule.getAdminRight())
 * res.addProperty(proRightsGranted,Right.ADMIN.toString()); if
 * (rule.getHistoryRight())
 * res.addProperty(proRightsGranted,Right.HISTORY.toString()); } // for rulz
 * 
 * ArrayList restrz = new ArrayList(Restriction.getRestrictions()); for (int ri =
 * 0; ri < restrz.size(); ri++) { Restriction r = (Restriction)restrz.get(ri);
 * com.hp.hpl.mesa.rdf.jena.model.Resource res =
 * model.createResource(BASE_KCS_URI+"Restriction_"+r.getName());
 * 
 * res.addProperty(proId,r.getId()); res.addProperty(proName,r.getName());
 * res.addProperty(proDescr,r.getDescription());
 * res.addProperty(proRestrictionType,Restriction.type2String(r.getType())); int
 * type = r.getType();
 * 
 * switch (type) { case 1 : res.addProperty(proRdfType,resRepositRestr);break;
 * case 2 : res.addProperty(proRdfType,resSchemaRestr);break; case 3 :
 * res.addProperty(proRdfType,resClassesRestr);break; case 4 :
 * res.addProperty(proRdfType,resInstancesRestr);break; case 5 :
 * res.addProperty(proRdfType,resPropertiesRestr);break; case 6 :
 * res.addProperty(proRdfType,resPaternRestr);break; case 7 :
 * res.addProperty(proRdfType,resQueryRestr);break; case 8 :
 * res.addProperty(proRdfType,resClassesOverSchemaRestr);break; }
 * 
 * switch (type) { case 5 : //properties restriction ArrayList props = new
 * ArrayList(((PropertiesRestriction)r).getProperties()); for ( int i = 0 ; i <
 * props.size() ; i++ ) {
 * res.addProperty(proIncludeProperty,model.createResource(
 * ((Resource)props.get(i)).getURI())) ; } break; case 6 : // PATTERn
 * restriction PatternRestriction pr = (PatternRestriction) r; ArrayList su =
 * new ArrayList(pr.getSubjectRestrictions()); for ( int i = 0 ; i < su.size();
 * i++ ) { res.addProperty(proSubjectRestr, model.createResource(
 * BASE_KCS_URI+"Restriction_"+((Restriction)su.get(i)).getName()) ); }
 * ArrayList pu = new ArrayList(pr.getPredicateRestrictions()); for ( int i = 0 ;
 * i < pu.size(); i++ ) { res.addProperty(proPredicateRestr,
 * model.createResource(
 * BASE_KCS_URI+"Restriction_"+((Restriction)pu.get(i)).getName())); } ArrayList
 * ou = new ArrayList(pr.getObjectRestrictions()); for ( int i = 0 ; i <
 * ou.size(); i++ ) { res.addProperty(proObjectRestr, model.createResource(
 * BASE_KCS_URI+"Restriction_"+((Restriction)ou.get(i)).getName())); } break;
 * case 7 : //QUERY Restriction ArrayList qList = new
 * ArrayList(((QueryRestriction)r).getQueries()); for ( int i = 0 ; i <
 * qList.size(); i++) { res.addProperty(proRestrOnQuery,(String)qList.get(i)); }
 * break; }
 * 
 * if ( r instanceof ResourceRestriction ) { ArrayList rlist = new
 * ArrayList(((ResourceRestriction)r).getResources()); for ( int i = 0 ; i <
 * rlist.size() ; i++ ) {
 * res.addProperty(proIncludeResource,model.createResource(((Resource)rlist.get(i)).getURI())); } }
 *  } // for restrictions
 * 
 * model.write(new PrintWriter(out), "RDF/XML-ABBREV");
 * 
 *  } catch (com.hp.hpl.mesa.rdf.jena.model.RDFException ex) { throw new
 * SailInternalException(ex); }
 *  // add the KCS rdf to the model // export all the setup to triples // export
 * this part of the repository to a file. }
 */

	/**
	 * Imports the security policy given a Graph. 
	 * 
	 * @param graph the Graph to read the security policy from
	 */
	public void importPolicy(Graph graph) {
		
		ValueFactory factory = graph.getValueFactory();
		
		URI proId = factory.createURI(BASE_KCS_URI, "id");
		URI proName = factory.createURI(BASE_KCS_URI, "name");
		URI proDescr = factory.createURI(BASE_KCS_URI, "descritpion");
		URI proSuperRole = factory.createURI(BASE_KCS_URI, "superRole");
		URI proIncludeRule = factory.createURI(BASE_KCS_URI, "includeRule");
		URI proRuleRestriction = factory.createURI(BASE_KCS_URI, "ruleRestriction");
		URI proRightsGranted = factory.createURI(BASE_KCS_URI, "rightsGranted");
		URI proRestrictionType = factory.createURI(BASE_KCS_URI, "restrictionType");
		URI proIncludeProperty = factory.createURI(BASE_KCS_URI, "includeProperty");

		URI proSubjectRestr = factory.createURI(BASE_KCS_URI, "subjectRestr");
		URI proPredicateRestr = factory.createURI(BASE_KCS_URI, "predicateRestr");
		URI proObjectRestr = factory.createURI(BASE_KCS_URI, "objectRestr");

		URI proRestrOnQuery = factory.createURI(BASE_KCS_URI, "restrOnQuery");

		URI proIncludeResource = factory.createURI(BASE_KCS_URI, "includeResource");

		URI resUser = factory.createURI(BASE_KCS_URI, "User");

		URI resRole = factory.createURI(BASE_KCS_URI, "Role");

		URI resRule = factory.createURI(BASE_KCS_URI, "SecurityRule");

		URI resRepositoryRestriction = factory.createURI(BASE_KCS_URI, "RepositoryRestriction");

		URI resSchemaRestrction = factory.createURI(BASE_KCS_URI, "SchemaRestrction");

		URI resClassesRestriction = factory.createURI(BASE_KCS_URI, "ClassesRestriction");

		URI resClassesOverSchemaRestriction = factory.createURI(BASE_KCS_URI, "ClassesOverSchemaRestriction");

		URI resInstanceRestriction = factory.createURI(BASE_KCS_URI, "InstancesRestriction");

		URI resPropertiesRestriction = factory.createURI(BASE_KCS_URI, "PropertiesRestriction");

		URI resPatternRestriction = factory.createURI(BASE_KCS_URI, "PatternRestriction");

		URI resQueryRestriction = factory.createURI(BASE_KCS_URI, "QueryRestriction");


		URI resRdfType = factory.createURI(RDF.TYPE);
		URI resRdfResource = factory.createURI(RDFS.RESOURCE);

		URI proHasRole = factory.createURI(BASE_KCS_URI, "hasRole");

		URI proHasRule = factory.createURI(BASE_KCS_URI, "hasRule");


		//contains uris vs role mapping
		Map uriVsRole = new HashMap();

		//uris vs rule mapping
		Map uriVsRule = new HashMap();

		//uris vs restriction mapping
		Map uriVsRestr = new HashMap();

		ArrayList usersList = new ArrayList();


		// iterate users
		try {
			StatementIterator sti = graph.getStatements(null,resRdfType,resUser);
			while ( sti.hasNext() ) {
				Resource resCUser = sti.next().getSubject();
				int id = Integer.parseInt(
						graph.getStatements(resCUser,proId,null).next().getObject().toString());
				User user = new UserImpl(id,"","","");
				// FIXME cast to URI appropriate?
				user.setUri(((URI)resCUser).getURI());

				//get rules
				StatementIterator rui = graph.getStatements(resCUser,proHasRule,null);
				Set rulzSet = new HashSet();
				while (rui.hasNext()){
					// FIXME cast to URI appropriate?
					URI rul = (URI)rui.next().getObject();
					Rule rule = (Rule) uriVsRule.get(rul.getURI());
					if ( null == rule ) {
						rule = new RuleImpl(
								Integer.parseInt(graph.getStatements(rul,proId,null)
								.next().getObject().toString())
								,"","");
						uriVsRule.put(rul.getURI(),rule);
						rule.setUri(rul.getURI());
					}
					rulzSet.add(rule);
				}
				user.setRules(rulzSet );

				StatementIterator roi
				= graph.getStatements(resCUser,proHasRole,null);
				Set rolzSet = new HashSet();
				while (roi.hasNext()){
					// FIXME cast to URI appropriate?
					URI rol = (URI)roi.next().getObject();
					Role role = (Role) uriVsRule.get(rol.getURI());
					if ( null == role ) {
						role = new RoleImpl(
								Integer.parseInt(graph.getStatements(rol,proId,null).next().
								getObject().toString())
								,"","");
						role.setUri(rol.getURI());
						uriVsRole.put(rol.getURI(),role);
					}
					rolzSet.add(role);
				}
				user.setRoles(rolzSet );
				usersList.add(user);
				usersById.put(new Integer(user.getId()),user);
				users.add(user);
			} // user
		} catch (Exception x) {
			throw new SailInternalException(x);
		}

		// iterate roles
		try {
			StatementIterator sti = graph.getStatements(null,resRdfType,resRole);
			while ( sti.hasNext() ) {
				Resource resCRole = sti.next().getSubject();
				int id = Integer.parseInt(
						graph.getStatements(resCRole,proId,null).next().getObject().toString());
				String name = graph.getStatements(resCRole,proName,null).next().getObject().toString();
				String descr = graph.getStatements(resCRole,proDescr,null).next().getObject().toString();

				StatementIterator parenti =
						graph.getStatements(resCRole,proSuperRole,null);

				//check the existence of the role
				// FIXME cast to URI appropriate?
				Role role = (Role)uriVsRole.get(((URI)resCRole).getURI());
				if (null==role) {
					//create the new role
					role = new RoleImpl(id,name,descr);
				} else {
					role.setId(id);
					role.setName(name);
					role.setDescription(descr);
				}
				// FIXME cast to URI appropriate?
				role.setUri(((URI)resCRole).getURI());


				// store in the map for further reference by uri
				// FIXME cast to URI appropriate?
				uriVsRole.put(((URI)resCRole).getURI(),role);

				Set prnts = new HashSet();
				while (parenti.hasNext()) {
				// FIXME cast to URI appropriate?
					URI parent =
							(URI)parenti.next().getObject();
					Role mappedRole = (Role) uriVsRole.get(parent.getURI());
					if (mappedRole == null) {
						mappedRole = new RoleImpl();
				// FIXME cast to URI appropriate?
						uriVsRole.put(((URI)resCRole).getURI(),mappedRole);
						mappedRole.setUri(((URI)resCRole).getURI());
					}
					prnts.add(mappedRole);
				}

				role.setParentRoles(prnts);

			} // ROLES
		} catch (Exception x) {
			throw new SailInternalException(x);
		}

		// rules
		try {
			StatementIterator sti = graph.getStatements(null,resRdfType,resRule);
			while (sti.hasNext()) {
				Resource resCRule = sti.next().getSubject();
			int id = Integer.parseInt(
					graph.getStatements(resCRule,proId,null).next().getObject().toString()
				);

			String name = graph.getStatements(resCRule,proName,null).next().getObject().toString();
			String descr = graph.getStatements(resCRule,proDescr,null).next().getObject().toString();

			// FIXME cast to URI appropriate?
			URI restrResource = (URI)
					graph.getStatements(resCRule,proRuleRestriction,null).next().getObject();
			String restrUri = restrResource.getURI();

			Restriction restr = (Restriction)uriVsRestr.get(restrUri);
			if ( null == restr) {
				String restrType =
						graph.getStatements(restrResource,proRestrictionType,null).next()
						.getObject().toString();
				int restrId = Integer.parseInt(
						graph.getStatements(restrResource,proId,null).next().getObject().toString()
					);
				restr = Restriction.createRestriction(Restriction.type2Int(restrType),
						restrId);
				uriVsRestr.put(restrUri,restr);
			}
			restr.setUri(restrResource.getURI());

			// FIXME cast to URI appropriate?
			Rule rule = (Rule)uriVsRule.get(((URI)resCRule).getURI());
			if ( null == rule ) {
				rule = new RuleImpl(id,name,descr);
			// FIXME cast to URI appropriate?
				uriVsRule.put(((URI)resCRule).getURI(),rule);
			}
			rule.setId(id);
			rule.setName(name);
			rule.setDescription(descr);

			rule.setRestriction(restr);
			// FIXME cast to URI appropriate?
			rule.setUri(((URI)resCRule).getURI());

			ArrayList rights = new ArrayList();
			StatementIterator ri =
				graph.getStatements(resCRule,proRightsGranted,null);
			while (ri.hasNext()) {
				String right = ri.next().getObject().toString();

				if (right.equals(Right.ADD.toString())) {
					rule.setAddRight(true);
				}
				else if (right.equals(Right.ADMIN.toString())) {
					rule.setAdminRight(true);
				}
				else if (right.equals(Right.HISTORY.toString())) {
					rule.setHistoryRight(true);
				}
				else if (right.equals(Right.READ.toString())) {
					rule.setReadRight(true);
				}
				else if (right.equals(Right.REMOVE.toString())) {
					rule.setRemoveRight(true);
				}
			}
		} // RULE
		}catch(Exception x) {
			throw new SailInternalException(x);
		}

		///// Restrictions

		// Instance
		try {
			StatementIterator sti = graph.getStatements(null,resRdfType,resInstanceRestriction);
			while (sti.hasNext()) {
				Resource resCIR = sti.next().getSubject();
				int id = Integer.parseInt(
						graph.getStatements(resCIR,proId,null).next().getObject().toString()
						);
				String name = graph.getStatements(resCIR,proName,null).next().getObject().toString();
				String descr = graph.getStatements(resCIR,proDescr,null).next().getObject().toString();

			// FIXME cast to URI appropriate?
				Restriction restr = (Restriction)uriVsRestr.get(((URI)resCIR).getURI());
				if (null== restr) {
					restr = Restriction.createRestriction(Restriction.INSTANCES,id);
				}
			// FIXME cast to URI appropriate?
				restr.setUri(((URI)resCIR).getURI());
				restr.setName(name);
				restr.setDescription(descr);
			// FIXME cast to URI appropriate?
				uriVsRestr.put(((URI)resCIR).getURI(),restr);
				StatementIterator clses =
						graph.getStatements(resCIR,proIncludeResource,null);
				Set resSet = new HashSet();
				while (clses.hasNext()) {
					Resource cls = (Resource)clses.next().getObject();
					resSet.add(cls);
					baseRdfRepository.addStatement(cls,resRdfType,resRdfResource);
				}
				((ResourceRestriction)restr).setResources(resSet);
			}
		} catch (Exception x) {
			throw new SailInternalException(x);
		}

		// Properies
		try {
			StatementIterator sti = graph.getStatements(null,resRdfType,resPropertiesRestriction);
			while (sti.hasNext()) {
				Resource resCPR = sti.next().getSubject();
				int id = Integer.parseInt(
						graph.getStatements(resCPR,proId,null).next().getObject().toString()
						);
				String name = graph.getStatements(resCPR,proName,null).next().getObject().toString();
				String descr = graph.getStatements(resCPR,proDescr,null).next().getObject().toString();

			// FIXME cast to URI appropriate?
				Restriction restr = (Restriction)uriVsRestr.get(((URI)resCPR).getURI());
				if (null== restr) {
					restr = Restriction.createRestriction(Restriction.PROPERTIES,id);
				}
			// FIXME cast to URI appropriate?
				restr.setUri(((URI)resCPR).getURI());
				restr.setName(name);
				restr.setDescription(descr);
			// FIXME cast to URI appropriate?
				uriVsRestr.put(((URI)resCPR).getURI(),restr);

				StatementIterator pies =
					graph.getStatements(resCPR,proIncludeProperty,null);
				Set propList = new HashSet();
				while (pies.hasNext()) {
					Resource py = (Resource) pies.next().getObject();
					baseRdfRepository.addStatement(py,resRdfType,resRdfResource);
					propList.add(py);
				}
				((PropertiesRestriction)restr).setProperties(propList);

			}
		} catch (Exception x) {
			throw new SailInternalException(x);
		}

		//Classes
		try {
			StatementIterator sti = graph.getStatements(null,resRdfType,resClassesRestriction);
			while (sti.hasNext()) {
				Resource resCIR = sti.next().getSubject();
				int id = Integer.parseInt(
						graph.getStatements(resCIR,proId,null).next().getObject().toString()
						);
				String name = graph.getStatements(resCIR,proName,null).next().getObject().toString();
				String descr = graph.getStatements(resCIR,proDescr,null).next().getObject().toString();

			// FIXME cast to URI appropriate?
				Restriction restr = (Restriction)uriVsRestr.get(((URI)resCIR).getURI());
				if (null== restr) {
					restr = Restriction.createRestriction(Restriction.CLASSES,id);
				}
			// FIXME cast to URI appropriate?
				restr.setUri(((URI)resCIR).getURI());
				restr.setName(name);
				restr.setDescription(descr);
			// FIXME cast to URI appropriate?
				uriVsRestr.put(((URI)resCIR).getURI(),restr);
				StatementIterator clses =
						graph.getStatements(resCIR,proIncludeResource,null);
				Set resSet = new HashSet();
				while (clses.hasNext()) {
					Resource cls = (Resource)clses.next().getObject();
					resSet.add(cls);
					baseRdfRepository.addStatement(cls,resRdfType,resRdfResource);
				}
				((ResourceRestriction)restr).setResources(resSet);
			}
		} catch (Exception x) {
			throw new SailInternalException(x);
		}

		//Classes over schema
		try {
			StatementIterator sti = graph.getStatements(null,resRdfType,resClassesOverSchemaRestriction);
			while (sti.hasNext()) {
				Resource resCIR = sti.next().getSubject();
				int id = Integer.parseInt(
						graph.getStatements(resCIR,proId,null).next().getObject().toString()
						);
				String name = graph.getStatements(resCIR,proName,null).next().getObject().toString();
				String descr = graph.getStatements(resCIR,proDescr,null).next().getObject().toString();

			// FIXME cast to URI appropriate?
				Restriction restr = (Restriction)uriVsRestr.get(((URI)resCIR).getURI());
				if (null== restr) {
					restr = Restriction.createRestriction(Restriction.CLASSES_OVER_SCHEMA,id);
				}
			// FIXME cast to URI appropriate?
				restr.setUri(((URI)resCIR).getURI());
				restr.setName(name);
				restr.setDescription(descr);
			// FIXME cast to URI appropriate?
				uriVsRestr.put(((URI)resCIR).getURI(),restr);
				StatementIterator clses =
						graph.getStatements(resCIR,proIncludeResource,null);
				Set resSet = new HashSet();
				while (clses.hasNext()) {
					Resource cls = (Resource)clses.next().getObject();
					resSet.add(cls);
					baseRdfRepository.addStatement(cls,resRdfType,resRdfResource);
				}
				((ResourceRestriction)restr).setResources(resSet);
			}
		} catch (Exception x) {
			throw new SailInternalException(x);
		}

		// pattern
		try{
			StatementIterator sti = graph.getStatements(null,resRdfType,resPatternRestriction);
			while (sti.hasNext()) {
				Resource resCPR = sti.next().getSubject();
				int id = Integer.parseInt(
						graph.getStatements(resCPR,proId,null).next().getObject().toString()
						);
				String name = graph.getStatements(resCPR,proName,null).next().getObject().toString();
				String descr = graph.getStatements(resCPR,proDescr,null).next().getObject().toString();

			// FIXME cast to URI appropriate?
				Restriction restr = (Restriction)uriVsRestr.get(((URI)resCPR).getURI());
				if (null== restr) {
					restr = Restriction.createRestriction(Restriction.PATTERN,id);
				}
			// FIXME cast to URI appropriate?
				restr.setUri(((URI)resCPR).getURI());
				restr.setName(name);
				restr.setDescription(descr);
			// FIXME cast to URI appropriate?
				uriVsRestr.put(((URI)resCPR).getURI(),restr);

				// subject
				StatementIterator subjI =
						graph.getStatements(resCPR,proSubjectRestr,null);
				while (subjI.hasNext()) {

					Resource cls = (Resource)	subjI.next().getObject();
					baseRdfRepository.addStatement(cls,resRdfType,resRdfResource);

					StatementIterator restrTypesI = graph.getStatements(cls,resRdfType,null);
					String rType = "";
					while (restrTypesI.hasNext()){
						rType = restrTypesI.next().getObject().toString();
						if (rType.indexOf("Restriction")!=-1) break;
					}

			// FIXME cast to URI appropriate?
					Restriction sRestr = (Restriction)uriVsRestr.get(((URI)cls).getURI());
					if (sRestr==null) {
						rType = graph.getStatements(cls,proRestrictionType,null).next().getObject().toString();
						int sRestrId = Integer.parseInt(
								graph.getStatements(cls,proId,null).next().getObject().toString()
								);
						sRestr = Restriction.createRestriction(Restriction.type2Int(rType),sRestrId );
			// FIXME cast to URI appropriate?
						uriVsRestr.put(((URI)cls).getURI(),sRestr);
					}
			// FIXME cast to URI appropriate?
					sRestr.setUri(((URI)cls).getURI());
					((PatternRestriction) restr).addSubjectRestriction((ResourceRestriction)sRestr);
				} // subject

				// predicate restrs
				StatementIterator predI = graph.getStatements(resCPR,proPredicateRestr,null);
				while (predI.hasNext()) {
					Resource cls = (Resource)predI.next().getObject();
					baseRdfRepository.addStatement(cls,resRdfType,resRdfResource);

					StatementIterator restrTypesI =
							graph.getStatements(cls,resRdfType,null);
					String rType = "";
					while (restrTypesI.hasNext()){
						rType = restrTypesI.next().getObject().toString();
						if (rType.indexOf("Restriction")!=-1) break;
					}

			// FIXME cast to URI appropriate?
					Restriction sRestr = (Restriction)uriVsRestr.get(((URI)cls).getURI());
					if (sRestr==null) {
						rType = graph.getStatements(cls,proRestrictionType,null).next().getObject().toString();
						int sRestrId = Integer.parseInt(
								graph.getStatements(cls,proId,null).next().getObject().toString()
								);
						sRestr = Restriction.createRestriction(Restriction.type2Int(rType),sRestrId);
			// FIXME cast to URI appropriate?
						uriVsRestr.put(((URI)cls).getURI(),sRestr);
					}
			// FIXME cast to URI appropriate?
					sRestr.setUri(((URI)cls).getURI());
					((PatternRestriction) restr).addPredicateRestriction((PropertiesRestriction)sRestr);
				} // predicate


				// object restrictions
				StatementIterator objI =
						graph.getStatements(resCPR,proObjectRestr,null);
				Set objRestrSet = new HashSet();
				while (objI.hasNext()) {

					Resource cls = (Resource)objI.next().getObject();
					baseRdfRepository.addStatement(cls,resRdfType,resRdfResource);

					objRestrSet.add(cls);

					StatementIterator restrTypesI =
							graph.getStatements(cls,resRdfType,null);
					String rType = "";
					while (restrTypesI.hasNext()){
						rType = restrTypesI.next().toString();
						if (rType.indexOf("Restriction")!=-1) break;
					}

			// FIXME cast to URI appropriate?
					Restriction sRestr = (Restriction)uriVsRestr.get(((URI)cls).getURI());
					if (sRestr==null) {
						rType = graph.getStatements(cls,proRestrictionType,null).next().getObject().toString();
						int sRestrId = Integer.parseInt(
								graph.getStatements(cls,proId,null).next().getObject().toString()
								);

						sRestr = Restriction.createRestriction(Restriction.type2Int(rType),sRestrId);
			// FIXME cast to URI appropriate?
						uriVsRestr.put(((URI)cls).getURI(),sRestr);
					}
			// FIXME cast to URI appropriate?
					sRestr.setUri(((URI)cls).getURI());
					((PatternRestriction) restr).addObjectRestriction((ResourceRestriction)sRestr);
				}

			} // while pattern restrictions
		} catch (Exception x) {
			throw new SailInternalException(x);
		}

		//query
		try {
			StatementIterator sti = graph.getStatements(null,resRdfType,resQueryRestriction);
			while (sti.hasNext()) {
				Resource resCQR = sti.next().getSubject();
				int id = Integer.parseInt(
						graph.getStatements(resCQR,proId,null).next().getObject().toString()
						);
				String name = graph.getStatements(resCQR,proName,null).next().getObject().toString();
				String descr = graph.getStatements(resCQR,proDescr,null).next().getObject().toString();

			// FIXME cast to URI appropriate?
				Restriction restr = (Restriction)uriVsRestr.get(((URI)resCQR).getURI());
				if (null== restr) {
					restr = Restriction.createRestriction(Restriction.QUERY,id);
				}
			// FIXME cast to URI appropriate?
				restr.setUri(((URI)resCQR).getURI());
				restr.setName(name);
				restr.setDescription(descr);
			// FIXME cast to URI appropriate?
				uriVsRestr.put(((URI)resCQR).getURI(),restr);

				StatementIterator queryI =
					graph.getStatements(resCQR,proRestrOnQuery,null);
				while (queryI.hasNext()) {
					String q = queryI.next().getObject().toString();
					((QueryRestriction)restr).addQuery(q);
				}

			}

		} catch (Exception x) {
			throw new SailInternalException(x);
		}//query

		// refresh the security maps
		rulesById = new HashMap();
		ArrayList vlist = new ArrayList(uriVsRule.values());
		for ( int i = 0 ; i < vlist.size(); i++ ) {
			Rule r = (Rule) vlist.get(i);
			rulesById.put(new Integer(r.getId()),r);
		}

		restrictionsById = new HashMap();
		vlist = new ArrayList(uriVsRestr.values());
		for ( int i = 0 ; i < vlist.size(); i++ ) {
			Restriction r = (Restriction) vlist.get(i);
			restrictionsById.put(new Integer(r.getId()),r);
		}

		rolesByName = new HashMap();
		rolesById =	new HashMap();
		roles = new HashSet();
		vlist = new ArrayList(uriVsRole.values());
		for ( int i = 0 ; i < vlist.size(); i++ ) {
			Role r = (Role) vlist.get(i);
			rolesById.put(new Integer(r.getId()),r);
			rolesByName.put(r.getName(),r);
			roles.add(r);
		}

	} // importPolicy()


/*
 * using Jena public void importSecurityPolicy(Reader in) {
 * com.hp.hpl.mesa.rdf.jena.model.Model model = new ModelMem();
 * 
 * try { com.hp.hpl.mesa.rdf.jena.model.Property proId =
 * model.createProperty(BASE_KCS_URI+"id");
 * com.hp.hpl.mesa.rdf.jena.model.Property proName =
 * model.createProperty(BASE_KCS_URI+"name");
 * com.hp.hpl.mesa.rdf.jena.model.Property proDescr =
 * model.createProperty(BASE_KCS_URI+"descritpion");
 * com.hp.hpl.mesa.rdf.jena.model.Property proSuperRole =
 * model.createProperty(BASE_KCS_URI+"superRole");
 * com.hp.hpl.mesa.rdf.jena.model.Property proIncludeRule =
 * model.createProperty(BASE_KCS_URI+"includeRule");
 * com.hp.hpl.mesa.rdf.jena.model.Property proRuleRestriction =
 * model.createProperty(BASE_KCS_URI+"ruleRestriction");
 * com.hp.hpl.mesa.rdf.jena.model.Property proRightsGranted =
 * model.createProperty(BASE_KCS_URI+"rightsGranted");
 * com.hp.hpl.mesa.rdf.jena.model.Property proRestrictionType =
 * model.createProperty(BASE_KCS_URI+"restrictionType");
 * com.hp.hpl.mesa.rdf.jena.model.Property proIncludeProperty =
 * model.createProperty(BASE_KCS_URI+"includeProperty");
 * 
 * com.hp.hpl.mesa.rdf.jena.model.Property proSubjectRestr =
 * model.createProperty(BASE_KCS_URI+"subjectRestr");
 * com.hp.hpl.mesa.rdf.jena.model.Property proPredicateRestr =
 * model.createProperty(BASE_KCS_URI+"predicateRestr");
 * com.hp.hpl.mesa.rdf.jena.model.Property proObjectRestr =
 * model.createProperty(BASE_KCS_URI+"objectRestr");
 * 
 * com.hp.hpl.mesa.rdf.jena.model.Property proRestrOnQuery =
 * model.createProperty(BASE_KCS_URI+"restrOnQuery");
 * 
 * com.hp.hpl.mesa.rdf.jena.model.Property proIncludeResource =
 * model.createProperty(BASE_KCS_URI+"includeResource");
 * 
 * com.hp.hpl.mesa.rdf.jena.model.Property proRdfType =
 * model.createProperty(RDF.TYPE); // com.hp.hpl.mesa.rdf.jena.model.Resource
 * resRdfResource = model.createResource(RDFS.RESOURCE);
 * 
 * Resource resRdfType = new Resource(RDF.TYPE); Resource resRdfResource = new
 * Resource(RDFS.RESOURCE);
 * 
 * 
 * com.hp.hpl.mesa.rdf.jena.model.Property proHasRole =
 * model.createProperty(BASE_KCS_URI+"hasRole");
 * 
 * com.hp.hpl.mesa.rdf.jena.model.Property proHasRule =
 * model.createProperty(BASE_KCS_URI+"hasRule");
 * 
 * 
 * //contains uris vs role mapping Map uriVsRole = new HashMap();
 * 
 * //uris vs rule mapping Map uriVsRule = new HashMap();
 * 
 * //uris vs restriction mapping Map uriVsRestr = new HashMap();
 * 
 * ArrayList usersList = new ArrayList();
 * 
 * 
 * model.read(in,KCS_URI); com.hp.hpl.mesa.rdf.jena.model.ResIterator resi =
 * model.listSubjectsWithProperty(proRdfType); while ( resi.hasNext() ) {
 * com.hp.hpl.mesa.rdf.jena.model.Resource subj; subj =
 * (com.hp.hpl.mesa.rdf.jena.model.Resource)resi.next();
 * com.hp.hpl.mesa.rdf.jena.model.NodeIterator ni =
 * model.listObjectsOfProperty(subj,proRdfType); while ( ni.hasNext() ) {
 * com.hp.hpl.mesa.rdf.jena.model.RDFNode node =
 * (com.hp.hpl.mesa.rdf.jena.model.RDFNode) ni.next(); if ( node instanceof
 * com.hp.hpl.mesa.rdf.jena.model.Resource ) { String type = ((
 * com.hp.hpl.mesa.rdf.jena.model.Resource) node) .getLocalName();
 * 
 * 
 * if (type.equals("User")) { int id =
 * Integer.parseInt(model.listObjectsOfProperty(subj,proId).next().toString());
 * User user = new UserImpl(id,"","","");
 * com.hp.hpl.mesa.rdf.jena.model.NodeIterator rui =
 * model.listObjectsOfProperty(subj,proHasRule); Set rulzSet = new HashSet();
 * while (rui.hasNext()){ com.hp.hpl.mesa.rdf.jena.model.Resource rul =
 * (com.hp.hpl.mesa.rdf.jena.model.Resource)rui.next(); Rule rule = (Rule)
 * uriVsRule.get(rul.getURI()); if ( null == rule ) { rule = new RuleImpl(
 * Integer.parseInt(model.listObjectsOfProperty(rul,proId).next().toString())
 * ,"",""); uriVsRule.put(rul.getURI(),rule); } rulzSet.add(rule); }
 * user.setRules(rulzSet );
 * 
 * com.hp.hpl.mesa.rdf.jena.model.NodeIterator roi =
 * model.listObjectsOfProperty(subj,proHasRole); Set rolzSet = new HashSet();
 * while (roi.hasNext()){ com.hp.hpl.mesa.rdf.jena.model.Resource rol =
 * (com.hp.hpl.mesa.rdf.jena.model.Resource)roi.next(); Role role = (Role)
 * uriVsRule.get(rol.getURI()); if ( null == role ) { role = new RoleImpl(
 * Integer.parseInt(model.listObjectsOfProperty(rol,proId).next().toString())
 * ,"",""); uriVsRole.put(rol.getURI(),role); } rolzSet.add(role); }
 * user.setRoles(rolzSet ); usersList.add(user); usersById.put(new
 * Integer(user.getId()),user); users.add(user); } // user
 * 
 * 
 * 
 * if (type.equals("Role")) {
 * 
 * int id =
 * Integer.parseInt(model.listObjectsOfProperty(subj,proId).next().toString());
 * String name = model.listObjectsOfProperty(subj,proName).next().toString();
 * String descr = model.listObjectsOfProperty(subj,proDescr).next().toString();
 * 
 * com.hp.hpl.mesa.rdf.jena.model.NodeIterator parenti =
 * model.listObjectsOfProperty(subj,proSuperRole);
 * 
 * 
 * 
 * //check the existence of the role Role role =
 * (Role)uriVsRole.get(subj.getURI()); if (null==role) { //create the new role
 * role = new RoleImpl(id,name,descr); } else { role.setId(id);
 * role.setName(name); role.setDescription(descr); }
 * 
 *  // store in the map for further reference by uri
 * uriVsRole.put(subj.getURI(),role);
 * 
 * Set prnts = new HashSet(); while (parenti.hasNext()) {
 * com.hp.hpl.mesa.rdf.jena.model.Resource parent =
 * (com.hp.hpl.mesa.rdf.jena.model.Resource) parenti.next(); Role mappedRole =
 * (Role) uriVsRole.get(parent.getURI()); if (mappedRole == null) { mappedRole =
 * new RoleImpl(); uriVsRole.put(subj.getURI(),mappedRole); }
 * prnts.add(mappedRole); }
 * 
 * role.setParentRoles(prnts);
 *  } // ROLE
 * 
 * if (type.equals("SecurityRule")) { int id = Integer.parseInt(
 * model.listObjectsOfProperty(subj,proId).next().toString() );
 * 
 * String name = model.listObjectsOfProperty(subj,proName).next().toString();
 * String descr = model.listObjectsOfProperty(subj,proDescr).next().toString();
 * 
 * com.hp.hpl.mesa.rdf.jena.model.Resource restrResource = model.getResource(
 * model.listObjectsOfProperty(subj,proRuleRestriction).next().toString() );
 * String restrUri = restrResource.getURI();
 * 
 * Restriction restr = (Restriction)uriVsRestr.get(restrUri); if ( null ==
 * restr) { String restrType = (model.listObjectsOfProperty
 * (restrResource,proRestrictionType).next().toString()); int restrId =
 * Integer.parseInt(
 * model.listObjectsOfProperty(restrResource,proId).next().toString() ); restr =
 * Restriction.createRestriction(Restriction.type2Int(restrType), restrId);
 * uriVsRestr.put(restrUri,restr); }
 * 
 * Rule rule = (Rule)uriVsRule.get(subj.getURI()); if ( null == rule ) { rule =
 * new RuleImpl(id,name,descr); uriVsRule.put(subj.getURI(),rule); }
 * rule.setId(id); rule.setName(name); rule.setDescription(descr);
 * 
 * rule.setRestriction(restr);
 * 
 * ArrayList rights = new ArrayList();
 * com.hp.hpl.mesa.rdf.jena.model.NodeIterator ri =
 * model.listObjectsOfProperty(subj,proRightsGranted); while (ri.hasNext()) {
 * String right = ri.next().toString(); if (right.equals(Right.ADD_STR))
 * rule.setAddRight(true); if (right.equals(Right.ADMIN_STR))
 * rule.setAdminRight(true); if (right.equals(Right.HISTORY_STR))
 * rule.setHistoryRight(true); if (right.equals(Right.READ_STR))
 * rule.setReadRight(true); if (right.equals(Right.REMOVE_STR))
 * rule.setRemoveRight(true); } } // RULE
 * 
 * String restriction = "Restriction"; if (type.indexOf(restriction) != -1 &&
 * type.length()>restriction.length()) {
 * 
 * int id = Integer.parseInt(
 * model.listObjectsOfProperty(subj,proId).next().toString() );
 * 
 * String name = model.listObjectsOfProperty(subj,proName).next().toString();
 * String descr = model.listObjectsOfProperty(subj,proDescr).next().toString();
 * 
 * 
 * int intType = Restriction.type2Int(type);
 * 
 * Restriction restr = (Restriction)uriVsRestr.get(subj.getURI()); if (null==
 * restr) { restr = Restriction.createRestriction(intType,id); }
 * restr.setName(name); restr.setDescription(descr);
 * uriVsRestr.put(subj.getURI(),restr);
 * 
 * switch (intType) { case Restriction.INSTANCES : {
 * com.hp.hpl.mesa.rdf.jena.model.NodeIterator clses =
 * model.listObjectsOfProperty(subj,proIncludeResource); Set resSet = new
 * HashSet(); while (clses.hasNext()) { Resource cls = new Resource(
 * ((com.hp.hpl.mesa.rdf.jena.model.Resource ) clses.next()).getURI());
 * baseRdfRepository.addStatement(cls,resRdfType,resRdfResource); }
 * ((ResourceRestriction)restr).setResources(resSet); break; } case
 * Restriction.PROPERTIES : { com.hp.hpl.mesa.rdf.jena.model.NodeIterator pies =
 * model.listObjectsOfProperty(subj,proIncludeProperty); Set propList = new
 * HashSet(); while (pies.hasNext()) { Resource py = new Resource(
 * ((com.hp.hpl.mesa.rdf.jena.model.Resource ) pies.next()).getURI());
 * baseRdfRepository.addStatement(py,resRdfType,resRdfResource);
 * propList.add(py); } ((PropertiesRestriction)restr).setProperties(propList);
 * break; }
 * 
 * case Restriction.CLASSES : { com.hp.hpl.mesa.rdf.jena.model.NodeIterator
 * clses = model.listObjectsOfProperty(subj,proIncludeResource); Set resSet =
 * new HashSet(); while (clses.hasNext()) { Resource cls = new Resource(
 * ((com.hp.hpl.mesa.rdf.jena.model.Resource ) clses.next()).getURI());
 * baseRdfRepository.addStatement(cls,resRdfType,resRdfResource);
 * 
 * resSet.add(cls); } ((ResourceRestriction)restr).setResources(resSet); break; }
 * case Restriction.CLASSES_OVER_SCHEMA : {
 * com.hp.hpl.mesa.rdf.jena.model.NodeIterator clses =
 * model.listObjectsOfProperty(subj,proIncludeResource); Set resSet = new
 * HashSet(); while (clses.hasNext()) { Resource cls = new Resource(
 * ((com.hp.hpl.mesa.rdf.jena.model.Resource ) clses.next()).getURI());
 * baseRdfRepository.addStatement(cls,resRdfType,resRdfResource);
 * 
 * resSet.add(cls); } ((ResourceRestriction)restr).setResources(resSet); break; }
 * case Restriction.PATTERN : { // subject restrictions
 * com.hp.hpl.mesa.rdf.jena.model.NodeIterator subjI =
 * model.listObjectsOfProperty(subj,proSubjectRestr); while (subjI.hasNext()) {
 * 
 * com.hp.hpl.mesa.rdf.jena.model.Resource rdfRes =
 * ((com.hp.hpl.mesa.rdf.jena.model.Resource ) subjI.next()); Resource cls = new
 * Resource(rdfRes.getURI());
 * baseRdfRepository.addStatement(cls,resRdfType,resRdfResource);
 * 
 * com.hp.hpl.mesa.rdf.jena.model.NodeIterator restrTypesI =
 * model.listObjectsOfProperty(rdfRes,proRdfType); String rType = ""; while
 * (restrTypesI.hasNext()){ rType = restrTypesI.next().toString(); if
 * (rType.indexOf(restriction)!=-1) break; }
 * 
 * Restriction sRestr = (Restriction)uriVsRestr.get(cls.getURI()); if
 * (sRestr==null) { rType =
 * model.listObjectsOfProperty(rdfRes,proRestrictionType).next().toString(); int
 * sRestrId = Integer.parseInt(
 * model.listObjectsOfProperty(rdfRes,proId).next().toString() );
 * 
 * sRestr = Restriction.createRestriction(Restriction.type2Int(rType),sRestrId );
 * uriVsRestr.put(cls.getURI(),sRestr); } ((PatternRestriction)
 * restr).addSubjectRestriction((ResourceRestriction)sRestr);
 *  }
 *  // predicate restrs com.hp.hpl.mesa.rdf.jena.model.NodeIterator predI =
 * model.listObjectsOfProperty(subj,proPredicateRestr); Set predRestrSet = new
 * HashSet(); while (predI.hasNext()) { com.hp.hpl.mesa.rdf.jena.model.Resource
 * rdfRes = ((com.hp.hpl.mesa.rdf.jena.model.Resource ) predI.next()); Resource
 * cls = new Resource(rdfRes.getURI());
 * baseRdfRepository.addStatement(cls,resRdfType,resRdfResource);
 * 
 * com.hp.hpl.mesa.rdf.jena.model.NodeIterator restrTypesI =
 * model.listObjectsOfProperty(rdfRes,proRdfType); String rType = ""; while
 * (restrTypesI.hasNext()){ rType = restrTypesI.next().toString(); if
 * (rType.indexOf(restriction)!=-1) break; }
 * 
 * Restriction sRestr = (Restriction)uriVsRestr.get(cls.getURI()); if
 * (sRestr==null) { rType =
 * model.listObjectsOfProperty(rdfRes,proRestrictionType).next().toString(); int
 * sRestrId = Integer.parseInt(
 * model.listObjectsOfProperty(rdfRes,proId).next().toString() ); sRestr =
 * Restriction.createRestriction(Restriction.type2Int(rType),sRestrId);
 * uriVsRestr.put(cls.getURI(),sRestr); } ((PatternRestriction)
 * restr).addPredicateRestriction((PropertiesRestriction)sRestr); }
 *  // object restrictions com.hp.hpl.mesa.rdf.jena.model.NodeIterator objI =
 * model.listObjectsOfProperty(subj,proObjectRestr); Set objRestrSet = new
 * HashSet(); while (objI.hasNext()) {
 * 
 * com.hp.hpl.mesa.rdf.jena.model.Resource rdfRes =
 * ((com.hp.hpl.mesa.rdf.jena.model.Resource ) objI.next()); Resource cls = new
 * Resource(rdfRes.getURI());
 * baseRdfRepository.addStatement(cls,resRdfType,resRdfResource);
 * 
 * objRestrSet.add(cls);
 * 
 * com.hp.hpl.mesa.rdf.jena.model.NodeIterator restrTypesI =
 * model.listObjectsOfProperty(rdfRes,proRdfType); String rType = ""; while
 * (restrTypesI.hasNext()){ rType = restrTypesI.next().toString(); if
 * (rType.indexOf(restriction)!=-1) break; }
 * 
 * Restriction sRestr = (Restriction)uriVsRestr.get(cls.getURI()); if
 * (sRestr==null) { rType =
 * model.listObjectsOfProperty(rdfRes,proRestrictionType).next().toString(); int
 * sRestrId = Integer.parseInt(
 * model.listObjectsOfProperty(rdfRes,proId).next().toString() );
 * 
 * sRestr = Restriction.createRestriction(Restriction.type2Int(rType),sRestrId);
 * uriVsRestr.put(cls.getURI(),sRestr); } ((PatternRestriction)
 * restr).addObjectRestriction((ResourceRestriction)sRestr); } break; }
 * 
 * case Restriction.QUERY : { com.hp.hpl.mesa.rdf.jena.model.NodeIterator queryI =
 * model.listObjectsOfProperty(subj,proRestrOnQuery); Set qSet = new HashSet();
 * while (queryI.hasNext()) { String q = queryI.next().toString();
 * ((QueryRestriction)restr).addQuery(q); } break; }
 *  } } // Restriction
 *  } //.. if node is resource } // for all types } // while there are subjects
 * 
 *  // refresh the security maps rulesById = new HashMap(); ArrayList vlist =
 * new ArrayList(uriVsRule.values()); for ( int i = 0 ; i < vlist.size(); i++ ) {
 * Rule r = (Rule) vlist.get(i); rulesById.put(new Integer(r.getId()),r); }
 * 
 * restrictionsById = new HashMap(); vlist = new ArrayList(uriVsRestr.values());
 * for ( int i = 0 ; i < vlist.size(); i++ ) { Restriction r = (Restriction)
 * vlist.get(i); restrictionsById.put(new Integer(r.getId()),r); }
 * 
 * rolesByName = new HashMap(); rolesById = new HashMap(); roles = new
 * HashSet(); vlist = new ArrayList(uriVsRole.values()); for ( int i = 0 ; i <
 * vlist.size(); i++ ) { Role r = (Role) vlist.get(i); rolesById.put(new
 * Integer(r.getId()),r); rolesByName.put(r.getName(),r); roles.add(r); }
 *  } catch (com.hp.hpl.mesa.rdf.jena.model.RDFException ex) { throw new
 * SailInternalException(ex); } catch ( Exception x) { throw new
 * SailInternalException(x); } catch ( Exception x) { throw new
 * SailInternalException( "Wrong format of the imported RDF reader",x); }
 *  }
 */

	/* < SecurityServices implementation */

	/* > TableQueryResultListener implementation */

	public void startTableQueryResult() throws IOException {
	}

	public void startTableQueryResult(String[] columnHeaders)
		throws IOException {
		queryResult = new ArrayList();
		isQueryReady = false;
	}

	public void endTableQueryResult() throws IOException {
		isQueryReady = true;
	}

	public void startTuple() throws IOException {
	}

	public void endTuple() throws IOException {
	}

	public void tupleValue(Value value) throws IOException {
		queryResult.add(value);
	}

	public void error(QueryErrorType errType, String msg) throws IOException {
		throw new SailInternalException(msg);
	}

	/* < TableQueryResultListener implementation */

	/* > Addition of Statements Handling */
	/**
	 * Adds transient statement. The added statements will be revised on commit
	 * of the current transaction. Then their relevance to the security setup
	 * will be tested and the irrelevant will be deleted in a new transaction.
	 * 
	 * @param subject
	 * @param predicate
	 * @param object
	 */
	private void addTransient(Resource subject, URI predicate, Value object) {
		org.openrdf.model.Statement st = new org.openrdf.model.impl.StatementImpl(
			subject, predicate, object);
		transients.add(st);
	} // addTransient(Resource,URI,Value)

	/**
	 * @return true if there are transient statements for the current
	 *         transaction
	 */
	private boolean areThereTransients() {
		return transients.size() != 0;
	}

	/**
	 * Removes the irrelevant transient statements. This to be called after the
	 * inference took place. Clears the transients member.
	 * 
	 * @return dump of the added and filtered statements : to be used for the
	 *         status reporter.
	 */
	private String removeIrrelevantTransients() {
		org.openrdf.model.Statement st;

		StringBuffer added = new StringBuffer();
		StringBuffer filtered = new StringBuffer();
		added.append("\nThe following statements were ADDed: \n");
		filtered.append("\nThe following statements were FILTERed: \n");
		boolean isAdded = false;
		boolean isFiltered = false;

		for (int ix = 0; ix < transients.size(); ix++) {
			st = (org.openrdf.model.Statement) transients.get(ix);
			Resource subject = st.getSubject();
			URI predicate = st.getPredicate();
			Value object = st.getObject();
			if (!isStatementAccessible(subject, predicate, object, Right.ADD)) {
				try {
					if (baseRdfRepository.removeStatements(subject, predicate, object) > 0) {
						filtered.append("\n").append(st.toString()).append("\n");
						isFiltered = true;
					}
				}
				catch (SailUpdateException e) {
					throw new SailInternalException(
						"Cannot finish the filtering of Tranisient Added Statements.\n"
							+ "Caused by :" + e.getClass() + "\n"
							+ e.getMessage());
				}
			}
			else {
				added.append("\n").append(st.toString()).append("\n");
				isAdded = true;
			}
		} // for
		transients = new ArrayList();

		return (isAdded ? added.toString() : "\nNo Statements ADDed.\n")
			+ (isFiltered ? filtered.toString() : "\nNo FILTERed Statements.\n");

	} //removeIrrelevantTransients();

	/* < Addition of Statements Handling */

	/**
	 * Loads the Literals table.
	 * 
	 * @param con connection to be used.
	 * @param st statement to be used.
	 * @throws SQLException if something goes wrong during the execution of this
	 *         method.
	 */
	private void loadLiteralsTable(Connection con, java.sql.Statement st)
		throws SQLException {
		ResultSet result = st.executeQuery(SELECT + LITERALS_TABLE);

		while (result.next()) {
			int id = result.getInt("id");
			String lang = result.getString("language");
			String value = result.getString("label");
			Literal literal = new LiteralImpl(value, lang);
			literalById.put(new Integer(id), literal);
			idByLiteral.put(literal, new Integer(id));
		} // while

		ThreadLog.trace(LITERALS_TABLE + " LOADED");
	}

	/**
	 * Loads the namespaces table.
	 * 
	 * @param con connection to be used.
	 * @param st statement to be used.
	 * @throws SQLException if something goes wrong during the execution of this
	 *         method.
	 */
	private void loadNameSpaceTable(Connection con, java.sql.Statement st)
		throws SQLException {
		ResultSet result = st.executeQuery(SELECT + NAMESPACES_TABLE);

		while (result.next()) {
			int id = result.getInt(1);
			String prefix = result.getString(2);
			String name = result.getString(3);
			RdbmsNamespace ns = new RdbmsNamespace(id, prefix, name, false);
			namespaces.put(new Integer(id), ns);
		} // while

		ThreadLog.trace(NAMESPACES_TABLE + " LOADED");
	}

	/**
	 * Loads the Resources table.
	 * 
	 * @param con connection to be used.
	 * @param st statement to be used.
	 * @throws SQLException if something goes wrong during the execution of this
	 *         method.
	 */
	private void loadResourcesTable(Connection con, java.sql.Statement st)
		throws SQLException {
		ResultSet result = st.executeQuery(SELECT + RESOURCES_TABLE);
		ValueFactory factory = new ValueFactoryImpl();

		while (result.next()) {
			int id = result.getInt(1);

			if (id > lastResId)
				lastResId = id;

			int namespace = result.getInt(2);
			String name = result.getString(3);
			RdbmsNamespace ns = (RdbmsNamespace) namespaces.get(new Integer(
				namespace));
			if (null == ns)
				throw new SailInternalException(
					"[Unknown] resource [namespace]. Namespace [id]="
						+ namespace);
			Resource rs = null;
			if (namespace != 0)
				rs = new URIImpl(ns.getName(), name);
			else
				// FIXME this is probably not correct, since it does not take
				// the bNode ID into account!
				rs = factory.createBNode();
			resById.put(new Integer(id), rs);
			idByRes.put(rs, new Integer(id));
			if (id > lastResourceId) {
				lastResourceId = id;
			}

		} // while

		ThreadLog.trace(RESTRICTIONS_TABLE + " LOADED");
	}

	/** Loads Security Setup from database */
	private void loadSecuritySetup() {
		ThreadLog.trace(">>>Loading of Security Setup");
		Connection con = null;
		java.sql.Statement st = null;
		try {
			con = conPool.getConnection();
			st = con.createStatement();

			loadNameSpaceTable(con, st);
			loadResourcesTable(con, st);
			loadLiteralsTable(con, st);

			loadRestrictionsTable(con, st);
			loadSecurityRulesTable(con, st);
			loadRolesTable(con, st);
			loadRolesHierarchyTable(con, st);
			loadRolesRulesTable(con, st);
			loadResourceOrPropertyRestrictionsTable(con, st);
			loadQueryRestrictionsTable(con, st);
			loadPatternRestrictionsTable(con, st);
			loadUsersRolesTable(con, st);
			loadUsersRulesTable(con, st);
		}
		catch (SQLException sqle) {
			throw new SailInternalException(sqle);
		}
		finally {
			try {
				st.close();
				con.close();
			}
			catch (SQLException e) {
				throw new SailInternalException(e);
			}
		} // finally

		ThreadLog.trace("<<<Loading of Security Setup");
	} // loadSecuritySetup()

	/**
	 * Initializes database : creates tables, if they do not exist.
	 * 
	 * @throws SailInternalException if something goes wrong with the db
	 *         initialization
	 */
	private void initDatabase() throws SailInternalException {
		ThreadLog.trace(">>>initializing Database");
		Connection con = null;
		java.sql.Statement st = null;
		try {
			con = conPool.getConnection();
			st = con.createStatement();

			createSecurityTables(con, st);

			ThreadLog.trace("<<<initializing Database");

		}
		catch (SQLException e) {
			throw new SailInternalException(e);
		}
		finally {
			try {
				st.close();
				con.close();
			}
			catch (SQLException e) {
				throw new SailInternalException(e);
			}
		}
	} // initDatabase()

	/**
	 * Checks whether a table exists, given a table name and a connection.
	 * 
	 * @param con the connection
	 * @param tableName the name of the table
	 * @return true if the table exists
	 * @throws SQLException if something goes wrong in this method
	 */
	private boolean tableExists(Connection con, String tableName)
		throws SQLException {
		boolean tableExists = false;
		DatabaseMetaData metadata = con.getMetaData();
		ResultSet rs = metadata.getTables(null, null, tableName, null);
		tableExists = rs.next();
		metadata = null;
		rs.close();
		rs = null;
		return tableExists;
	}

	//	/** Creates the users table if not already created.
	//	 * NOTE: this table should be created outside of the repository.
	//		* @param con connection to be used.
	//		* @param st statement to be used.
	//		* @throws SQLException if something goes wrong during the execution of
	// this method.
	//		*/
	//	private void createUsersTable(Connection con, java.sql.Statement st)
	//		throws SQLException
	//	{
	//		if (!tableExists(con, USERS_TABLE))
	//		{
	//			String query = "CREATE TABLE "+USERS_TABLE+" ("+
	//				"usid "+INT+" NOT NULL,"+
	//				"login "+CHR20+" ,"+
	//				"password "+CHR20+" ,"+
	//				"name "+CHR255+
	//				" );";
	//			st.executeUpdate(query);
	//
	//			st.executeUpdate(
	//				"CREATE UNIQUE INDEX XPKUsers ON "+ USERS_TABLE
	//				+"( usid );");
	//
	//			st.executeUpdate("insert into "+USERS_TABLE+
	//			" VALUES ("+1+", 'admin', 'admin', 'Administrator' )");
	//			}
	//
	//			ThreadLog.trace(USERS_TABLE + " CREATED");
	//
	//	} // createUsersTable(Connection,java.sql.Statement)

	/** Creates the users' roles table if not already created.
	 * @param con connection to be used.
	 * @parzam st statement to be used.
	 * @throws SQLException if something goes wrong during the execution of this method.
	 */
	private void createUsersRolesTable(Connection con, java.sql.Statement st)
		throws SQLException {
		if (!tableExists(con, USERS_ROLES_TABLE)) {
			st.executeUpdate("CREATE TABLE " + USERS_ROLES_TABLE + " ("
				+ "user_id " + INT + " NOT NULL," + "role_id " + INT
				+ " NOT NULL" + " );");

			st.executeUpdate("CREATE UNIQUE INDEX XPKUsersRoles ON "
				+ USERS_ROLES_TABLE + "( user_id, role_id );");
			ThreadLog.trace(USERS_ROLES_TABLE + " CREATED");
		}
	} // createUsersRolesTable(Connection,java.sql.Statement)

	/** Load the users' roles table.
	 * @param con connection to be used.
	 * @parzam st statement to be used.
	 * @throws SQLException if something goes wrong during the execution of this method.
	 */
	private void loadUsersRolesTable(Connection con, java.sql.Statement st)
		throws SQLException {
		ResultSet result = st.executeQuery(SELECT + USERS_ROLES_TABLE);
		while (result.next()) {
			int user_id = result.getInt(1);
			int role_id = result.getInt(2);
			User user = getUser(user_id);
			Role role = (Role) rolesById.get(new Integer(role_id));

			if (null == role)
				throw new SailInternalException("Unknown [role id] in table "
					+ USERS_ROLES_TABLE + "\n[id]=" + role_id);
			if (null == user) {
				//				throw new SailInternalException(
				//					"Unknown [user id] in table "+USERS_ROLES_TABLE+
				//					"\n[id]="+user_id);
				try {
					user = new UserImpl(user_id, UserImpl.ANONYMOUS + user_id,
						"", UserImpl.ANONYMOUS + user_id);

					users.add(user);
					usersById.put(new Integer(user_id), user);
					usersByLogin.put(user.getLogin(), user);

				}
				catch (Exception e) {
					throw new SailInternalException(e);
				}

			} // if null user
			Set roles = user.getRoles();
			roles.add(role);

			try {
				user.setRoles(roles);
			}
			catch (Exception e) {
				throw new SailInternalException(e);
			}

		} // while

		ThreadLog.trace(USERS_ROLES_TABLE + " LOADED");
	} // loadUsersRolesTable(Connection,java.sql.Statement)

	/**
	 * Creates the users' rules table if not already created.
	 * 
	 * @param con connection to be used.
	 * @param st statement to be used.
	 * @throws SQLException if something goes wrong during the execution of this
	 *         method.
	 */
	private void createUsersRulesTable(Connection con, java.sql.Statement st)
		throws SQLException {
		if (!tableExists(con, USERS_RULES_TABLE)) {
			st.executeUpdate("CREATE TABLE " + USERS_RULES_TABLE + " ("
				+ "user_id " + INT + " NOT NULL," + "rule_id " + INT
				+ " NOT NULL" + " );");

			st.executeUpdate("CREATE UNIQUE INDEX XPKUsersRules ON "
				+ USERS_RULES_TABLE + "( user_id, rule_id );");
			ThreadLog.trace(USERS_RULES_TABLE + " CREATED");
		}
	} // createUsersRulesTable(Connection,java.sql.Statement)

	/** Load the users' rules table.
	 * @param con connection to be used.
	 * @parzam st statement to be used.
	 * @throws SQLException if something goes wrong during the execution of this method.
	 */
	private void loadUsersRulesTable(Connection con, java.sql.Statement st)
		throws SQLException {
		ResultSet result = st.executeQuery(SELECT + USERS_RULES_TABLE);
		while (result.next()) {
			int user_id = result.getInt(1);
			int rule_id = result.getInt(2);
			User user = getUser(user_id);
			Rule rule = (Rule) rulesById.get(new Integer(rule_id));

			if (null == rule)
				throw new SailInternalException("Unknown [rule id] in table "
					+ USERS_RULES_TABLE + "\n[id]=" + rule_id);
			if (null == user) {
				//				throw new SailInternalException(
				//					"Unknown [user id] in table "+USERS_RULES_TABLE+
				//					"\n[id]="+user_id);
				try {
					user = new UserImpl(user_id, UserImpl.ANONYMOUS + user_id,
						"", UserImpl.ANONYMOUS + user_id);

					users.add(user);
					usersById.put(new Integer(user_id), user);
					usersByLogin.put(user.getLogin(), user);

				}
				catch (Exception e) {
					throw new SailInternalException(e);
				}
			}//if null user
			Set rules = user.getRules();
			rules.add(rule);
			try {
				user.setRules(rules);
			}
			catch (Exception e) {
				throw new SailInternalException(e);
			}

		} // while

		ThreadLog.trace(USERS_RULES_TABLE + " LOADED");
	} // loadUsersRulesTable(Connection,java.sql.Statement)

	/**
	 * Creates the roles table if not already created.
	 * 
	 * @param con connection to be used.
	 * @param st statement to be used.
	 * @throws SQLException if something goes wrong during the execution of this
	 *         method.
	 */
	private void createRolesTable(Connection con, java.sql.Statement st)
		throws SQLException {
		if (!tableExists(con, ROLES_TABLE)) {
			st.executeUpdate("CREATE TABLE " + ROLES_TABLE + " (" + "id " + INT
				+ " NOT NULL," + "name " + CHR20 + ",description " + CHR255
				+ " );");

			st.executeUpdate("CREATE UNIQUE INDEX XPKRoles ON " + ROLES_TABLE
				+ "( id );");
			ThreadLog.trace(ROLES_TABLE + " CREATED");
		}
	} // createRolesTable(Connection,java.sql.Statement)

	/**
	 * Loads the roles table.
	 * 
	 * @param con connection to be used.
	 * @param st statement to be used.
	 * @throws SQLException if something goes wrong during the execution of this
	 *         method.
	 */
	private void loadRolesTable(Connection con, java.sql.Statement st)
		throws SQLException {
		ResultSet result = st.executeQuery(SELECT + ROLES_TABLE);

		while (result.next()) {
			int id = result.getInt(1);
			String name = result.getString(2);
			String description = result.getString(3);
			Role role = createRole(id, name, description, new HashSet(1));
		} // while

		ThreadLog.trace(ROLES_TABLE + " LOADED");
	} // loadRolesTable(Connection,java.sql.Statement)

	/**
	 * Creates the roles hierarchy table if not already created.
	 * 
	 * @param con connection to be used.
	 * @param st statement to be used.
	 * @throws SQLException if something goes wrong during the execution of this
	 *         method.
	 */
	private void createRolesHierarchyTable(Connection con, java.sql.Statement st)
		throws SQLException {
		if (!tableExists(con, ROLES_HIERARCHY_TABLE)) {
			st.executeUpdate("CREATE TABLE " + ROLES_HIERARCHY_TABLE + " ("
				+ "role_id " + INT + " NOT NULL," + "parent_role_id " + INT
				+ " NOT NULL" + " );");

			st.executeUpdate("CREATE UNIQUE INDEX XPKRolesHierarchy ON "
				+ ROLES_HIERARCHY_TABLE + "( role_id, parent_role_id );");
			ThreadLog.trace(ROLES_HIERARCHY_TABLE + " CREATED");

		}
	} // createRolesHierarchyTable(Connection,java.sql.Statement)

	/**
	 * Loads the Roles Hierarchy Table
	 * 
	 * @param con connection to be used.
	 * @param st statement to be used.
	 * @throws SQLException if something goes wrong during the execution of this
	 *         method.
	 */
	private void loadRolesHierarchyTable(Connection con, java.sql.Statement st)
		throws SQLException {
		ResultSet result = st.executeQuery(SELECT + ROLES_HIERARCHY_TABLE);

		while (result.next()) {
			int role_id = result.getInt(1);
			int parent_id = result.getInt(2);
			Role role = (Role) rolesById.get(new Integer(role_id));
			Role parent = (Role) rolesById.get(new Integer(parent_id));
			if (null == role)
				throw new SailInternalException("Unknown [role id] in table "
					+ ROLES_HIERARCHY_TABLE + "\n[id]=" + role_id);
			if (null == parent)
				throw new SailInternalException(
					"Unknown [parent role id] in table "
						+ ROLES_HIERARCHY_TABLE + "\n[id]=" + parent_id);
			Set parents = role.getParentRoles();
			parents.add(parent);

			try {
				role.setParentRoles(parents);
			}
			catch (Exception e) {
				throw new SailInternalException(e);
			}
		} // while

		ThreadLog.trace(ROLES_HIERARCHY_TABLE + " LOADED");
	} // loadRolesHierarchyTable(Connection,java.sql.Statement)

	/**
	 * Creates the roles' rules table if not already created.
	 * 
	 * @param con connection to be used.
	 * @param st statement to be used.
	 * @throws SQLException if something goes wrong during the execution of this
	 *         method.
	 */
	private void createRolesRulesTable(Connection con, java.sql.Statement st)
		throws SQLException {
		if (!tableExists(con, ROLES_RULES_TABLE)) {
			st.executeUpdate("CREATE TABLE " + ROLES_RULES_TABLE + " ("
				+ "role_id " + INT + " NOT NULL," + "rule_id " + INT
				+ " NOT NULL" + " );");

			st.executeUpdate("CREATE UNIQUE INDEX XPKRolesRules ON "
				+ ROLES_RULES_TABLE + "( role_id, rule_id );");
			ThreadLog.trace(ROLES_RULES_TABLE + " CREATED");

		}
	} // createRolesRulesTable(Connection,java.sql.Statement)

	/**
	 * Loads the Roles Rules Table
	 * 
	 * @param con connection to be used.
	 * @param st statement to be used.
	 * @throws SQLException if something goes wrong during the execution of this
	 *         method.
	 */
	private void loadRolesRulesTable(Connection con, java.sql.Statement st)
		throws SQLException {
		ResultSet result = st.executeQuery(SELECT + ROLES_RULES_TABLE);

		while (result.next()) {
			int role_id = result.getInt(1);
			int rule_id = result.getInt(2);
			Role role = (Role) rolesById.get(new Integer(role_id));
			Rule rule = (Rule) rulesById.get(new Integer(rule_id));

			if (null == role)
				throw new SailInternalException("Unknown [role id] in table "
					+ ROLES_RULES_TABLE + "\n[id]=" + role_id);
			if (null == rule)
				throw new SailInternalException("Unknown [rule id] in table "
					+ ROLES_RULES_TABLE + "\n[id]=" + rule_id);

			try {
				role.addRule(rule);
			}
			catch (Exception e) {
				throw new SailInternalException(e);
			}

		} // while

		ThreadLog.trace(ROLES_RULES_TABLE + " LOADED");
	} // loadRolesRulesTable(Connection,java.sql.Statement)

	/**
	 * Creates the Security Rules table if not already created.
	 * 
	 * @param con connection to be used.
	 * @param st statement to be used.
	 * @throws SQLException if something goes wrong during the execution of this
	 *         method.
	 */
	private void createSecurityRulesTable(Connection con, java.sql.Statement st)
		throws SQLException {
		if (!tableExists(con, SECURITY_RULES_TABLE)) {
			String query = "CREATE TABLE "
				+ SECURITY_RULES_TABLE
				+ " ("
				+ "id "
				+ INT
				+ " NOT NULL,"
				+ "name "
				+ CHR20
				+ ", "
				+ "description "
				+ CHR255
				+ ", "
				+ "restriction_id "
				+ INT
				+ " NOT NULL,"
				+ "read_right BOOL , add_right BOOL, remove_right BOOL, admin_right BOOL, history_right BOOL"
				+ " );";
			st.executeUpdate(query);

			st.executeUpdate("CREATE UNIQUE INDEX XPKSecurityRules ON "
				+ SECURITY_RULES_TABLE + "( id );");
			ThreadLog.trace(SECURITY_RULES_TABLE + " CREATED");
		}
	} // createSecurityRulesTable(Connection,java.sql.Statement)

	/**
	 * Loads the Security Rules table.
	 * 
	 * @param con connection to be used.
	 * @param st statement to be used.
	 * @throws SQLException if something goes wrong during the execution of this
	 *         method.
	 */
	private void loadSecurityRulesTable(Connection con, java.sql.Statement st)
		throws SQLException {
		ResultSet result = st.executeQuery(SELECT + SECURITY_RULES_TABLE);

		while (result.next()) {
			int id = result.getInt(1);
			String name = result.getString(2);
			String description = result.getString(3);
			Rule rule;
			try {
				rule = new RuleImpl(id, name, description);
			}
			catch (Exception e) {
				throw new SailInternalException(e);
			}

			int restrId = result.getInt(4);
			Restriction restr = getRestriction(restrId);
			try {
				rule.setRestriction(restr);
			}
			catch (Exception e) {
				throw new SailInternalException(e);
			}

			rule.setReadRight(result.getBoolean(5));
			rule.setAddRight(result.getBoolean(6));
			rule.setRemoveRight(result.getBoolean(7));
			rule.setAdminRight(result.getBoolean(8));
			rule.setHistoryRight(result.getBoolean(9));

			rulesById.put(new Integer(id), rule);
		} // while

		ThreadLog.trace(SECURITY_RULES_TABLE + " LOADED");
	} // loadSecurityRulesTable(Connection,java.sql.Statement)

	/**
	 * Creates the Restrictions table if not already created.
	 * 
	 * @param con connection to be used.
	 * @param st statement to be used.
	 * @throws SQLException if something goes wrong during the execution of this
	 *         method.
	 */
	private void createRestrictionsTable(Connection con, java.sql.Statement st)
		throws SQLException {
		if (!tableExists(con, RESTRICTIONS_TABLE)) {
			String query = "CREATE TABLE " + RESTRICTIONS_TABLE + " (" + "id "
				+ INT + " NOT NULL," + "type " + INT + " NOT NULL," + "name "
				+ CHR20 + ", " + "description " + CHR255 + " );";
			st.executeUpdate(query);

			st.executeUpdate("CREATE UNIQUE INDEX XPKRestrictions ON "
				+ RESTRICTIONS_TABLE + "( id );");
			ThreadLog.trace(RESTRICTIONS_TABLE + " CREATED");

		}
	} // createRestrictionsTable(Connection,java.sql.Statement)

	/**
	 * Loads the Restrictions table.
	 * 
	 * @param con connection to be used.
	 * @param st statement to be used.
	 * @throws SQLException if something goes wrong during the execution of this
	 *         method.
	 */
	private void loadRestrictionsTable(Connection con, java.sql.Statement st)
		throws SQLException {
		ResultSet result = st.executeQuery(SELECT + RESTRICTIONS_TABLE);

		while (result.next()) {
			int id = result.getInt(1);
			int type = result.getInt(2);
			String name = result.getString(3);
			String description = result.getString(4);
			Restriction restr;
			try {
				restr = createRestriction(id, type, name, description);
			}
			catch (Exception e) {
				throw new SailInternalException(e);
			}
			restrictionsById.put(new Integer(id), restr);
		} // while

		ThreadLog.trace(RESTRICTIONS_TABLE + " LOADED");
	} // loadRestrictionsTable(Connection,java.sql.Statement)

	/**
	 * Creates the Resource or Property Restrictions table if not already
	 * created.
	 * 
	 * @param con connection to be used.
	 * @param st statement to be used.
	 * @throws SQLException if something goes wrong during the execution of this
	 *         method.
	 */
	private void createResourceOrPropertyRestrictionsTable(Connection con, java.sql.Statement st)
		throws SQLException {
		if (!tableExists(con, RES_PROP_RESTRS_TABLE)) {
			String query = "CREATE TABLE " + RES_PROP_RESTRS_TABLE + " ("
				+ "restr_id " + INT + " NOT NULL," + "resource_id " + INT
				+ " NOT NULL" + " );";
			st.executeUpdate(query);

			st.executeUpdate("CREATE UNIQUE INDEX XPKResOrPropRestrictions ON "
				+ RES_PROP_RESTRS_TABLE + "( restr_id,resource_id );");
			ThreadLog.trace(RES_PROP_RESTRS_TABLE + " CREATED");
		}
	} // createResourceOrPropertyRestrictionsTable(Connection,java.sql.Statement)

	/**
	 * Loads the ResourceOrPropertyRestrictionsTable.
	 * 
	 * @param con connection to be used.
	 * @param st statement to be used.
	 * @throws SQLException if something goes wrong during the execution of this
	 *         method.
	 */
	private void loadResourceOrPropertyRestrictionsTable(Connection con, java.sql.Statement st)
		throws SQLException {
		ResultSet result = st.executeQuery(SELECT + RES_PROP_RESTRS_TABLE);

		while (result.next()) {
			int restrId = result.getInt(1);
			int resourceId = result.getInt(2);

			Restriction restr = getRestriction(restrId);
			Resource res = (Resource) resById.get(new Integer(resourceId));

			if (null == restr)
				throw new SailInternalException(
					"Unknown Restriction [id] in table "
						+ RES_PROP_RESTRS_TABLE + "\n[id] = " + restrId);
			if (null == res)
				throw new SailInternalException(
					"Unknown Resource [id] in table " + RES_PROP_RESTRS_TABLE
						+ "\n[id] = " + resourceId);

			int type = restr.getType();
			try {
				switch (type) {
					case Restriction.CLASSES : {
						ClassesRestriction cr = (ClassesRestriction) restr;
						Set rss = cr.getResources();
						rss.add(res);
						cr.setResources(rss);
						break;
					}
					case Restriction.CLASSES_OVER_SCHEMA : {
						ClassesOverSchemaRestriction cr = (ClassesOverSchemaRestriction) restr;
						Set rss = cr.getResources();
						rss.add(res);
						cr.setResources(rss);
						break;
					}
					case Restriction.INSTANCES : {
						InstancesRestriction ir = (InstancesRestriction) restr;
						Set rss = ir.getResources();
						rss.add(res);
						ir.setResources(rss);
						break;
					}
					case Restriction.PROPERTIES : {
						PropertiesRestriction pr = (PropertiesRestriction) restr;
						Set rss = pr.getProperties();
						rss.add(res);
						pr.setProperties(rss);
						break;
					}
					default : {
						throw new SailInternalException(RES_PROP_RESTRS_TABLE
							+ " contains a restriction which type [" + type
							+ "] is neither INSTANCES type, \n"
							+ "neither CLASSES type, neither PROPERTIES type.");
					}
				}
			}
			catch (Exception e) {
				throw new SailInternalException(e);
			}
		} // while

		ThreadLog.trace(RESTRICTIONS_TABLE + " LOADED");
	} // loadResourceOrPropertyRestrictionsTable(Connection,java.sql.Statement)

	/**
	 * Creates the Pattern Restrictions table if not already created.
	 * 
	 * @param con connection to be used.
	 * @param st statement to be used.
	 * @throws SQLException if something goes wrong during the execution of this
	 *         method.
	 */
	private void createPatternRestrictionsTable(Connection con, java.sql.Statement st)
		throws SQLException {
		if (!tableExists(con, PATTERN_RESTRS_TABLE)) {
			st.executeUpdate("CREATE TABLE " + PATTERN_RESTRS_TABLE + " ("
				+ "pattern_restr_id " + INT + " NOT NULL," + "cip_restr_id "
				+ INT + " NOT NULL," + "spo_type " + INT + " NOT NULL" + " );");

			st.executeUpdate("CREATE UNIQUE INDEX XPKPatternRestrictions ON "
				+ PATTERN_RESTRS_TABLE
				+ "( pattern_restr_id, cip_restr_id, spo_type );");
			ThreadLog.trace(PATTERN_RESTRS_TABLE + " CREATED");
		}
	} // createPatternRestrictionsTable(Connection,java.sql.Statement)

	/**
	 * Loads the Pattern Restrictions Table
	 * 
	 * @param con connection to be used.
	 * @param st statement to be used.
	 * @throws SQLException if something goes wrong during the execution of this
	 *         method.
	 */
	private void loadPatternRestrictionsTable(Connection con, java.sql.Statement st)
		throws SQLException {
		ResultSet result = st.executeQuery(SELECT + PATTERN_RESTRS_TABLE);

		while (result.next()) {
			int restr_id = result.getInt(1);
			/* classes, instances or properties restriction id, or literal id */
			int cip_restr_id = result.getInt(2);
			/* subject, predicate, object, or object literal restriction */
			int spo = result.getInt(3);

			Restriction restr = (Restriction) restrictionsById.get(new Integer(
				restr_id));

			if (null == restr)
				throw new SailInternalException(
					"Unknown Restriction [id] in table " + PATTERN_RESTRS_TABLE
						+ "\n[id] = " + restr_id);

			Restriction cip = null;
			if (spo != PatternRestriction.SPO_OBJECT_LITERAL) {
				cip = getRestriction(cip_restr_id);
				if (null == cip)
					throw new SailInternalException(
						"Unknown Restriction [id] in table "
							+ PATTERN_RESTRS_TABLE + "\n[id] = " + cip_restr_id);
			}// if not literal

			if (!(restr instanceof PatternRestriction))
				throw new SailInternalException("All restrictions in the "
					+ PATTERN_RESTRS_TABLE
					+ "\n should be PatternRestriction.\n"
					+ "Restriction [id]=" + restr_id + "\n+class:"
					+ restr.getClass());

			PatternRestriction pr = (PatternRestriction) restr;

			switch (spo) {
				case PatternRestriction.SPO_SUBJECT : {
					if (!(cip instanceof ResourceRestriction))
						throw new SailInternalException(
							"Subject Restrictions should be instances of ResourceRestriction\n"
								+ "class:" + cip.getClass());
					pr.addSubjectRestriction((ResourceRestriction) cip);
					break;
				}
				case PatternRestriction.SPO_PREDICATE : {
					if (!(cip instanceof PropertiesRestriction))
						throw new SailInternalException(
							"Predicate Restrictions should be instances of PropertiesRestriction\n"
								+ "class:" + cip.getClass());
					pr.addPredicateRestriction((PropertiesRestriction) cip);
					break;
				}
				case PatternRestriction.SPO_OBJECT : {
					if (!(cip instanceof ResourceRestriction))
						throw new SailInternalException(
							"Object Restrioctions should be instances of ResourceRestriction\n"
								+ "class:" + cip.getClass());
					pr.addObjectRestriction((ResourceRestriction) cip);
					break;
				}
				case PatternRestriction.SPO_OBJECT_LITERAL : {
					Literal lit = (Literal) literalById.get(new Integer(
						cip_restr_id));
					if (null == lit)
						throw new SailInternalException(
							"Unknown literal id in table "
								+ PATTERN_RESTRS_TABLE + "\n[id]="
								+ cip_restr_id);

					try {
						pr.addObjectRestriction(lit);
					}
					catch (Exception e) {
						throw new SailInternalException(e);
					}

					break;
				}
				default : {
					throw new SailInternalException(
						"Unknown Subject_Predicate_Object Type\n" + "in table "
							+ PATTERN_RESTRS_TABLE + "\n[spo_type]=" + spo);
				}
			} //switch

		} // while

		ThreadLog.trace(PATTERN_RESTRS_TABLE + " LOADED");
	} // loadPatternRestrictionsTable(Connection,java.sql.Statement)

	/**
	 * Creates the Query Restrictions table if not already created.
	 * 
	 * @param con connection to be used.
	 * @param st statement to be used.
	 * @throws SQLException if something goes wrong during the execution of this
	 *         method.
	 */
	private void createQueryRestrictionsTable(Connection con, java.sql.Statement st)
		throws SQLException {
		if (!tableExists(con, QUERY_RESTRS_TABLE)) {
			st.executeUpdate("CREATE TABLE " + QUERY_RESTRS_TABLE + " ("
				+ "restr_id " + INT + " NOT NULL," + "query " + CHR255 + " );");
			ThreadLog.trace(QUERY_RESTRS_TABLE + " CREATED");
		}
	} // createQueryRestrictionsTable(Connection,java.sql.Statement)

	/**
	 * Loads the Query Restrictions Table
	 * 
	 * @param con connection to be used.
	 * @param st statement to be used.
	 * @throws SQLException if something goes wrong during the execution of this
	 *         method.
	 */
	private void loadQueryRestrictionsTable(Connection con, java.sql.Statement st)
		throws SQLException {
		ResultSet result = st.executeQuery(SELECT + QUERY_RESTRS_TABLE);

		while (result.next()) {
			int id = result.getInt(1);
			String query = result.getString(2);
			Restriction restr = getRestriction(id);
			if (null == restr)
				throw new SailInternalException(
					"Unknown Restriction [id] in table " + QUERY_RESTRS_TABLE
						+ "\n[id] = " + id);
			if (!(restr instanceof QueryRestriction))
				throw new SailInternalException("All restrictions in the "
					+ QUERY_RESTRS_TABLE + "\n should be QueryRestrictions.\n"
					+ "Restriction [id]=" + id + "\n+class:" + restr.getClass());

			QueryRestriction qr = (QueryRestriction) restr;
			qr.addQuery(query);
		} // while

		ThreadLog.trace(QUERY_RESTRS_TABLE + " LOADED");
	} // loadQueryRestrictionsTable(Connection,java.sql.Statement)

	/**
	 * Creates the Security Rules table if not already created.
	 * 
	 * @param con connection to be used.
	 * @param st statement to be used.
	 * @throws SQLException if something goes wrong during the execution of this
	 *         method.
	 * @throws NullParameterException if some of the parameters are null.
	 */
	private void dropSecurityTables(Connection con, java.sql.Statement st)
		throws SQLException, NullParameterException {
		if (null == con || null == st)
			throw new NullParameterException(
				"the Connection [con] and java.sql.Statement [st] should not be [null]");

		try {
			st.executeUpdate(DROP_TABLE + USERS_ROLES_TABLE);
		}
		catch (SQLException e) {
		}
		try {
			st.executeUpdate(DROP_TABLE + USERS_RULES_TABLE);
		}
		catch (SQLException e) {
		}
		try {
			st.executeUpdate(DROP_TABLE + SECURITY_RULES_TABLE);
		}
		catch (SQLException e) {
		}
		try {
			st.executeUpdate(DROP_TABLE + ROLES_TABLE);
		}
		catch (SQLException e) {
		}
		try {
			st.executeUpdate(DROP_TABLE + ROLES_RULES_TABLE);
		}
		catch (SQLException e) {
		}
		try {
			st.executeUpdate(DROP_TABLE + ROLES_HIERARCHY_TABLE);
		}
		catch (SQLException e) {
		}
		try {
			st.executeUpdate(DROP_TABLE + RESTRICTIONS_TABLE);
		}
		catch (SQLException e) {
		}
		try {
			st.executeUpdate(DROP_TABLE + RES_PROP_RESTRS_TABLE);
		}
		catch (SQLException e) {
		}
		try {
			st.executeUpdate(DROP_TABLE + PATTERN_RESTRS_TABLE);
		}
		catch (SQLException e) {
		}
		try {
			st.executeUpdate(DROP_TABLE + QUERY_RESTRS_TABLE);
		}
		catch (SQLException e) {
		}
		ThreadLog.trace("Security Tables DROPPED.");
	} // dropSecurityTables(Connection,java.sql.Statement)

	/**
	 * Creates security tables.
	 * 
	 * @param con
	 * @param st
	 * @throws SQLException
	 */
	private void createSecurityTables(Connection con, java.sql.Statement st)
		throws SQLException {

		createUsersRolesTable(con, st);
		createUsersRulesTable(con, st);

		createRolesTable(con, st);
		createRolesHierarchyTable(con, st);
		createRolesRulesTable(con, st);

		createSecurityRulesTable(con, st);

		createRestrictionsTable(con, st);
		createResourceOrPropertyRestrictionsTable(con, st);
		createPatternRestrictionsTable(con, st);
		createQueryRestrictionsTable(con, st);
	}

	/**
	 * Uploads sample security setup via creting security setup objects and
	 * dumping them to sql.
	 */
	private void uploadSampleSetup() {

		Connection con = null;
		java.sql.Statement st = null;

		try {
			con = conPool.getConnection();
			st = con.createStatement();
			loadNameSpaceTable(con, st);
			loadResourcesTable(con, st);
			loadLiteralsTable(con, st);

			User uAdmin = new UserImpl(1, "", "", "");
			User uAnon = new UserImpl(2, "", "", "");
			User uTest = new UserImpl(3, "", "", "");
			User uHR = new UserImpl(4, "", "", "");
			User uRD = new UserImpl(5, "", "", "");
			User uEmp = new UserImpl(6, "", "", "");
			User uQue = new UserImpl(7, "", "", "");
			User uMitac = new UserImpl(8, "", "", "");
			users.add(uAdmin);
			users.add(uAnon);
			users.add(uTest);
			users.add(uHR);
			users.add(uRD);
			users.add(uEmp);
			users.add(uQue);
			users.add(uMitac);

			// ADMIN
			Restriction rsRepository = Restriction.createRepositoryRestriction("repository", "repository");

			Rule ruAdmin = new RuleImpl("doEverything",
				"do everything with the repository rule");
			ruAdmin.setAddRight(true);
			ruAdmin.setAdminRight(true);
			ruAdmin.setHistoryRight(true);
			ruAdmin.setRemoveRight(true);
			ruAdmin.setReadRight(true);
			ruAdmin.setRestriction(rsRepository);

			{
				Set rulz = uAdmin.getRules();
				rulz.add(ruAdmin);
				uAdmin.setRules(rulz);
			}

			// TESTUSER
			Restriction rsSchema = Restriction.createSchemaRestriction("schema", "schema");
			Rule ruReadSchema = new RuleImpl("readschema", "readschema");
			ruReadSchema.setReadRight(true);
			ruReadSchema.setRestriction(rsSchema);
			{
				Set rulz = uTest.getRules();
				rulz.add(ruReadSchema);
				uTest.setRules(rulz);
			}

			//HR Manager
			URI resSkill = null;
			ResourceRestriction rsSkillsSchema = null;
			{
				resSkill = new URIImpl(SKILL);
				rsSkillsSchema = (ResourceRestriction) ResourceRestriction.createClassesOverSchemaRestriction("skillsSchema", "skills hierarchy");

				{
					Set rSet = rsSkillsSchema.getResources();
					rSet.add(resSkill);
					rsSkillsSchema.setResources(rSet);
				}

				ResourceRestriction rsSkills = (ResourceRestriction) ResourceRestriction.createClassesRestriction("skillsInstances", "skills instances");

				{
					Set rSet = rsSkills.getResources();
					rSet.add(resSkill);
					rsSkills.setResources(rSet);
				}

				Rule ruSkillsSchema = new RuleImpl("skillsSchema",
					"skills schema");
				ruSkillsSchema.setRestriction(rsSkillsSchema);
				ruSkillsSchema.setAddRight(true);
				ruSkillsSchema.setReadRight(true);
				ruSkillsSchema.setRemoveRight(true);

				Rule ruSkills = new RuleImpl("skills", "skills");
				ruSkills.setRestriction(rsSkills);
				ruSkills.setAddRight(true);
				ruSkills.setReadRight(true);
				ruSkills.setRemoveRight(true);

				{
					Set rulz = uHR.getRules();
					rulz.add(ruSkills);
					rulz.add(ruSkillsSchema);
					uHR.setRules(rulz);
				}
			}

			// RandD Manager
			URI resTechSkill = null;
			ResourceRestriction rsTSkills = null;
			{
				resTechSkill = new URIImpl(TECH_SKILL);
				ResourceRestriction rsTSkillsSchema = (ResourceRestriction) ResourceRestriction.createClassesOverSchemaRestriction("TechskillsSchema", "Techskills hierarchy");

				{
					Set rSet = rsTSkillsSchema.getResources();
					rSet.add(resTechSkill);
					rsTSkillsSchema.setResources(rSet);
				}

				rsTSkills = (ResourceRestriction) ResourceRestriction.createClassesRestriction("TechskillsInstances", "Techskills instances");

				{
					Set rSet = rsTSkills.getResources();
					rSet.add(resTechSkill);
					rsTSkills.setResources(rSet);
				}

				Rule ruSkillsSchema = new RuleImpl("TechskillsSchema",
					"Techskills schema");
				ruSkillsSchema.setRestriction(rsTSkillsSchema);
				ruSkillsSchema.setAddRight(true);
				ruSkillsSchema.setReadRight(true);
				ruSkillsSchema.setRemoveRight(true);

				Rule ruSkills = new RuleImpl("Techskills", "Techskills");
				ruSkills.setRestriction(rsTSkills);
				ruSkills.setAddRight(true);
				ruSkills.setReadRight(true);
				ruSkills.setRemoveRight(true);

				{
					Set rulz = uRD.getRules();
					rulz.add(ruSkills);
					rulz.add(ruSkillsSchema);
					uRD.setRules(rulz);
				}
			}

			//Employee
			URI resPerson = null;
			URI resHasPosition = null;
			URI resSubClassOf = null;
			URI resHasSkill = null;
			{
				// <iPerson, pHasPosition, _ >
				resPerson = new URIImpl(PERSON);
				ResourceRestriction rsPersons = (ResourceRestriction) Restriction.createClassesRestriction("Persons", "Persons");
				{
					Set rSet = rsPersons.getResources();
					rSet.add(resPerson);
					rsPersons.setResources(rSet);
				}

				resHasPosition = new URIImpl(HAS_POSITION);
				PropertiesRestriction rsHasPosition = (PropertiesRestriction) Restriction.createPropertiesRestriction("hasPosition", "hasPosition");
				{
					Set rSet = rsHasPosition.getProperties();
					rSet.add(resHasPosition);
					rsHasPosition.setProperties(rSet);
				}

				PatternRestriction rsPersonHasPosition = (PatternRestriction) ResourceRestriction.createPatternRestriction("PersonHasPosition", "PersonHasPosition");
				rsPersonHasPosition.addSubjectRestriction(rsPersons);
				rsPersonHasPosition.addPredicateRestriction(rsHasPosition);

				Rule ruPersonHasPosition = new RuleImpl(
					"PersonHasPositionRead", "PersonHasPositionRead");
				ruPersonHasPosition.setRestriction(rsPersonHasPosition);
				ruPersonHasPosition.setReadRight(true);

				//< scSkill, pSubClassOf, _ >
				resSubClassOf = new URIImpl(SUB_CLASS_OF);
				PropertiesRestriction rsSubClassOf = (PropertiesRestriction) Restriction.createPropertiesRestriction("subClassOf", "subClassOf");
				{
					Set props = rsSubClassOf.getProperties();
					props.add(resSubClassOf);
					rsSubClassOf.setProperties(props);
				}

				PatternRestriction rsSkillSubClassOf = (PatternRestriction) Restriction.createPatternRestriction("SkillSubClassOf", "rsSkillSubClassOf");
				{
					rsSkillSubClassOf.addSubjectRestriction(rsSkillsSchema);
					rsSkillSubClassOf.addPredicateRestriction(rsSubClassOf);
				}
				Rule ruSkillSubClassOf = new RuleImpl("SkillSubClassOfRead",
					"SkillSubClassOfRead");
				ruSkillSubClassOf.setRestriction(rsSkillSubClassOf);
				ruSkillSubClassOf.setReadRight(true);

				//< iPerson, pHasSkill, iTechnicalSkill >
				resHasSkill = new URIImpl(HAS_SKILL);
				PropertiesRestriction rsHasSkill = (PropertiesRestriction) Restriction.createPropertiesRestriction("HasSkill", "HasSkill");
				{
					Set props = rsHasSkill.getProperties();
					props.add(resHasSkill);
					rsHasSkill.setProperties(props);
				}
				PatternRestriction rsPersonHasTechSkill = (PatternRestriction) Restriction.createPatternRestriction("PersonHasTechSkill", "PersonHasTechSkill");
				{
					rsPersonHasTechSkill.addSubjectRestriction(rsPersons);
					rsPersonHasTechSkill.addPredicateRestriction(rsHasSkill);
					rsPersonHasTechSkill.addObjectRestriction(rsTSkills);
				}
				Rule ruPersonHasTechSkill = new RuleImpl(
					"PersonHasTechSkillRead", "PersonHasTechSkillRead");
				ruPersonHasTechSkill.setRestriction(rsPersonHasTechSkill);
				ruPersonHasTechSkill.setReadRight(true);

				Set rulz = uEmp.getRules();
				rulz.add(ruPersonHasPosition);
				rulz.add(ruSkillSubClassOf);
				rulz.add(ruPersonHasTechSkill);
				uEmp.setRules(rulz);

			}

			//QueryTester
			{
				QueryRestriction rsQuSchema = (QueryRestriction) Restriction.createQueryRestriction("QuSchema", "QuSchema");
				rsQuSchema.addQuery(HAS_POSITION_QUERY);

				Rule ruQuReadSchema = new RuleImpl("QuReadSchema",
					"QuReadSchema");
				ruQuReadSchema.setRestriction(rsQuSchema);
				ruQuReadSchema.setReadRight(true);

				Set rulz = uQue.getRules();
				rulz.add(ruQuReadSchema);
				uQue.setRules(rulz);
			}

			//Dimitar
			URI resMitac = new URIImpl(MITAC);
			{
				InstancesRestriction rsMitac = (InstancesRestriction) Restriction.createInstancesRestriction("Mitac", "Mitac");
				Set resz = rsMitac.getResources();
				resz.add(resMitac);
				rsMitac.setResources(resz);
				Rule ruMitac = new RuleImpl("MitacRead", "MitacRead");
				ruMitac.setReadRight(true);
				ruMitac.setRestriction(rsMitac);
				Set rulz = uMitac.getRules();
				rulz.add(ruMitac);
				uMitac.setRules(rulz);
			}

			//check all resources and store them if they do not exist
			{
				ArrayList rez = new ArrayList();
				rez.add(resHasPosition);
				rez.add(resHasSkill);
				rez.add(resPerson);
				rez.add(resSkill);
				rez.add(resSubClassOf);
				rez.add(resTechSkill);
				rez.add(resMitac);
				storeMissingResources(rez);
				loadNameSpaceTable(con, st);
				loadResourcesTable(con, st);
			}

		}
		catch (SQLException sqle) {
			throw new SailInternalException(sqle);
		}
		catch (Exception npe) {
			throw new SailInternalException(npe);
		}
	} // upload sample setup

	/** Uploads sample security setup data via direct sql queries */
	private void uploadSampleSetupDirectSQL() {
		Connection con = null;
		java.sql.Statement st = null;
		try {
			con = conPool.getConnection();
			st = con.createStatement();
			dropSecurityTables(con, st);
			createSecurityTables(con, st);

			Map resIdsByUri = new HashMap();

			/* load resources */
			ResultSet rset = st.executeQuery(" select r.id, concat(n.name, r.localname) "
				+ " from namespaces n, resources r "
				+ " where n.id = r.namespace ");
			while (rset.next()) {
				Integer id = new Integer(rset.getInt(1));
				String uri = rset.getString(2);
				resIdsByUri.put(uri, id);
			}

			/** uses these through out the uploading as last ids */
			int restrId = 0;
			int ruleId = 0;

			int schemaRestrId;
			int reposRestrId;

			int readSchemaRuleId;
			int doEverythingRuleId;

			/** schema read for all users */

			/* SCHEMA RESTR */
			st.executeUpdate(INSERT + RESTRICTIONS_TABLE + VALUES + "( "
				+ (schemaRestrId = ++restrId) + "," + Restriction.SCHEMA + ","
				+ "'Schema'" + "," + "'Schema Restriction'" + " );");
			/* REPOSITORY RESTR */
			st.executeUpdate(INSERT + RESTRICTIONS_TABLE + VALUES + "( "
				+ (reposRestrId = ++restrId) + "," + Restriction.REPOSITORY
				+ "," + "'Repository'" + "," + "'Repository Restriction'"
				+ " );");

			/* READ SCHEMA RULE */
			st.executeUpdate(INSERT + SECURITY_RULES_TABLE + VALUES + "( "
				+ (readSchemaRuleId = ++ruleId) + "," + "'ReadSchema'" + ","
				+ "'Allows reading of the Schema of the repository.'" + ","
				+ schemaRestrId + ",1, 0,0,0,0);");

			/* DO EVERYTHING WITH THE REPOSITROY RULE */
			st.executeUpdate(INSERT + SECURITY_RULES_TABLE + VALUES + "( "
				+ (doEverythingRuleId = ++ruleId) + "," + "'DoEverything'"
				+ "," + "'Allows all actions over the whole repository'" + ","
				+ reposRestrId + ",1,1,1,1,1);");

			/* READ the REPOSITROY RULE */
			int readRepositoryRuleId = 0;
			st.executeUpdate(INSERT + SECURITY_RULES_TABLE + VALUES + "( "
				+ (readRepositoryRuleId = ++ruleId) + "," + "'ReadRepository'"
				+ "," + "'Allows readind over the whole repository'" + ","
				+ reposRestrId + ",1,0,0,0,0);");

			/* ASSIGN DOEVERYTHING RULE TO ADMIN */
			st.executeUpdate(INSERT + USERS_RULES_TABLE + VALUES + "(" + 1
				+ "," + doEverythingRuleId + ");");

			/* ASSIGN READ SCHMEA RULE TO TESTUSER */
			st.executeUpdate(INSERT + USERS_RULES_TABLE + VALUES + "(" + 3
				+ "," + readSchemaRuleId + ");");

			/* read everything from the repository for HRManager and RandD */
			st.executeUpdate(INSERT + USERS_RULES_TABLE + VALUES + "(" + 4
				+ "," + readRepositoryRuleId + ");");
			st.executeUpdate(INSERT + USERS_RULES_TABLE + VALUES + "(" + 5
				+ "," + readRepositoryRuleId + ");");

			/*---HRManager Rules: allowed: editing of skills hier-----*/
			boolean skipSkillsRules = false;

			int skillId = 0;
			int skillRestrictionId = 0;
			int skillLevelId = 0;
			try {
				skillId = ((Integer) resIdsByUri.get(SKILL)).intValue();
				skillLevelId = ((Integer) resIdsByUri.get(SKILL_LEVEL)).intValue();
			}
			catch (NullPointerException npe) {
				skipSkillsRules = true;
			}
			if (!skipSkillsRules) {
				int hRCOSRestrId = 0;
				int hRSkills1Rule = 0;

				st.executeUpdate(INSERT + RESTRICTIONS_TABLE + VALUES + "("
					+ (hRCOSRestrId = ++restrId) + ","
					+ Restriction.CLASSES_OVER_SCHEMA + ","
					+ "'SkillsHierarchy','Skills Hierarchy');");

				// not so local
				skillRestrictionId = hRCOSRestrId;

				st.executeUpdate(INSERT
					+ SECURITY_RULES_TABLE
					+ VALUES
					+ "("
					+ (hRSkills1Rule = ++ruleId)
					+ ","
					+ "'EditSkillsHierarchy','Allows Editing of the Skill Hierarchy'"
					+ "," + hRCOSRestrId + ",1,1,1,0,0);");

				st.execute(INSERT + RES_PROP_RESTRS_TABLE + VALUES + "("
					+ hRCOSRestrId + "," + skillId + ");");

				int hRClassRestrId = 0;

				st.executeUpdate(INSERT + RESTRICTIONS_TABLE + VALUES + "("
					+ (hRClassRestrId = ++restrId) + "," + Restriction.CLASSES
					+ "," + "'SkillsInstances','Skills Instances');");

				st.execute(INSERT + RES_PROP_RESTRS_TABLE + VALUES + "("
					+ hRClassRestrId + "," + skillId + ");");

				st.execute(INSERT + RES_PROP_RESTRS_TABLE + VALUES + "("
					+ hRClassRestrId + "," + skillLevelId + ");");

				int hRSkills1Rule2 = 0;

				st.executeUpdate(INSERT
					+ SECURITY_RULES_TABLE
					+ VALUES
					+ "("
					+ (hRSkills1Rule2 = ++ruleId)
					+ ","
					+ "'EditSkillsInstances','Allows Editing of the Skill Instances'"
					+ "," + hRClassRestrId + ",1,1,1,0,0);");

				st.executeUpdate(INSERT + USERS_RULES_TABLE + VALUES + "(" + 4
					+ "," + hRSkills1Rule + ");");

				st.executeUpdate(INSERT + USERS_RULES_TABLE + VALUES + "(" + 4
					+ "," + hRSkills1Rule2 + ");");
			} // do not skipSkillsRules

			skipSkillsRules = false;
			/* RandDManager : allowed :editing of TechSkills */
			skillId = 0;
			try {
				skillId = ((Integer) resIdsByUri.get(TECH_SKILL)).intValue();
				skillLevelId = ((Integer) resIdsByUri.get(SKILL_LEVEL)).intValue();
			}
			catch (NullPointerException npe) {
				skipSkillsRules = true;
			}
			if (!skipSkillsRules) {
				int hrCOSRestrId = 0;
				int hRSkills1Rule = 0;

				st.executeUpdate(INSERT + RESTRICTIONS_TABLE + VALUES + "("
					+ (hrCOSRestrId = ++restrId) + ","
					+ Restriction.CLASSES_OVER_SCHEMA + ","
					+ "'TechSkillsHierarchy','Technical Skills Hierarchy');");

				st.executeUpdate(INSERT
					+ SECURITY_RULES_TABLE
					+ VALUES
					+ "("
					+ (hRSkills1Rule = ++ruleId)
					+ ","
					+ "'EditTechSkillsHierarchy','Allows Editing of the Tech Skill Hierarchy'"
					+ "," + hrCOSRestrId + ",1,1,1,0,0);");

				st.execute(INSERT + RES_PROP_RESTRS_TABLE + VALUES + "("
					+ hrCOSRestrId + "," + skillId + ");");

				int hRClassRestrId = 0;

				st.executeUpdate(INSERT + RESTRICTIONS_TABLE + VALUES + "("
					+ (hRClassRestrId = ++restrId) + "," + Restriction.CLASSES
					+ ","
					+ "'TechSkillsInstances','Technical Skills Instances');");

				st.execute(INSERT + RES_PROP_RESTRS_TABLE + VALUES + "("
					+ hRClassRestrId + "," + skillId + ");");
				st.execute(INSERT + RES_PROP_RESTRS_TABLE + VALUES + "("
					+ hRClassRestrId + "," + skillLevelId + ");");

				int hRSkills1Rule2 = 0;

				st.executeUpdate(INSERT
					+ SECURITY_RULES_TABLE
					+ VALUES
					+ "("
					+ (hRSkills1Rule2 = ++ruleId)
					+ ","
					+ "'EditSkillsInstances','Allows Editing of the Skill Instances'"
					+ "," + hRClassRestrId + ",1,1,1,0,0);");

				st.executeUpdate(INSERT + USERS_RULES_TABLE + VALUES + "(" + 5
					+ "," + hRSkills1Rule + ");");

				st.executeUpdate(INSERT + USERS_RULES_TABLE + VALUES + "(" + 5
					+ "," + hRSkills1Rule2 + ");");
			} // do not skipSkillsRules

			int techSkillId = 0;
			skillId = 0;
			int personId = 0;
			int subClassOfId = 0;
			int hasPositionId = 0;
			int hasSkillId = 0;
			boolean skip = false;
			try {
				skillId = ((Integer) resIdsByUri.get(SKILL)).intValue();
				techSkillId = ((Integer) resIdsByUri.get(TECH_SKILL)).intValue();
				personId = ((Integer) resIdsByUri.get(PERSON)).intValue();
				subClassOfId = ((Integer) resIdsByUri.get(SUB_CLASS_OF)).intValue();
				hasSkillId = ((Integer) resIdsByUri.get(HAS_SKILL)).intValue();
				hasPositionId = ((Integer) resIdsByUri.get(HAS_POSITION)).intValue();
			}
			catch (NullPointerException npe) {
				skip = true;
			}
			if (!skip) {

				int personRstr = 0;
				st.executeUpdate(INSERT + RESTRICTIONS_TABLE + VALUES + "("
					+ (personRstr = ++restrId) + "," + Restriction.CLASSES
					+ "," + "'Persons','Persons Restriction');");

				st.execute(INSERT + RES_PROP_RESTRS_TABLE + VALUES + "("
					+ personRstr + "," + personId + ");");

				int hasPositionRestr = 0;
				st.executeUpdate(INSERT + RESTRICTIONS_TABLE + VALUES + "("
					+ (hasPositionRestr = ++restrId) + ","
					+ Restriction.PROPERTIES + ","
					+ "'hasPosition','hasPosition Restriction');");
				st.execute(INSERT + RES_PROP_RESTRS_TABLE + VALUES + "("
					+ hasPositionRestr + "," + hasPositionId + ");");

				int subClassRestr = 0;
				st.executeUpdate(INSERT + RESTRICTIONS_TABLE + VALUES + "("
					+ (subClassRestr = ++restrId) + ","
					+ Restriction.PROPERTIES + ","
					+ "'SubClassOf','SubClassOf Restriction');");
				st.execute(INSERT + RES_PROP_RESTRS_TABLE + VALUES + "("
					+ subClassRestr + "," + subClassOfId + ");");

				int hasSkillRestr = 0;
				st.executeUpdate(INSERT + RESTRICTIONS_TABLE + VALUES + "("
					+ (hasSkillRestr = ++restrId) + ","
					+ Restriction.PROPERTIES + ","
					+ "'hasSkill','hasSkill Restriction');");
				st.execute(INSERT + RES_PROP_RESTRS_TABLE + VALUES + "("
					+ hasSkillRestr + "," + hasSkillId + ");");

				int techSkillsRestr = 0;
				st.executeUpdate(INSERT + RESTRICTIONS_TABLE + VALUES + "("
					+ (techSkillsRestr = ++restrId) + "," + Restriction.CLASSES
					+ "," + "'TechSkillInstances','TechSkillInstances');");
				st.execute(INSERT + RES_PROP_RESTRS_TABLE + VALUES + "("
					+ techSkillsRestr + "," + techSkillId + ");");

				// pattern restrictions
				int ptrnPersonPositionsId = 0;
				st.executeUpdate(INSERT + RESTRICTIONS_TABLE + VALUES + "("
					+ (ptrnPersonPositionsId = ++restrId) + ","
					+ Restriction.PATTERN + ","
					+ "'Person Positions','Person Positions Restriction');");

				st.executeUpdate(INSERT + PATTERN_RESTRS_TABLE + VALUES + "("
					+ ptrnPersonPositionsId + "," + personRstr + ","
					+ PatternRestriction.SPO_SUBJECT + ");");

				st.executeUpdate(INSERT + PATTERN_RESTRS_TABLE + VALUES + "("
					+ ptrnPersonPositionsId + "," + hasPositionRestr + ","
					+ PatternRestriction.SPO_PREDICATE + ");");

				int personPositionsRule = 0;
				st.executeUpdate(INSERT
					+ SECURITY_RULES_TABLE
					+ VALUES
					+ "("
					+ (personPositionsRule = ++ruleId)
					+ ","
					+ "'Read Person Positions Rule','Read Person Positions Rule'"
					+ "," + ptrnPersonPositionsId + ",1,0,0,0,0);");

				int ptrnSkillsHierarchyRstr = 0;
				st.executeUpdate(INSERT
					+ RESTRICTIONS_TABLE
					+ VALUES
					+ "("
					+ (ptrnSkillsHierarchyRstr = ++restrId)
					+ ","
					+ Restriction.PATTERN
					+ ","
					+ "'Skills Hierarchy Pattern','Skills Hierarchy Pattern Restriction');");

				st.executeUpdate(INSERT + PATTERN_RESTRS_TABLE + VALUES + "("
					+ ptrnSkillsHierarchyRstr + "," + skillRestrictionId + ","
					+ PatternRestriction.SPO_SUBJECT + ");");

				st.executeUpdate(INSERT + PATTERN_RESTRS_TABLE + VALUES + "("
					+ ptrnSkillsHierarchyRstr + "," + subClassRestr + ","
					+ PatternRestriction.SPO_PREDICATE + ");");

				int readSkillsHierarchyRule = 0;
				st.executeUpdate(INSERT
					+ SECURITY_RULES_TABLE
					+ VALUES
					+ "("
					+ (readSkillsHierarchyRule = ++ruleId)
					+ ","
					+ "'Read Skills Hierarchy Rule','Read Skills Hierarchy Rule'"
					+ "," + ptrnSkillsHierarchyRstr + ",1,0,0,0,0);");

				int ptrnTechSkillInstancesRstr = 0;
				st.executeUpdate(INSERT
					+ RESTRICTIONS_TABLE
					+ VALUES
					+ "("
					+ (ptrnTechSkillInstancesRstr = ++restrId)
					+ ","
					+ Restriction.PATTERN
					+ ","
					+ "'Technical Skill Instances Pattern','Technical Skill Instances Pattern Restriction');");

				st.executeUpdate(INSERT + PATTERN_RESTRS_TABLE + VALUES + "("
					+ ptrnTechSkillInstancesRstr + "," + personRstr + ","
					+ PatternRestriction.SPO_SUBJECT + ");");

				st.executeUpdate(INSERT + PATTERN_RESTRS_TABLE + VALUES + "("
					+ ptrnTechSkillInstancesRstr + "," + hasSkillRestr + ","
					+ PatternRestriction.SPO_PREDICATE + ");");

				st.executeUpdate(INSERT + PATTERN_RESTRS_TABLE + VALUES + "("
					+ ptrnTechSkillInstancesRstr + "," + techSkillsRestr + ","
					+ PatternRestriction.SPO_OBJECT + ");");

				int readTechSkillsRule = 0;
				st.executeUpdate(INSERT
					+ SECURITY_RULES_TABLE
					+ VALUES
					+ "("
					+ (readTechSkillsRule = ++ruleId)
					+ ","
					+ "'Read Tech Skills Hierarchy Rule','Read Skills Hierarchy Rule'"
					+ "," + ptrnTechSkillInstancesRstr + ",1,0,0,0,0);");

				//assign rules to employee
				st.executeUpdate(INSERT + USERS_RULES_TABLE + VALUES + "(" + 6
					+ "," + personPositionsRule + ");");

				st.executeUpdate(INSERT + USERS_RULES_TABLE + VALUES + "(" + 6
					+ "," + readSkillsHierarchyRule + ");");

				st.executeUpdate(INSERT + USERS_RULES_TABLE + VALUES + "(" + 6
					+ "," + readTechSkillsRule + ");");

			} // employee rules

			ThreadLog.trace("Sample Security Setup UPLOADED.");
		}
		catch (SQLException sqle) {
			throw new SailInternalException(sqle);
		}
		catch (Exception npe) {
			throw new SailInternalException(npe);
		} //catch
		finally {
			try {
				st.close();
				con.close();
			}
			catch (SQLException e) {
				throw new SailInternalException(e);
			}
		} // finally
	} // UploadSampleSecuritySetup()

	/**
	 * Stores the security setup cia update sql queries to the repository.
	 */
	private void storeSecuritySetup() {
		Connection con = null;
		java.sql.Statement st = null;
		try {
			ThreadLog.trace("Store Security Setup ...");
			con = conPool.getConnection();
			st = con.createStatement();

			/* load resources */
			ResultSet rset = st.executeQuery(" select r.id, concat(n.name, r.localname) "
				+ " from namespaces n, resources r "
				+ " where n.id = r.namespace ");
			while (rset.next()) {
				Integer id = new Integer(rset.getInt(1));
				String uri = rset.getString(2);
				resIdsByUri.put(uri, id);
			}

			dropSecurityTables(con, st);
			createSecurityTables(con, st);

			ArrayList userz = new ArrayList(users);
			Set queSet = new HashSet();
			ArrayList queList = new ArrayList();
			ArrayList rolez = new ArrayList(roles);
			Set rulez = new HashSet(RoleImpl.getRules(roles, false));

			for (int ui = 0; ui < userz.size(); ui++) {
				User usr = (User) userz.get(ui);
				queSet.addAll(usr.toSql());
				rulez.addAll(usr.getRules());
			} // for userz

			ArrayList rolz = new ArrayList(rolez);
			for (int roi = 0; roi < rolz.size(); roi++) {
				Role role = (Role) rolz.get(roi);
				rulez.addAll(role.getRules(true));
				queSet.addAll(role.toSql());
			} // for rolz

			ArrayList rulz = new ArrayList(rulez);
			for (int rui = 0; rui < rulz.size(); rui++) {
				Rule rule = (Rule) rulz.get(rui);
				queSet.addAll(rule.toSql());
			} // for rulz

			ArrayList restrz = new ArrayList(Restriction.getRestrictions());
			for (int ri = 0; ri < restrz.size(); ri++) {
				Restriction r = (Restriction) restrz.get(ri);
				queSet.addAll(r.toSql(idByLiteral, idByRes));
			} // for restrictions

			// execute queries
			queList = new ArrayList(queSet);
			for (int qi = 0; qi < queList.size(); qi++) {
				String q = queList.get(qi).toString();
				int result;
				try {
					result = st.executeUpdate(q);
				}
				catch (Exception x) {
					result = st.executeUpdate(q);
				}
				if (result <= 0)
					throw new SecurityException(
						"The following query did not change the repository :\n"
							+ q);
			} // for queries
			ThreadLog.trace("Security Setup STORED.");

		}
		catch (SQLException sqle) {
			throw new SailInternalException(sqle);
		}
		catch (SecurityException se) {
			throw new SailInternalException(se);
		}
		catch (Exception npe) {
			throw new SailInternalException(npe);
		}
		finally {
			try {
				st.close();
				con.close();
			}
			catch (SQLException e) {
				throw new SailInternalException(e);
			}
		} // finally

	} // storeSecuritySetup()

	/** stores the missing resources from the security setup */
	private void storeMissingResources(ArrayList resourcez) {
		try {
			baseRdfRepository.startTransaction();
			URI resType = new URIImpl(RDF_TYPE);
			URI resResource = new URIImpl(RESOURCE);

			for (int i = 0; i < resourcez.size(); i++) {
				Resource res = (Resource) resourcez.get(i);

				if (idByRes.get(res) == null) {
					baseRdfRepository.addStatement(res, resType, resResource);
				}

			}
			baseRdfRepository.commitTransaction();
		}
		catch (SailUpdateException e) {
			throw new SailInternalException(e);
		}

	}

	/**
	 * Answers the question : is this resource accessible through this query restriction.
	 * @param rst
	 * @param res
	 * @return
	 * @todo : make it private
	 * @todo implement it to work both for literals(instances restr) and resources
	 */
	boolean isQuAccessible(QueryRestriction rst, Resource res) {
		boolean is = false;
		String query;
		ArrayList qus = new ArrayList(rst.getQueries());
		for (int i = 0; i < qus.size(); i++) {
			query = qus.get(i).toString();
			queryResult = (ArrayList) queryResults.get(query);
			if (queryResult == null) {
				// execute query
				try {
					org.openrdf.sesame.query.rql.RqlEngine queryEngine = new org.openrdf.sesame.query.rql.RqlEngine(
						baseRdfSchemaSource);

					org.openrdf.sesame.query.rql.model.Query queryModel = null;

					queryModel = queryEngine.parseQuery(query);

					queryEngine.evaluateQuery(queryModel, this);
				}
				catch (MalformedQueryException me) {
					// Incorrect query
					throw new SailInternalException(me);
				}
				catch (org.openrdf.sesame.query.QueryEvaluationException qee) {
					// Error during evaluation
					throw new SailInternalException(qee);
				}
				catch (IOException ioe) {
					throw new SailInternalException(ioe);
				}

				while (!isQueryReady) {
					int k = 0;
				}

				//				for (int j = 0 ; j < queryResult.size() ; j++){
				//					System.out.println(queryResult.get(j).toString());
				//				}

				queryResults.put(query, queryResult);
			} // if null query result

			for (int ri = 0; ri < queryResult.size(); ri++) {
				Value v = (Value) queryResult.get(ri);
				URI rr;
				Literal lit;
				if (v instanceof URI) {
					rr = (URI) v;
					// FIXME cast to URI appropriate?
					if (((URI) res).getLocalName().equals(rr.getLocalName())
						&& ((URI) res).getNamespace().equals(rr.getNamespace())) {
						is = true;
						break;
					}
				}
			}
			if (is)
				break;
		}

		return is;
	}
	
	/* (non-Javadoc)
	 * @see org.openrdf.sesame.sail.RdfRepository#addListener(org.openrdf.sesame.sail.SailChangedListener)
	 */
	public void addListener(SailChangedListener listener) {
		baseRdfRepository.addListener(listener);
	}

	/* (non-Javadoc)
	 * @see org.openrdf.sesame.sail.RdfRepository#removeListener(org.openrdf.sesame.sail.SailChangedListener)
	 */
	public void removeListener(SailChangedListener listener) {
		baseRdfRepository.removeListener(listener);
	}


	/** main method for test purposes */
	/*
	 * public static void main(String[] args) {
	 *  // sesame startup initialization via a config file and security setup
	 * test //------------------------------ try {
	 * SesameStartup.initialize("system.conf");
	 * RequestRouter.login("Dimitar","mitac"); Sail sail =
	 * SystemConfigCenter.getSail("mysql-omm","Dimitar","mitac",true,false);
	 * SecurityServices security = (SecuritySail) QuerySailStack
	 * .queryInterface(sail,"org.openrdf.sesame.omm.SecurityServices");
	 * 
	 * System.out.println(SessionContext.getContext().userID);
	 * System.out.println("Statements"); StatementIterator sti
	 * =((RdfSchemaRepository)security).getStatements(null,null,null); while
	 * (sti.hasNext()) { System.out.println(sti.next()); }
	 * 
	 *  } catch (Exception x) { x.printStackTrace(); }
	 * 
	 * 
	 * 
	 *  // ---------------------- // initialize sails and test them
	 *  // VersioningTmsMySQLSail verSail= new VersioningTmsMySQLSail(); // //
	 * SecuritySail securitySail = new SecuritySail(); //
	 * //securitySail.setBaseSail(verSail); // // HashMap map = new HashMap(); //
	 * map.put("jdbcDriver", "org.gjt.mm.mysql.Driver"); // map.put("jdbcUrl",
	 * "jdbc:mysql://localhost:3306/ommdemo"); // map.put("user", "sesame"); //
	 * map.put("password", "sesame"); // map.put("security_setup",
	 * "http://www.ontotext.com/otk/2002/05/SecurityDemo/DemoSecuritySetup.rdf"); // //
	 * try { // SessionContext context = new SessionContext(); // context.user =
	 * "admin"; // context.pass = "admin"; // context.userID=1; //
	 * context.baseUrl =
	 * "http://www.ontotext.com/otk/2002/05/SecurityDemo/newSwingSkill.rdf"; //
	 * context.repository = "mysql-omm"; // SessionContext.put("",context); //
	 * SessionContext.setContext(context); // // verSail.initialize(map); // // }
	 * catch (Exception e) { // e.printStackTrace(System.out); //
	 * System.out.println(e.getMessage()); // } // verSail.shutDown(); }
	 */
} // class SecuritySail
