/*  OMM - Ontology Middleware Module
 *  Copyright (C) 2002 OntoText Lab, Sirma AI OOD
 *
 *  Contact:
 *	Sirma AI OOD, OntoText Lab.
 *	38A, Christo Botev Blvd.
 *  1000 Sofia, Bulgaria
 *	tel. +359(2)981 00 18
 *	fax. +359(2)981 90 58
 *	info@ontotext.com
 *
 * 	http://www.ontotext.com/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package org.openrdf.sesame.sailimpl.omm.security;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
/**Restriction.java
 * <p>Title: Knowledge Control System</p>
 * <p> </p>
 * <p> </p>
 * <p>Company: OntoText Lab. Sirma AI. </p>
 * @author  damyan ognyanoff, borislav popov
 * @version 1.0
 * This is the base class for all kinds of restrictions supported in the
 * repository
 */
public abstract class Restriction {

  /** set of all the restrictions */
  private static Set restrictions = new HashSet();

  /**REPOSITORY Restriction Type*/
  public static final int REPOSITORY = 1;
  /**SCHEMA Restriction Type*/
  public static final int SCHEMA = 2;
  /**CLASSES Restriction Type*/
  public static final int CLASSES = 3;
  /**INSTANCES Restriction Type*/
  public static final int INSTANCES = 4;
  /**PROPERTIES Restriction Type*/
  public static final int PROPERTIES = 5;
  /**PATTERN Restriction Type*/
  public static final int PATTERN = 6;
  /**QUERY Restriction Type*/
  public static final int QUERY = 7;
  /**CLASSES Over Schema Restriction Type*/
  public static final int CLASSES_OVER_SCHEMA = 8;


  /**REPOSITORY Restriction name*/
  public static final String REPOSITORY_RESTRICTION = "RepositoryRestriction";
  /**SCHEMA Restriction name*/
  public static final String SCHEMA_RESTRICTION = "SchemaRestriction";
  /**CLASSES Restriction name*/
  public static final String CLASSES_RESTRICTION = "ClassesRestriction";
  /**INSTANCES Restriction name*/
  public static final String INSTANCES_RESTRICTION = "InstancesRestriction";
  /**PROPERTIES Restriction name*/
  public static final String PROPERTIES_RESTRICTION = "PropertiesRestriction";
  /**PATTERN Restriction name*/
  public static final String PATTERN_RESTRICTION = "PatternRestriction";
  /**QUERY Restriction name*/
  public static final String QUERY_RESTRICTION = "QueryRestriction";
  /**CLASSES Over Schema Restriction name*/
  public static final String CLASSES_OVER_SCHEMA_RESTRICTION
    = "ClassesOverSchemaRestriction";

  /** the lastId of all types of restrictions */
  private static int lastId = 0;

  /** the type of the restriction */
  int type;

  /** the id of the restriction */
  int id;

  /** the name of the restriction */
  String name = null;

  /** the descritpion of the restriction */
  String description = null;

	/** the URI */
	private String uri = null;

  /** construction of a restriction
   * @param id id of the restriction
   * @param kind restriction type
   * @param name description
   * @param desc description
   */
  protected Restriction(int id, int kind,String name, String desc) {
    this.id = id;
    type = kind;
    this.name = name;
    description = desc;
    restrictions.add(this);
  }//CONsTRUCTion

  /**
   * Constructs a restriction given its type.
   * @param type the type of the restriction
   */
  protected Restriction(int type) {
    this.type = type;
    this.id = ++lastId;
  }

  /**
   * Creates a restriction given only its type. Other members should be set explicitly.
   * @param type the type of the restriction
   * @return the created restriction if a valid type was specified, otherwise - null.
   */
  public static Restriction createRestriction(int type) {
    Restriction restr = null;
    switch ( type ) {
      case (1) : restr = new RepositoryRestriction(++lastId,"",""); break;
      case (2) : restr = new SchemaRestriction(++lastId,"","");break;
      case (3) : restr = new ClassesRestriction(++lastId,"","");break;
      case (4) : restr = new InstancesRestriction(++lastId,"","");break;
      case (5) : restr = new PropertiesRestriction(++lastId,"","");break;
      case (6) : restr = new PatternRestriction(++lastId,"","");break;
      case (7) : restr = new QueryRestriction(++lastId,"","");break;
      case (8) : restr = new ClassesOverSchemaRestriction(++lastId,"","");break;
    }
    return restr;
  }

  /**
   * Creates a restriction given its type and id. Other members should be set explicitly.
   * @param type the type of the restriction
   * @param id the id of the restriction
   * @return the created restriction if a valid type was specified, otherwise - null.
   */
  public static Restriction createRestriction(int type, int id) {
    Restriction restr = null;
    switch ( type ) {
      case (1) : restr = new RepositoryRestriction(id,"",""); break;
      case (2) : restr = new SchemaRestriction(id,"","");break;
      case (3) : restr = new ClassesRestriction(id,"","");break;
      case (4) : restr = new InstancesRestriction(id,"","");break;
      case (5) : restr = new PropertiesRestriction(id,"","");break;
      case (6) : restr = new PatternRestriction(id,"","");break;
      case (7) : restr = new QueryRestriction(id,"","");break;
      case (8) : restr = new ClassesOverSchemaRestriction(id,"","");break;
    }
    if ( id > lastId) lastId = id;
    return restr;
  }

  /**
   * Create a Repository Restriction.
   * @param id id of the restriction.
   * @param name name of the restriction.
   * @param descr description of the restriction.
   * @return the created restriction
   */
  static public Restriction createRepositoryRestriction(int id, String name, String descr)
    {
      if ( id > lastId )
       lastId = id;
      return new RepositoryRestriction(id,name,descr);
    }

  /**
   * Create a Repository Restriction.
   * @param name name of the restriction.
   * @param descr description of the restriction.
   * @return the created restriction
   */
  static public Restriction createRepositoryRestriction( String name, String descr)
    {
      return new RepositoryRestriction(++lastId,name,descr);
    }

  /**
   * Create a Schema Restriction.
   * @param id id of the restriction.
   * @param name name of the restriction.
   * @param descr description of the restriction.
   * @return the created restriction
   */
  static public Restriction createSchemaRestriction(int id, String name, String descr)
    {
      if ( id > lastId )
        lastId = id;

      return new SchemaRestriction(id,name,descr);
    }

  /**
   * Create a Schema Restriction.
   * @param name name of the restriction.
   * @param descr description of the restriction.
   * @return the created restriction
   */
  static public Restriction createSchemaRestriction( String name, String descr)
    {
      return new SchemaRestriction(++lastId,name,descr);
    }

  /**
   * Create a Classes Restriction.
   * @param id id of the restriction.
   * @param name name of the restriction.
   * @param descr description of the restriction.
   * @return the created restriction
   */
  static public Restriction createClassesRestriction(int id, String name, String descr)
    {
      if ( id > lastId )
        lastId = id;
      return new ClassesRestriction(id,name,descr);
    }

  /**
   * Create a Classes Restriction.
   * @param name name of the restriction.
   * @param descr description of the restriction.
   * @return the created restriction
   */
  static public Restriction createClassesRestriction(String name, String descr)
    {
      return new ClassesRestriction(++lastId,name,descr);
    }

  /**
   * Create a Classes Over Schema Restriction.
   * @param id id of the restriction.
   * @param name name of the restriction.
   * @param descr description of the restriction.
   * @return the created restriction
   */
  static public Restriction createClassesOverSchemaRestriction(
    int id, String name, String descr)
    {
      if ( id > lastId)
        lastId = id;
      return new ClassesOverSchemaRestriction(id,name,descr);
    }

  /**
   * Create a Classes Over Schema Restriction.
   * @param name name of the restriction.
   * @param descr description of the restriction.
   * @return the created restriction
   */
  static public Restriction createClassesOverSchemaRestriction(
    String name, String descr)
    {
      return new ClassesOverSchemaRestriction(
        ++lastId,name,descr);
    }

  /**
   * Create an Instances Restriction.
   * @param id id of the restriction.
   * @param name name of the restriction.
   * @param descr description of the restriction.
   * @return the created restriction
   */
  static public Restriction createInstancesRestriction(int id, String name, String descr)
    {
      if ( id > lastId)
        lastId = id;
      return new InstancesRestriction(id,name,descr);
    }

  /**
   * Create an Instances Restriction.
   * @param name name of the restriction.
   * @param descr description of the restriction.
   * @return the created restriction
   */
  static public Restriction createInstancesRestriction(String name, String descr)
    {
      return new InstancesRestriction(++lastId,name,descr);
    }


  /**
   * Create a Properties Restriction.
   * @param id id of the restriction.
   * @param name name of the restriction.
   * @param descr description of the restriction.
   * @return the created restriction
   */
  static public Restriction createPropertiesRestriction(int id, String name, String descr)
    {
      if ( id > lastId )
        lastId = id;

      return new PropertiesRestriction(id,name,descr);}

  /**
   * Create a Properties Restriction.
   * @param name name of the restriction.
   * @param descr description of the restriction.
   * @return the created restriction
   */
  static public Restriction createPropertiesRestriction(String name, String descr)
    {

      return new PropertiesRestriction(++lastId,name,descr);
    }


  /**
   * Create a Pattern Restriction.
   * @param id id of the restriction.
   * @param name name of the restriction.
   * @param descr description of the restriction.
   * @return the created restriction
   */
  static public Restriction createPatternRestriction(int id, String name, String descr)
    {
      if ( id > lastId )
        lastId = id;
      return new PatternRestriction(id,name,descr);
    }

  /**
   * Create a Pattern Restriction.
   * @param name name of the restriction.
   * @param descr description of the restriction.
   * @return the created restriction
   */
  static public Restriction createPatternRestriction( String name, String descr)
    {
      return new PatternRestriction(++lastId,name,descr);
    }


 /**
   * Create a Query Restriction.
   * @param id id of the restriction.
   * @param name name of the restriction.
   * @param descr description of the restriction.
   * @return the created restriction
   */
  static public Restriction createQueryRestriction(int id, String name, String descr){
    if ( id > lastId )
      lastId = id;

    return new QueryRestriction(id,name,descr);
  }

 /**
   * Create a Query Restriction.
   * @param name name of the restriction.
   * @param descr description of the restriction.
   * @return the created restriction
   */
  static public Restriction createQueryRestriction(String name, String descr){

    return new QueryRestriction(++lastId,name,descr);
  }

  /**
   * Retrieve the set of all restrictions.
   * @return the set of all restrictions.
   */
  public static Set getRestrictions(){
    return restrictions;
  }


  /**
   * Gets the string representation of the type of the restriction. e.g. 1 is Repository.
   * @param type the type to be converted
   * @return the string representing the converted restriction type
   */
  public static String type2String(int type) {
    String result = ""+type;
    switch ( type ) {
      case (1) : result = "Repository"; break;
      case (2) : result = "Schema"; break;
      case (3) : result = "Classes"; break;
      case (4) : result = "Instances"; break;
      case (5) : result = "Properties"; break;
      case (6) : result = "Pattern"; break;
      case (7) : result = "Query"; break;
      case (8) : result = "ClassesOverSchema"; break;
    }
    return result+"Restriction";
  }

  /**
   * Converts the type to its integer value.
   * @param type the String representation of the Restriction type
   * @return 0 is returned if unknown type,
   * otherwise the int associated with the string type is returned.
   */
  public static int type2Int(String type){
    int result = 0;
    if (type.equals("RepositoryRestriction")) result =1 ;
    if (type.equals("SchemaRestriction")) result =2 ;
    if (type.equals("ClassesRestriction")) result =3 ;
    if (type.equals("InstancesRestriction")) result =4 ;
    if (type.equals("PropertiesRestriction")) result =5 ;
    if (type.equals("PatternRestriction")) result =6 ;
    if (type.equals("QueryRestriction")) result =7 ;
    if (type.equals("ClassesOverSchemaRestriction")) result =8 ;

    return result;
  }


  /**
   * Gets the restriction id.
   * @return the id of the restriction
   */
  public int getId() {
    return id;
  }

  /**
   * Sets restriction's id.
   * @param id the id to be set.
   */
  public void setId(int id) {
    this.id = id;
  }

  /**
   * Gets the restriction's description.
   * @return the restriction's description
   */
  public String getDescription() {
    return description;
  }

  /**
   * Sets the restriction's description.
   * @param descr the restriction's description
   */
  public void setDescription(String descr) {
    description = descr;
  }

  /**
   * Gets the restriciton's name.
   * @return the restriciton's name
   */
  public String getName() {
    return name;
  }

  /**
   * Sets the restriciton's name.
   * @param name the name to be set
   */
  public void setName(String name) {
    this.name = name;
  }

  /**
   * Gets restriction type.
   * @return type of the restriction
   */
  public int getType() {
    return type;
  }

  /**
   * Sets the type of the restriction.
   * @param type restriction type
   */
  public void setType(int type) {
    this.type = type;
  }

  /** Creates update queries over the restrictions, res_prop_restrs, query_restrs,
   *  pattern_restrs tables.
   * @return A list of update SQL queries, to be used to mirror the objects'
   *  properties from memory to an SQL repository.
   */
  public ArrayList toSql(Map idByLiteral, Map idByRes) throws NullParameterException,SecurityException  {
    if (idByRes == null )
      throw new NullParameterException("[ids by resources] map should not be [null].");
    if (idByLiteral == null )
      throw new NullParameterException("[ids by literals] map should not be [null].");

    ArrayList list = new ArrayList(3);
    StringBuffer query = new StringBuffer();
    query.append(SecuritySail.INSERT);
    query.append(SecuritySail.RESTRICTIONS_TABLE);
    query.append(SecuritySail.VALUES);
    query.append("(").append(id);
    query.append(",");
    query.append(type);
    query.append(",'");
    query.append(name);
    query.append("','");
    query.append(description).append("');");

    list.add(query.toString());

    return list;
  }// toSql()


	public void setUri(String uri) {
		this.uri = uri;
	}

	public String getUri() {
		return uri;
	}


} // RestrictionType class
