/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sailimpl.nativerdf.datastore;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;

import org.openrdf.util.log.ThreadLog;

/**
 * Utility class for transfering bytes to/from file channels. The utility
 * normally will use the <tt>transferTo()</tt> and <tt>transferFrom()</tt>
 * methods that are available in class FileChannel, but will fall back to more
 * conventional means on platforms on which this doesn't work (JDK 1.4.x with
 * Linux kernel 2.6.x) as a result of
 * <a href="http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=5056395">bug 5056395</a>.
 *
 * @author Arjohn Kampman
 * @version $Revision: 1.3.4.2 $
 **/
public class TransferUtil {
	
	private static boolean supportsTransfer = true;

	private static ByteBuffer _byteBuffer = null;

	static {
		String os_name = System.getProperty("os.name");
		String os_version = System.getProperty("os.version");
		String java_version = System.getProperty("java.version");

		if (os_name != null && os_name.equalsIgnoreCase("linux") &&
			os_version != null && os_version.startsWith("2.6.") &&
			java_version != null && java_version.startsWith("1.4."))
		{
			supportsTransfer = false;
			_byteBuffer = ByteBuffer.allocateDirect(16 * 1024);
			ThreadLog.warning("Transfers on file channels not supported on this platform, falling back to possibly slower, more conventional means");
			System.err.println("Transfers on file channels not supported on this platform, falling back to possibly slower, more conventional means");
		}
	}

	public static long transferTo(FileChannel sourceChannel, long startPos, long count, WritableByteChannel targetChannel)
		throws IOException
	{
		if (supportsTransfer) {
			return sourceChannel.transferTo(startPos, count, targetChannel);
		}
		else {
			long maxPos = startPos + count;
			long curPos = startPos;

			synchronized (_byteBuffer) {
				while (curPos < maxPos) {
					if (maxPos - curPos < _byteBuffer.capacity()) {
						_byteBuffer.limit((int) (maxPos - curPos));
					}

					int c = sourceChannel.read(_byteBuffer, curPos);

					if (c < 0) {
						// End of stream reached
						break;
					}

					_byteBuffer.flip();
					targetChannel.write(_byteBuffer);
					_byteBuffer.clear();

					curPos += c;
				}
			}

			return curPos - startPos;
		}
	}

	public static long transferFrom(ReadableByteChannel sourceChannel, long startPos, long count, FileChannel targetChannel)
		throws IOException
	{
		if (supportsTransfer) {
			return targetChannel.transferFrom(sourceChannel, startPos, count);
		}
		else {
			long maxPos = startPos + count;
			long curPos = startPos;

			synchronized (_byteBuffer) {
				while (curPos < maxPos) {
					if (maxPos - curPos < _byteBuffer.capacity()) {
						_byteBuffer.limit((int) (maxPos - curPos));
					}

					int c = sourceChannel.read(_byteBuffer);

					if (c < 0) {
						// End of stream reached
						break;
					}

					_byteBuffer.flip();
					targetChannel.write(_byteBuffer, curPos);
					_byteBuffer.clear();

					curPos += c;
				}
			}

			return curPos - startPos;
		}
	}
}
