/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sailimpl.nativerdf.datastore;

import java.io.File;
import java.io.IOException;
import java.util.zip.CRC32;

import org.openrdf.util.ByteArrayUtil;

/**
 * Class that provides indexed storage and retrieval of RDF values.
 **/
public class DataStore {

/*-------------+
| Constants    |
+-------------*/

	private static final int COUNT_IDX = 0;
	private static final int ID_IDX = 4;
	private static final int DATA_IDX = 8;

/*-------------+
| Variables    |
+-------------*/

	private DataFile _dataFile;
	private IDFile _idFile;
	private HashFile _hashFile;

	private CRC32 _crc32 = new CRC32();

/*-------------+
| Constructors |
+-------------*/

	public DataStore(File dataDir, String filePrefix)
		throws IOException
	{
		_dataFile = new DataFile(new File(dataDir, filePrefix + ".dat"));
		_idFile = new IDFile(new File(dataDir, filePrefix + ".id"));
		_hashFile = new HashFile(new File(dataDir, filePrefix + ".hash"));
	}

/*----------+
| Methods   |
+----------*/

	/**
	 * Gets the value for the specified ID.
	 *
	 * @param id A value ID.
	 * @return The value for the ID, or <tt>null</tt> no such value could be
	 * found.
	 * @exception IOException If an I/O error occurred.
	 **/
	public byte[] getData(int id)
		throws IOException
	{
		return getData(id, false);
	}

	/**
	 * Gets the value for the specified ID, optionally searching the
	 * non-commited data.
	 *
	 * @param id A value ID.
	 * @param dirtyReads Flag indicating whether the non-commited data should
	 * be searched.
	 * @return The value for the ID, or <tt>null</tt> if no such value could be
	 * found.
	 * @exception IOException If an I/O error occurred.
	 **/
	public byte[] getData(int id, boolean dirtyReads)
		throws IOException
	{
		long offset = _idFile.getOffset(id, dirtyReads);
		if (offset != 0L) {
			byte[] combinedData = _dataFile.getData(offset, dirtyReads);
			return _getDataFromCombinedArray(combinedData);
		}
		return null;
	}

	/**
	 * Gets the ID for the specified value.
	 *
	 * @param queryData A value.
	 * @return The ID for the specified value, or <tt>0</tt> if no such ID could
	 * be found.
	 * @exception IOException If an I/O error occurred.
	 **/
	public int getID(byte[] queryData)
		throws IOException
	{
		return getID(queryData, false);
	}

	/**
	 * Gets the ID for the specified value, optionally search the non-committed
	 * data.
	 *
	 * @param queryData A value.
	 * @param dirtyReads Flag indicating whether the non-commited data should
	 * be searched.
	 * @return The ID for the specified value, or <tt>0</tt> if no such ID could
	 * be found.
	 * @exception IOException If an I/O error occurred.
	 **/
	public int getID(byte[] queryData, boolean dirtyReads)
		throws IOException
	{
		return _getID(queryData, _getHash(queryData), dirtyReads);
	}

	private int _getID(byte[] queryData, int hash, boolean dirtyReads)
		throws IOException
	{
		HashFile.OffsetIterator iter = _hashFile.getOffsetIterator(hash, dirtyReads);
		long offset = iter.next();
		while (offset >= 0L) {
			byte[] combinedData = _dataFile.getData(offset, dirtyReads);

			// First 8 bytes in combinedData form the count and ID, the rest of the bytes should match
			if (queryData.length + 8 == combinedData.length &&
				ByteArrayUtil.regionMatches(queryData, combinedData, DATA_IDX))
			{
				// Stored data matches, return this data's ID
				return _getIdFromCombinedArray(combinedData);
			}

			// Data doesn't match, try next offset
			offset = iter.next();
		}

		// Value was not found
		return 0;		
	}

	/**
	 * Returns the maximum ID value.
	 **/
	public int getMaxID()
		throws IOException
	{
		return _idFile.getMaxID();
	}

	/**
	 * Starts a transaction. This prepares the DataStore for upcoming updates
	 * to the stored data.
	 *
	 * @exception IOException If an I/O error occurred.
	 **/
	public void startTransaction()
		throws IOException
	{
		_dataFile.startTransaction(false);
		_idFile.startTransaction(false);
		_hashFile.startTransaction(false);
	}

	/**
	 * Commits a transaction, applying all updates that have been performed
	 * during the transaction.
	 *
	 * @exception IOException If an I/O error occurred.
	 **/
	public void commitTransaction()
		throws IOException
	{
		_hashFile.commitTransaction();
		_idFile.commitTransaction();
		_dataFile.commitTransaction();
	}

	/**
	 * Rolls back all updates that have been performed in the current
	 * transaction and closes it.
	 *
	 * @exception IOException If an I/O error occurred.
	 **/
	public void rollbackTransaction()
		throws IOException
	{
		_hashFile.rollbackTransaction();
		_idFile.rollbackTransaction();
		_dataFile.rollbackTransaction();
	}

	/**
	 * Stores the supplied value and returns the ID that has been assigned to
	 * it. In case the data to store is already present, the ID of this existing
	 * data is returned. This method can only be called as part of a
	 * transaction.
	 *
	 * @param data The data to store.
	 * @return The ID that has been assigned to the value.
	 * @exception IOException If an I/O error occurred.
	 * @see #startTransaction
	 * @see #commitTransaction
	 **/
	public int storeData(byte[] data)
		throws IOException
	{
		int hash = _getHash(data);
		int id = _getID(data, hash, true);

		if (id == 0) {
			// Data not stored yet, store it under a new ID.
			id = _idFile.getNewID();
	
			byte[] combinedData = _createCombinedArray(1, id, data);
	
			long offset = _dataFile.storeData(combinedData);
			_hashFile.storeOffset(hash, offset);
			_idFile.storeOffset(id, offset);
		}
		
		return id;
	}

	/**
	 * Removes all values from the DataStore. This method can only be called as
	 * part of a transaction. The DataStore will not be cleared until the
	 * transaction is committed.
	 *
	 * @exception IOException If an I/O error occurred.
	 * @see #startTransaction
	 * @see #commitTransaction
	 **/
	public void clear()
		throws IOException
	{
		_hashFile.clear();
		_idFile.clear();
		_dataFile.clear();
	}

	/**
	 * Closes the DataStore, releasing any file references, etc. In case a
	 * transaction is currently open, it will be rolled back. Once closed, the
	 * DataStore can no longer be used.
	 *
	 * @exception IOException If an I/O error occurred.
	 **/
	public void close()
		throws IOException
	{
		_hashFile.close();
		_idFile.close();
		_dataFile.close();
	}

	private int _getHash(byte[] data) {
		synchronized (_crc32) {
			_crc32.update(data);
			int crc = (int)_crc32.getValue();
			_crc32.reset();
			return crc;
		}
	}

/*--------------------------------------+
| Methods for combining/filtering an ID |
| and data in/from a single byte array  |
+--------------------------------------*/

	/**
	 * Creates a combined data array that contains the supplied count, ID and data.
	 **/
	private static byte[] _createCombinedArray(int count, int id, byte[] data) {
		byte[] result = new byte[8 + data.length];
		ByteArrayUtil.putInt(count, result, COUNT_IDX);
		ByteArrayUtil.putInt(id, result, ID_IDX);
		ByteArrayUtil.put(data, result, DATA_IDX);
		return result;
	}

	private static int _getCountFromCombinedArray(byte[] data) {
		return ByteArrayUtil.getInt(data, COUNT_IDX);
	}

	/**
	 * Gets the ID from a combined data array.
	 **/
	private static int _getIdFromCombinedArray(byte[] data) {
		return ByteArrayUtil.getInt(data, ID_IDX);
	}

	/**
	 * Gets the data from a combined data array.
	 **/
	private static byte[] _getDataFromCombinedArray(byte[] data) {
		return ByteArrayUtil.get(data, DATA_IDX);
	}

/*-------------------+
| Test/debug methods |
+-------------------*/

	public static void main(String[] args)
		throws Exception
	{
		if (args.length == 2) {
			_printContents(args);
		}
		else {
			_test(args);
		}
	}

	private static void _test(String[] args)
		throws Exception
	{
		System.out.println("Running DataStore test...");
		File dataDir = new File(args[0]);
		DataStore dataStore = new DataStore(dataDir, "strings");

		System.out.println("Creating strings...");
		int stringCount = Integer.parseInt(args[1]);
		String[] strings = new String[stringCount];
		for (int i = 0; i < stringCount; i++) {
			strings[i] = String.valueOf(i + 1);
		}

		System.out.println("Starting transaction...");
		long startTime = System.currentTimeMillis();
		dataStore.startTransaction();

		for (int i = 0; i < strings.length; i++) {
			String s = strings[i];
			byte[] sBytes = s.getBytes();
			int sID = dataStore.getID(sBytes, true);
			if (sID == 0) {
				sID = dataStore.storeData(sBytes);
			}
		}

		System.out.println("Commiting transaction...");
		dataStore.commitTransaction();
		long endTime = System.currentTimeMillis();
		System.out.println("Transaction finished in " + (endTime-startTime) + " ms");

		System.out.println("Fetching IDs for all strings...");
		startTime = System.currentTimeMillis();

		for (int i = 0; i < strings.length; i++) {
			String s = strings[i];
			int sID = dataStore.getID(s.getBytes());
			if (sID != i+1) {
				System.out.println("Unexpected ID for string \"" + s + "\": " + sID);
			}
		}

		endTime = System.currentTimeMillis();
		System.out.println("All IDs fetched in " + (endTime-startTime) + " ms");

		System.out.println("Fetching data for all IDs...");
		startTime = System.currentTimeMillis();

		for (int i = 0; i < strings.length; i++) {
			String s = new String(dataStore.getData(i+1));
			if (!s.equals(strings[i])) {
				System.out.println("Unexpected string for ID " + (i + 1) + ": \"" + s + "\"");
			}
		}

		endTime = System.currentTimeMillis();
		System.out.println("All data fetched in " + (endTime - startTime) + " ms");

		System.out.println("Closing DataStore...");
		dataStore.close();
		System.out.println("Done.");
	}

	private static void _printContents(String[] args)
		throws Exception
	{
		System.out.println("Dumping DataStore contents...");
		File dataDir = new File(args[0]);
		DataStore dataStore = new DataStore(dataDir, args[1]);

		DataFile.DataIterator iter = dataStore._dataFile.iterator();
		while (iter.hasNext()) {
			byte[] combinedData = iter.next();

			int count = _getCountFromCombinedArray(combinedData);
			int id = _getIdFromCombinedArray(combinedData);
			byte[] data = _getDataFromCombinedArray(combinedData);

			System.out.println("id="+id+" count="+count+": " + ByteArrayUtil.toHexString(data));
		}
	}
}
