/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sailimpl.nativerdf;

import java.io.File;
import java.io.IOException;
import java.util.LinkedHashMap;
import java.util.Map;

import org.openrdf.util.ByteArrayUtil;

import org.openrdf.model.BNode;
import org.openrdf.model.Literal;
import org.openrdf.model.Resource;
import org.openrdf.model.Statement;
import org.openrdf.model.URI;
import org.openrdf.model.Value;
import org.openrdf.model.ValueFactory;
import org.openrdf.model.impl.StatementImpl;

import org.openrdf.sesame.sailimpl.nativerdf.datastore.DataStore;
import org.openrdf.sesame.sailimpl.nativerdf.model.NativeBNode;
import org.openrdf.sesame.sailimpl.nativerdf.model.NativeLiteral;
import org.openrdf.sesame.sailimpl.nativerdf.model.NativeURI;
import org.openrdf.sesame.sailimpl.nativerdf.model.NativeValue;

/**
 * Class that provides indexed storage and retrieval of RDF values.
 */
public class ValueStore implements ValueFactory {

	/*-------------+
	 | Constants    |
	 +-------------*/

	private static final String FILENAME_PREFIX = "values";

	private static final int VALUE_CACHE_SIZE = 256;

	private static final int ID_CACHE_SIZE = 256;

	private static final byte VALUE_TYPE_MASK = 0x3; // 0000 0011
	private static final byte URI_VALUE = 0x1; // 0000 0001
	private static final byte BNODE_VALUE = 0x2; // 0000 0010
	private static final byte LITERAL_VALUE = 0x3; // 0000 0011

	/*-------------+
	 | Variables    |
	 +-------------*/

	private DataStore _dataStore;

	private NamespaceStore _namespaceStore;

	private NativeRdfRepository _repository;

	/**
	 * An object that indicates the revision of the value store, which is used to
	 * check if cached value IDs are still valid. In order to be valid, the
	 * ValueStoreRevision object of a NativeValue needs to be equal to this
	 * object.
	 */
	private ValueStoreRevision _revision;

	/**
	 * A simple cache containing the [VALUE_CACHE_SIZE] most-recently used values
	 * stored by their ID (an Integer).
	 */
	private LinkedHashMap _valueCache;

	/**
	 * A simple cache containing the [ID_CACHE_SIZE] most-recently used IDs (an
	 * Integer) stored by their value.
	 */
	private LinkedHashMap _idCache;

	/**
	 * The prefix for any new bnode IDs.
	 */
	private String _bnodePrefix;

	/**
	 * The ID for the next bnode that is created.
	 */
	private int _nextBNodeID;

	/*-------------+
	 | Constructors |
	 +-------------*/

	public ValueStore(File dataDir, NamespaceStore namespaceStore, NativeRdfRepository repository)
		throws IOException
	{
		_dataStore = new DataStore(dataDir, FILENAME_PREFIX);
		_namespaceStore = namespaceStore;
		_repository = repository;

		_valueCache = new LinkedHashMap(VALUE_CACHE_SIZE, 0.75f, true) {
			protected boolean removeEldestEntry(Map.Entry eldest) {
				return size() > VALUE_CACHE_SIZE;
			}
		};

		_idCache = new LinkedHashMap(ID_CACHE_SIZE, 0.75f, true) {
			protected boolean removeEldestEntry(Map.Entry eldest) {
				return size() > ID_CACHE_SIZE;
			}
		};

		_updateBNodePrefix();

		_setNewRevision();
	}

	/**
	 * Generates a new bnode prefix based on <tt>currentTimeMillis()</tt> and
	 * resets <tt>_nextBNodeID</tt> to <tt>1</tt>.
	 */
	private void _updateBNodePrefix() {
		// BNode prefix is based on currentTimeMillis(). Combined with a
		// sequential number per session, this gives a unique identifier.
		_bnodePrefix = "node" + Long.toString(System.currentTimeMillis(), 32) + "x";
		_nextBNodeID = 1;
	}

	/**
	 * Creates a new revision object for this value store, invalidating any IDs
	 * cached in NativeValue objects that were created by this value store.
	 */
	private void _setNewRevision() {
		_revision = new ValueStoreRevision(this);
	}

	/*----------+
	 | Methods   |
	 +----------*/

	/**
	 * Gets the value for the specified ID.
	 * 
	 * @param id
	 *        A value ID.
	 * @return The value for the ID, or <tt>null</tt> no such value could be
	 *         found.
	 * @exception IOException
	 *            If an I/O error occurred.
	 */
	public Value getValue(int id)
		throws IOException
	{
		NativeValue resultValue = null;
		Integer cacheID = new Integer(id);

		// Check cache
		synchronized (_valueCache) {
			resultValue = (NativeValue)_valueCache.get(cacheID);
		}

		if (resultValue == null) {
			// Value not in cache, fetch it from file
			byte[] data = _dataStore.getData(id);

			if (data != null) {
				resultValue = _data2value(id, data);

				// Store value in cache
				synchronized (_valueCache) {
					_valueCache.put(cacheID, resultValue);
				}
			}
		}

		return resultValue;
	}

	/**
	 * Gets the ID for the specified value.
	 * 
	 * @param value
	 *        A value.
	 * @return The ID for the specified value, or <tt>0</tt> if no such ID
	 *         could be found.
	 * @exception IOException
	 *            If an I/O error occurred.
	 */
	public int getID(Value value)
		throws IOException
	{
		return getID(value, false);
	}

	public int getID(Value value, boolean dirtyReads)
		throws IOException
	{
		// Check if the value stores the ID internally
		boolean isOwnValue = _isOwnValue(value);
		if (isOwnValue) {
			NativeValue nativeValue = (NativeValue)value;

			if (_revision.equals(nativeValue.getValueStoreRevision())) {
				// Value's ID is still current
				return nativeValue.getInternalId();
			}
		}

		// Check cache
		Integer cachedID = null;
		synchronized (_idCache) {
			cachedID = (Integer)_idCache.get(value);
		}

		if (cachedID != null) {
			int id = cachedID.intValue();

			if (isOwnValue) {
				// Store id in value for fast access in any consecutive calls
				((NativeValue)value).setInternalId(id, _revision);
			}

			return id;
		}

		// ID not cached, search in file
		byte[] data = _value2data(value, dirtyReads, false);

		if (data != null) {
			int id = _dataStore.getID(data, dirtyReads);

			if (id != 0) {
				if (isOwnValue) {
					// Store id in value for fast access in any consecutive calls
					((NativeValue)value).setInternalId(id, _revision);
				}
				else {
					// Store id in cache
					synchronized (_idCache) {
						_idCache.put(value, new Integer(id));
					}
				}
			}

			return id;
		}

		// Unknown value
		return 0;
	}

	/**
	 * Starts a transaction. This prepares the ValueStore for upcoming updates to
	 * the stored data.
	 * 
	 * @exception IOException
	 *            If an I/O error occurred.
	 */
	public void startTransaction()
		throws IOException
	{
		_namespaceStore.startTransaction();
		_dataStore.startTransaction();
	}

	/**
	 * Commits a transaction, applying all updates that have been performed
	 * during the transaction.
	 * 
	 * @exception IOException
	 *            If an I/O error occurred.
	 */
	public void commitTransaction()
		throws IOException
	{
		_dataStore.commitTransaction();
		_namespaceStore.commitTransaction();
	}

	/**
	 * Rolls back all updates that have been performed in the current transaction
	 * and closes it.
	 * 
	 * @exception IOException
	 *            If an I/O error occurred.
	 */
	public void rollbackTransaction()
		throws IOException
	{
		synchronized (_idCache) {
			_idCache.clear();
		}

		_dataStore.rollbackTransaction();
		_namespaceStore.rollbackTransaction();
	}

	/**
	 * Stores the supplied value and returns the ID that has been assigned to it.
	 * In case the value was already present, the value will not be stored again
	 * and the ID of the existing value is returned. This method can only be
	 * called as part of a transaction. The supplied value will not be stored
	 * until the transaction has been committed.
	 * 
	 * @param value
	 *        The Value to store.
	 * @return The ID that has been assigned to the value.
	 * @exception IOException
	 *            If an I/O error occurred.
	 * @see #startTransaction
	 * @see #commitTransaction
	 */
	public int storeValue(Value value)
		throws IOException
	{
		// Try to get the internal ID from the value itself
		boolean isOwnValue = _isOwnValue(value);
		if (isOwnValue) {
			NativeValue nativeValue = (NativeValue)value;

			if (_revision.equals(nativeValue.getValueStoreRevision())) {
				// Value's ID is still current
				return nativeValue.getInternalId();
			}
		}

		// Check cache
		Integer cachedID = null;
		synchronized (_idCache) {
			cachedID = (Integer)_idCache.get(value);
		}
		
		if (cachedID != null) {
			// value has already been stored
			int id = cachedID.intValue();

			if (isOwnValue) {
				// Store id in value for fast access in any consecutive calls
				((NativeValue)value).setInternalId(id, _revision);
			}

			return id;
		}

		// ID not in cache, just store it in the data store which will handle
		// duplicates
		byte[] valueData = _value2data(value, true, true);

		int id = _dataStore.storeData(valueData);

		if (isOwnValue) {
			// Store id in value for fast access in any consecutive calls
			((NativeValue)value).setInternalId(id, _revision);
		}
		else {
			// Update cache
			synchronized (_idCache) {
				_idCache.put(value, new Integer(id));
			}
		}

		return id;
	}

	/**
	 * Removes all values from the ValueStore. This method can only be called as
	 * part of a transaction. The ValueStore will not be cleared until the
	 * transaction is committed.
	 * 
	 * @exception IOException
	 *            If an I/O error occurred.
	 * @see #startTransaction
	 * @see #commitTransaction
	 */
	public void clear()
		throws IOException
	{
		_dataStore.clear();
		_namespaceStore.clear();

		synchronized (_valueCache) {
			_valueCache.clear();
		}

		synchronized (_idCache) {
			_idCache.clear();
		}

		_nextBNodeID = 1;

		_setNewRevision();
	}

	/**
	 * Closes the ValueStore, releasing any file references, etc. In case a
	 * transaction is currently open, it will be rolled back. Once closed, the
	 * ValueStore can no longer be used.
	 * 
	 * @exception IOException
	 *            If an I/O error occurred.
	 */
	public void close()
		throws IOException
	{
		_dataStore.close();
		_namespaceStore.close();
		_valueCache = null;
		_idCache = null;
		_repository = null;
	}

	/**
	 * Checks if the supplied Value object is a NativeValue object that has been
	 * created by this ValueStore.
	 */
	private boolean _isOwnValue(Value value) {
		return value instanceof NativeValue && ((NativeValue)value).getRepository() == _repository;
	}

	private byte[] _value2data(Value value, boolean dirtyReads, boolean create)
		throws IOException
	{
		if (value instanceof URI) {
			return _uri2data((URI)value, dirtyReads, create);
		}
		else if (value instanceof BNode) {
			return _bnode2data((BNode)value, dirtyReads, create);
		}
		else if (value instanceof Literal) {
			return _literal2data((Literal)value, dirtyReads, create);
		}
		else {
			throw new IllegalArgumentException("value parameter should be a URI, BNode or Literal");
		}
	}

	private byte[] _uri2data(URI uri, boolean dirtyReads, boolean create)
		throws IOException
	{
		// Get namespace ID
		int nsID = _namespaceStore.getID(uri.getNamespace(), dirtyReads);
		if (nsID == 0) {
			// Unknown namespace
			if (create) {
				// Add namespace
				nsID = _namespaceStore.storeNamespace(uri.getNamespace());
			}
			else {
				// Unknown namespace means unknown URI
				return null;
			}
		}

		// Get local name in UTF-8
		byte[] localNameData = uri.getLocalName().getBytes("UTF-8");

		// Combine parts in a single byte array
		byte[] uriData = new byte[5 + localNameData.length];
		uriData[0] = URI_VALUE;
		ByteArrayUtil.putInt(nsID, uriData, 1);
		ByteArrayUtil.put(localNameData, uriData, 5);

		return uriData;
	}

	private byte[] _bnode2data(BNode bNode, boolean dirtyReads, boolean create)
		throws IOException
	{
		byte[] idData = bNode.getID().getBytes("UTF-8");

		byte[] bNodeData = new byte[1 + idData.length];
		bNodeData[0] = BNODE_VALUE;
		ByteArrayUtil.put(idData, bNodeData, 1);

		return bNodeData;
	}

	private byte[] _literal2data(Literal literal, boolean dirtyReads, boolean create)
		throws IOException
	{
		// Get datatype ID
		int datatypeID = 0;
		if (literal.getDatatype() != null) {
			if (create) {
				datatypeID = storeValue(literal.getDatatype());
			}
			else {
				datatypeID = getID(literal.getDatatype(), dirtyReads);
				if (datatypeID == 0) {
					// Unknown datatype means unknown literal
					return null;
				}
			}
		}

		// Get language tag in UTF-8
		byte[] langData = null;
		int langDataLength = 0;
		if (literal.getLanguage() != null) {
			langData = literal.getLanguage().getBytes("UTF-8");
			langDataLength = langData.length;
		}

		// Get label in UTF-8
		byte[] labelData = literal.getLabel().getBytes("UTF-8");

		// Combine parts in a single byte array
		byte[] literalData = new byte[6 + langDataLength + labelData.length];
		literalData[0] = LITERAL_VALUE;
		ByteArrayUtil.putInt(datatypeID, literalData, 1);
		literalData[5] = (byte)langDataLength;
		if (langData != null) {
			ByteArrayUtil.put(langData, literalData, 6);
		}
		ByteArrayUtil.put(labelData, literalData, 6 + langDataLength);

		return literalData;
	}

	private NativeValue _data2value(int id, byte[] data)
		throws IOException
	{
		NativeValue nativeValue;

		switch ((data[0] & VALUE_TYPE_MASK)) {
			case URI_VALUE:
				nativeValue = _data2uri(data); break;
			case BNODE_VALUE:
				nativeValue = _data2bnode(data); break;
			case LITERAL_VALUE:
				nativeValue = _data2literal(data); break;
			default:
				throw new IllegalArgumentException("data does not specify a known value type");
		}
		
		nativeValue.setInternalId(id, _revision);
		
		return nativeValue;
	}

	private NativeURI _data2uri(byte[] data)
		throws IOException
	{
		int nsID = ByteArrayUtil.getInt(data, 1);
		String namespace = _namespaceStore.getNamespaceName(nsID);
		String localName = new String(data, 5, data.length - 5, "UTF-8");
		return new NativeURI(_repository, namespace, localName);
	}

	private NativeBNode _data2bnode(byte[] data)
		throws IOException
	{
		String nodeID = new String(data, 1, data.length - 1, "UTF-8");
		return new NativeBNode(_repository, nodeID);
	}

	private NativeLiteral _data2literal(byte[] data)
		throws IOException
	{
		// Get datatype
		int datatypeID = ByteArrayUtil.getInt(data, 1);
		URI datatype = null;
		if (datatypeID != 0) {
			datatype = (URI)getValue(datatypeID);
		}

		// Get language tag
		String lang = null;
		int langLength = data[5];
		if (langLength > 0) {
			lang = new String(data, 6, langLength, "UTF-8");
		}

		// Get label
		String label = new String(data, 6 + langLength, data.length - 6 - langLength, "UTF-8");

		if (datatype != null) {
			return new NativeLiteral(_repository, label, datatype);
		}
		else if (lang != null) {
			return new NativeLiteral(_repository, label, lang);
		}
		else {
			return new NativeLiteral(_repository, label);
		}
	}

	/*------------------------------------+
	 | Methods from interface ValueFactory |
	 +------------------------------------*/

	// Implements ValueFactory.createURI(String)
	public URI createURI(String uri) {
		return new NativeURI(_repository, uri);
	}

	public URI createURI(String namespace, String localName) {
		return new NativeURI(_repository, namespace, localName);
	}

	// Implements ValueFactory.createBNode()
	public BNode createBNode() {
		if (_nextBNodeID == Integer.MAX_VALUE) {
			// Start with a new bnode prefix
			_updateBNodePrefix();
		}

		return createBNode(_bnodePrefix + _nextBNodeID++);
	}

	// Implements ValueFactory.createBNode(String)
	public BNode createBNode(String nodeId) {
		return new NativeBNode(_repository, nodeId);
	}

	// Implements ValueFactory.createLiteral(String)
	public Literal createLiteral(String value) {
		return new NativeLiteral(_repository, value);
	}

	// Implements ValueFactory.createLiteral(String, String)
	public Literal createLiteral(String value, String language) {
		return new NativeLiteral(_repository, value, language);
	}

	// Implements ValueFactory.createLiteral(String, URI)
	public Literal createLiteral(String value, URI datatype) {
		return new NativeLiteral(_repository, value, datatype);
	}

	// Implements ValueFactory.createStatement(Resource, URI, Value)
	public Statement createStatement(Resource subject, URI predicate, Value object) {
		return new StatementImpl(subject, predicate, object);
	}

	/*-------------------+
	 | Test/debug methods |
	 +-------------------*/

	public static void main(String[] args)
		throws Exception
	{
		File dataDir = new File(args[0]);
		NamespaceStore namespaceStore = new NamespaceStore(dataDir);
		ValueStore valueStore = new ValueStore(dataDir, namespaceStore, null);

		int maxID = valueStore._dataStore.getMaxID();
		for (int id = 1; id <= maxID; id++) {
			Value value = valueStore.getValue(id);
			System.out.println("ID " + id + " : " + value.toString());
		}
	}
}
