/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sailimpl.nativerdf;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Arrays;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Properties;
import java.util.Set;
import java.util.StringTokenizer;

import org.openrdf.util.ByteArrayUtil;
import org.openrdf.util.log.ThreadLog;

import org.openrdf.sesame.sail.SailInitializationException;
import org.openrdf.sesame.sailimpl.nativerdf.btree.BTree;
import org.openrdf.sesame.sailimpl.nativerdf.btree.BTreeIterator;
import org.openrdf.sesame.sailimpl.nativerdf.btree.BTreeValueComparator;

/**
 **/
public class TripleStore {

/*-------------+
| Constants    |
+-------------*/

	private static final String PROPERTIES_FILE = "triples.prop";
	private static final String VERSION_KEY = "version";
	private static final String INDEXES_KEY = "triple-indexes";

	private static final int SCHEME_VERSION = 1;

	private static final String OLD_IDX_FILE = "triples.dat";

	/* 13 bytes are used to represent a triple:
     *  byte 0-3 : subject
     *  byte 4-7 : predicate
     *  byte 8-11: object
     *  byte 12  : additional flag(s)
	 */
	static final int RECORD_LENGTH = 13;
	static final int SUBJ_IDX = 0;
	static final int PRED_IDX = 4;
	static final int OBJ_IDX  = 8;
	static final int FLAG_IDX = 12;

	static final byte EXPLICIT_FLAG = (byte)0x1; // 0000 0001

/*-------------+
| Variables    |
+-------------*/

	/**
	 * The directory that is used to store the index files.
	 */
	private File _dir;

	/**
	 * Object containing meta-data for the triple store. This includes 
	 */
	private Properties _properties;

	/**
	 * The array of triple indexes that are used to store and retrieve triples.
	 **/
	private TripleIndex[] _indexes;

/*-------------+
| Constructors |
+-------------*/

	public TripleStore(File dir, String indexSpecStr)
		throws IOException, SailInitializationException
	{
		_dir = dir;
		_properties = new Properties();

		// Read triple properties file, upgrade if not present, restore indexes, reindex
		File propFile = new File(dir, PROPERTIES_FILE);

		if (propFile.exists()) {
			_loadProperties(propFile);
		}
		else {
			// Check for presence of old index file and rename it if necessary
			File oldIndexFile = new File(dir, OLD_IDX_FILE);

			if (oldIndexFile.exists()) {
				// Old index file contains an spo index
				File spoFile = _getIndexFile("spo");

				if (!spoFile.exists()) {
					ThreadLog.log("Updating old triple indexing scheme...");
					oldIndexFile.renameTo(spoFile);
				}

				_properties.setProperty(VERSION_KEY, String.valueOf(SCHEME_VERSION));
				_properties.setProperty(INDEXES_KEY, "spo");
			}
		}

		Set indexSpecs = _parseIndexSpecList(indexSpecStr);

		if (indexSpecs.isEmpty()) {
			// Create default spo index
			ThreadLog.log("No indexes specified, defaulting to single spo index");
			indexSpecs.add("spo");
			indexSpecStr = "spo";
		}

		// Initialize added indexes and delete removed ones:
		_reindex(indexSpecs);

		// Store up-to-date properties
		_properties.setProperty(VERSION_KEY, String.valueOf(SCHEME_VERSION));
		_properties.setProperty(INDEXES_KEY, indexSpecStr);
		_storeProperties(propFile);

		// Create specified indexes
		_indexes = new TripleIndex[indexSpecs.size()];
		int i = 0;
		Iterator iter = indexSpecs.iterator();
		while (iter.hasNext()) {
			String fieldSeq = (String)iter.next();
			ThreadLog.trace("Activating index '" + fieldSeq + "'...");
			_indexes[i++] = new TripleIndex(fieldSeq);
		}
	}

/*----------+
| Methods   |
+----------*/

	/**
	 * Parses a comma/whitespace-separated list of index specifications. Index
	 * specifications are required to consists of 3 characters: 's', 'p' and 'o'.
	 *
	 * @param indexSpecStr A string like "spo, pos, osp".
	 * @return A Set containing the parsed index specifications.
	 */
	private Set _parseIndexSpecList(String indexSpecStr)
		throws SailInitializationException
	{
		Set indexes = new HashSet();

		if (indexSpecStr != null) {
			StringTokenizer tok = new StringTokenizer(indexSpecStr, ", \t");
			while (tok.hasMoreTokens()) {
				String index = tok.nextToken().toLowerCase();

				// sanity checks
				if (index.length() != 3 ||
					index.indexOf('s') == -1 ||
					index.indexOf('p') == -1 ||
					index.indexOf('o') == -1)
				{
					throw new SailInitializationException(
							"invalid value '" + index + "' in index specification: " + indexSpecStr);
				}

				indexes.add(index);
			}
		}

		return indexes;
	}

	private void _reindex(Set newIndexSpecs)
		throws IOException, SailInitializationException
	{
		// Check if the index specification has changed and update indexes if necessary
		String currentIndexSpecStr = _properties.getProperty(INDEXES_KEY);
		if (currentIndexSpecStr == null) {
			return;
		}

		Set currentIndexSpecs = _parseIndexSpecList(currentIndexSpecStr);

		if (currentIndexSpecs.isEmpty()) {
			throw new SailInitializationException("Invalid index specification found in index properties");
		}

		// Determine the set of newly added indexes
		Set addedIndexSpecs = new HashSet(newIndexSpecs);
		addedIndexSpecs.removeAll(currentIndexSpecs);

		if (!addedIndexSpecs.isEmpty()) {
			// Initialize new indexes using an existing index as source
			String sourceIndexSpec = (String)currentIndexSpecs.iterator().next();
			TripleIndex sourceIndex = new TripleIndex(sourceIndexSpec);

			try {
				Iterator fieldSeqIter = addedIndexSpecs.iterator();
				while (fieldSeqIter.hasNext()) {
					String fieldSeq = (String)fieldSeqIter.next();

					ThreadLog.trace("Initializing new index '" + fieldSeq + "'...");

					TripleIndex addedIndex = new TripleIndex(fieldSeq);
					BTree addedBTree = addedIndex.getBTree();

					addedBTree.startTransaction();

					BTreeIterator sourceIter = sourceIndex.getBTree().iterateAll();
					try {
						byte[] value = null;
						while ( (value = sourceIter.next()) != null) {
							addedBTree.insert(value);
						}
					}
					finally {
						sourceIter.close();
					}

					addedBTree.commitTransaction();
					addedBTree.close();
				}

				ThreadLog.trace("New index(es) initialized");
			}
			finally {
				sourceIndex.getBTree().close();
			}
		}

		// Determine the set of removed indexes
		Set removedIndexSpecs = new HashSet(currentIndexSpecs);
		removedIndexSpecs.removeAll(newIndexSpecs);

		// Delete files for removed indexes
		Iterator iter = removedIndexSpecs.iterator();
		while (iter.hasNext()) {
			String fieldSeq = (String)iter.next();

			boolean deleted = _getIndexFile(fieldSeq).delete();

			if (deleted) {
				ThreadLog.trace("Deleted file for removed " + fieldSeq + " index");
			}
			else {
				ThreadLog.warning("Unable to delete file for removed " + fieldSeq + " index");
			}
		}
	}

	public void close()
		throws IOException
	{
		for (int i = 0; i < _indexes.length; i++) {
			_indexes[i].getBTree().close();
		}
		_indexes = null;
	}

	public BTreeIterator getTriples(int subj, int pred, int obj)
		throws IOException
	{
		// Get best matching index
		int bestScore = -1;
		TripleIndex bestIndex = null;
		for (int i = 0; i < _indexes.length; i++) {
			int score = _indexes[i].getPatternScore(subj, pred, obj);
			if (score > bestScore) {
				bestScore = score;
				bestIndex = _indexes[i];
			}
		}

		byte[] searchKey = _getSearchKey(subj, pred, obj);
		byte[] searchMask = _getSearchMask(subj, pred, obj);

		if (bestScore > 0) {
			// Use ranged search
			byte[] minValue = _getMinValue(subj, pred, obj);
			byte[] maxValue = _getMaxValue(subj, pred, obj);

			//ThreadLog.trace("using " + bestIndex.getFieldSeq() + " index with a score of " +
			//		bestScore + " for pattern (" + subj + ", " + pred + ", " + obj + ")");

			return bestIndex.getBTree().iterateValues(searchKey, searchMask, minValue, maxValue);
		}
		else {
			//ThreadLog.trace("using sequential search for pattern (" + subj + ", " + pred + ", " + obj + ")");
			// Use sequential scan
			return bestIndex.getBTree().iterateValues(searchKey, searchMask);
		}
	}

	public void startTransaction()
		throws IOException
	{
		for (int i = 0; i < _indexes.length; i++) {
			_indexes[i].getBTree().startTransaction();
		}
	}

	public void commitTransaction()
		throws IOException
	{
		for (int i = 0; i < _indexes.length; i++) {
			_indexes[i].getBTree().commitTransaction();
		}
	}

	public void rollbackTransaction()
		throws IOException
	{
		// FIXME: transaction rollback not implemented yet
		throw new IOException("transaction rollback not implemented yet");
	}

	public void clear()
		throws IOException
	{
		for (int i = 0; i < _indexes.length; i++) {
			_indexes[i].getBTree().clear();
		}
	}

	public byte[] storeTriple(int subj, int pred, int obj)
		throws IOException
	{
		byte[] data = _getData(subj, pred, obj);

		byte[] oldData = _indexes[0].getBTree().insert(data);

		if (oldData == null || !Arrays.equals(data, oldData)) {
			// new or changed data was inserted into the first index,
			// also insert it into the other indexes
			for (int i = 1; i < _indexes.length; i++) {
				_indexes[i].getBTree().insert(data);
			}
		}

		return oldData;
	}

	public int removeTriples(int subj, int pred, int obj)
		throws IOException
	{
		// FIXME: naive implementation below
		ArrayList removeList = new ArrayList();

		BTreeIterator iter = getTriples(subj, pred, obj);
		byte[] value = iter.next();
		while (value != null) {
			removeList.add(value);
			value = iter.next();
		}

		for (int i = 0; i < _indexes.length; i++) {
			BTree btree = _indexes[i].getBTree();

			for (int j = 0; j < removeList.size(); j++) {
				btree.remove( (byte[])removeList.get(j) );
			}
		}

		return removeList.size();
	}

	private byte[] _getData(int subj, int pred, int obj) {
		byte[] data = new byte[RECORD_LENGTH];

		ByteArrayUtil.putInt(subj, data, SUBJ_IDX);
		ByteArrayUtil.putInt(pred, data, PRED_IDX);
		ByteArrayUtil.putInt(obj, data, OBJ_IDX);
		data[FLAG_IDX] = EXPLICIT_FLAG;

		return data;
	}

	private byte[] _getSearchKey(int subj, int pred, int obj) {
		byte[] searchKey = _getData(subj, pred, obj);
		searchKey[FLAG_IDX] = 0;
		return searchKey;
	}

	private byte[] _getSearchMask(int subj, int pred, int obj) {
		byte[] mask = new byte[RECORD_LENGTH];

		if (subj != 0) {
			ByteArrayUtil.putInt(0xffffffff, mask, SUBJ_IDX);
		}
		if (pred != 0) {
			ByteArrayUtil.putInt(0xffffffff, mask, PRED_IDX);
		}
		if (obj != 0) {
			ByteArrayUtil.putInt(0xffffffff, mask, OBJ_IDX);
		}

		return mask;
	}

	private byte[] _getMinValue(int subj, int pred, int obj) {
		byte[] minValue = new byte[RECORD_LENGTH];

		// No need to check for values being equal to 0 as
		// this happens to be the minimum value anyway
		ByteArrayUtil.putInt(subj, minValue, SUBJ_IDX);
		ByteArrayUtil.putInt(pred, minValue, PRED_IDX);
		ByteArrayUtil.putInt(obj, minValue, OBJ_IDX);

		return minValue;
	}

	private byte[] _getMaxValue(int subj, int pred, int obj) {
		byte[] maxValue = new byte[RECORD_LENGTH];

		ByteArrayUtil.putInt( (subj == 0 ? 0xffffffff : subj), maxValue, SUBJ_IDX);
		ByteArrayUtil.putInt( (pred == 0 ? 0xffffffff : pred), maxValue, PRED_IDX);
		ByteArrayUtil.putInt( (obj  == 0 ? 0xffffffff : obj),  maxValue, OBJ_IDX);

		return maxValue;
	}

	private File _getIndexFile(String fieldSeq) {
		return new File(_dir, "triples-" + fieldSeq + ".dat");
	}

	private void _loadProperties(File propFile)
		throws IOException
	{
		InputStream in = new FileInputStream(propFile);
		try {
			_properties.clear();
			_properties.load(in);
		}
		finally {
			in.close();
		}
	}

	private void _storeProperties(File propFile)
		throws IOException
	{
		OutputStream out = new FileOutputStream(propFile);
		try {
			_properties.store(out, "triple indexes meta-data, DO NOT EDIT!");
		}
		finally {
			out.close();
		}
	}

/*------------------------+
| Inner class TripleIndex |
+------------------------*/

	private class TripleIndex {

		private String _fieldSeq;
		private File _file;
		private BTree _btree;

		public TripleIndex(String fieldSeq)
			throws IOException
		{
			_fieldSeq = fieldSeq;
			_file = _getIndexFile(fieldSeq);
			_btree = new BTree(_file, 2048, RECORD_LENGTH, new TripleComparator(_fieldSeq));
		}

		public String getFieldSeq() {
			return _fieldSeq;
		}

		public File getFile() {
			return _file;
		}

		public BTree getBTree() {
			return _btree;
		}

		/**
		 * Determines the 'score' of this index on the supplied pattern of
		 * subject, predicate and object IDs. The higher the score, the better
		 * the index is suited for matching the pattern. Lowest score is 0,
		 * which means that the index will perform a sequential scan.
		 */
		public int getPatternScore(int subj, int pred, int obj) {
			int score = 0;

			for (int i = 0; i < _fieldSeq.length(); i++) {
				char field = _fieldSeq.charAt(i);
				if (field == 's') {
					if (subj == 0) {
						break;
					} else {
						score++;
					}
				}
				else if (field == 'p') {
					if (pred == 0) {
						break;
					} else {
						score++;
					}
				}
				else if (field == 'o') {
					if (obj == 0) {
						break;
					} else {
						score++;
					}
				}
			}

			return score;
		}
	}

/*-----------------------------+
| Inner class TripleComparator |
+-----------------------------*/

	/**
	 * A BTreeValueComparator that can be used to create indexes with a
	 * configurable order of the subject, predicate and object fields.
	 */
	private static class TripleComparator implements BTreeValueComparator {

		private String _fieldSeq;

		public TripleComparator(String fieldSeq) {
			_fieldSeq = fieldSeq;
		}

		// implements BTreeValueComparator.compareBTreeValues()
		public final int compareBTreeValues(byte[] key, byte[] data, int offset, int length) {
			int result = 0;

			for (int i = 0; result == 0 && i < _fieldSeq.length(); i++) {
				switch (_fieldSeq.charAt(i)) {
					case 's': result = ByteArrayUtil.compareRegion(key, SUBJ_IDX, data, offset + SUBJ_IDX, 4); break;
					case 'p': result = ByteArrayUtil.compareRegion(key, PRED_IDX, data, offset + PRED_IDX, 4); break;
					case 'o': result = ByteArrayUtil.compareRegion(key, OBJ_IDX, data, offset + OBJ_IDX, 4); break;
					default: throw new IllegalArgumentException(
							 "invalid character '" + _fieldSeq.charAt(i) + "' in field sequence: " + _fieldSeq);
				}
			}

			return result;
		}
	}
}
