/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sailimpl.nativerdf;

import java.io.IOException;

import org.openrdf.util.ByteArrayUtil;

import org.openrdf.model.Resource;
import org.openrdf.model.Statement;
import org.openrdf.model.URI;
import org.openrdf.model.Value;
import org.openrdf.model.ValueFactory;

import org.openrdf.sesame.sail.SailInternalException;
import org.openrdf.sesame.sail.StatementIterator;
import org.openrdf.sesame.sailimpl.nativerdf.btree.BTreeIterator;

/**
 * An implementation of the RdfRepository interface from the RDF Sail API that
 * stores its data in, and queries it from files on disk.
 * 
 * @author Arjohn Kampman
 * @version $Revision: 1.6.4.2 $
 */
public class NativeStatementIterator implements StatementIterator {

/*--------------+
| Variables     |
+--------------*/

	private BTreeIterator _btreeIter;
	private ValueStore _valueStore;
	private ValueFactory _valueFactory;

	private Resource _subj;
	private URI _pred;
	private Value _obj;

	private byte[] _nextValue;

/*--------------+
| Constructors  |
+--------------*/

	/**
	 * Creates a new NativeStatementIterator.
	 **/
	public NativeStatementIterator(
			TripleStore tripleStore, ValueStore valueStore, ValueFactory valueFactory,
			Resource subj, URI pred, Value obj, int subjID, int predID, int objID)
		throws IOException
	{
		_valueStore = valueStore;
		_valueFactory = valueFactory;

		_subj = subj;
		_pred = pred;
		_obj = obj;

		_btreeIter = tripleStore.getTriples(subjID, predID, objID);

		_nextValue = _btreeIter.next();
	}

/*--------------+
| Methods       |
+--------------*/

	public boolean hasNext() {
		return _nextValue != null;
	}

	public Statement next() {
		try {
			Resource subj = _subj;
			if (subj == null) {
				int subjID = ByteArrayUtil.getInt(_nextValue, 0);
				subj = (Resource)_valueStore.getValue(subjID);
			}

			URI pred = _pred;
			if (pred == null) {
				int predID = ByteArrayUtil.getInt(_nextValue, 4);
				pred = (URI)_valueStore.getValue(predID);
			}

			Value obj = _obj;
			if (obj == null) {
				int objID  = ByteArrayUtil.getInt(_nextValue, 8);
				obj = (Value)_valueStore.getValue(objID);
			}

			_nextValue = _btreeIter.next();

			return _valueFactory.createStatement(subj, pred, obj);
		}
		catch (IOException e) {
			throw new SailInternalException(e);
		}
	}

	public void close() {
		_nextValue = null;

		if (_btreeIter != null) {
			try {
				_btreeIter.close();
			}
			catch (IOException e) {
				throw new SailInternalException(e);
			}
			finally {
				_btreeIter = null;
			}
		}
	}
}
