/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sailimpl.nativerdf;

import org.openrdf.sesame.config.SailConfig;

/**
 * Extension of SailConfig offering specific utility methods for configuring an
 * <tt>org.openrdf.sesame.sailimpl.nativerdf.NativeRdfRepository</tt> Sail.
 **/
public class NativeRdfRepositoryConfig extends SailConfig {

/*--------------+
| Constructors  |
+--------------*/

	/**
	 * Creates a new and unconfigured NativeRdfRepositoryConfig object for an
	 * <tt>org.openrdf.sesame.sailimpl.nativerdf.NativeRdfRepository</tt> Sail.
	 * A data directory should be specified before using this configuration
	 * object.
	 *
	 * @see #setDataDir(java.lang.String)
	 **/
	public NativeRdfRepositoryConfig() {
		super("org.openrdf.sesame.sailimpl.nativerdf.NativeRdfRepository");
	}

	/**
	 * Creates a new NativeRdfRepositoryConfig object for an
	 * <tt>org.openrdf.sesame.sailimpl.nativerdf.NativeRdfRepository</tt> Sail
	 * that uses the specified directory for storing its data.
	 *
	 * @param dataDir The data directory.
	 **/
	public NativeRdfRepositoryConfig(String dataDir) {
		this();
		setDataDir(dataDir);
	}

	/**
	 * Creates a new NativeRdfRepositoryConfig object for an
	 * <tt>org.openrdf.sesame.sailimpl.nativerdf.NativeRdfRepository</tt> Sail
	 * that uses the specified directory for storing its data and the specified
	 * indexes for searching triples.
	 *
	 * @param dataDir The data directory.
	 * @param indexSpec A set of comma/whitespace-separated field sequences,
	 * e.g. <tt>"spo, pos"</tt>. See {@link #setTripleIndexes} for a description
	 * of the string format.
	 **/
	public NativeRdfRepositoryConfig(String dataDir, String indexSpec) {
		this();
		setDataDir(dataDir);
		setTripleIndexes(indexSpec);
	}

/*--------------+
| Methods       |
+--------------*/

	/**
	 * Sets the data directory for the native RDF repository.
	 *
	 * @param dataDir The data directory.
	 **/
	public void setDataDir(String dataDir) {
    	setParameter(NativeRdfRepository.DATA_DIR_KEY, dataDir);
	}

	/**
	 * Gets the fully qualified name of the data directory.
	 **/
	public String getDataDir() {
		return getParameter(NativeRdfRepository.DATA_DIR_KEY);
	}

	/**
	 * Specifies which triple indexes the native RDF repository should use. Each
	 * index consists of three fields: subject, predicate and object. The order
	 * of these fields determines the usability of an index on a specific triple
	 * query pattern: searching triples with a specific subject in an index that
	 * has the subject as the first field is a lot faster than searching these
	 * triples in an index where the subject field is second or third.
	 * <p>
	 * This method allows one to specify which indexes to use. The subject-,
	 * predicate- and object fields are represented by the characters 's', 'p'
	 * and 'o', respectively. Indexes can be specified by creating 3-letter
	 * words from these three characters. Multiple indexes can be specified by
	 * separating these words with comma's, spaces and/or tabs. E.g.: the string
	 * <tt>"spo, pos"</tt> specifies two indexes; a subject-predicate-object
	 * index and a predicate-object-subject index.
	 *
	 * @param indexSpec A set of comma/whitespace-separated field sequences,
	 * e.g. <tt>"spo, pos"</tt>.
	 **/
	public void setTripleIndexes(String indexSpec) {
    	setParameter(NativeRdfRepository.TRIPLES_INDEXES_KEY, indexSpec);
	}

	/**
	 * Gets the triple index specification to be used by the native RDF
	 * repository.
	 *
	 * @see #setTripleIndexes
	 **/
	public String getTripleIndexes() {
		return getParameter(NativeRdfRepository.TRIPLES_INDEXES_KEY);
	}
}
