/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sailimpl.nativerdf;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import org.openrdf.sesame.sail.NamespaceIterator;

/**
 * An in-memory store for namespace information that uses a file for persistence.
 * Namespaces are encoded in the file as records as follows:
 * <pre>
 * byte 1 - 3     : internal namespace ID
 * byte 4         : reserved byte for storing some boolean properties
 * byte 5 - 6     : the length of the encoded namespace prefix
 * byte 7 - A     : the UTF-8 encoded namespace prefix
 * byte A+1 - A+2 : the length of the encoded namespace name
 * byte A+3 - end : the UTF-8 encoded namespace name
 * </pre>
 *
 * @author Arjohn Kampman
 * @version $Revision: 1.9.4.3 $
 **/
public class NamespaceStore {

/*-------------+
| Constants    |
+-------------*/

	private static final String FILE_NAME = "namespaces.dat";
	private static final int EXPORT_FLAG = 1;	// 0000 0001

/*-------------+
| Variables    |
+-------------*/

	// The data file for this NamespaceStore.
	private File _file;

	// Arrays storing namespace info using their ID. As ID 0 is
	// not used, the first element will always be empty.
	private Namespace[] _namespaces;
	private Namespace[] _txnNamespaces;

	// Map storing Namespace objects using the namespace name (String) as key.
	private Map _namespacesMap;
	private Map _txnNamespacesMap;

	// Flag indicating whether the contents of this NamespaceStore
	// are different from what is stored on disk.
	private boolean _contentsChanged;

/*-------------+
| Constructors |
+-------------*/

	public NamespaceStore(File dataDir)
		throws IOException
	{
		_file = new File(dataDir, FILE_NAME);

		// Make sure the file exists
		_file.createNewFile();

		_namespaces = new Namespace[16];
		_namespacesMap = new HashMap(16);

		_readNamespacesFromFile();
	}

/*----------+
| Methods   |
+----------*/

	public int getID(String namespace) {
		return getID(namespace, false);
	}

	public int getID(String namespace, boolean dirtyReads) {
		Map nsMap = dirtyReads ? _txnNamespacesMap : _namespacesMap;

		Namespace ns = (Namespace)nsMap.get(namespace);

		if (ns != null) {
			return ns.getID();
		}

		return 0;
	}

	public Namespace getNamespace(int id) {
		return getNamespace(id, false);
	}

	public Namespace getNamespace(int id, boolean dirtyReads) {
		Namespace[] namespaces = dirtyReads ? _txnNamespaces : _namespaces;

		if (id >= 0 && id < namespaces.length) {
			return namespaces[id];
		}

		return null;
	}

	public String getNamespaceName(int id) {
		return getNamespaceName(id, false);
	}

	public String getNamespaceName(int id, boolean dirtyReads) {
		Namespace ns = getNamespace(id, dirtyReads);

		if (ns != null) {
			return ns.getName();
		}

		return null;
	}

	public NamespaceIterator getNamespaces() {
		return new NI(_namespaces);
	}

	public void startTransaction() {
		// Create working copies of the normal data structures
		_txnNamespaces = new Namespace[_namespaces.length];
		System.arraycopy(_namespaces, 0, _txnNamespaces, 0, _namespaces.length);

		_txnNamespacesMap = new HashMap(_namespacesMap);

		_contentsChanged = false;
	}

	public void commitTransaction()
		throws IOException
	{
		_commitTransaction(true);
	}

	private void _commitTransaction(boolean flushToFile)
		throws IOException
	{
		if (_contentsChanged) {
			// Replace the normal data structures with the working copies
			_namespaces = _txnNamespaces;
			_namespacesMap = _txnNamespacesMap;

			if (flushToFile) {
				// Flush the changes to disk
				_writeNamespacesToFile();
			}
			
			_contentsChanged = false;
		}

		// Discard the working copies
		_txnNamespaces = null;
		_txnNamespacesMap = null;
	}

	public void rollbackTransaction() {
		// Discard the working copies
		_txnNamespaces = null;
		_txnNamespacesMap = null;
		_contentsChanged = false;
	}

	public void close() {
		if (_txnNamespaces != null) {
			rollbackTransaction();
		}
		_namespaces = null;
		_namespacesMap = null;
		_file = null;
	}

	public void clear() {
		// Create new working copies
		_txnNamespaces = new Namespace[16];
		_txnNamespacesMap = new HashMap(16);
		_contentsChanged = true;
	}

	public int storeNamespace(String namespace) {
		// Find first available ID (but not 0)
		int id = 1;
		for (; id < _txnNamespaces.length; id++) {
			if (_txnNamespaces[id] == null) {
				break;
			}
		}

		Namespace ns = new Namespace(id, "ns"+id, namespace);
		_storeNamespace(ns);

		return id;
	}

	private void _storeNamespace(Namespace ns) {
		int id = ns.getID();

		if (id >= _txnNamespaces.length) {
			// Grow namespaces array
			int newSize = Math.max(2*_txnNamespaces.length, id+1);
			Namespace[] newArray = new Namespace[newSize];
			System.arraycopy(_txnNamespaces, 0, newArray, 0, _txnNamespaces.length);
			_txnNamespaces = newArray;
		}

		_txnNamespaces[id] = ns;
		_txnNamespacesMap.put(ns.getName(), ns);

		_contentsChanged = true;
	}

	public void setNamespacePrefix(String prefix, String name) {
		Namespace ns = (Namespace)_txnNamespacesMap.get(name);

		if (ns != null && _getNamespaceForPrefix(prefix) == null) {
			// Namespace found and prefix is not used yet
			ns.setPrefix(prefix);
			ns.setExported(true);
			_contentsChanged = true;
		}
	}

	private Namespace _getNamespaceForPrefix(String prefix) {
		for (int i = 0; i < _txnNamespaces.length; i++) {
			Namespace ns = _txnNamespaces[i];

			if (ns != null && ns.getPrefix().equals(prefix)) {
				return ns;
			}
		}
		return null;
	}

/*-----------+
| File I/O   |
+-----------*/

	private void _writeNamespacesToFile()
		throws IOException
	{
		synchronized (_file) {
			DataOutputStream out = new DataOutputStream(new FileOutputStream(_file));

			try {
				for (int i = 0; i < _namespaces.length; i++) {
					Namespace ns = _namespaces[i];
					if (ns != null) {
						int flag = 0;
						if (ns.exported()) {
							flag |= EXPORT_FLAG;
						}

						out.writeInt(ns.getID());
						out.writeByte(flag);
						out.writeUTF(ns.getPrefix());
						out.writeUTF(ns.getName());
					}
				}
			}
			finally {
				out.close();
			}
		}
	}

	private void _readNamespacesFromFile()
		throws IOException
	{
		synchronized (_file) {
			startTransaction();
			DataInputStream in = new DataInputStream(new FileInputStream(_file));

			try {
				while (true) {
					try {
						int id = in.readInt();
						byte flag = in.readByte(); // not yet used
						String prefix = in.readUTF();
						String name = in.readUTF();

						boolean export = (flag & EXPORT_FLAG) == EXPORT_FLAG;

						Namespace ns = new Namespace(id, prefix, name, export);

						_storeNamespace(ns);
					}
					catch (EOFException e) {
						break;
					}
				}
			}
			finally {
				in.close();
			}

			_commitTransaction(false);
		}
	}

/*---------------+
| Inner class NI |
+---------------*/

	private static class NI implements NamespaceIterator {

		private Namespace[] _namespaces;
		private int _nextNamespace;
		private Namespace _currentNamespace;

		public NI(Namespace[] namespaces) {
			_namespaces = namespaces;

			_nextNamespace = 0;
			_findNextNamespace();
		}

		private void _findNextNamespace() {
			_nextNamespace++;

			while (_nextNamespace < _namespaces.length
				&& _namespaces[_nextNamespace] == null)
			{
				_nextNamespace++;
			}
		}

		public boolean hasNext() {
			return _nextNamespace < _namespaces.length;
		}

		public void next() {
			_currentNamespace = _namespaces[_nextNamespace];
			_findNextNamespace();
		}

		public String getName() {
			return _currentNamespace.getName();
		}

		public String getPrefix() {
			return _currentNamespace.getPrefix();
		}

		public void close() {
			_currentNamespace = null;
			_nextNamespace = _namespaces.length;
		}
	}

/*------------------+
| Debugging methods |
+------------------*/

	public static void main(String[] args)
		throws Exception
	{
		NamespaceStore nsStore = new NamespaceStore(new File(args[0]));
		Namespace[] namespaces = nsStore._namespaces;
		for (int i = 0; i < namespaces.length; i++) {
			Namespace ns = namespaces[i];
			if (ns != null) {
				System.out.println("ID "+ns.getID()+": prefix='"+ns.getPrefix()+"', name='"+ns.getName()+"'");
			}
		}
	}
}
