/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sailimpl.memory;

import java.util.ArrayList;
import java.util.List;

import org.openrdf.model.GraphException;
import org.openrdf.model.Resource;
import org.openrdf.model.Statement;
import org.openrdf.model.URI;
import org.openrdf.model.Value;
import org.openrdf.model.impl.URIImpl;

import org.openrdf.sesame.sail.SailUpdateException;
import org.openrdf.sesame.sail.StatementIterator;
import org.openrdf.sesame.sail.util.EmptyStatementIterator;

/**
 * An extension of URI giving it node properties.
 *
 * @author Arjohn Kampman
 * @version $Revision: 1.15.4.2 $
 **/
public class URINode extends URIImpl implements ResourceNode {
	
/*---------------+
| Variables      |
+---------------*/
	
	transient protected RdfSource _source;
	
	/** The list of statements for which this URINode is the subject. **/
	transient protected StatementList _subjectStatements;
	
	/** The list of statements for which this URINode is the predicate. **/
	transient protected StatementList _predicateStatements;
	
	/** The list of statements for which this URINode is the object. **/
	transient protected StatementList _objectStatements;
	
	transient protected StatementList _directTypeStatements;
	
	transient protected StatementList _directSubClassStatements;
	
	transient protected StatementList _directSubPropertyStatements;
	
/*---------------+
| Constructors   |
+---------------*/
	
	/**
	 * Creates a new URINode for a URI.
	 *
	 * @param source The Sail instance that created this URINode.
	 * @param namespace namespace part of URI
	 * @param localName localname part of URI
	 */
	URINode(RdfSource source, String namespace, String localName) {
		super(namespace, localName);
		_source = source;
	}
	
	URINode(RdfSource source, String uri) {
		super(uri);
		_source = source;
	}
	
/*-----------------------------------+
| Methods for _subjectStatements     |
+-----------------------------------*/

	public StatementList getSubjectStatementList() {
		if (_subjectStatements == null) {
			return EMPTY_LIST;
		}
		else {
			return _subjectStatements;
		}
	}
	
	/**
	 * Gets the list of Statements for which this URINode is the subject.
	 * @return A List of Statements.
	 **/
	public StatementIterator getSubjectStatements() {
		if (_subjectStatements == null) {
			return new EmptyStatementIterator();
		}
		else {
			return new MemStatementIterator(_subjectStatements);
		}
	}
	
	/**
	 * Gets the amount of Statements for which this URINode is the subject.
	 * @return An integer larger than or equal to 0.
	 **/
	public int getSubjectStatementCount() {
		if (_subjectStatements == null) {
			return 0;
		}
		else {
			return _subjectStatements.size();
		}
	}
	
	/**
	 * Adds a statement to this URINode's list of statements for which
	 * it is the subject.
	 **/
	public void addSubjectStatement(Statement st) {
		if (_subjectStatements == null) {
			_subjectStatements = new StatementList(4);
		}
		_subjectStatements.add(st);
	}
	
	/**
	 * Removes a statement from this URINode's list of statements for which
	 * it is the subject.
	 **/
	public void removeSubjectStatement(Statement st) {
		_subjectStatements.remove(st);
		if (_subjectStatements.isEmpty()) {
			_subjectStatements = null;
		}
	}
	
/*-----------------------------------+
| Methods for _predicateStatements   |
+-----------------------------------*/

	public StatementList getPredicateStatementList() {
		if (_predicateStatements == null) {
			return EMPTY_LIST;
		}
		else {
			return _predicateStatements;
		}
	}
	
	/**
	 * Gets the list of Statements for which this URINode is the predicate.
	 * @return A List of Statements.
	 **/
	public StatementIterator getPredicateStatements() {
		if (_predicateStatements == null) {
			return new EmptyStatementIterator();
		}
		else {
			return new MemStatementIterator(_predicateStatements);
		}
	}
	
	/**
	 * Gets the amount of Statements for which this URINode is the predicate.
	 * @return An integer larger than or equal to 0.
	 **/
	public int getPredicateStatementCount() {
		if (_predicateStatements == null) {
			return 0;
		}
		else {
			return _predicateStatements.size();
		}
	}
	
	/**
	 * Adds a statement to this URINode's list of statements for which
	 * it is the predicate.
	 **/
	public void addPredicateStatement(Statement st) {
		if (_predicateStatements == null) {
			_predicateStatements = new StatementList(4);
		}
		_predicateStatements.add(st);
	}
	
	/**
	 * Removes a statement from this URINode's list of statements for which
	 * it is the predicate.
	 **/
	public void removePredicateStatement(Statement st) {
		_predicateStatements.remove(st);
		if (_predicateStatements.isEmpty()) {
			_predicateStatements = null;
		}
	}
	
/*-----------------------------------+
| Methods for _objectStatements      |
+-----------------------------------*/

	public StatementList getObjectStatementList() {
		if (_objectStatements == null) {
			return EMPTY_LIST;
		}
		else {
			return _objectStatements;
		}
	}
	
	/**
	 * Gets the list of Statements for which this URINode is the object.
	 * @return A List of Statements, or null if no such statement exists.
	 **/
	public StatementIterator getObjectStatements() {
		if (_objectStatements == null) {
			return new EmptyStatementIterator();
		}
		else {
			return new MemStatementIterator(_objectStatements);
		}
	}
	
	/**
	 * Gets the amount of Statements for which this URINode is the object.
	 * @return An integer larger than or equal to 0.
	 **/
	public int getObjectStatementCount() {
		if (_objectStatements == null) {
			return 0;
		}
		else {
			return _objectStatements.size();
		}
	}
	
	/**
	 * Adds a statement to this URINode's list of statements for which
	 * it is the object.
	 **/
	public void addObjectStatement(Statement st) {
		if (_objectStatements == null) {
			_objectStatements = new StatementList(4);
		}
		_objectStatements.add(st);
	}
	
	/**
	 * Removes a statement from this URINode's list of statements for which
	 * it is the object.
	 **/
	public void removeObjectStatement(Statement st) {
		_objectStatements.remove(st);
		if (_objectStatements.isEmpty()) {
			_objectStatements = null;
		}
	}
	
	public RdfSource getRdfSource() {
		return _source;
	}
	
	
	public void addProperty(URI property, Value value)
		throws GraphException
	{
		if (_source instanceof RdfRepository) {
			RdfRepository rep = (RdfRepository)_source;
			rep.startTransaction();
			try {
				rep.addStatement(this, property, value);
			}
			catch (SailUpdateException e) {
				throw new GraphException(e);
			}
			finally {
				rep.commitTransaction();
			}
		}
		else {
			throw new GraphException("source not writable");
		}
	}
	
	public StatementIterator getDirectSubClassStatements() {
		if (_directSubClassStatements == null) {
			//ThreadLog.trace("Determinining directSubClass cache for " + this.toString());
			_directSubClassStatements = new StatementList();
			
			RdfSchemaRepository repository = (RdfSchemaRepository)_source;
	
			StatementIterator subClassOfIter = repository.getSubClassOf(null, this);
			List subClasses = new ArrayList();
			
			while (subClassOfIter.hasNext()) {
				Statement st = subClassOfIter.next();
				// skip equivalent classes.
				if (!repository.isSubClassOf(this, st.getSubject())) { 
					subClasses.add(st.getSubject());
				}
			}
			subClassOfIter.close();
			
			subClassOfIter = repository.getSubClassOf(null, this);
			while (subClassOfIter.hasNext()) {
				
				Statement stat = subClassOfIter.next();
				
				if (_isDirectSubClassStat(stat, subClasses)) {
					//ThreadLog.trace("found direct subClass statement: " + stat.toString());
					_directSubClassStatements.add(stat);
				}
			}
		}
		//ThreadLog.trace("Returning directSubClass cache for " + this.toString());
		return new MemStatementIterator(_directSubClassStatements);
	}
	
	public boolean isDirectSubClass(Resource aClass) {
		boolean result = false;
		
		StatementIterator subClasses = getDirectSubClassStatements();
		
		while (subClasses.hasNext()) {
			Statement st = subClasses.next();
			if (st.getSubject().equals(aClass)) {
				result = true;
				break;
			}
		}
		subClasses.close();
		
		return result;
	}
	
	private boolean _isDirectSubClassStat(Statement stat, List subClasses) {
		RdfSchemaRepository repository = (RdfSchemaRepository)_source;
		
		//ThreadLog.trace("Determining if " + stat.toString() + " is a direct subclass of " + this.toString());
		ResourceNode subNode = (ResourceNode)stat.getSubject();
		
		if (subNode == this) {
			// a class is never a direct subclass of itself.
			//ThreadLog.trace("not a direct subclass");
			return false;
		}

		if (repository.isSubClassOf(this, subNode)) {
			// cycle in the class hierarchy, classes are equivalent
			//ThreadLog.trace("not a direct subclass");
			return false;
		}

		// 'subClass' is a direct subclass of 'superClass' when the
		// set of superclasses of 'subClass' and the set of subclasses
		// of 'superClass' are disjoint (when ignoring 'subClass' and
		// 'superClass' in the sets).
		boolean isDirect = true;

		// Check whether one of the superclasses of 'subClass' is
		// present in 'subClasses', again ignoring 'subClass'
		// and 'superClass'
		StatementIterator supsOfSubClassIter = repository.getSubClassOf(subNode, null);
		while (supsOfSubClassIter.hasNext()) {
			Statement st = supsOfSubClassIter.next();

			if (st.getObject() instanceof Resource) {
				Resource r = (Resource)st.getObject();

				if (r != subNode && r != this &&
					subClasses.contains(r) &&
					!repository.isSubClassOf(r, subNode)) // ignore equivalent classes
				{
					isDirect = false;
					break;
				}
			}
		}
		supsOfSubClassIter.close();

		//ThreadLog.trace("Returning direct subClass: " + isDirect);
		return isDirect;
	}
	
	public StatementIterator getDirectTypeStatements() {
		if (_directTypeStatements == null) {
			//ThreadLog.trace("determining direct types cache for " + this.toString());
			RdfSchemaRepository repository = (RdfSchemaRepository)_source;
			
			_directTypeStatements = new StatementList();
			
			StatementList typeStatements = new StatementList();
			List classes = new ArrayList();

			//ThreadLog.trace("Getting all type statements...");
			StatementIterator iter = repository.getStatements(this, repository.RDF_TYPE_NODE, null, false);
			while (iter.hasNext()) {
				Statement st = iter.next();
				if (st.getPredicate().equals(repository.RDF_TYPE_NODE)){
					typeStatements.add(st);
					classes.add(st.getObject());
				}
			}
			iter.close();
			//ThreadLog.trace("type statements retrieved: " + classes.size());
			
			StatementIterator typeIter = new MemStatementIterator(typeStatements);
			
			Statement rdfsResourceStatement = null;
			while (typeIter.hasNext()) {
				Statement typeSt = typeIter.next();
				
				ResourceNode type = (ResourceNode)typeSt.getObject();
				if (type.equals(repository.RDFS_RESOURCE_NODE)) {
					// skip the top class: it is only a direct type if no other direct types
					// can be found.
					rdfsResourceStatement = typeSt;
					continue;
				}
				
				StatementIterator subClassIter = type.getDirectSubClassStatements();
				
				boolean directType = true;
				while (directType && subClassIter.hasNext()) {
					Resource aClass = subClassIter.next().getSubject();
					
					if (classes.contains(aClass) ) {
						directType = false;
					}
				}
				
				if (directType) {
					//ThreadLog.trace("found a direct type statement: " + typeSt.toString());
					_directTypeStatements.add(typeSt);
				}
			}
			typeIter.close();
			
			// if no direct type relations were found,the only direct type is the top class, rdfs:Resource
			if (_directTypeStatements.size() == 0) {
				_directTypeStatements.add(rdfsResourceStatement);
			}
		}
		
		
		//ThreadLog.trace("Returning direct type cache for " + this.toString());
		return new MemStatementIterator(_directTypeStatements);
	}
	
	public boolean isDirectType(Resource aClass) {
		
		boolean result = false;
		
		StatementIterator directTypes = getDirectTypeStatements();
		
		while (directTypes.hasNext()) {
			Statement st = directTypes.next();
			if (st.getObject().equals(aClass)) {
				result = true;
				break;
			}
		}
		directTypes.close();
		return result;
	}

	public StatementIterator getDirectSubPropertyStatements() {
		if (_directSubPropertyStatements == null) {
			//ThreadLog.trace("Determinining directSubProp cache for " + this.toString());
			_directSubPropertyStatements = new StatementList();
			
			RdfSchemaRepository repository = (RdfSchemaRepository)_source;
	
			StatementIterator subPropertyOfIter = repository.getSubPropertyOf(null, this);
			List subProps = new ArrayList();
			
			while (subPropertyOfIter.hasNext()) {
				Statement st = subPropertyOfIter.next();
				//skip equivalent properties
				if (!repository.isSubPropertyOf(this, st.getSubject())) { 
					subProps.add(st.getSubject());
				}
			}
			subPropertyOfIter.close();
			
			subPropertyOfIter = repository.getSubPropertyOf(null, this);
			while (subPropertyOfIter.hasNext()) {
				
				Statement stat = subPropertyOfIter.next();
				
				if (_isDirectSubPropertyStat(stat, subProps)) {
					//ThreadLog.trace("found direct subProp statement: " + stat.toString());
					_directSubPropertyStatements.add(stat);
				}
			}
		}
		//ThreadLog.trace("Returning directSubProp cache for " + this.toString());
		return new MemStatementIterator(_directSubPropertyStatements);
	}
	
	public boolean isDirectSubProperty(Resource aProperty) {
		boolean result = false;
		
		StatementIterator subProps = getDirectSubPropertyStatements();
		
		while (subProps.hasNext()) {
			Statement st = subProps.next();
			if (st.getSubject().equals(aProperty)) {
				result = true;
				break;
			}
		}
		subProps.close();
		
		return result;
	}
	
	private boolean _isDirectSubPropertyStat(Statement stat, List subProps) {
		RdfSchemaRepository repository = (RdfSchemaRepository)_source;
		
		//ThreadLog.trace("Determining if " + stat.toString() + " is a direct subprop of " + this.toString());
		ResourceNode subNode = (ResourceNode)stat.getSubject();
		
		if (subNode == this) {
			// a prop is never a direct subprop of itself.
			//ThreadLog.trace("not a direct subprop");
			return false;
		}

		if (repository.isSubPropertyOf(this, subNode)) {
			// cycle in the property hierarchy, props are equivalent
			//ThreadLog.trace("not a direct subprop");
			return false;
		}

		// 'subProp' is a direct subclass of 'superProp' when the
		// set of superproperties of 'subProp' and the set of subproperties
		// of 'superProp' are disjoint (when ignorning 'subProp' and
		// 'superProp' in the sets).
		boolean isDirect = true;

		// Check whether one of the superproperties of 'subProp' is
		// present in 'subProps', again ignoring 'subProp'
		// and 'superProp'
		StatementIterator supsOfSubPropIter = repository.getSubPropertyOf(subNode, null);
		while (supsOfSubPropIter.hasNext()) {
			Statement st = supsOfSubPropIter.next();

			if (st.getObject() instanceof Resource) {
				Resource r = (Resource)st.getObject();

				if (r != subNode && r != this &&
					subProps.contains(r) &&
					!repository.isSubPropertyOf(r, subNode)) // ignore equivalent properties
				{
					isDirect = false;
					break;
				}
			}
		}
		supsOfSubPropIter.close();

		//ThreadLog.trace("Returning direct subProperty: " + isDirect);
		return isDirect;
	}

	public void clearCache() {
		_directSubClassStatements = null;
		_directTypeStatements = null;
		_directSubPropertyStatements = null;
	}
}
