/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sailimpl.memory;

import org.openrdf.sesame.config.SailConfig;
import org.openrdf.sesame.constants.RDFFormat;

/**
 * Extension of SailConfig offering specific utility methods for configuring an
 * <tt>org.openrdf.sesame.sailimpl.memory.RdfSource</tt> Sail. This class can be
 * used to configure a memory Sail that wraps an RDF data file.
 **/
public class RdfSourceConfig extends SailConfig {

/*--------------+
| Constructors  |
+--------------*/

	/**
	 * Creates a new and unconfigured RdfSourceConfig object for the
	 * specified Sail class. This is a protected constructor that can be used
	 * by subclasses to specify a different Sail class.
	 *
	 * @param className A fully qualified Sail class name.
	 * @param dummy Dummy parameter to distinguish this constructor from the
	 * other constructor with a single String parameter.
	 **/
	protected RdfSourceConfig(String className, boolean dummy) {
		super(className);
	}

	/**
	 * Creates a new RdfSourceConfig object for an
	 * <tt>org.openrdf.sesame.sailimpl.memory.RdfSource</tt> Sail that will
	 * contain the XML-encoded RDF data from specified file.
	 *
	 * @param file The data file.
	 **/
	public RdfSourceConfig(String file) {
		this(file, RDFFormat.RDFXML);
	}

	/**
	 * Creates a new RdfSourceConfig object for an
	 * <tt>org.openrdf.sesame.sailimpl.memory.RdfSource</tt> Sail that will
	 * contain the data from specified file.
	 *
	 * @param file The data file.
	 * @param dataFormat The format of the data in the data file. Legal values
	 * are <tt>RDFFormat.RDFXML</tt>, <tt>RDFFormat.NTRIPLES</tt> and
	 * <tt>RDFFormat.TURTLE</tt>.
	 **/
	public RdfSourceConfig(String file, RDFFormat dataFormat) {
		this("org.openrdf.sesame.sailimpl.memory.RdfSource", true);
		setFile(file);
		setDataFormat(dataFormat);
	}

/*--------------+
| Methods       |
+--------------*/

	/**
	 * Sets the data file for the Sail.
	 **/
	public void setFile(String file) {
		setParameter(RdfSource.FILE_KEY, file);
	}

	/**
	 * Gets the data file that has been configured for the Sail.
	 *
	 * @return The file.
	 **/
	public String getFile() {
		return getParameter(RdfSource.FILE_KEY);
	}

	/**
	 * Sets the format of the data in the data file.
	 *
	 * @param dataFormat One of <tt>RDFFormat.RDFXML</tt>,
	 * <tt>RDFFormat.NTRIPLES</tt> and <tt>RDFFormat.TURTLE</tt>.
	 **/
	public void setDataFormat(RDFFormat dataFormat) {
		if (dataFormat == RDFFormat.RDFXML ||
			dataFormat == RDFFormat.NTRIPLES ||
			dataFormat == RDFFormat.TURTLE)
		{
			setParameter(RdfSource.DATA_FORMAT_KEY, dataFormat.toString());
		}
		else {
			throw new IllegalArgumentException(
					"Illegal value for dataFormat parameter: " + dataFormat.toString());
		}
	}

	/**
	 * Gets the data format of the data in the data file that has been
	 * configured for the Sail.
	 *
	 * @return The data format, or <tt>null</tt> if no data format has been
	 * configured.
	 **/
	public RDFFormat getDataFormat() {
		String dataFormatStr = getParameter(RdfSource.DATA_FORMAT_KEY);

		if (dataFormatStr != null) {
			return RDFFormat.forValue(dataFormatStr);
		}

		return null;
	}
}
