/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2007 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sailimpl.memory;

import java.io.File;
import java.io.IOException;

import org.openrdf.model.Resource;
import org.openrdf.model.Statement;
import org.openrdf.model.URI;
import org.openrdf.model.Value;
import org.openrdf.model.impl.URIImpl;

import org.openrdf.rio.RdfDocumentWriter;

import org.openrdf.sesame.constants.RDFFormat;
import org.openrdf.sesame.export.RdfExport;
import org.openrdf.sesame.sail.LiteralIterator;
import org.openrdf.sesame.sail.SailInitializationException;
import org.openrdf.sesame.sail.SailUpdateException;
import org.openrdf.sesame.sail.StatementIterator;
import org.openrdf.sesame.sail.util.EmptyStatementIterator;

/**
 * An implementation of the RdfSchemaRepository interface extending the
 * memory repository with Schema inferencing capabilities.
 *
 * @author Jeen Broekstra
 * @author Arjohn Kampman
 * @version $Revision: 1.19.2.4 $
 */
public class RdfSchemaRepository extends RdfRepository
	implements org.openrdf.sesame.sail.RdfSchemaRepository
{

/*----------------------------------+
| Constants                         |
+----------------------------------*/

	protected URINode RDF_TYPE_NODE;
	protected URINode RDF_PROPERTY_NODE;
	protected URINode RDF_SUBJECT_NODE;
	protected URINode RDF_PREDICATE_NODE;
	protected URINode RDF_OBJECT_NODE;
	protected URINode RDF_STATEMENT_NODE;
	protected URINode RDF_LIST_NODE;
	protected URINode RDF_FIRST_NODE;
	protected URINode RDF_REST_NODE;
	protected URINode RDF_NIL_NODE;
	protected URINode RDF_BAG_NODE;
	protected URINode RDF_SEQ_NODE;
	protected URINode RDF_ALT_NODE;
	protected URINode RDF_XMLLITERAL_NODE;
	protected URINode RDF_VALUE_NODE;

	protected URINode RDFS_RESOURCE_NODE;
	protected URINode RDFS_LITERAL_NODE;
	protected URINode RDFS_CLASS_NODE;
	protected URINode RDFS_CONTAINER_NODE;
	protected URINode RDFS_CONTAINERMEMBERSHIPPROPERTY_NODE;
	protected URINode RDFS_DOMAIN_NODE;
	protected URINode RDFS_RANGE_NODE;
	protected URINode RDFS_SUBCLASSOF_NODE;
	protected URINode RDFS_SUBPROPERTYOF_NODE;
	protected URINode RDFS_LABEL_NODE;
	protected URINode RDFS_COMMENT_NODE;
	protected URINode RDFS_SEEALSO_NODE;
	protected URINode RDFS_ISDEFINEDBY_NODE;
	protected URINode RDFS_MEMBER_NODE;
	protected URINode RDFS_DATATYPE_NODE;

/*---------------------------------+
| Variables                        |
+---------------------------------*/

	private boolean _statementsRemoved = false;

	private RdfMTInferencer _inferencer;

/*---------------------------------+
| Constructors                     |
+---------------------------------*/

	public RdfSchemaRepository() {
		super();

		_inferencer = new RdfMTInferencer(this);
	}

/*---------------------------------+
| Methods                          |
+---------------------------------*/

	// overrides RdfRepository.initialize(File, RDFFormat, boolean)
	public void initialize(File file, RDFFormat dataFormat, boolean compressFile)
		throws SailInitializationException
	{
		_initRdfPrimitives();

		super.initialize(file, dataFormat, compressFile);

		// Any statements loaded from a file are input for the inferencer
		_inferencer.notifyNewStatements(_statements);

		// Add axioms and apply inferencing
		_inferencer.initialize();
	}

	protected void _initRdfPrimitives() {
		RDF_TYPE_NODE = _createURINode(URIImpl.RDF_TYPE);
		RDF_PROPERTY_NODE = _createURINode(URIImpl.RDF_PROPERTY);
		RDF_SUBJECT_NODE = _createURINode(URIImpl.RDF_SUBJECT);
		RDF_PREDICATE_NODE = _createURINode(URIImpl.RDF_PREDICATE);
		RDF_OBJECT_NODE = _createURINode(URIImpl.RDF_OBJECT);
		RDF_STATEMENT_NODE = _createURINode(URIImpl.RDF_STATEMENT);
		RDF_LIST_NODE = _createURINode(URIImpl.RDF_LIST);
		RDF_FIRST_NODE = _createURINode(URIImpl.RDF_FIRST);
		RDF_REST_NODE = _createURINode(URIImpl.RDF_REST);
		RDF_NIL_NODE = _createURINode(URIImpl.RDF_NIL);
		RDF_BAG_NODE = _createURINode(URIImpl.RDF_BAG);
		RDF_SEQ_NODE = _createURINode(URIImpl.RDF_SEQ);
		RDF_ALT_NODE = _createURINode(URIImpl.RDF_ALT);
		RDF_XMLLITERAL_NODE = _createURINode(URIImpl.RDF_XMLLITERAL);
		RDF_VALUE_NODE = _createURINode(URIImpl.RDF_VALUE);

		RDFS_RESOURCE_NODE = _createURINode(URIImpl.RDFS_RESOURCE);
		RDFS_LITERAL_NODE = _createURINode(URIImpl.RDFS_LITERAL);
		RDFS_CLASS_NODE = _createURINode(URIImpl.RDFS_CLASS);
		RDFS_CONTAINER_NODE = _createURINode(URIImpl.RDFS_CONTAINER);
		RDFS_CONTAINERMEMBERSHIPPROPERTY_NODE = _createURINode(URIImpl.RDFS_CONTAINERMEMBERSHIPPROPERTY);
		RDFS_DOMAIN_NODE = _createURINode(URIImpl.RDFS_DOMAIN);
		RDFS_RANGE_NODE = _createURINode(URIImpl.RDFS_RANGE);
		RDFS_SUBCLASSOF_NODE = _createURINode(URIImpl.RDFS_SUBCLASSOF);
		RDFS_SUBPROPERTYOF_NODE = _createURINode(URIImpl.RDFS_SUBPROPERTYOF);
		RDFS_LABEL_NODE = _createURINode(URIImpl.RDFS_LABEL);
		RDFS_COMMENT_NODE = _createURINode(URIImpl.RDFS_COMMENT);
		RDFS_SEEALSO_NODE = _createURINode(URIImpl.RDFS_SEEALSO);
		RDFS_ISDEFINEDBY_NODE = _createURINode(URIImpl.RDFS_ISDEFINEDBY);
		RDFS_MEMBER_NODE = _createURINode(URIImpl.RDFS_MEMBER);
		RDFS_DATATYPE_NODE = _createURINode(URIImpl.RDFS_DATATYPE);
	}

/*--------------------------------------------------------+
| Overridden methods from RdfSource                       |
+--------------------------------------------------------*/

	// inherit javadoc
	public StatementIterator getStatements(Resource subj, URI pred, Value obj) {
		return getStatements(subj, pred, obj, false);
	}

	// inherit javadoc
	public boolean hasStatement(Resource subj, URI pred, Value obj) {
		return hasStatement(subj, pred, obj, false);
	}

/*--------------------------------------------------------+
| Methods from org.openrdf.sesame.sail.RdfSchemaSource |
+--------------------------------------------------------*/

	// inherit javadoc
	public StatementIterator getExplicitStatements(Resource subj, URI pred, Value obj) {
		return getStatements(subj, pred, obj, true);
	}

	public StatementIterator getStatements(Resource subj, URI pred, Value obj, boolean explicitOnly) {
		ResourceNode subjNode = null;
		URINode predNode = null;
		ValueNode objNode = null;

		if (subj != null) {
			subjNode = _getResourceNode(subj);
			if (subjNode == null) {
				return new EmptyStatementIterator();
			}
		}
		if (pred != null) {
			predNode = _getURINode(pred);
			if (predNode == null) {
				return new EmptyStatementIterator();
			}
		}
		if (obj != null) {
			objNode = _getValueNode(obj);
			if (objNode == null) {
				return new EmptyStatementIterator();
			}
		}

		StatementList smallestList = _statements;

		if (subjNode != null) {
			StatementList l = subjNode.getSubjectStatementList();
			if (l.size() < smallestList.size()) {
				smallestList = l;
			}
		}
		if (predNode != null) {
			StatementList l = predNode.getPredicateStatementList();
			if (l.size() < smallestList.size()) {
				smallestList = l;
			}
		}
		if (objNode != null) {
			StatementList l = objNode.getObjectStatementList();
			if (l.size() < smallestList.size()) {
				smallestList = l;
			}
		}

		// Iterate over 'smallestList'
		return new MemStatementIterator(smallestList, subjNode, predNode, objNode, explicitOnly);
	}

	// inherit javadoc
	public boolean hasExplicitStatement(Resource subj, URI pred, Value obj) {
		return hasStatement(subj, pred, obj, true);
	}

	public boolean hasStatement(Resource subj, URI pred, Value obj, boolean explicitOnly) {
		StatementIterator stIter = getStatements(subj, pred, obj, explicitOnly);
		boolean result = stIter.hasNext();
		stIter.close();
		return result;
	}

	// inherit javadoc
	public StatementIterator getClasses() {
		return getStatements(null, RDF_TYPE_NODE, RDFS_CLASS_NODE);
	}

	// inherit javadoc
	public boolean isClass(Resource resource) {
		return hasStatement(resource, RDF_TYPE_NODE, RDFS_CLASS_NODE);
	}

	// inherit javadoc
	public StatementIterator getProperties() {
		return getStatements(null, RDF_TYPE_NODE, RDF_PROPERTY_NODE);
	}

	// inherit javadoc
	public boolean isProperty(Resource resource) {
		return hasStatement(resource, RDF_TYPE_NODE, RDF_PROPERTY_NODE);
	}

	// inherit javadoc
	public StatementIterator getSubClassOf(Resource subClass, Resource superClass) {
		return getStatements(subClass, RDFS_SUBCLASSOF_NODE, superClass);
	}

	// inherit javadoc
	public boolean isSubClassOf(Resource subClass, Resource superClass) {
		return hasStatement(subClass, RDFS_SUBCLASSOF_NODE, superClass);
	}

	// inherit javadoc
	public StatementIterator getDirectSubClassOf(Resource subClass, Resource superClass) {
		ResourceNode subClassNode = null;
		ResourceNode superClassNode = null;

		if (superClass != null && subClass != null) {
			superClassNode = _getResourceNode(superClass);
			if (superClassNode == null) {
				return new EmptyStatementIterator();
			}
			subClassNode = _getResourceNode(subClass);
			if (subClassNode == null) {
				return new EmptyStatementIterator();
			}
			StatementIterator iter = superClassNode.getDirectSubClassStatements();
			StatementList list = new StatementList();
			while (iter.hasNext()) {
				Statement st = iter.next();
				if (st.getSubject().equals(subClass)) {
					list.add(st);
					break;
				}
			}
			iter.close();
			return new MemStatementIterator(list);
		}
		
		if (superClass != null) {
			superClassNode = _getResourceNode(superClass);
			if (superClassNode == null) {
				return new EmptyStatementIterator();
			}
			if (subClass == null) { 
				return superClassNode.getDirectSubClassStatements();
			}
		}
		if (subClass != null) {
			subClassNode = _getResourceNode(subClass);
			if (subClassNode == null) {
				return new EmptyStatementIterator();
			}
			if (superClass == null) {
				StatementIterator superClassIter = getStatements(subClass, RDFS_SUBCLASSOF_NODE, null, false);
				StatementList directSuperClasses = new StatementList();
				
				if (! superClassIter.hasNext()) {
					// supplied subClass is not a class.
					return new EmptyStatementIterator();
				}
				
				Statement rdfsResourceStatement = null;
				
				while (superClassIter.hasNext()) {
					Statement st = superClassIter.next();
					superClassNode = (ResourceNode)st.getObject();
					if (superClassNode.equals(RDFS_RESOURCE_NODE)) {
						// skip rdfs:Resource, the check for direct subclass is very expensive, and it
						// is only a direct subclass if no others can be found anyway.
						rdfsResourceStatement = st;
						continue;
					}
					if (superClassNode.isDirectSubClass(subClass)) {
						directSuperClasses.add(st);
					}
				}
				superClassIter.close();
				
				// if the list of direct superclasses is empty, we have to add rdfs:Resource as
				// a direct superclass.
				if (directSuperClasses.size() == 0) {
					if (rdfsResourceStatement != null) {
						directSuperClasses.add(rdfsResourceStatement);
					}
				}
				return new MemStatementIterator(directSuperClasses);
			}
		}

		// both arguments null: retrieve all direct subClass relations in the graph.
		StatementIterator subClassIter = new MemStatementIterator(RDFS_SUBCLASSOF_NODE.getPredicateStatementList());
		StatementList directSubClassStatements = new StatementList();
		
		while (subClassIter.hasNext()) {
			Statement st = subClassIter.next();
	
			ResourceNode subject = (ResourceNode)st.getSubject();
			
			StatementIterator iter = subject.getDirectSubClassStatements();
			while (iter.hasNext()) {
				directSubClassStatements.add(iter.next());
			}
			iter.close();
		}
		subClassIter.close();
		
		return new MemStatementIterator(directSubClassStatements);
	}

	// inherit javadoc
	public boolean isDirectSubClassOf(Resource subClass, Resource superClass) {
		StatementIterator stIter = getDirectSubClassOf(subClass, superClass);
		boolean result = stIter.hasNext();
		stIter.close();
		return result;
	}

	// inherit javadoc
	public StatementIterator getSubPropertyOf(Resource subProperty, Resource superProperty) {
		return getStatements(subProperty, RDFS_SUBPROPERTYOF_NODE, superProperty);
	}

	// inherit javadoc
	public boolean isSubPropertyOf(Resource subProperty, Resource superProperty) {
		return hasStatement(subProperty, RDFS_SUBPROPERTYOF_NODE, superProperty);
	}

	// inherit javadoc
	public StatementIterator getDirectSubPropertyOf(Resource subProp, Resource superProp) {
		ResourceNode subPropNode = null;
		ResourceNode superPropNode = null;
		if (superProp != null && subProp != null) {
			superPropNode = _getResourceNode(superProp);
			if (superPropNode == null) {
				return new EmptyStatementIterator();
			}
			subPropNode = _getResourceNode(subProp);
			if (subPropNode == null) {
				return new EmptyStatementIterator();
			}
			StatementIterator iter = superPropNode.getDirectSubPropertyStatements();
			StatementList list = new StatementList();
			while (iter.hasNext()) {
				Statement st = iter.next();
				if (st.getSubject().equals(subProp)) {
					list.add(st);
					break;
				}
			}
			iter.close();
			return new MemStatementIterator(list);
		}
		if (superProp != null) {
			superPropNode = _getResourceNode(superProp);
			if (superPropNode == null) {
				return new EmptyStatementIterator();
			}
			if (subProp == null) { 
				return superPropNode.getDirectSubPropertyStatements();
			}
		}
		if (subProp != null) {
			subPropNode = _getResourceNode(subProp);
			if (subPropNode == null) {
				return new EmptyStatementIterator();
			}
			if (superProp == null) {
				StatementIterator superPropIter = getStatements(subProp, RDFS_SUBPROPERTYOF_NODE, null, false);
				StatementList directSuperProps = new StatementList();
				
				while (superPropIter.hasNext()) {
					Statement st = superPropIter.next();
					superPropNode = (ResourceNode)st.getObject();
					
					if (superPropNode.isDirectSubProperty(subProp)) {
						directSuperProps.add(st);
					}
				}
				superPropIter.close();
				
				return new MemStatementIterator(directSuperProps);
			}
		}

		// both arguments null: retrieve all direct subClass relations in the graph.
		StatementIterator subPropIter = new MemStatementIterator(RDFS_SUBPROPERTYOF_NODE.getPredicateStatementList());
		StatementList directSubPropStatements = new StatementList();
		
		while (subPropIter.hasNext()) {
			Statement st = subPropIter.next();
	
			ResourceNode subject = (ResourceNode)st.getSubject();
			
			StatementIterator iter = subject.getDirectSubPropertyStatements();
			while (iter.hasNext()) {
				directSubPropStatements.add(iter.next());
			}
			iter.close();
		}
		subPropIter.close();
		
		return new MemStatementIterator(directSubPropStatements);
	}

	// inherit javadoc
	public boolean isDirectSubPropertyOf(Resource subProperty, Resource superProperty) {
		StatementIterator stIter = getDirectSubPropertyOf(subProperty, superProperty);
		boolean result = stIter.hasNext();
		stIter.close();
		return result;
	}

	// inherit javadoc
	public StatementIterator getDomain(Resource prop, Resource domain) {
		return getStatements(prop, RDFS_DOMAIN_NODE, domain);
	}

	// inherit javadoc
	public StatementIterator getRange(Resource prop, Resource range) {
		return getStatements(prop, RDFS_RANGE_NODE, range);
	}

	// inherit javadoc
	public StatementIterator getType(Resource anInstance, Resource aClass) {
		return getStatements(anInstance, RDF_TYPE_NODE, aClass);
	}

	// inherit javadoc
	public boolean isType(Resource anInstance, Resource aClass) {
		return hasStatement(anInstance, RDF_TYPE_NODE, aClass);
	}

	public StatementIterator getDirectType(Resource anInstance, Resource aClass) {
		ResourceNode instanceNode = null;
		ResourceNode classNode = null;
		if (anInstance != null && aClass != null) {
			classNode = _getResourceNode(aClass);
			if (classNode == null) {
				return new EmptyStatementIterator();
			}
			instanceNode = _getResourceNode(anInstance);
			if (instanceNode == null) {
				return new EmptyStatementIterator();
			}
			StatementIterator iter = instanceNode.getDirectTypeStatements();
			StatementList list = new StatementList();
			while (iter.hasNext()) {
				Statement st = iter.next();
				if (st.getObject().equals(classNode)) {
					list.add(st);
					break;
				}
			}
			iter.close();
			return new MemStatementIterator(list);
		}
		if (anInstance != null) {
			instanceNode = _getResourceNode(anInstance);
			if (instanceNode == null) {
				return new EmptyStatementIterator();
			}
			if (aClass == null) {
				return instanceNode.getDirectTypeStatements();
			}
		}
		if (aClass != null) {
			classNode = _getResourceNode(aClass);
			if (classNode == null) {
				return new EmptyStatementIterator();
			}
			if (anInstance == null) {
				StatementList directInstances = new StatementList();
				StatementIterator instanceIter = getStatements(null, RDF_TYPE_NODE, classNode, false);
				
				while (instanceIter.hasNext()) {
					Statement st = instanceIter.next();
					if (((ResourceNode)st.getSubject()).isDirectType(aClass)) {
						directInstances.add(st);
					}
				}
				instanceIter.close();
				return new MemStatementIterator(directInstances);
			}
		}

		// both arguments null: retrieve all direct type relations in the graph.
		StatementIterator typeIter = new MemStatementIterator(RDF_TYPE_NODE.getPredicateStatementList());
		StatementList directTypeStatements = new StatementList();
		
		while (typeIter.hasNext()) {
			Statement st = typeIter.next();
			ResourceNode subject = (ResourceNode)st.getSubject();
			
			StatementIterator iter = subject.getDirectTypeStatements();
			
			while (iter.hasNext()) {
				directTypeStatements.add(iter.next());
			}
			iter.close();
		}
		typeIter.close();
		
		return new MemStatementIterator(directTypeStatements);
	}

	// inherit javadoc
	public boolean isDirectType(Resource anInstance, Resource aClass) {
		StatementIterator stIter = getDirectType(anInstance, aClass);
		boolean result = stIter.hasNext();
		stIter.close();
		return result;
	}

	public LiteralIterator getLiterals(String label, String language, URI datatype) {
		return new MemLiteralIterator(
				_literalNodesMap.values(), label, language, datatype);
	}

	// inherit javadoc
	public void commitTransaction() {

		_clearAllCaches();
		
		if (_statementsRemoved) {
			_removeInferredStatements();

			// Mark all remaining (=explicit) statements as new
			_inferencer.notifyNewStatements(_statements);

			// Add axioms and apply inferencing
			_inferencer.initialize();
		}
		else {
			_inferencer.doInferencing();
		}

		_statementsRemoved = false;

		super.commitTransaction();
	}

	// overrides RdfRepository.addStatement()
	public void addStatement(Resource subj, URI pred, Value obj)
		throws SailUpdateException
	{
		if (!transactionStarted()) {
			throw new SailUpdateException("no transaction started.");
		}

		Statement st = _addStatement(subj, pred, obj);

		_inferencer.notifyNewStatement(st);
		_sailChangedEvent.setStatementsAdded(true);
	}

	// overrides RdfRepository.removeStatements()
	public int removeStatements(Resource subj, URI pred, Value obj)
		throws SailUpdateException
	{
		if (!transactionStarted()) {
			throw new SailUpdateException("no transaction started.");
		}

		int nrOfStatements = super.removeStatements(subj, pred, obj);

		if (nrOfStatements > 0) {
			_statementsRemoved = true;
			_sailChangedEvent.setStatementsRemoved(true);
		}

		return nrOfStatements;
	}

	/**
	 * Overrides <tt>RdfRepository._matchesForRemoval(...)</tt> such that it
	 * only matches explicit statements. Inferred statements cannot be removed
	 * by hand, removal of these statements are handled by the inferencer.
	 *
	 * @see RdfRepository#_matchesForRemoval
	 **/
	protected boolean _matchesForRemoval(Statement st, Resource subj, URI pred, Value obj) {
		return ((MemStatement)st).isExplicit() &&
				super._matchesForRemoval(st, subj, pred, obj);
	}


	public void clearRepository()
		throws SailUpdateException
	{
		super.clearRepository();

		_initRdfPrimitives();

		// Add axioms and apply inferencing
		_inferencer.initialize();
	}

	// Method called by RdfMTInferencer
	protected MemStatement _getMemStatement(Resource subj, URI pred, Value obj) {
		MemStatement result = null;

		StatementIterator iter = getStatements(subj, pred, obj, false);
		if (iter.hasNext()) {
			result = (MemStatement)iter.next();
		}
		iter.close();

		return result;
	}

	// overrides RdfSource._addStatement
	protected MemStatement _addStatement(Resource subj, URI pred, Value obj) {
		MemStatement st = super._addStatement(subj, pred, obj);

		// An implicit statement could already have been present, make
		// sure the added statement is explicit.
		st.setExplicit(true);

		return st;
	}

	// override RdfRepository._exportData(RdfDocumentWriter): only write explicit statements.
	protected void _exportData(RdfDocumentWriter docWriter)
		throws IOException
	{
		RdfExport rdfExport = new RdfExport();
		rdfExport.exportRdf(this, docWriter, true, true, true, false);
	}

/*----------------+
| private methods |
+----------------*/

	/**
	 * Removes all inferred statements.
	 **/
	private void _removeInferredStatements() {
		int size = _statements.size();
		for (int i = size - 1; i >= 0; --i) {
			MemStatement st = (MemStatement)_statements.get(i);

			if (!st.isExplicit()) {
				// Inferred statement, remove all references to it from ValueNodes.
				ResourceNode rn = (ResourceNode)st.getSubject();
				rn.removeSubjectStatement(st);

				URINode un = (URINode)st.getPredicate();
				un.removePredicateStatement(st);

				ValueNode vn = (ValueNode)st.getObject();
				vn.removeObjectStatement(st);

				_statements.remove(i);
			}
		}
	}


	private void _clearAllCaches() {
		
		// a single sweep over all subjects of type relations is enough: if something is a 
		// subclass or subproperty, it is by definition present as subject of a type relation.
		StatementIterator allTypeRelations = new MemStatementIterator(RDF_TYPE_NODE.getPredicateStatementList());
		while (allTypeRelations.hasNext()) {
			Statement st = allTypeRelations.next();
			((ResourceNode)st.getSubject()).clearCache();
		}
		allTypeRelations.close();

		/*
		StatementIterator allSubClassRelations = new MemStatementIterator(RDFS_SUBCLASSOF_NODE.getPredicateStatementList());
		
		while (allSubClassRelations.hasNext()) {
			Statement st = allSubClassRelations.next();
			((ResourceNode)st.getSubject()).clearCache();
		}
		allSubClassRelations.close();
		
		StatementIterator allSubPropRelations = new MemStatementIterator(RDFS_SUBPROPERTYOF_NODE.getPredicateStatementList());
		while (allSubPropRelations.hasNext()) {
			Statement st = allSubPropRelations.next();
			((ResourceNode)st.getSubject()).clearCache();
		}
		allSubPropRelations.close();
		*/		
	}
}
