/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sailimpl.memory;

import org.openrdf.sesame.constants.RDFFormat;

/**
 * Extension of RdfSourceConfig offering additional utility methods for
 * configuring a <tt>org.openrdf.sesame.sailimpl.memory.RdfRepository</tt> Sail.
 **/
public class RdfRepositoryConfig extends RdfSourceConfig {

/*--------------+
| Constructors  |
+--------------*/

	/**
	 * Creates a new and unconfigured RdfRepositoryConfig object for the
	 * specified Sail class. This is a protected constructor that can be used
	 * by subclasses to specify a different Sail class.
	 *
	 * @param className A fully qualified Sail class name.
	 * @param dummy Dummy parameter to distinguish this constructor from the
	 * other constructor with a single String parameter.
	 **/
	protected RdfRepositoryConfig(String className, boolean dummy) {
		super(className, true);
	}

	/**
	 * Creates a new RdfRepositoryConfig object for an
	 * <tt>org.openrdf.sesame.sailimpl.memory.RdfRepository</tt> Sail. The
	 * data in the configured sail will be volatile if no data file is
	 * specified; all data contained in the Sail will be lost when the Sail
	 * object is destroyed.
	 *
	 * @see #setFile
	 **/
	public RdfRepositoryConfig() {
		this("org.openrdf.sesame.sailimpl.memory.RdfRepository", true);
	}

	/**
	 * Creates a new RdfRepositoryConfig object for an
	 * <tt>org.openrdf.sesame.sailimpl.memory.RdfRepository</tt> Sail that will
	 * contain the XML-encoded RDF data from specified file.
	 *
	 * @param file The data file.
	 **/
	public RdfRepositoryConfig(String file) {
		this(file, RDFFormat.RDFXML);
	}

	/**
	 * Creates a new RdfRepositoryConfig object for an
	 * <tt>org.openrdf.sesame.sailimpl.memory.RdfRepository</tt> Sail that will
	 * contain the data from specified file.
	 *
	 * @param file The data file.
	 * @param dataFormat The format of the data in the data file. Legal values
	 * are <tt>RDFFormat.RDFXML</tt>, <tt>RDFFormat.NTRIPLES</tt> and
	 * <tt>RDFFormat.TURTLE</tt>.
	 **/
	public RdfRepositoryConfig(String file, RDFFormat dataFormat) {
		this();
		setFile(file);
		setDataFormat(dataFormat);
	}

/*--------------+
| Methods       |
+--------------*/

	/**
	 * Sets whether the data in the data file is/should be compressed.
	 **/
	public void setCompressFile(boolean compressFile) {
		String compressFileStr = compressFile ? "yes" : "no";
		setParameter(RdfSource.COMPRESS_FILE_KEY, compressFileStr);
	}

	/**
	 * Checks whether this configuration object specifies that the Sail should
	 * compress its data before storing it in the data file.
	 **/
	public boolean getCompressFile() {
		return "yes".equalsIgnoreCase( getParameter(RdfSource.COMPRESS_FILE_KEY) );
	}

	/**
	 * Sets the synchronization delay for the Sail. See
	 * RdfRepositoryConfig.setSyncDelay(long) for more details.
	 *
	 * @see RdfRepositoryConfig#setSyncDelay
	 **/
	public void setSyncDelay(long syncDelay) {
		setParameter(RdfRepository.SYNC_DELAY_KEY, String.valueOf(syncDelay));
	}

	/**
	 * Gets the synchronization delay that has been configured for the Sail.
	 **/
	public long getSyncDelay() {
		String syncDelayStr = getParameter(RdfRepository.SYNC_DELAY_KEY);

		if (syncDelayStr != null) {
			return Long.parseLong(syncDelayStr);
		}

		return 0L;
	}
}
