/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sailimpl.memory;

import org.openrdf.util.log.ThreadLog;
import org.openrdf.vocabulary.RDF;

import org.openrdf.model.Statement;

import org.openrdf.sesame.sail.StatementIterator;


/**
 * RDF(S) Inferencer for in-memory repositories using the rules from the 
 * RDF Semantics Recommendation (10 February 2004). 
 * See http://www.w3.org/TR/2004/REC-rdf-mt-20040210/
 *
 * @author Jeen Broekstra
 * @version $Revision: 1.11.4.2 $
 */
public class RdfMTInferencer {

/*-------------------------------------+
| Constants                            |
+-------------------------------------*/

	private static final int Rdf1    = 0;
	private static final int Rdfs2_1  = 1;
	private static final int Rdfs2_2  = 2;
	private static final int Rdfs3_1  = 3;
	private static final int Rdfs3_2  = 4;
	private static final int Rdfs4a   = 5;
	private static final int Rdfs4b   = 6;
	private static final int Rdfs5_1 = 7;
	private static final int Rdfs5_2 = 8;
	private static final int Rdfs6   = 9;
	private static final int Rdfs7_1  = 10;
	private static final int Rdfs7_2  = 11;
	private static final int Rdfs8   = 12;
	private static final int Rdfs9_1  = 13;
	private static final int Rdfs9_2  = 14;
	private static final int Rdfs10   = 15;
	private static final int Rdfs11_1  = 16;
	private static final int Rdfs11_2  = 17;
	private static final int Rdfs12   = 18;
	private static final int Rdfs13   = 19;
	private static final int RX1   = 20;

	private static final int RULECOUNT = 21;

	private static final String[] RULENAMES = {
			"   Rdf1",
			" Rdfs2_1",
			" Rdfs2_2",
			" Rdfs3_1",
			" Rdfs3_2",
			"  Rdfs4a",
			"  Rdfs4b",
			"Rdfs5_1",
			"Rdfs5_2",
			"  Rdfs6",
			" Rdfs7_1",
			" Rdfs7_2",
			"  Rdfs8",
			" Rdfs9_1",
			" Rdfs9_2",
			"  Rdfs10",
			" Rdfs11_1",
			" Rdfs11_2",
			"  Rdfs12",
			"  Rdfs13",
			"  RX1"};

	private static final boolean _ = false;
	private static final boolean X = true;

	protected static final boolean[][] TRIGGERS = {
//		    1    2_2   3_2    4b   5_2   7_1    8    9_2   11_1   12   X1
//		      2_1   3_1    4a   5_1    6    7_2   9_1   10   11_2   13
			{ _, X, _, X, _, X, _, _, _, X, X, _, _, _, X, _, _, _, _, _, _},// 1
			{ _, X, _, X, _, _, _, _, _, X, X, _, X, _, X, X, _, _, X, X, _},// 2_1
			{ _, X, _, X, _, _, _, _, _, X, X, _, X, _, X, X, _, _, X, X, _},// 2_2
			{ _, X, _, X, _, _, _, _, _, X, X, _, X, _, X, X, _, _, X, X, _},// 3_1
			{ _, X, _, X, _, _, _, _, _, X, X, _, X, _, X, X, _, _, X, X, _},// 3_2
			{ _, X, _, X, _, _, _, _, _, _, X, _, _, _, X, _, _, _, _, _, _},// 4a
			{ _, X, _, X, _, _, _, _, _, _, X, _, _, _, X, _, _, _, _, _, _},// 4b
			{ _, _, _, _, _, _, _, X, X, _, X, X, _, _, _, _, _, _, _, _, _},// 51
			{ _, _, _, _, _, _, _, X, X, _, X, X, _, _, _, _, _, _, _, _, _},// 52
			{ _, X, _, X, _, _, _, _, _, _, X, _, _, _, _, _, _, _, _, _, _},// 6
			{ _, X, X, X, X, _, _, X, X, X, X, X, X, X, X, X, X, X, X, X, X},// 7_1
			{ _, X, X, X, X, _, _, X, X, X, X, X, X, X, X, X, X, X, X, X, X},// 7_2
			{ _, X, _, X, _, _, _, _, _, _, X, _, _, X, _, _, X, X, _, _, _},// 8
			{ _, _, _, X, _, _, _, _, _, X, X, _, X, _, X, X, _, _, X, X, _},// 9_1
			{ _, _, _, X, _, _, _, _, _, X, X, _, X, _, X, X, _, _, X, X, _},// 9_2
			{ _, X, _, X, _, _, _, _, _, _, X, _, _, _, _, _, _, _, _, _, _},// 10
			{ _, _, _, _, _, _, _, _, _, _, X, _, _, X, _, _, X, X, _, _, _},// 11_1
			{ _, _, _, _, _, _, _, _, _, _, X, _, _, X, _, _, X, X, _, _, _},// 11_2
			{ _, X, _, X, _, _, X, X, X, _, X, X, _, _, _, _, _, _, _, _, _},// 12
			{ _, X, _, X, _, _, _, _, _, _, X, _, _, X, _, _, X, X, _, _, _},// 13
			{ _, X, _, X, _, _, _, _, _, _, X, _, _, _, X, _, _, _, X, _, _},// X1
			};

/*-------------------------------------+
| Variables                            |
+-------------------------------------*/

	private RdfSchemaRepository _repository;

	private StatementList _newStatements;
	private StatementList _newThisIteration;

	/** Flags indicating which rules should be evaluated. **/
	private boolean[] _checkRule = new boolean[RULECOUNT];

	/** Flags indicating which rules should be evaluated next iteration. **/
	private boolean[] _checkRuleNextIter = new boolean[RULECOUNT];

	private int _totalInferred = 0;

	// number of inferred statements per rule.
	private int[] _ruleCount = new int[RULECOUNT];

/*-------------------------------------+
| Constructors                         |
+-------------------------------------*/

	protected RdfMTInferencer(RdfSchemaRepository repository) {
		_repository = repository;
	}

/*-------------------------------------+
| Methods                              |
+-------------------------------------*/

	protected void initialize() {
		_addAxiomStatements();
		doInferencing();
	}

	protected void notifyNewStatement(Statement newStatement) {
		if (_newStatements == null) {
			_newStatements = new StatementList(512);
		}
		_newStatements.add(newStatement);
	}

	protected void notifyNewStatements(StatementList newStatements) {
		if (_newStatements == null) {
			_newStatements = new StatementList(newStatements);
		}
		else {
			_newStatements.addAll(newStatements);
		}
	}

	protected void doInferencing() {
		if (_newStatements == null) {
			// There's nothing to do
			return;
		}

		// initialize some vars
		_totalInferred = 0;
		int iteration = 0;
		int nofInferred = 1;

		// All rules needs to be checked:
		for (int i = 0; i < RULECOUNT; i++) {
			_ruleCount[i] = 0;
			_checkRuleNextIter[i] = true;
		}

		while (nofInferred > 0) {
			iteration++;
			_prepareIteration();

			nofInferred = 0;
			nofInferred += _applyRuleRdf1();
			nofInferred += _applyRuleRdfs2_1();
			nofInferred += _applyRuleRdfs2_2();
			nofInferred += _applyRuleRdfs3_1();
			nofInferred += _applyRuleRdfs3_2();
			nofInferred += _applyRuleRdfs4a();
			nofInferred += _applyRuleRdfs4b();
			nofInferred += _applyRuleRdfs5_1();
			nofInferred += _applyRuleRdfs5_2();
			nofInferred += _applyRuleRdfs6();
			nofInferred += _applyRuleRdfs7_1();
			nofInferred += _applyRuleRdfs7_2();
			nofInferred += _applyRuleRdfs8();
			nofInferred += _applyRuleRdfs9_1();
			nofInferred += _applyRuleRdfs9_2();
			nofInferred += _applyRuleRdfs10();
			nofInferred += _applyRuleRdfs11_1();
			nofInferred += _applyRuleRdfs11_2();
			nofInferred += _applyRuleRdfs12();
			nofInferred += _applyRuleRdfs13();
			nofInferred += _applyRuleX1();

			ThreadLog.trace("iteration " + iteration + " done; " +
					"inferred " + nofInferred + " new statements");
			_totalInferred += nofInferred;
		}

		// Print some statistics
		ThreadLog.trace("---RdfMTInferencer statistics:---");
		ThreadLog.trace("total statements inferred = " + _totalInferred);
		for (int i = 0; i < RULECOUNT; i++) {
			ThreadLog.trace(
					"rule " + RULENAMES[i] + ":\t#inferred=" + _ruleCount[i]);
		}
		ThreadLog.trace("---end of statistics:---");

		// clean up
		_newStatements = null;
		_newThisIteration = null;
	}

	private void _prepareIteration() {
		_newThisIteration = _newStatements;
		_newStatements = new StatementList();
		for (int i = 0; i < RULECOUNT; i++) {
			_checkRule[i] = _checkRuleNextIter[i];

			// reset for next iteration:
			_checkRuleNextIter[i] = false;
		}
	}

	private void _updateTriggers(int ruleNo, int nofInferred) {
		if (nofInferred > 0) {
			_ruleCount[ruleNo] += nofInferred;

			// Check which rules are triggered by this one.
			boolean[] triggers = TRIGGERS[ruleNo];

			for (int i = 0; i < RULECOUNT; i++) {
				if (triggers[i] == true) {
					_checkRuleNextIter[i] = true;
				}
			}
		}
	}

	/* rdf1. 
	 * xxx aaa yyy --> aaa rdf:type rdf:Property
	 */
	private int _applyRuleRdf1() {
		if (! _checkRule[Rdf1]) {
			return 0;
		}
		int nofInferred = 0;

		StatementIterator iter = new MemStatementIterator(_newThisIteration);

		while (iter.hasNext()) {
			Statement st = iter.next();

			boolean added = _addInferredStatement(
					(URINode)st.getPredicate(), _repository.RDF_TYPE_NODE, _repository.RDF_PROPERTY_NODE);
			if (added) {
				nofInferred++;
			}
		}

		iter.close();

		_updateTriggers(Rdf1, nofInferred);

		return nofInferred;
	}

	/* rdfs2.
	 * 2_1. xxx aaa yyy &&           (nt)
	 *      aaa rdfs:domain zzz -->  (t1)
	 *      xxx rdf:type zzz         (t2)
	 */
	private int _applyRuleRdfs2_1() {
		if (! _checkRule[Rdfs2_1]) {
			return 0;
		}
		int nofInferred = 0;

		StatementIterator ntIter = new MemStatementIterator(_newThisIteration);

		while (ntIter.hasNext()) {
			Statement nt = ntIter.next();

			ResourceNode xxx = (ResourceNode)nt.getSubject();
			URINode aaa = (URINode)nt.getPredicate();

			StatementIterator t1Iter = _repository.getStatements(aaa, _repository.RDFS_DOMAIN_NODE, null);

			while (t1Iter.hasNext()) {
				Statement t1 = t1Iter.next();

				ValueNode zzz = (ValueNode)t1.getObject();
				if (zzz instanceof ResourceNode) {
					boolean added = _addInferredStatement(xxx, _repository.RDF_TYPE_NODE, zzz);
					if (added) {
						nofInferred++;
					}
				}
			}

			t1Iter.close();
		}

		ntIter.close();

		_updateTriggers(Rdfs2_1, nofInferred);

		return nofInferred;
	}

	/* rdfs2.
	 * 2_2. aaa rdfs:domain zzz &&  (nt)
	 *     xxx aaa yyy -->         (t1)
	 *     xxx rdf:type zzz        (t2)
	 */
	private int _applyRuleRdfs2_2() {
		if (! _checkRule[Rdfs2_2]) {
			return 0;
		}
		int nofInferred = 0;

		StatementIterator ntIter = new MemStatementIterator(_newThisIteration, null, _repository.RDFS_DOMAIN_NODE, null);

		while (ntIter.hasNext()) {
			Statement nt = ntIter.next();

			ResourceNode aaa = (ResourceNode)nt.getSubject();
			ValueNode zzz = (ValueNode)nt.getObject();

			if (aaa instanceof URINode && zzz instanceof ResourceNode) {
				StatementIterator t1Iter = _repository.getStatements(null, (URINode)aaa, null);

				while (t1Iter.hasNext()) {
					Statement t1 = t1Iter.next();

					ResourceNode xxx = (ResourceNode)t1.getSubject();
					boolean added = _addInferredStatement(xxx, _repository.RDF_TYPE_NODE, zzz);
					if (added) {
						nofInferred++;
					}
				}

				t1Iter.close();
			}
		}

		ntIter.close();

		_updateTriggers(Rdfs2_2, nofInferred);

		return nofInferred;
	}

	/* rdfs3.
	 * 3_1. xxx aaa uuu &&          (nt)
	 *     aaa rdfs:range zzz -->  (t1)
	 *     uuu rdf:type zzz        (t2)
	 */
	private int _applyRuleRdfs3_1() {
		if (! _checkRule[Rdfs3_1]) {
			return 0;
		}
		int nofInferred = 0;

		StatementIterator ntIter = new MemStatementIterator(_newThisIteration);

		while (ntIter.hasNext()) {
			Statement nt = ntIter.next();

			URINode aaa = (URINode)nt.getPredicate();
			ValueNode uuu = (ValueNode)nt.getObject();

			if (uuu instanceof ResourceNode) {
				StatementIterator t1Iter = _repository.getStatements(aaa, _repository.RDFS_RANGE_NODE, null);

				while (t1Iter.hasNext()) {
					Statement t1 = t1Iter.next();

					ValueNode zzz = (ValueNode)t1.getObject();
					if (zzz instanceof ResourceNode) {
						boolean added = _addInferredStatement((ResourceNode)uuu, _repository.RDF_TYPE_NODE, zzz);
						if (added) {
							nofInferred++;
						}
					}
				}

				t1Iter.close();
			}
		}

		ntIter.close();

		_updateTriggers(Rdfs3_1, nofInferred);

		return nofInferred;
	}

	/* rdfs3.
	 * 3_2. aaa rdfs:range zzz &&  (nt)
	 *     xxx aaa uuu -->        (t1)
	 *     uuu rdf:type zzz       (t2)
	 */
	private int _applyRuleRdfs3_2() {
		if (! _checkRule[Rdfs3_2]) {
			return 0;
		}
		int nofInferred = 0;

		StatementIterator ntIter = new MemStatementIterator(_newThisIteration, null, _repository.RDFS_RANGE_NODE, null);

		while (ntIter.hasNext()) {
			Statement nt = ntIter.next();

			ResourceNode aaa = (ResourceNode)nt.getSubject();
			ValueNode zzz = (ValueNode)nt.getObject();

			if (aaa instanceof URINode && zzz instanceof ResourceNode) {
				StatementIterator t1Iter = _repository.getStatements(null, (URINode)aaa, null);

				while (t1Iter.hasNext()) {
					Statement t1 = t1Iter.next();

					ValueNode uuu = (ValueNode)t1.getObject();
					if (uuu instanceof ResourceNode) {
						boolean added = _addInferredStatement((ResourceNode)uuu, _repository.RDF_TYPE_NODE, zzz);
						if (added) {
							nofInferred++;
						}
					}
				}

				t1Iter.close();
			}
		}

		ntIter.close();

		_updateTriggers(Rdfs3_2, nofInferred);

		return nofInferred;

	}

	/* rdfs4a. 
	 * xxx aaa yyy --> xxx rdf:type rdfs:Resource
	 */
	private int _applyRuleRdfs4a() {
		if (! _checkRule[Rdfs4a]) {
			return 0;
		}
		int nofInferred = 0;

		StatementIterator iter = new MemStatementIterator(_newThisIteration);

		while (iter.hasNext()) {
			Statement st = iter.next();

			boolean added = _addInferredStatement(
					(ResourceNode)st.getSubject(), _repository.RDF_TYPE_NODE, _repository.RDFS_RESOURCE_NODE);
			if (added) {
				nofInferred++;
			}
		}

		iter.close();

		_updateTriggers(Rdfs4a, nofInferred);

		return nofInferred;
	}

	/* rdfs4b. 
	 * xxx aaa uuu --> uuu rdf:type rdfs:Resource
	 */
	private int _applyRuleRdfs4b() {
		if (! _checkRule[Rdfs4b]) {
			return 0;
		}
		int nofInferred = 0;

		StatementIterator iter = new MemStatementIterator(_newThisIteration);

		while (iter.hasNext()) {
			Statement st = iter.next();

			ValueNode uuu = (ValueNode)st.getObject();
			if (uuu instanceof ResourceNode) {
				boolean added = _addInferredStatement(
						(ResourceNode)uuu, _repository.RDF_TYPE_NODE, _repository.RDFS_RESOURCE_NODE);
				if (added) {
					nofInferred++;
				}
			}
		}

		iter.close();

		_updateTriggers(Rdfs4b, nofInferred);

		return nofInferred;
	}

	/* rdfs5. 
	 * 5_1. aaa rdfs:subPropertyOf bbb &&   (nt)
	 *     bbb rdfs:subPropertyOf ccc -->  (t1)
	 *     aaa rdfs:subPropertyOf ccc      (t2)
	 */
	private int _applyRuleRdfs5_1() {
		if (! _checkRule[Rdfs5_1]) {
			return 0;
		}

		int nofInferred = 0;

		StatementIterator ntIter = new MemStatementIterator(_newThisIteration,
				null, _repository.RDFS_SUBPROPERTYOF_NODE, null);

		while (ntIter.hasNext()) {
			Statement nt = ntIter.next();

			ResourceNode aaa = (ResourceNode)nt.getSubject();
			ValueNode bbb = (ValueNode)nt.getObject();

			if (bbb instanceof ResourceNode) {
				StatementIterator t1Iter = _repository.getStatements(
						(ResourceNode)bbb, _repository.RDFS_SUBPROPERTYOF_NODE, null);

				while (t1Iter.hasNext()) {
					Statement t1 = t1Iter.next();

					ValueNode ccc = (ValueNode)t1.getObject();
					if (ccc instanceof ResourceNode) {
						boolean added = _addInferredStatement(aaa, _repository.RDFS_SUBPROPERTYOF_NODE, ccc);
						if (added) {
							nofInferred++;
						}
					}
				}

				t1Iter.close();
			}
		}

		ntIter.close();

		_updateTriggers(Rdfs5_1, nofInferred);

		return nofInferred;
	}

	/* rdfs5.
	 * 5_2. bbb rdfs:subPropertyOf ccc && (nt)
	 *     aaa rdfs:subPropertyOf bbb -->  (t1)
	 *     aaa rdfs:subPropertyOf ccc      (t2)
	 */
	private int _applyRuleRdfs5_2() {
		if (! _checkRule[Rdfs5_2]) {
			return 0;
		}

		int nofInferred = 0;

		StatementIterator ntIter = new MemStatementIterator(_newThisIteration,
				null, _repository.RDFS_SUBPROPERTYOF_NODE, null);

		while (ntIter.hasNext()) {
			Statement nt = ntIter.next();

			ResourceNode bbb = (ResourceNode)nt.getSubject();
			ValueNode ccc = (ValueNode)nt.getObject();

			if (ccc instanceof ResourceNode) {
				StatementIterator t1Iter = _repository.getStatements(null, _repository.RDFS_SUBPROPERTYOF_NODE, bbb);

				while (t1Iter.hasNext()) {
					Statement t1 = t1Iter.next();

					ResourceNode aaa = (ResourceNode)t1.getSubject();
					boolean added = _addInferredStatement(aaa, _repository.RDFS_SUBPROPERTYOF_NODE, ccc);
					if (added) {
						nofInferred++;
					}
				}

				t1Iter.close();
			}
		}

		ntIter.close();

		_updateTriggers(Rdfs5_2, nofInferred);

		return nofInferred;
	}

	/* rdfs6. 
	 * xxx rdf:type rdf:Property --> xxx rdfs:subPropertyOf xxx
	 * reflexivity of rdfs:subPropertyOf
	 */
	private int _applyRuleRdfs6() {
		if (! _checkRule[Rdfs6]) {
			return 0;
		}
		int nofInferred = 0;

		StatementIterator iter = new MemStatementIterator(_newThisIteration,
				null, _repository.RDF_TYPE_NODE, _repository.RDF_PROPERTY_NODE);

		while (iter.hasNext()) {
			Statement st = iter.next();

			ResourceNode xxx = (ResourceNode)st.getSubject();
			boolean added = _addInferredStatement(xxx, _repository.RDFS_SUBPROPERTYOF_NODE, xxx);
			if (added) {
				nofInferred++;
			}
		}

		iter.close();

		_updateTriggers(Rdfs6, nofInferred);

		return nofInferred;
	}

	/* rdfs7.
	 * 7_1. xxx aaa yyy &&                  (nt)
	 *     aaa rdfs:subPropertyOf bbb -->  (t1)
	 *     xxx bbb yyy                     (t2)
	 */
	private int _applyRuleRdfs7_1() {
		if (! _checkRule[Rdfs7_1]) {
			return 0;
		}

		int nofInferred = 0;

		StatementIterator ntIter = new MemStatementIterator(_newThisIteration);

		while (ntIter.hasNext()) {
			Statement nt = ntIter.next();

			ResourceNode xxx = (ResourceNode)nt.getSubject();
			URINode aaa = (URINode)nt.getPredicate();
			ValueNode yyy = (ValueNode)nt.getObject();

			StatementIterator t1Iter = _repository.getStatements(aaa, _repository.RDFS_SUBPROPERTYOF_NODE, null);

			while (t1Iter.hasNext()) {
				Statement t1 = t1Iter.next();

				ValueNode bbb = (ValueNode)t1.getObject();
				if (bbb instanceof URINode) {
					boolean added = _addInferredStatement(xxx, (URINode)bbb, yyy);
					if (added) {
						nofInferred++;
					}
				}
			}

			t1Iter.close();
		}

		ntIter.close();

		_updateTriggers(Rdfs7_1, nofInferred);

		return nofInferred;
	}

	/* rdfs7.
	 * 7_2. aaa rdfs:subPropertyOf bbb &&  (nt)
	 *     xxx aaa yyy -->                (t1)
	 *     xxx bbb yyy                    (t2)
	 */
	private int _applyRuleRdfs7_2() {
		if (! _checkRule[Rdfs7_2]) {
			return 0;
		}

		int nofInferred = 0;

		StatementIterator ntIter = new MemStatementIterator(_newThisIteration,
				null, _repository.RDFS_SUBPROPERTYOF_NODE, null);

		while (ntIter.hasNext()) {
			Statement nt = ntIter.next();

			ResourceNode aaa = (ResourceNode)nt.getSubject();
			ValueNode bbb = (ValueNode)nt.getObject();

			if (aaa instanceof URINode && bbb instanceof URINode) {
				StatementIterator t1Iter = _repository.getStatements(null, (URINode)aaa, null);

				while (t1Iter.hasNext()) {
					Statement t1 = t1Iter.next();

					ResourceNode xxx = (ResourceNode)t1.getSubject();
					ValueNode yyy = (ValueNode)t1.getObject();

					boolean added = _addInferredStatement(xxx, (URINode)bbb, yyy);
					if (added) {
						nofInferred++;
					}
				}

				t1Iter.close();
			}
		}

		ntIter.close();

		_updateTriggers(Rdfs7_2, nofInferred);

		return nofInferred;
	}

	/* rdfs8.
	 *  xxx rdf:type rdfs:Class --> xxx rdfs:subClassOf rdfs:Resource */
	private int _applyRuleRdfs8() {
		if (! _checkRule[Rdfs8]) {
			return 0;
		}
		int nofInferred = 0;

		StatementIterator iter = new MemStatementIterator(_newThisIteration,
				null, _repository.RDF_TYPE_NODE, _repository.RDFS_CLASS_NODE);

		while (iter.hasNext()) {
			Statement st = iter.next();

			ResourceNode xxx = (ResourceNode)st.getSubject();

			boolean added = _addInferredStatement(xxx, _repository.RDFS_SUBCLASSOF_NODE, _repository.RDFS_RESOURCE_NODE);
			if (added) {
				nofInferred++;
			}
		}

		iter.close();

		_updateTriggers(Rdfs8, nofInferred);

		return nofInferred;
	}

	/* rdfs10.
	 * xxx rdf:type rdfs:Class --> xxx rdfs:subClassOf xxx
	 * reflexivity of rdfs:subClassOf
	 */
	private int _applyRuleRdfs10() {
		if (! _checkRule[Rdfs10]) {
			return 0;
		}
		int nofInferred = 0;

		StatementIterator iter = new MemStatementIterator(_newThisIteration,
				null, _repository.RDF_TYPE_NODE, _repository.RDFS_CLASS_NODE);

		while (iter.hasNext()) {
			Statement st = iter.next();

			ResourceNode xxx = (ResourceNode)st.getSubject();

			boolean added = _addInferredStatement(xxx, _repository.RDFS_SUBCLASSOF_NODE, xxx);
			if (added) {
				nofInferred++;
			}
		}

		iter.close();

		_updateTriggers(Rdfs10, nofInferred);

		return nofInferred;
	}

	/* rdfs11.
	 * 11_1. xxx rdfs:subClassOf yyy &&  (nt)
	 *     yyy rdfs:subClassOf zzz -->  (t1)
	 *     xxx rdfs:subClassOf zzz      (t2)
	 */
	private int _applyRuleRdfs11_1() {
		if (! _checkRule[Rdfs11_1]) {
			return 0;
		}

		int nofInferred = 0;

		StatementIterator ntIter = new MemStatementIterator(_newThisIteration,
				null, _repository.RDFS_SUBCLASSOF_NODE, null);

		while (ntIter.hasNext()) {
			Statement nt = ntIter.next();

			ResourceNode xxx = (ResourceNode)nt.getSubject();
			ValueNode yyy = (ValueNode)nt.getObject();

			if (yyy instanceof ResourceNode) {
				StatementIterator t1Iter = _repository.getStatements(
						(ResourceNode)yyy, _repository.RDFS_SUBCLASSOF_NODE,  null, false);

				while (t1Iter.hasNext()) {
					Statement t1 = t1Iter.next();

					ValueNode zzz = (ValueNode)t1.getObject();

					if (zzz instanceof ResourceNode) {
						boolean added = _addInferredStatement(xxx, _repository.RDFS_SUBCLASSOF_NODE, zzz);
						if (added) {
							nofInferred++;
						}
					}
				}

				t1Iter.close();
			}
		}

		ntIter.close();

		_updateTriggers(Rdfs11_1, nofInferred);

		return nofInferred;
	}

	/* rdfs11.
	 * 11_2. yyy rdfs:subClassOf zzz &&  (nt)
	 *     xxx rdfs:subClassOf yyy -->  (t1)
	 *     xxx rdfs:subClassOf zzz      (t2)
	 */
	private int _applyRuleRdfs11_2() {
		if (! _checkRule[Rdfs11_2]) {
			return 0;
		}

		int nofInferred = 0;

		StatementIterator ntIter = new MemStatementIterator(_newThisIteration, null, _repository.RDFS_SUBCLASSOF_NODE, null);

		while (ntIter.hasNext()) {
			Statement nt = ntIter.next();

			ResourceNode yyy = (ResourceNode)nt.getSubject();
			ValueNode zzz = (ValueNode)nt.getObject();

			if (zzz instanceof ResourceNode) {
				StatementIterator t1Iter = _repository.getStatements(null, _repository.RDFS_SUBCLASSOF_NODE, yyy);

				while (t1Iter.hasNext()) {
					Statement t1 = t1Iter.next();

					ResourceNode xxx = (ResourceNode)t1.getSubject();

					boolean added = _addInferredStatement(xxx, _repository.RDFS_SUBCLASSOF_NODE, zzz);
					if (added) {
						nofInferred++;
					}
				}

				t1Iter.close();
			}
		}

		ntIter.close();

		_updateTriggers(Rdfs11_2, nofInferred);

		return nofInferred;
	}

	/* rdfs9.
	 * 9_1. xxx rdfs:subClassOf yyy &&  (nt)
	 *     aaa rdf:type xxx -->        (t1)
	 *     aaa rdf:type yyy            (t2)
	 */
	private int _applyRuleRdfs9_1() {
		if (! _checkRule[Rdfs9_1]) {
			return 0;
		}

		int nofInferred = 0;

		StatementIterator ntIter = new MemStatementIterator(_newThisIteration, null, _repository.RDFS_SUBCLASSOF_NODE, null);

		while (ntIter.hasNext()) {
			Statement nt = ntIter.next();

			ResourceNode xxx = (ResourceNode)nt.getSubject();
			ValueNode yyy = (ValueNode)nt.getObject();

			if (yyy instanceof ResourceNode) {
				StatementIterator t1Iter = _repository.getStatements(null, _repository.RDF_TYPE_NODE, xxx, false);

				while (t1Iter.hasNext()) {
					Statement t1 = t1Iter.next();

					ResourceNode aaa = (ResourceNode)t1.getSubject();

					boolean added = _addInferredStatement(aaa, _repository.RDF_TYPE_NODE, yyy);
					if (added) {
						nofInferred++;
					}
				}

				t1Iter.close();
			}
		}

		ntIter.close();

		_updateTriggers(Rdfs9_1, nofInferred);

		return nofInferred;
	}

	/* rdfs9.
	 * 9_2. aaa rdf:type xxx &&          (nt)
	 *     xxx rdfs:subClassOf yyy -->  (t1)
	 *     aaa rdf:type yyy             (t2)
	 */
	private int _applyRuleRdfs9_2() {
		if (! _checkRule[Rdfs9_2]) {
			return 0;
		}

		int nofInferred = 0;

		StatementIterator ntIter = new MemStatementIterator(_newThisIteration, null, _repository.RDF_TYPE_NODE, null);

		while (ntIter.hasNext()) {
			Statement nt = ntIter.next();

			ResourceNode aaa = (ResourceNode)nt.getSubject();
			ValueNode xxx = (ValueNode)nt.getObject();

			if (xxx instanceof ResourceNode) {
				StatementIterator t1Iter = _repository.getStatements(
						(ResourceNode)xxx, _repository.RDFS_SUBCLASSOF_NODE,  null);

				while (t1Iter.hasNext()) {
					Statement t1 = t1Iter.next();

					ValueNode yyy = (ValueNode)t1.getObject();

					if (yyy instanceof ResourceNode) {
						boolean added = _addInferredStatement(aaa, _repository.RDF_TYPE_NODE, yyy);
						if (added) {
							nofInferred++;
						}
					}
				}

				t1Iter.close();
			}
		}

		ntIter.close();

		_updateTriggers(Rdfs9_2, nofInferred);

		return nofInferred;
	}

	/* rdfs12.
	 * xxx rdf:type rdfs:ContainerMembershipProperty -->
	 *     xxx rdfs:subPropertyOf rdfs:member
	 */
	private int _applyRuleRdfs12() {
		if (! _checkRule[Rdfs12]) {
			return 0;
		}
		int nofInferred = 0;

		StatementIterator iter = new MemStatementIterator(_newThisIteration,
				null, _repository.RDF_TYPE_NODE, _repository.RDFS_CONTAINERMEMBERSHIPPROPERTY_NODE);

		while (iter.hasNext()) {
			Statement st = iter.next();

			ResourceNode xxx = (ResourceNode)st.getSubject();

			boolean added = _addInferredStatement(xxx, _repository.RDFS_SUBPROPERTYOF_NODE, _repository.RDFS_MEMBER_NODE);
			if (added) {
				nofInferred++;
			}
		}

		iter.close();

		_updateTriggers(Rdfs12, nofInferred);

		return nofInferred;
	}

	/* rdfs13.  
	 * xxx rdf:type rdfs:Datatype --> xxx rdfs:subClassOf rdfs:Literal
	 */
	private int _applyRuleRdfs13() {
		if (! _checkRule[Rdfs13]) {
			return 0;
		}
		int nofInferred = 0;

		StatementIterator iter = new MemStatementIterator(_newThisIteration,
				null, _repository.RDF_TYPE_NODE, _repository.RDFS_DATATYPE_NODE);

		while (iter.hasNext()) {
			Statement st = iter.next();

			ResourceNode xxx = (ResourceNode)st.getSubject();

			boolean added = _addInferredStatement(xxx, _repository.RDFS_SUBCLASSOF_NODE, _repository.RDFS_LITERAL_NODE);
			if (added) {
				nofInferred++;
			}
		}

		iter.close();

		_updateTriggers(Rdfs13, nofInferred);

		return nofInferred;
	}

	/* X1. xxx rdf:_* yyy -->
	 *     rdf:_* rdf:type rdfs:ContainerMembershipProperty
	 *
	 * This is an extra rule for list membership properties (_1, _2, _3,
	 * ...). The RDF MT does not specificy a production for this.
	 */
	private int _applyRuleX1() {
		if (! _checkRule[RX1]) {
			return 0;
		}
		int nofInferred = 0;

		StatementIterator iter = new MemStatementIterator(_newThisIteration);

		while (iter.hasNext()) {
			Statement st = iter.next();

			URINode predNode = (URINode)st.getPredicate();

			if (predNode.getNamespace().equals(RDF.NAMESPACE) &&
				predNode.getLocalName().startsWith("_"))
			{
				boolean added = _addInferredStatement(
						predNode, _repository.RDF_TYPE_NODE, _repository.RDFS_CONTAINERMEMBERSHIPPROPERTY_NODE);
				if (added) {
					nofInferred++;
				}
			}
		}

		iter.close();

		_updateTriggers(RX1, nofInferred);

		return nofInferred;
	}

	private void _addAxiomStatements() {
		
		// RDF axiomatic triples (from RDF Semantics, section 3.1):
		
		_addInferredStatement(_repository.RDF_TYPE_NODE, _repository.RDF_TYPE_NODE, _repository.RDF_PROPERTY_NODE);
		_addInferredStatement(_repository.RDF_SUBJECT_NODE, _repository.RDF_TYPE_NODE, _repository.RDF_PROPERTY_NODE);
		_addInferredStatement(_repository.RDF_PREDICATE_NODE, _repository.RDF_TYPE_NODE, _repository.RDF_PROPERTY_NODE);
		_addInferredStatement(_repository.RDF_OBJECT_NODE, _repository.RDF_TYPE_NODE, _repository.RDF_PROPERTY_NODE);
		
		_addInferredStatement(_repository.RDF_FIRST_NODE, _repository.RDF_TYPE_NODE, _repository.RDF_PROPERTY_NODE);
		_addInferredStatement(_repository.RDF_REST_NODE, _repository.RDF_TYPE_NODE, _repository.RDF_PROPERTY_NODE);
		_addInferredStatement(_repository.RDF_VALUE_NODE, _repository.RDF_TYPE_NODE, _repository.RDF_PROPERTY_NODE);
		
		_addInferredStatement(_repository.RDF_NIL_NODE, _repository.RDF_TYPE_NODE, _repository.RDF_LIST_NODE);
		
		// RDFS axiomatic triples (from RDF Semantics, section 4.1):
		
		_addInferredStatement(_repository.RDF_TYPE_NODE, _repository.RDFS_DOMAIN_NODE, _repository.RDFS_RESOURCE_NODE);
		_addInferredStatement(_repository.RDFS_DOMAIN_NODE, _repository.RDFS_DOMAIN_NODE, _repository.RDF_PROPERTY_NODE);
		_addInferredStatement(_repository.RDFS_RANGE_NODE, _repository.RDFS_DOMAIN_NODE, _repository.RDF_PROPERTY_NODE);
		_addInferredStatement(_repository.RDFS_SUBPROPERTYOF_NODE, _repository.RDFS_DOMAIN_NODE, _repository.RDF_PROPERTY_NODE);
		_addInferredStatement(_repository.RDFS_SUBCLASSOF_NODE, _repository.RDFS_DOMAIN_NODE, _repository.RDFS_CLASS_NODE);
		_addInferredStatement(_repository.RDF_SUBJECT_NODE, _repository.RDFS_DOMAIN_NODE, _repository.RDF_STATEMENT_NODE);
		_addInferredStatement(_repository.RDF_PREDICATE_NODE, _repository.RDFS_DOMAIN_NODE, _repository.RDF_STATEMENT_NODE);
		_addInferredStatement(_repository.RDF_OBJECT_NODE, _repository.RDFS_DOMAIN_NODE, _repository.RDF_STATEMENT_NODE);
		_addInferredStatement(_repository.RDFS_MEMBER_NODE, _repository.RDFS_DOMAIN_NODE, _repository.RDFS_RESOURCE_NODE);
		_addInferredStatement(_repository.RDF_FIRST_NODE, _repository.RDFS_DOMAIN_NODE, _repository.RDF_LIST_NODE);
		_addInferredStatement(_repository.RDF_REST_NODE, _repository.RDFS_DOMAIN_NODE, _repository.RDF_LIST_NODE);
		_addInferredStatement(_repository.RDFS_SEEALSO_NODE, _repository.RDFS_DOMAIN_NODE, _repository.RDFS_RESOURCE_NODE);
		_addInferredStatement(_repository.RDFS_ISDEFINEDBY_NODE, _repository.RDFS_DOMAIN_NODE, _repository.RDFS_RESOURCE_NODE);
		_addInferredStatement(_repository.RDFS_COMMENT_NODE, _repository.RDFS_DOMAIN_NODE, _repository.RDFS_RESOURCE_NODE);
		_addInferredStatement(_repository.RDFS_LABEL_NODE, _repository.RDFS_DOMAIN_NODE, _repository.RDFS_RESOURCE_NODE);
		_addInferredStatement(_repository.RDF_VALUE_NODE, _repository.RDFS_DOMAIN_NODE, _repository.RDFS_RESOURCE_NODE);
		
		_addInferredStatement(_repository.RDF_TYPE_NODE, _repository.RDFS_RANGE_NODE, _repository.RDFS_CLASS_NODE);
		_addInferredStatement(_repository.RDFS_DOMAIN_NODE, _repository.RDFS_RANGE_NODE, _repository.RDFS_CLASS_NODE);
		_addInferredStatement(_repository.RDFS_RANGE_NODE, _repository.RDFS_RANGE_NODE, _repository.RDFS_CLASS_NODE);
		_addInferredStatement(_repository.RDFS_SUBPROPERTYOF_NODE, _repository.RDFS_RANGE_NODE, _repository.RDF_PROPERTY_NODE);
		_addInferredStatement(_repository.RDFS_SUBCLASSOF_NODE, _repository.RDFS_RANGE_NODE, _repository.RDFS_CLASS_NODE);
		_addInferredStatement(_repository.RDF_SUBJECT_NODE, _repository.RDFS_RANGE_NODE, _repository.RDFS_RESOURCE_NODE);
		_addInferredStatement(_repository.RDF_PREDICATE_NODE, _repository.RDFS_RANGE_NODE, _repository.RDFS_RESOURCE_NODE);
		_addInferredStatement(_repository.RDF_OBJECT_NODE, _repository.RDFS_RANGE_NODE, _repository.RDFS_RESOURCE_NODE);
		_addInferredStatement(_repository.RDFS_MEMBER_NODE, _repository.RDFS_RANGE_NODE, _repository.RDFS_RESOURCE_NODE);
		_addInferredStatement(_repository.RDF_FIRST_NODE, _repository.RDFS_RANGE_NODE, _repository.RDFS_RESOURCE_NODE);
		_addInferredStatement(_repository.RDF_REST_NODE, _repository.RDFS_RANGE_NODE, _repository.RDF_LIST_NODE);
		_addInferredStatement(_repository.RDFS_SEEALSO_NODE, _repository.RDFS_RANGE_NODE, _repository.RDFS_RESOURCE_NODE);
		_addInferredStatement(_repository.RDFS_ISDEFINEDBY_NODE, _repository.RDFS_RANGE_NODE, _repository.RDFS_RESOURCE_NODE);
		_addInferredStatement(_repository.RDFS_COMMENT_NODE, _repository.RDFS_RANGE_NODE, _repository.RDFS_LITERAL_NODE);
		_addInferredStatement(_repository.RDFS_LABEL_NODE, _repository.RDFS_RANGE_NODE, _repository.RDFS_LITERAL_NODE);
		_addInferredStatement(_repository.RDF_VALUE_NODE, _repository.RDFS_RANGE_NODE, _repository.RDFS_RESOURCE_NODE);
		
		_addInferredStatement(_repository.RDF_ALT_NODE, _repository.RDFS_SUBCLASSOF_NODE, _repository.RDFS_CONTAINER_NODE);
		_addInferredStatement(_repository.RDF_BAG_NODE, _repository.RDFS_SUBCLASSOF_NODE, _repository.RDFS_CONTAINER_NODE);
		_addInferredStatement(_repository.RDF_SEQ_NODE, _repository.RDFS_SUBCLASSOF_NODE, _repository.RDFS_CONTAINER_NODE);
		_addInferredStatement(_repository.RDFS_CONTAINERMEMBERSHIPPROPERTY_NODE, _repository.RDFS_SUBCLASSOF_NODE, _repository.RDF_PROPERTY_NODE);
		
		_addInferredStatement(_repository.RDFS_ISDEFINEDBY_NODE, _repository.RDFS_SUBPROPERTYOF_NODE, _repository.RDFS_SEEALSO_NODE);

		_addInferredStatement(_repository.RDF_XMLLITERAL_NODE, _repository.RDF_TYPE_NODE, _repository.RDFS_DATATYPE_NODE);
		_addInferredStatement(_repository.RDF_XMLLITERAL_NODE, _repository.RDFS_SUBCLASSOF_NODE, _repository.RDFS_LITERAL_NODE);
		_addInferredStatement(_repository.RDFS_DATATYPE_NODE, _repository.RDFS_SUBCLASSOF_NODE, _repository.RDFS_CLASS_NODE);

	}

	private boolean _addInferredStatement(ResourceNode subjNode, URINode predNode, ValueNode objNode) {
		// Check if the inferred statement is already known
		if (_repository.hasStatement(subjNode, predNode, objNode)) {
			return false;
		}

		// This is a new inferred statement, add it and mark it inferred
		MemStatement newStatement = _repository._addStatement(subjNode, predNode, objNode);
		newStatement.setExplicit(false);

		// notify the inferencer of this new statement
		notifyNewStatement(newStatement);

		return true;
	}
}
