/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sailimpl.memory;

import org.openrdf.model.Statement;

import org.openrdf.sesame.sail.StatementIterator;

/**
 * An implementation of org.openrdf.sesame.sail.StatementIterator that
 * can iterate over a list of Statement objects. This iterator compares
 * Resource and Literal objects using the '==' operator, which is possible
 * thanks to the extensive sharing of these objects in this implementation
 * of the RDF Sail API.
 *
 * @author Jeen Broekstra
 * @author Arjohn Kampman
 * @version $Revision: 1.9.4.2 $
 **/
class MemStatementIterator
	implements StatementIterator
{

/*--------------+
| Variables     |
+--------------*/

	/** The statements over which to iterate. **/
	private StatementList _statements;

	/** The number of statements in _statements. **/
	private int _statementCount;

	/** The index of the next statement to return. **/
	private int _nextStatementIdx;

	/** Subject of statement to look for, or null if any subject is OK. **/
	private ResourceNode _subject;

	/** Predicate of statement to look for, or null if any predicate is OK. **/
	private URINode _predicate;

	/** Object of statement to look for, or null if any object is OK. **/
	private ValueNode _object;

	/** Flag indicating whether this iterator should only return explicit statements. **/
	private boolean _explicitOnly;

/*----------------+
| Constructors    |
+----------------*/

	/**
	 * Constructor.
	 *
	 * @param statements the statements over which to iterate.
	 */
	public MemStatementIterator(StatementList statements) {
		this(statements, null, null, null, false);
	}

	/**
	 * Constructor.
	 *
	 * @param statements the statements over which to iterate.
	 * @param subject subject of pattern.
	 * @param predicate predicate of pattern.
	 * @param object object of pattern.
	 */
	public MemStatementIterator(StatementList statements,
		ResourceNode subject, URINode predicate, ValueNode object)
	{
		this(statements, subject, predicate, object, false);
	}

	public MemStatementIterator(StatementList statements,
		ResourceNode subject, URINode predicate, ValueNode object, boolean explicitOnly)
	{
		_statements = statements;
		_statementCount = _statements.size();

		_subject = subject;
		_predicate = predicate;
		_object = object;

		_explicitOnly = explicitOnly;

		_nextStatementIdx = -1;
		_findNextStatement();
	}


/*--------------+
| Methods       |
+--------------*/

	/**
	 * Searches through _statements, starting from index _nextStatementIdx + 1,
	 * for statements that match the specified pattern of _subject, _predicate,
	 * _object and _explicitOnly. If a matching statements has been found,
	 * _nextStatementIdx points to the index of this statement in _statements
	 * and this method will return 'true'. Otherwise, _nextStatementIdx will be
	 * equal to _statementCount and 'false' will be returned.
	 **/
	private boolean _findNextStatement() {
		_nextStatementIdx++;
		for (; _nextStatementIdx < _statementCount; _nextStatementIdx++) {
			MemStatement st = (MemStatement)_statements.get(_nextStatementIdx);

			if (_explicitOnly && !st.isExplicit()) {
				// explicit statements only; skip inferred ones
				continue;
			}

			if ( (_subject == null   || _subject == st.getSubject()) &&
			     (_predicate == null || _predicate == st.getPredicate()) &&
			     (_object == null    || _object == st.getObject()) )
			{
				// A matching statement has been found, _nextStatementIdx
				// now points to this statement.
				return true;
			}
		}

		// No more matching statements.
		return false;
	}

	// Imlements StatementIterator.hasNext()
	public boolean hasNext() {
		// Iterator has more results when _nextStatementIdx points to a statement in _statements
		return _nextStatementIdx < _statementCount;
	}

	// Imlements StatementIterator.next()
	public Statement next() {
		Statement result = (Statement)_statements.get(_nextStatementIdx);
		_findNextStatement();
		return result;
	}
	
	// Imlements StatementIterator.close()
	public void close() {
		_statements = null;
		_nextStatementIdx = _statementCount;
		
		_subject = null;
		_predicate = null;
		_object = null;
	}
}
