/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sailimpl.memory;

import java.util.Collection;
import java.util.Iterator;
import java.util.NoSuchElementException;

import org.openrdf.model.Literal;
import org.openrdf.model.URI;
import org.openrdf.model.Value;

import org.openrdf.sesame.sail.LiteralIterator;

/**
 * An implementation of org.openrdf.sesame.sail.LiteralIterator that
 * can iterate over a list of Literal objects. 
 *
 * @author Arjohn Kampman
 * @version $Revision: 1.4.4.2 $
 */
public class MemLiteralIterator implements LiteralIterator {
	
/*---------------------------+
| Variables                  |
+---------------------------*/

	private Iterator _iter;

	private String _label;

	private String _language;

	private URI _datatype;

	private Literal _nextLiteral;
	
/*---------------------------+
| Constructors               |
+---------------------------*/

	/**
	 * Creates a new LiteralIterator that iterates over all objects
	 * in the supplied list.
	 *
	 * @param literals A Collection containing Literal objects
	 **/
	public MemLiteralIterator(Collection literals) {
		this(literals, null, null, null);
	}

	/**
	 * Creates a new LiteralIterator that iterates over all objects
	 * in the supplied list, only returning the ones that have the
	 * specified label, language and/or datatype.
	 *
	 * @param literals A Collection containing Literal objects
	 * @param label The label that the returned literals should have,
	 * or <tt>null</tt> if literals with any label should be returned.
	 * @param language The language that the returned literals should have,
	 * or <tt>null</tt> if literals with any language should be returned.
	 * @param datatype The datatype that the returned literals should have,
	 * or <tt>null</tt> if literals with any datatype should be returned.
	 **/
	public MemLiteralIterator(
			Collection literals, String label, String language, URI datatype)
	{
		_iter = literals.iterator();

		_label = label;
		_language = language;
		_datatype = datatype;

		_findNextLiteral();
	}
	
/*---------------------------+
| Methods                    |
+---------------------------*/

	private void _findNextLiteral() {
		if (_iter != null) {
			while (_iter.hasNext()) {
				_nextLiteral = (Literal)_iter.next();
				
				boolean matchesPattern = 
					(_label == null || _label.equals(_nextLiteral.getLabel())) &&
					(_language == null || _language.equals(_nextLiteral.getLanguage())) &&
					(_datatype == null || _datatype.equals(_nextLiteral.getDatatype()));

				if (matchesPattern) {
					return;
				}
			}
		}

		// No next literal found
		_nextLiteral = null;
	}
	
	public boolean hasNext() {
		return _nextLiteral != null;
	}

	public Value next() {
		return nextLiteral();
	}

	public Literal nextLiteral() {
		if (_nextLiteral != null) {
			Literal result = _nextLiteral;
			_findNextLiteral();
			return result;
		}

		throw new NoSuchElementException("No more objects in iterator");
	}

	public void close() {
		_iter = null;
		_nextLiteral = null;
	}
}
