/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sailimpl.memory;

import org.openrdf.model.Statement;
import org.openrdf.model.URI;
import org.openrdf.model.impl.LiteralImpl;

import org.openrdf.sesame.sail.StatementIterator;

/**
 * An extension of Literal giving it node properties.
 *
 * @author Arjohn Kampman
 * @version $Revision: 1.12.4.2 $
 **/
public class LiteralNode extends LiteralImpl implements ValueNode {

/*-----------------------------------+
| Variables                          |
+-----------------------------------*/

	transient protected RdfSource _source;

	/** The list of statements for which this LiteralNode is the object. **/
	transient protected StatementList _objectStatements = new StatementList(1);

/*-----------------------------------+
| Constructors                       |
+-----------------------------------*/

	/**
	 * Creates a new Literal which will get the supplied label.
	 *
	 * @param source The Sail instance that created this LiteralNode.
	 * @param label The label for this literal.
	 **/
	LiteralNode(RdfSource source, String label) {
		super(label);
		_source = source;
	}

	/**
	 * Creates a new Literal which will get the supplied label and
	 * language code.
	 *
	 * @param source The Sail instance that created this LiteralNode.
	 * @param label The label for this literal.
	 * @param lang The language code of the supplied label, or null if not
	 * specified.
	 **/
	public LiteralNode(RdfSource source, String label, String lang) {
		super(label, lang);
		_source = source;
	}

	/**
	 * Creates a new Literal which will get the supplied label and datatype.
	 *
	 * @param source The Sail instance that created this LiteralNode.
	 * @param label The label for this literal.
	 * @param datatype The datatype of the supplied label, or null if not
	 * specified.
	 **/
	public LiteralNode(RdfSource source, String label, URI datatype) {
		super(label, datatype);
		_source = source;
	}


/*-----------------------------------+
| Methods for _objectStatements      |
+-----------------------------------*/

	public StatementList getObjectStatementList() {
		return _objectStatements;
	}

	/**
	 * Gets the list of Statements for which this LiteralNode is the object.
	 * @return A List of Statements, or null if no such statement exists.
	 **/
	public StatementIterator getObjectStatements() {
		return new MemStatementIterator(_objectStatements);
	}

	/**
	 * Gets the amount of Statements for which this LiteralNode is the object.
	 * @return An integer larger than or equal to 0.
	 **/
	public int getObjectStatementCount() {
		return _objectStatements.size();
	}

	/**
	 * Adds a statement to this LiteralNode's list of statements for which
	 * it is the object.
	 **/
	public void addObjectStatement(Statement st) {
		_objectStatements.add(st);
	}

	/**
	 * Removes a statement from this ResourceNode's list of statements for which
	 * it is the object.
	 **/
	public void removeObjectStatement(Statement st) {
		_objectStatements.remove(st);
	}

    public RdfSource getRdfSource() {
        return _source;
    }
}
