/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sail.query;

import java.util.Collection;

import org.openrdf.util.xml.XmlDatatypeUtil;
import org.openrdf.vocabulary.XmlSchema;

import org.openrdf.model.Literal;
import org.openrdf.model.URI;
import org.openrdf.model.Value;
import org.openrdf.model.impl.URIImpl;

import org.openrdf.sesame.sail.RdfSource;

/**
 * A comparison between two values.
 **/
public class ValueCompare implements BooleanExpr {

/*----------+
| Constants |
+----------*/

	/** equal to **/
	public static final int EQ = 1;

	/** not equal to **/
	public static final int NE = 2;

	/** lower than **/
	public static final int LT = 3;

	/** lower than or equal to **/
	public static final int LE = 4;

	/** greater than or equal to **/
	public static final int GE = 5;

	/** greater than **/
	public static final int GT = 6;

/*----------+
| Variables |
+----------*/
 
	private ValueExpr _leftArg;
	private ValueExpr _rightArg;
	private int _operator;

/*-------------+
| Constructors |
+-------------*/

	public ValueCompare(ValueExpr leftArg, ValueExpr rightArg) {
		this(leftArg, rightArg, EQ);
	}

	public ValueCompare(ValueExpr leftArg, ValueExpr rightArg, int op) {
		if (op < EQ || op > GT) {
			throw new IllegalArgumentException("Illegal operator value: " + op);
		}
		_leftArg = leftArg;
		_rightArg = rightArg;
		_operator = op;
	}

/*--------+
| Methods |
+--------*/

	public ValueExpr getLeftArg() {
		return _leftArg;
	}

	public ValueExpr getRightArg() {
		return _rightArg;
	}

	public int getOperator() {
		return _operator;
	}

	public void getVariables(Collection variables) {
		_leftArg.getVariables(variables);
		_rightArg.getVariables(variables);
	}

	public boolean isTrue(RdfSource rdfSource)
		throws BooleanExprEvaluationException
	{
		Value leftVal = _leftArg.getValue();
		Value rightVal = _rightArg.getValue();
		
		return isTrue(leftVal, rightVal, _operator);
	}
	
	public static boolean isTrue(Value leftVal, Value rightVal, int operator)
		throws BooleanExprEvaluationException
	{
		boolean result = false;

		if (leftVal instanceof Literal && rightVal instanceof Literal) {
			// Both left and right argument is a Literal
			result = _compareLiterals((Literal)leftVal, (Literal)rightVal, operator);
		}
		else {
			// All other value combinations
			switch (operator) {
				case EQ: result = _valuesEqual(leftVal, rightVal); break;
				case NE: result = !_valuesEqual(leftVal, rightVal); break;
				default: throw new BooleanExprEvaluationException(
						"Only literals with compatible, ordered datatypes can be compared using <, <=, > and >= operators");
			}
		}

		return result;
	}

	private static boolean _valuesEqual(Value leftVal, Value rightVal) {
		boolean result = false;

		if (leftVal == null) {
			result = (rightVal == null);
		}
		else {
			result = leftVal.equals(rightVal);
		}

		return result;
	}

	private static boolean _compareLiterals(Literal leftLit, Literal rightLit, int operator)
		throws BooleanExprEvaluationException
	{
		String leftLabel = leftLit.getLabel();
		String leftLang = leftLit.getLanguage();
		URI leftDatatype = leftLit.getDatatype();

		String rightLabel = rightLit.getLabel();
		String rightLang = rightLit.getLanguage();
		URI rightDatatype = rightLit.getDatatype();

		// apply type casting if necessary
		if (leftDatatype == null && rightDatatype != null) {
			// left argument has no datatype, assume it is
			// of the same datatype as the right argument
			leftDatatype = rightDatatype;
		}
		else if (rightDatatype == null && leftDatatype != null) {
			// right argument has no datatype, assume it is
			// of the same datatype as the left argument
			rightDatatype = leftDatatype;
		}
		else if (rightDatatype != null && leftDatatype != null &&
				!leftDatatype.equals(rightDatatype))
		{
			// left and right arguments have different datatypes,
			// try to cast them to a more general, shared datatype
			if (XmlDatatypeUtil.isDecimalDatatype(rightDatatype.getURI()) && 
				XmlDatatypeUtil.isDecimalDatatype(leftDatatype.getURI()))
			{
				leftDatatype = rightDatatype = new URIImpl(XmlSchema.DECIMAL);
			}
		}

		boolean result = false;

		if (leftDatatype != null &&
			leftDatatype.equals(rightDatatype) &&
			XmlDatatypeUtil.isOrderedDatatype(leftDatatype.getURI()))
		{
			// Both arguments have the same, ordered datatype
			try {
				int compareTo = XmlDatatypeUtil.compare(leftLabel, rightLabel, leftDatatype.getURI());
				switch (operator) {
					case LT: result = compareTo < 0; break;
					case LE: result = compareTo <= 0; break;
					case EQ: result = compareTo == 0; break;
					case NE: result = compareTo != 0; break;
					case GE: result = compareTo >= 0; break;
					case GT: result = compareTo > 0; break;
					default: throw new BooleanExprEvaluationException("Unknown operator value: " + operator);
				}
			}
			catch (IllegalArgumentException e) {
				// One or both of the arguments was invalid, probably due to an
				// invalid cast earlier in this method. Return false.
				throw new BooleanExprEvaluationException(e.getMessage());
			}
		}
		else if (  (leftDatatype  == null && leftLang  == null ||
					leftDatatype  != null && leftDatatype.getURI().equals(XmlSchema.STRING))
				&& (rightDatatype == null && rightLang == null ||
					rightDatatype != null && rightDatatype.getURI().equals(XmlSchema.STRING)) )
		{
			// Both arguments are either plain literals (i.e. have no language
			// or datatype), or are of type xsd:string. Compare the labels
			int compareTo = leftLabel.compareTo(rightLabel);
			switch (operator) {
				case LT: result = compareTo < 0; break;
				case LE: result = compareTo <= 0; break;
				case EQ: result = compareTo == 0; break;
				case NE: result = compareTo != 0; break;
				case GE: result = compareTo >= 0; break;
				case GT: result = compareTo > 0; break;
				default: throw new BooleanExprEvaluationException("Unknown operator value: " + operator);
			}
		}
		else {
			// All other cases, e.g. literals with languages, unequal or
			// unordered datatypes, etc. These arguments can only be compared
			// using the operators 'EQ' and 'NE'.
			switch (operator) {
				case EQ: result = leftLit.equals(rightLit); break;
				case NE: result = !leftLit.equals(rightLit); break;
				default: throw new BooleanExprEvaluationException(
						"Only literals with compatible, ordered datatypes can be compared using <, <=, > and >= operators");
			}
		}

		return result;
	}

	/**
	 * Gives a string representation of the specified operator code.
	 * @param op One of <tt>Compare.EQ</tt>, <tt>Compare.NE</tt>,
	 * <tt>ValueCompare.LT</tt>, <tt>ValueCompare.LE</tt>,
	 * <tt>ValueCompare.GE</tt> or <tt>ValueCompare.GT</tt>.
	 * @exception IllegalArgumentException If the specified operator code is
	 * an illegal code.
	 **/
	public static String operator2string(int op) {
		switch (op) {
			case EQ: return "=";
			case NE: return "!=";
			case LT: return "<";
			case LE: return "<=";
			case GE: return ">";
			case GT: return ">=";
			default: throw new IllegalArgumentException("Illegal operator value: " + op);
		}
	}

	public String toString() {
		StringBuffer buf = new StringBuffer(32);
		buf.append('(');
		buf.append(_leftArg.toString());
		buf.append(' ');
		buf.append( operator2string(_operator) );
		buf.append(' ');
		buf.append(_rightArg.toString());
		buf.append(')');

		return buf.toString();
	}
}
