/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sail.query;

import java.io.IOException;

import org.openrdf.sesame.sail.RdfSource;

/**
 * Determines the union of two sets of values that are produced by queries.
 *
 * @author Jeen Broekstra
 */
public class Union extends SetOperator {

	/**
	 * Creates a new Union object that reports the union of the query answers
	 * that are reported by both the left- and right argument.
	 **/
	public Union(Query leftArg, Query rightArg, boolean all) {
		super(leftArg, rightArg, all);
	}
	
	// implements Query.evaluate()
	public void evaluate(RdfSource rdfSource, QueryAnswerListener listener)
		throws SailQueryException
	{
		if (!_all) {
			listener = new DuplicatesFilter(listener);
		}

		ContinueEvaluationListener cel = new ContinueEvaluationListener(listener);

		_leftArg.evaluate(rdfSource, cel);

		if (cel.continueEvaluation()) {
			_rightArg.evaluate(rdfSource, listener);
		}
	}
	
	public String toString() {
		return
			"(" + _leftArg.toString() +
			" UNION " + (_all ? "ALL " : "") +
			_rightArg.toString() + ")";
	}

/*---------------------------------------+
| Inner class ContinueEvaluationListener |
+---------------------------------------*/

	private static class ContinueEvaluationListener implements QueryAnswerListener {

		private boolean _continueEvaluation = true;

		private QueryAnswerListener _listener;

		public ContinueEvaluationListener(QueryAnswerListener l) {
			_listener = l;
		}

		public boolean queryAnswer(QueryAnswer qa)
			throws IOException
		{
			_continueEvaluation = _listener.queryAnswer(qa);
			return _continueEvaluation;
		}

		public void clear() {
		}

		public boolean continueEvaluation() {
			return _continueEvaluation;
		}
	}
}
