/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sail.query;

import java.util.Collection;

import org.openrdf.model.Resource;
import org.openrdf.model.Statement;
import org.openrdf.model.URI;
import org.openrdf.model.Value;

import org.openrdf.sesame.sail.RdfSource;
import org.openrdf.sesame.sail.StatementIterator;
import org.openrdf.sesame.sail.util.EmptyStatementIterator;

/**
 * @author Arjohn Kampman
 */
public class TriplePattern implements PathExpression {

/*----------+
| Variables |
+----------*/

	private Var _subjectVar;
	private Var _predicateVar;
	private Var _objectVar;

	private StatementIterator _statementIter;

	private boolean _subjectVarLocked;
	private boolean _predicateVarLocked;
	private boolean _objectVarLocked;

/*-------------+
| Constructors |
+-------------*/

	public TriplePattern(Var subjectVar, Var predicateVar, Var objectVar) {
		_subjectVar = subjectVar;
		_predicateVar = predicateVar;
		_objectVar = objectVar;
	}

/*-------------------------------+
| Methods                        |
+-------------------------------*/

	public Var getSubjectVar() {
		return _subjectVar;
	}

	public Var getPredicateVar() {
		return _predicateVar;
	}

	public Var getObjectVar() {
		return _objectVar;
	}

	public void initialize(RdfSource rdfSource)
		throws SailQueryException
	{
		_subjectVarLocked = _subjectVar.hasValue();
		_predicateVarLocked = _predicateVar.hasValue();
		_objectVarLocked = _objectVar.hasValue();

		try {
			Resource subjectValue = (Resource)_subjectVar.getValue();
			URI predicateValue = (URI)_predicateVar.getValue();
			Value objectValue = _objectVar.getValue();

			_statementIter = _getStatementIterator(rdfSource, subjectValue, predicateValue, objectValue);
		}
		catch (ClassCastException e) {
			// subject is not a Resource or predicate is not a URI,
			// possibly due to a wrong comparison in the where clause.
			_statementIter = new EmptyStatementIterator();
		}
	}

	protected StatementIterator _getStatementIterator(
		RdfSource rdfSource, Resource subj, URI pred, Value obj)
		throws SailQueryException
	{
		return rdfSource.getStatements(subj, pred, obj);
	}

	public boolean selectNext(RdfSource rdfSource) {
		while (_statementIter.hasNext()) {
			Statement st = _statementIter.next();

			// Set subject
			if (!_subjectVarLocked) {
				_subjectVar.setValue(st.getSubject());
			}

			// Set predicate
			if (!_predicateVarLocked) {
				if (_predicateVar == _subjectVar) {
					// Identical vars, subject and predicate values should be equal
					if (!st.getSubject().equals(st.getPredicate())) {
						// Not equal, try next result
						continue;
					}
				}
				else {
					_predicateVar.setValue(st.getPredicate());
				}
			}

			// Set object
			if (!_objectVarLocked) {
				if (_objectVar == _subjectVar) {
					// Identical vars, subject and object values should be equal
					if (!st.getSubject().equals(st.getObject())) {
						// Not equal, try next result
						continue;
					}
				}
				else if (_objectVar == _predicateVar) {
					// Identical vars, predicate and object values should be equal
					if (!st.getPredicate().equals(st.getObject())) {
						// Not equal, try next result
						continue;
					}
				}
				else {
					_objectVar.setValue(st.getObject());
				}
			}

			return true;
		}

		// No more results, release locks
		_releaseLocks();

		return false;
	}

	public void clear() {
		if (_statementIter != null) {
			// Close the statement iterator
			_statementIter.close();

			// Release locks on vars
			_releaseLocks();
		}
	}

	private void _releaseLocks() {
		if (!_subjectVarLocked) {
			_subjectVar.setValue(null);
		}
		if (!_predicateVarLocked) {
			_predicateVar.setValue(null);
		}
		if (!_objectVarLocked) {
			_objectVar.setValue(null);
		}
	}

	public void getVariables(Collection variables) {
		variables.add(_subjectVar);
		variables.add(_predicateVar);
		variables.add(_objectVar);
	}

	public String toString() {
		StringBuffer result = new StringBuffer(128);

		result.append("(");
		if (_subjectVar.hasValue()) {
			result.append(_subjectVar.getString());
		}
		else {
			result.append(_subjectVar.toString());
		}
		result.append(",");
		if (_predicateVar.hasValue()) {
			result.append(_predicateVar.getString());
		}
		else {
			result.append(_predicateVar.toString());
		}
		result.append(",");
		if (_objectVar.hasValue()) {
			result.append(_objectVar.getString());
		}
		else {
			result.append(_objectVar.toString());
		}
		result.append(")");

		return result.toString();
	}
}
