/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sail.query;

import java.io.IOException;
import java.util.Collection;
import java.util.List;

import org.openrdf.model.Value;
import org.openrdf.sesame.sail.RdfSource;

/**
 * Class representing a select-from-where query.
 *
 * @author Arjohn Kampman
 **/
public class SelectQuery extends GraphPatternQuery {

/*----------+
| Variables |
+----------*/

	/**
	 * The query's projection determining which values to include in
	 * the query result.
	 **/
	private ProjectionElem[] _projection;

/*-------------+
| Constructors |
+-------------*/

	public SelectQuery(List projection, GraphPattern graphPattern) {
		this(false, projection, graphPattern);
	}

	public SelectQuery(boolean distinct, List projection, GraphPattern graphPattern) {
		super(distinct, graphPattern);
		setProjection(projection);
	}

	public SelectQuery(boolean distinct, int offset, int limit,
			List projection, GraphPattern graphPattern)
	{
		super(distinct, offset, limit, graphPattern);
		setProjection(projection);
	}

/*--------+
| Methods |
+--------*/

	public ProjectionElem[] getProjection() {
		return _projection;
	}

	public void setProjection(List projection) {
		ProjectionElem[] projArray = new ProjectionElem[projection.size()];
		projArray = (ProjectionElem[])projection.toArray(projArray);

		setProjection(projArray);
	}

	public void setProjection(ProjectionElem[] projection) {
		_projection = projection;
	}

	// implements GraphPatternQuery.getProjectionVariables()
	public void getProjectionVariables(Collection variables) {
		for (int i = 0; i < _projection.length; i++) {
			_projection[i].getValueExpr().getVariables(variables);
		}
	}

	// implements Query.getColumnHeaders()
	public String[] getColumnHeaders() {
		String columnHeaders[] = new String[_projection.length];

		for (int i = 0; i < _projection.length; i++) {
			columnHeaders[i] = _projection[i].getName();
		}

		return columnHeaders;
	}

	// implements GraphPatternQuery._reportQueryAnswer(RdfSource, QueryAnswerListener)
	protected boolean _reportQueryAnswer(RdfSource rdfSource, QueryAnswerListener listener)
		throws IOException
	{
		// Report the values from the projection
		Value[] values = new Value[_projection.length];

		for (int i = 0; i < _projection.length; i++) {
			values[i] = _projection[i].getValueExpr().getValue();
		}

		return listener.queryAnswer( new QueryAnswer(values) );
	}

	public String toString() {
		StringBuffer result = new StringBuffer(256);
		result.append("select ");

		if (isDistinct()) {
			result.append("distinct ");
		}

		for (int i = 0; i < _projection.length; i++) {
			if (i > 0) {
				result.append(", ");
			}

			result.append( _projection[i].toString() );
		}

		result.append("\nfrom\n");
		result.append( getGraphPattern().toString() );

		if (hasLimit()) {
			result.append("\nlimit ").append(getLimit());
		}

		if (hasOffset()) {
			result.append("\noffset ").append(getOffset());
		}

		return result.toString();
	}
}
