/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sail.query;

import java.io.IOException;

/**
 * A QueryAnswerListener that handles any OFFSET settings on a query. This
 * class drops the first <tt>OFFSET</tt> results, but forwards the rest.
 **/
public class OffsetFilter implements QueryAnswerListener {

/*----------+
| Variables |
+----------*/

	/**
	 * The offset (0-based) of the first query result to return. By default,
	 * all query results are returned (offset is 0). By setting the offset to
	 * N, the first N results will be dropped.
	 **/
	private int _offset;

	/**
	 * The number of query results that have been dropped so far.
	 **/
	private int _droppedResults;

	/**
	 * The listener to forward the filtered query results to.
	 **/
	private QueryAnswerListener _listener;

/*-------------+
| Constructors |
+-------------*/

	/**
	 * Creates a new OffsetFilter.
	 *
	 * @param l The QueryAnswerListener that will receive the filtered
	 * query results.
	 **/
	public OffsetFilter(QueryAnswerListener l, int offset) {
		_listener = l;
		_offset = offset;
		_droppedResults = 0;
	}

/*--------+
| Methods |
+--------*/

	// implements QueryAnswerListener.queryAnswer(QueryAnswer)
	public boolean queryAnswer(QueryAnswer qa)
		throws IOException
	{
		if (_droppedResults < _offset) {
			// Drop this result
			_droppedResults++;
			return true;
		}
		else {
			// forward the result
			return _listener.queryAnswer(qa);
		}
	}

	// implements QueryAnswerListener.clear()
	public void clear() {
		_droppedResults = 0;
		_listener.clear();
	}
}
