/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sail.query;

import java.util.Collection;

import org.openrdf.sesame.sail.RdfSource;

/**
 * Compares the string representation of an expression to a pattern.
 **/
public class Like implements BooleanExpr {

/*----------+
| Variables |
+----------*/

	private StringExpr _expr;
	private String _pattern;
	private boolean _ignoreCase;

/*-------------+
| Constructors |
+-------------*/

	public Like(StringExpr expr, StringConstant pattern, boolean ignoreCase) {
		_expr = expr;
		_pattern = pattern.getString();
		_ignoreCase = ignoreCase;

		if (ignoreCase) {
			_pattern = _pattern.toLowerCase();
		}
	}

/*--------+
| Methods |
+--------*/

	// implements StringExpr.getVariables(Collection)
	public void getVariables(Collection variables) {
		_expr.getVariables(variables);
	}
	
	public StringExpr getStringExpr() {
		return _expr;
	}

	public String getPattern() {
		return _pattern;
	}
	
	public boolean caseSensitive() {
		return !_ignoreCase;
	}
	
	/**
	 * Determines whether the two operands match according to the
	 * <code>like</code> operator. The operator is defined as a string
	 * comparison with the possible use of an asterisk (*) at the end
	 * and/or the start of the second operand to indicate substring
	 * matching.
	 *
	 * @return <code>true</code> if the operands match according to the
	 * <code>like</code> operator, <code>false</code> otherwise.
	 **/
	public boolean isTrue(RdfSource rdfSource)
		throws BooleanExprEvaluationException, SailQueryException
	{
		String val = _expr.getString();

		if (val == null) {
			throw new BooleanExprEvaluationException();
		}

		if (_ignoreCase) {
			// Convert val to lower case, just like the pattern has been done
			val = val.toLowerCase();
		}

		int valIndex = 0;
		int prevPatternIndex = -1;
		int patternIndex = _pattern.indexOf('*');

		if (patternIndex == -1) {
			// No wildcards
			return _pattern.equals(val);
		}

		String snippet;

		if (patternIndex > 0) {
			// Pattern does not start with a wildcard, first part must match
			snippet = _pattern.substring(0, patternIndex);
			if (!val.startsWith(snippet)) {
				return false;
			}

			valIndex += snippet.length();
			prevPatternIndex = patternIndex;
			patternIndex = _pattern.indexOf('*', patternIndex + 1);
		}

		while (patternIndex != -1) {
			// Get snippet between previous wildcard and this wildcard
			snippet = _pattern.substring(prevPatternIndex + 1, patternIndex);

			// Search for the snippet in the value
			valIndex = val.indexOf(snippet, valIndex);
			if (valIndex == -1) {
				return false;
			}

			valIndex += snippet.length();
			prevPatternIndex = patternIndex;
			patternIndex = _pattern.indexOf('*', patternIndex + 1);
		}

		// Part after last wildcard
		snippet = _pattern.substring(prevPatternIndex + 1);

		if (snippet.length() > 0) {
			// Pattern does not end with a wildcard.

			// Search last occurence of the snippet.
			valIndex = val.indexOf(snippet, valIndex);
			int i;
			while ((i = val.indexOf(snippet, valIndex + 1)) != -1) {
				// A later occurence was found.
				valIndex = i;
			}

			if (valIndex == -1) {
				return false;
			}

			valIndex += snippet.length();

			if (valIndex < val.length()) {
				// Some characters were not matched
				return false;
			}
		}

		return true;
	}

	public String toString() {
		String result = _expr.toString() + " like \"" + _pattern + "\"";

		if (_ignoreCase) {
			result += " ignore case";
		}

		return result;
	}
}
