/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sail.query;

import java.io.IOException;
import java.util.Collection;

import org.openrdf.sesame.sail.RdfSource;

/**
 * An abstract representation of an RDF query that works with path expressions.
 *
 * @author Arjan Kleine
 * @author Jeen Broekstra
 * @author Arjohn Kampman
 */
public abstract class GraphPatternQuery implements Query {

/*----------+
| Variables |
+----------*/

	/**
	 * Flag indicating whether any duplicates should be filtered from the
	 * results.
	 **/
	private boolean _distinct;

	/**
	 * The offset (0-based) of the first query result to return. By default, all
	 * query results are returned (offset is 0). By setting the offset to N, the
	 * first N results will not be returned.
	 **/
	private int _offset;

	/**
	 * The maximum number of results to return. By default, all query results
	 * are returned (limit is <tt>Integer.MAX_VALUE</tt>). By setting limit to
	 * N, the query engine will stop evaluating the query as soon as it has
	 * returned N results.
	 **/
	private int _limit;

	/**
	 * A graph pattern consisting of a mix of path expressions and boolean
	 * constraints on values of the path expressions.
	 **/
	private GraphPattern _graphPattern;

/*-------------+
| Constructors |
+-------------*/

	protected GraphPatternQuery(GraphPattern graphPattern) {
		this(false, graphPattern);
	}

	protected GraphPatternQuery(boolean distinct, GraphPattern graphPattern) {
		this(distinct, 0, Integer.MAX_VALUE, graphPattern);
	}

	protected GraphPatternQuery(boolean distinct, int offset, int limit, GraphPattern graphPattern) {
		if (offset < 0) {
			throw new IllegalArgumentException("offset must be non-negative, is: " + offset);
		}
		if (limit < 0) {
			throw new IllegalArgumentException("limit must be non-negative, is: " + limit);
		}

		_distinct = distinct;
		_offset = offset;
		_limit = limit;
		_graphPattern = graphPattern;
	}

/*--------+
| Methods |
+--------*/

	public boolean isDistinct() {
		return _distinct;
	}

	public void setDistinct(boolean distinct) {
		_distinct = distinct;
	}

	public int getOffset() {
		return _offset;
	}

	public boolean hasOffset() {
		return _offset > 0;
	}

	public void setOffset(int offset) {
		_offset = offset;
	}

	public int getLimit() {
		return _limit;
	}

	public boolean hasLimit() {
		return _limit < Integer.MAX_VALUE;
	}

	public void setLimit(int limit) {
		_limit = limit;
	}

	public void setGraphPattern(GraphPattern graphPattern) {
		_graphPattern = graphPattern;
	}

	public GraphPattern getGraphPattern() {
		return _graphPattern;
	}

	// implements Query.getVariables(Collection)
	public void getVariables(Collection variables) {
		_graphPattern.getVariables(variables);
	}

	/**
	 * Gets all variables that are used in the projection of this
	 * GraphPatternQuery.
	 *
	 * @param variables A Collection to add the Var objects to.
	 **/
	public abstract void getProjectionVariables(Collection variables);

	// implements Query.evaluate(RdfSource,QueryAnswerListener)
	public void evaluate(RdfSource rdfSource, QueryAnswerListener listener)
		throws SailQueryException
	{
		// If _limit is 0 or negative we don't have to do anything
		if (_limit <= 0) {
			return;
		}

		// Build relevant QueryAnswerListener chain from back to front
		QueryResultsCounter countListener = null;
		if (_limit < Integer.MAX_VALUE) {
			// LIMIT has been specified
			listener = countListener = new QueryResultsCounter(listener);
		}
		if (_offset > 0) {
			// OFFSET has been specified
			listener = new OffsetFilter(listener, _offset);
		}
		if (_distinct) {
			// DISTINCT has been specified
			listener = new DuplicatesFilter(listener);
		}

		_graphPattern.initialize(rdfSource);

		try {
			boolean instantiated = _graphPattern.findFirst(rdfSource);

			while (instantiated) {
				boolean continueEvaluation = _reportQueryAnswer(rdfSource, listener);

				if (!continueEvaluation) {
					break;
				}

				if (countListener == null || countListener.getCount() < _limit) {
					// No limit is set, or limit has not yet been reached. Get next result.
					instantiated = _graphPattern.findNext(rdfSource);
				}
				else {
					// limit reached, don't return any more results
					instantiated = false;
				}
			}
		}
		catch (IOException e) {
			throw new SailQueryException(e);
		}
		finally {
			_graphPattern.clear();
		}
	}

	/**
	 * Reports the query answer as it is currently instantiated in the path
	 * expression list to the supplied listener.
	 **/
	protected abstract boolean _reportQueryAnswer(RdfSource rdfSource, QueryAnswerListener listener)
		throws IOException;
}
