/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sail.query;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.openrdf.model.Value;
import org.openrdf.model.ValueFactory;
import org.openrdf.sesame.sail.RdfSource;

/**
 * Class representing a construct-from-where query. The conceptual result of a
 * ConstructQuery, a set of triples, is represented by QueryAnswers with three
 * values: the subject, predicate and object.
 *
 * @author Arjohn Kampman
 */
public class ConstructQuery extends GraphPatternQuery {

/*----------+
| Variables |
+----------*/

	/**
	 * The query's projection determining which triples to report in
	 * the query result.
	 **/
	private TriplePattern[] _projection;

	/**
	 * A Set of anonymous Var objects that potentially have to be assigned
	 **/
	private Set _anonymousVars;

/*-------------+
| Constructors |
+-------------*/

	public ConstructQuery(boolean distinct, List projection, GraphPattern graphPattern) {
		this(distinct, 0, Integer.MAX_VALUE, projection, graphPattern);
	}

	public ConstructQuery(boolean distinct, int offset, int limit,
			List projection, GraphPattern graphPattern)
	{
		super(distinct, offset, limit, graphPattern);
		_anonymousVars = new HashSet(8);
		setProjection(projection);
	}

/*--------+
| Methods |
+--------*/

	public TriplePattern[] getProjection() {
		return _projection;
	}

	public void setProjection(List projection) {
		TriplePattern[] projArray = new TriplePattern[projection.size()];
		projArray = (TriplePattern[])projection.toArray(projArray);

		setProjection(projArray);
	}

	public void setProjection(TriplePattern[] projection) {
		_projection = projection;

		_anonymousVars.clear();
		for (int i = 0; i < projection.length; i++) {
			TriplePattern tp = projection[i];

			if (tp.getSubjectVar().isAnonymous()) {
				_anonymousVars.add(tp.getSubjectVar());
			}

			if (tp.getPredicateVar().isAnonymous()) {
				_anonymousVars.add(tp.getPredicateVar());
			}

			if (tp.getObjectVar().isAnonymous()) {
				_anonymousVars.add(tp.getObjectVar());
			}
		}
	}

	// implements GraphPatternQuery.getProjectionVariables()
	public void getProjectionVariables(Collection variables) {
		for (int i = 0; i < _projection.length; i++) {
			_projection[i].getVariables(variables);
		}
	}

	// implements Query.getColumnHeaders()
	public String[] getColumnHeaders() {
		return new String[] {"subject", "predicate", "object"};
	}

	// implements GraphPatternQuery._reportQueryAnswer(RdfSource, QueryAnswerListener)
	protected boolean _reportQueryAnswer(RdfSource rdfSource, QueryAnswerListener listener)
		throws IOException
	{
		// Create new bnodes for any anonymous variables that have not been
		// given a value by the query. We need to keep track of which variables
		// have been filled to clear them for the next iterator.
		List filledVars = new ArrayList(8);
		ValueFactory valFactory = rdfSource.getValueFactory();

		Iterator iter = _anonymousVars.iterator();
		while (iter.hasNext()) {
			Var var = (Var)iter.next();

			if (!var.hasValue()) {
				var.setValue( valFactory.createBNode() );

				filledVars.add(var);
			}
		}

		boolean reportMore = true;

		for (int i = 0; reportMore && i < _projection.length; i++) {
			TriplePattern tp = _projection[i];

			Value subject = tp.getSubjectVar().getValue();
			Value predicate = tp.getPredicateVar().getValue();
			Value object = tp.getObjectVar().getValue();

			// We should skip any triples for which a value is null; these are
			// 'partial' matches for an optional path and should not be returned
			if (subject != null && predicate != null && object != null) {
				Value[] values = new Value[] { subject, predicate, object };
				reportMore = listener.queryAnswer( new QueryAnswer(values) );
			}
		}

		// Clear any anonymous variables that have been filled earlier
		for (int i = 0; i < filledVars.size(); i++) {
			Var var = (Var)filledVars.get(i);
			var.setValue(null);
		}

		return reportMore;
	}

	public String toString() {
		StringBuffer result = new StringBuffer(256);
		result.append("construct ");

		if (isDistinct()) {
			result.append("distinct ");
		}

		for (int i = 0; i < _projection.length; i++) {
			if (i > 0) {
				result.append(", ");
			}

			result.append( _projection[i].toString() );
		}

		result.append("\nfrom\n");
		result.append( getGraphPattern().toString() );

		if (hasLimit()) {
			result.append("\nlimit ").append(getLimit());
		}

		if (hasOffset()) {
			result.append("\noffset ").append(getOffset());
		}

		return result.toString();
	}
}
