/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sail;

import java.util.Map;

/**
 * An interface for an RDF-based Storage And Inference Layer. This interface
 * only defines methods for initializing and shutting down a repository.
 * The extensions of this interface (RdfSource, RdfRepository, RdfSchemaSource
 * and RdfSchemaRepository) define methods for querying and manipulating the
 * contents of the repository. These interfaces are organized in a hierarachy
 * like this:
 * <pre>
 *              Sail
 *                ^
 *                |
 *            RdfSource
 *              ^  ^
 *             /    \
 * RdfRepository   RdfSchemaSource
 *            ^     ^
 *             \   /
 *      RdfSchemaRepository
 * </pre>
 *
 * All methods in this interface, or in any extension of this interface, can
 * throw SailInternalException's (a RuntimeException) to indicate an error or
 * an unexpected situation in the RDF Sail internally (e.g. the database to
 * connect to does not exist).
 *
 * @author Arjohn Kampman
 * @version $Revision: 1.3.4.2 $
 */
public interface Sail {

	/**
	 * Initializes the Sail using a set of configuration parameters. The
	 * relevant names and values of the parameters are determined by the
	 * implementation of the interface.
	 *
	 * @param configParams A set a configuration parameters.
	 * @exception SailInitializationException If the Sail could not be
	 * initialized.
	 * @exception SailInternalException To indicate an internal error.
	 */
	public void initialize(Map configParams)
		throws SailInitializationException;

	/**
	 * Allow the SAIL to synchronize any stale data. The SAIL can assume that
	 * shutDown() is called before an application is stopped.
	 *
	 * @exception SailInternalException To indicate an internal error.
	 */
	public void shutDown();
}
