/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sail;

import org.openrdf.model.Resource;
import org.openrdf.model.URI;
import org.openrdf.model.Value;
import org.openrdf.model.ValueFactory;

import org.openrdf.sesame.sail.query.Query;

/**
 * An extension of the Storage And Inference Layer (Sail) interface that
 * defines methods for read-only RDF-repositories or other sources of
 * information that can be accessed as RDF.
 *
 * @author Arjohn Kampman
 * @version $Revision: 1.9.2.2 $
 */
public interface RdfSource extends Sail {

	/**
	 * Gets a ValueFactory object that can be used to create URI-, blank node-
	 * and literal objects.
	 *
	 * @return a ValueFactory object for this RdfSource instance.
	 */
	public ValueFactory getValueFactory();

	/**
	 * Gets all statements with a specific subject, predicate and/or object.
	 * All three parameters may be null to indicate wildcards.
	 *
	 * @param subj subject of pattern
	 * @param pred predicate of pattern
	 * @param obj object of pattern
	 * @return iterator over statements
	 * @exception SailInternalException To indicate an internal error.
	 */
	public StatementIterator getStatements(Resource subj, URI pred, Value obj);

	/**
	 * Checks whether some statement with a specific subject, predicate and/or
	 * object is present in the repository. All three parameters may be null
	 * to indicate wildcards.
	 *
	 * @param subj subject of statement
	 * @param pred predicate of statement
	 * @param obj object of statement
	 * @return boolean indicating if specified statement is present
	 * @exception SailInternalException To indicate an internal error.
	 */
	public boolean hasStatement(Resource subj, URI pred, Value obj);

	/**
	 * Gives the RdfSource the oportunity to optimize a Query to its specific
	 * storage model.
	 *
	 * @param qc The Query to optimize.
	 * @return The optimized query.
	 * @exception SailInternalException To indicate an internal error.
	 * 
	 * @return the optimized Query.
	 **/
	public Query optimizeQuery(Query qc);

	/**
	 * Gets the prefix and name of all namespaces.
	 *
	 * @return iterator over namespaces
	 * @exception SailInternalException To indicate an internal error.
	 */
	public NamespaceIterator getNamespaces();
	
}
