/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sail;

import org.openrdf.model.Resource;
import org.openrdf.model.URI;
import org.openrdf.model.Value;

/**
 * An extension of RdfSource offering RDF Schema specific methods.
 *
 * @author Arjohn Kampman
 * @version $Revision: 1.3.4.2 $
 */
public interface RdfSchemaSource extends RdfSource {

	/**	
	 * Gets all explicitly added statements with a specific subject, predicate
	 * and/or object. All of these parameters may be null to indicate
	 * wildcards.
	 *
	 * @param subj subject of pattern
	 * @param pred predicate of pattern
	 * @param obj object of pattern
	 * @return iterator over statements
	 * @exception SailInternalException To indicate an internal error.
	 * @see RdfSource#getStatements
	 */
	public StatementIterator getExplicitStatements(
			Resource subj, URI pred, Value obj);

	/**
	 * Checks if an explicitly added statement with a specific subject,
	 * predicate and/or object is present in the repository. All of these
	 * parameters may be null to indicate wildcards.
	 *
	 * @param subj subject of statement
	 * @param pred predicate of statement
	 * @param obj object of statement
	 * @return boolean indicating if the specified statement is present.
	 * @exception SailInternalException To indicate an internal error.
	 * @see RdfSource#hasStatement
	 */
	public boolean hasExplicitStatement(Resource subj, URI pred, Value obj);

	/**
	 * Gets all defined classes.
	 *
	 * @return A StatementIterator containing statements of the form
	 * (someClass, rdf:type, rdfs:Class).
	 * @exception SailInternalException To indicate an internal error.
	 */
	public StatementIterator getClasses();

	/**
	 * Checks whether the supplied resource represents a class.
	 *
	 * @param resource A resource
	 * @return <tt>true</tt> if <tt>resource</tt> is a class,
	 * <tt>false</tt> otherwise.
	 * @exception SailInternalException To indicate an internal error.
	 */
	public boolean isClass(Resource resource);

	/**
	 * Gets all defined properties.
	 *
	 * @return A StatementIterator containing statements of the form
	 * (someProperty, rdf:type, rdf:Property).
	 * @exception SailInternalException To indicate an internal error.
	 */
	public StatementIterator getProperties();

	/**
	 * Checks whether the supplied resource represents a property. 
	 *
	 * @param resource A resource
	 * @return <tt>true</tt> if <tt>resource</tt> is a property,
	 * <tt>false</tt> otherwise.
	 * @exception SailInternalException To indicate an internal error.
	 */
	public boolean isProperty(Resource resource);

	/**
	 * Gets all subClassOf relations with a specific sub- and/or superclass.
	 * Note that the subClassOf relation is reflexive: a class is implicitly
	 * always a subclass of itself.
	 *
	 * @param subClass The subclass of the relations that should be returned,
	 * or <tt>null</tt> if relations with any subclass should be returned.
	 * @param superClass The superclass of the relations that should be returned,
	 * or <tt>null</tt> if relations with any superclass should be returned.
	 * @return A StatementIterator containing statements of the form
	 * (someSubClass, rdfs:subClassOf, someSuperClass).
	 * @exception SailInternalException To indicate an internal error.
	 */
	public StatementIterator getSubClassOf(Resource subClass, Resource superClass);

	/**
	 * Gets all direct subClassOf relations with a specific sub- and/or superclass.
	 * A class <tt>A</tt> is a direct subclass of class <tt>B</tt> if
	 * there is <em>no</em> class <tt>C</tt> such that <tt>A</tt> is a subclass
	 * of <tt>C</tt> and <tt>C</tt> is a subclass of <tt>B</tt>.
	 *
	 * @param subClass The subclass of the relations that should be returned,
	 * or <tt>null</tt> if relations with any subclass should be returned.
	 * @param superClass The superclass of the relations that should be returned,
	 * or <tt>null</tt> if relations with any superclass should be returned.
	 * @return A StatementIterator containing statements of the form
	 * (someSubClass, rdfs:subClassOf, someSuperClass).
	 * @exception SailInternalException To indicate an internal error.
	 */
	public StatementIterator getDirectSubClassOf(Resource subClass, Resource superClass);

	/**
	 * Checks whether one resource is a subclass of another.
	 *
	 * @param subClass A class
	 * @param superClass A class
	 * @return <tt>true</tt> if <tt>subClass</tt> is a subclass of
	 * <tt>superClass</tt>, <tt>false</tt> otherwise.
	 * @exception SailInternalException To indicate an internal error.
	 */
	public boolean isSubClassOf(Resource subClass, Resource superClass);

	/**
	 * Checks whether one resource is a direct subclass of another.
	 *
	 * @param subClass A class
	 * @param superClass A class
	 * @return <tt>true</tt> if <tt>subClass</tt> is a direct subclass
	 * of <tt>superClass</tt>, <tt>false</tt> otherwise.
	 * @exception SailInternalException To indicate an internal error.
	 */
	public boolean isDirectSubClassOf(Resource subClass, Resource superClass);

	/**
	 * Gets all subPropertyOf relations with a specific sub- and/or superproperty.
	 * Note that the subPropertyOf relation is reflexive: a property is implicitly
	 * always a subproperty of itself.
	 *
	 * @param subProperty The subproperty of the relations that should be returned,
	 * or <tt>null</tt> if relations with any subproperty should be returned.
	 * @param superProperty The superproperty of the relations that should be returned,
	 * or <tt>null</tt> if relations with any superproperty should be returned.
	 * @return A StatementIterator containing statements of the form
	 * (someSubProperty, rdfs:subPropertyOf, someSuperProperty).
	 * @exception SailInternalException To indicate an internal error.
	 */
	public StatementIterator getSubPropertyOf(Resource subProperty, Resource superProperty);

	/**
	 * Gets all direct subPropertyOf relations with a specific sub- and/or superproperty.
	 * A property <tt>A</tt> is a direct subproperty of property <tt>B</tt> if
	 * there is <em>no</em> property <tt>C</tt> such that <tt>A</tt> is a subproperty
	 * of <tt>C</tt> and <tt>C</tt> is a subproperty of <tt>B</tt>.
	 *
	 * @param subProperty The subproperty of the relations that should be returned,
	 * or <tt>null</tt> if relations with any subproperty should be returned.
	 * @param superProperty The superproperty of the relations that should be returned,
	 * or <tt>null</tt> if relations with any superproperty should be returned.
	 * @return A StatementIterator containing statements of the form
	 * (someSubProperty, rdfs:subPropertyOf, someSuperProperty).
	 * @exception SailInternalException To indicate an internal error.
	 */
	public StatementIterator getDirectSubPropertyOf(Resource subProperty, Resource superProperty);

	/**
	 * Checks whether one resource is a subproperty of another.
	 *
	 * @param subProperty A property
	 * @param superProperty A property
	 * @return <tt>true</tt> if <tt>subProperty</tt> is a subproperty of
	 * <tt>superProperty</tt>, <tt>false</tt> otherwise.
	 * @exception SailInternalException To indicate an internal error.
	 */
	public boolean isSubPropertyOf(Resource subProperty, Resource superProperty);

	/**
	 * Checks whether one resource is a direct subproperty of another.
	 *
	 * @param subProperty A property
	 * @param superProperty A property
	 * @return <tt>true</tt> if <tt>subProperty</tt> is a direct subproperty
	 * of <tt>superProperty</tt>, <tt>false</tt> otherwise.
	 * @exception SailInternalException To indicate an internal error.
	 */
	public boolean isDirectSubPropertyOf(Resource subProperty, Resource superProperty);

	/**
	 * Gets all domain relations with a specific property and/or domain class.
	 *
	 * @param prop The property of the relations that should be returned,
	 * or <tt>null</tt> if relations with any property should be returned.
	 * @param domain The domain of the relations that should be returned,
	 * or <tt>null</tt> if relations with any domain should be returned.
	 * @return A StatementIterator containing statements of the form
	 * (someProperty, rdfs:domain, someClass).
	 * @exception SailInternalException To indicate an internal error.
	 */
	public StatementIterator getDomain(Resource prop, Resource domain);

	/**
	 * Gets all range relations with a specific property and/or range class.
	 *
	 * @param prop The property of the relations that should be returned,
	 * or <tt>null</tt> if relations with any property should be returned.
	 * @param range The range of the relations that should be returned,
	 * or <tt>null</tt> if relations with any range should be returned.
	 * @return A StatementIterator containing statements of the form
	 * (someProperty, rdfs:range, someClass).
	 * @exception SailInternalException To indicate an internal error.
	 */
	public StatementIterator getRange(Resource prop, Resource range);

	/**
	 * Gets all type relations with a specific instance and/or class.
	 *
	 * @param anInstance The instance of the relations that should be returned,
	 * or <tt>null</tt> if relations with any instance should be returned.
	 * @param aClass The class of the relations that should be returned,
	 * or <tt>null</tt> if relations with any class should be returned.
	 * @return A StatementIterator containing statements of the form
	 * (someInstance, rdf:type, someClass).
	 * @exception SailInternalException To indicate an internal error.
	 */
	public StatementIterator getType(Resource anInstance, Resource aClass);

	/**
	 * Gets all direct type relations with a specific instance and/or class.
	 *
	 * @param anInstance The instance of the relations that should be returned,
	 * or <tt>null</tt> if relations with any instance should be returned.
	 * @param aClass The class of the relations that should be returned,
	 * or <tt>null</tt> if relations with any class should be returned.
	 * @return A StatementIterator containing statements of the form
	 * (someInstance, rdf:type, someClass).
	 * @exception SailInternalException To indicate an internal error.
	 */
	public StatementIterator getDirectType(Resource anInstance, Resource aClass);

	/**
	 * Checks whether one resource is an instance of another.
	 *
	 * @param anInstance An instance
	 * @param aClass A class
	 * @return <tt>true</tt> if <tt>anInstance</tt> is a instance of
	 * <tt>aClass</tt>, <tt>false</tt> otherwise.
	 * @exception SailInternalException To indicate an internal error.
	 */
	public boolean isType(Resource anInstance, Resource aClass);

	/**
	 * Checks whether one resource is a direct instance of another.
	 *
	 * @param anInstance An instance
	 * @param aClass A class
	 * @return <tt>true</tt> if <tt>anInstance</tt> is a direct instance
	 * of <tt>aClass</tt>, <tt>false</tt> otherwise.
	 * @exception SailInternalException To indicate an internal error.
	 */
	public boolean isDirectType(Resource anInstance, Resource aClass);

	/**
	 * Gets all literals with a specific label, language and/or datatype.
	 *
	 * @param label The label of the literals that should be returned,
	 * or <tt>null</tt> if literals with any label should be returned.
	 * @param language The language of the literals that should be returned,
	 * or <tt>null</tt> if literals with any language should be returned.
	 * @param datatype The datatype of the literals that should be returned,
	 * or <tt>null</tt> if literals with any datatype should be returned.
	 * @return A LiteralIterator containing literals that match the specified
	 * pattern.
	 * @exception SailInternalException To indicate an internal error.
	 */
	public LiteralIterator getLiterals(
			String label, String language, URI datatype);
}
