/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.sail;

import org.openrdf.model.Resource;
import org.openrdf.model.URI;
import org.openrdf.model.Value;

/**
 * An extension of the read-only RdfSource interface that adds methods to
 * add data to, and remove data from RDF repositories.
 *
 * @author Arjohn Kampman
 * @version $Revision: 1.6.2.2 $
 */
public interface RdfRepository extends RdfSource {

/*----------------------------------------+
| Transactions                            |
+----------------------------------------*/

	/**
	 * Starts a transaction. A transaction needs to be started before
	 * data can be added to or removed from the repository.
	 * @exception SailInternalException To indicate an internal error.
	 **/
	public void startTransaction();

	/**
	 * Commits a started transaction. A transaction needs to be commited
	 * to be guaranteed that data has been added to, or removed from the
	 * repository.
	 * @exception SailInternalException To indicate an internal error.
	 **/
	public void commitTransaction();

	/**
	 * Checks whether a transaction has been started.
	 * @return true if a transaction has been started, false otherwise.
	 * @exception SailInternalException To indicate an internal error.
	 **/
	public boolean transactionStarted();

/*----------------------------------------+
| Add/delete methods                      |
+----------------------------------------*/

	/**
	 * Adds a statament to the repository. You must start a transaction
	 * before this method can be used.
	 *
	 * @param subj subject of statement
	 * @param pred predicate of statement
	 * @param obj object of statement
	 * @exception SailUpdateException If the statement could not be added.
	 * @exception SailInternalException To indicate an internal error.
	 * @see #startTransaction
	 */
	public void addStatement(Resource subj, URI pred, Value obj)
		throws SailUpdateException;

	/**
	 * Removes data statements that match the (subject, predicate, object)
	 * pattern from the repository. All three parameters may be null to
	 * indicate wildcards. You must start a transaction before this method
	 * can be used.
	 *
	 * @param subj subject of statement
	 * @param pred predicate of statement
	 * @param obj object of statement
	 * @return the number of statements that have been removed.
	 * @exception SailUpdateException If the statement(s) could not be removed.
	 * @exception SailInternalException To indicate an internal error.
	 * @see #startTransaction
	 */
	public int removeStatements(Resource subj, URI pred, Value obj) 
		throws SailUpdateException;

	/**
	 * Clears the repository. After the repository was cleared, it will
	 * be in the same state as after initialization. You must start a
	 * transaction before this method can be used.
	 * @exception SailUpdateException If the repository could not be
	 * cleared (for example when a transaction was not started first).
	 * @exception SailInternalException To indicate an internal error.
	 */
	public void clearRepository()
		throws SailUpdateException;


/*----------------------------------------+
| Namespaces                              |
+----------------------------------------*/

	/**
	 * Changes the prefix of a namespace. The new prefix must be unique in
	 * the repository; it is not allowed to be associated with any other
	 * namespace. You must start a transaction before this method can be
	 * used.
	 *
	 * @param namespace The namespace for which the associated prefix should
	 * be changed.
	 * @param prefix The new prefix.
	 * @exception SailUpdateException If the namespace prefix could not be
	 * changed, for example when no transaction was started.
	 * @exception SailInternalException To indicate an internal error.
	 **/
	public void changeNamespacePrefix(String namespace, String prefix)
		throws SailUpdateException;
	
	public void addListener(SailChangedListener listener);
	
	public void removeListener(SailChangedListener listener);

}
