/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.sesame.repository.remote;

import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.HashMap;

import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;

import org.openrdf.util.http.CookieManager;
import org.openrdf.util.http.HttpClientUtil;
import org.openrdf.util.xml.XMLReaderFactory;

import org.openrdf.sesame.config.AccessDeniedException;
import org.openrdf.sesame.config.RepositoryListBuilder;
import org.openrdf.sesame.config.RepositoryListListener;
import org.openrdf.sesame.config.RepositoryListReader;
import org.openrdf.sesame.config.UnknownRepositoryException;
import org.openrdf.sesame.constants.QueryResultFormat;
import org.openrdf.sesame.constants.RDFFormat;
import org.openrdf.sesame.repository.RepositoryList;
import org.openrdf.sesame.repository.SesameRepository;
import org.openrdf.sesame.repository.SesameService;

/**
 * A Sesame service for remote repositories that can be communicated with over
 * HTTP.
 *
 * @author Jeen Broekstra
 * @author Arjohn Kampman
 */
public class HTTPService implements SesameService {

/*-------------------+
| Constants          |
+-------------------*/

	private static final String LIST_REPOS_SERVLET = "servlets/listRepositories";
	private static final String LOGIN_SERVLET = "servlets/login";
	private static final String LOGOUT_SERVLET = "servlets/logout";

/*-------------------+
| Variables          |
+-------------------*/

	/**
	 * The Sesame server's URL.
	 **/
	private URL _serverURL;

	/**
	 * The URL of the 'list repositories' servlet.
	 **/
	private URL _listReposURL;

	/**
	 * The URL of the 'login' servlet.
	 **/
	private URL _loginURL;

	/**
	 * The URL of the 'logout' servlet.
	 **/
	private URL _logoutURL;

	/**
	 * The cookie manager.
	 **/
	private CookieManager _cookieManager;

	private QueryResultFormat _defaultTableQueryResultFormat = null;

	private RDFFormat _defaultGraphQueryResultFormat = null;

/*-------------------+
| Constructors       |
+-------------------*/

	/**
	 * Creates a new remote Sesame service that will communicate over
	 * HTTP with the server at the specified URL.
	 *
	 * @param serverURL The Sesame server's URL.
	 * @exception IllegalArgumentException If the supplied URL is
	 * <tt>null</tt>, if it does not specify <tt>http</tt> as the
	 * protocol, or if the combination of the specified server URL and
	 * the paths to its servlets results in an illegal URL.
	 **/
	public HTTPService(URL serverURL) {
		if (serverURL == null) {
			throw new IllegalArgumentException("serverURL must be specified");
		}

		_serverURL = serverURL;

		try {
			_listReposURL = resolveURL(serverURL, LIST_REPOS_SERVLET);
			_loginURL = resolveURL(serverURL, LOGIN_SERVLET);
			_logoutURL = resolveURL(serverURL, LOGOUT_SERVLET);
		}
		catch (MalformedURLException e) {
			throw new IllegalArgumentException("Invalid base URL");
		}

		_cookieManager = new CookieManager();
	}

/*-------------------+
| Methods            |
+-------------------*/

	/**
	 * Sets the default response format for table query results to be used by
	 * HTTPRepository objects of this service.
	 *
	 * @param format A table query result format. Allowed values are <tt>XML</tt> and
	 * <tt>BINARY</tt>.
	 * @see HTTPRepository#setTableQueryResultFormat
	 **/
	public void setDefaultTableQueryResultFormat(QueryResultFormat format) {
		_defaultTableQueryResultFormat = format;
	}

	/**
	 * Sets the default response format for graph query results to be used by
	 * HTTPRepository objects of this service.
	 *
	 * @param format An RDF serialization format. Allowed values are <tt>RDFXML</tt>,
	 * <tt>NTRIPLES</tt> and <tt>TURTLE</tt>.
	 * @see HTTPRepository#setGraphQueryResultFormat
	 **/
	public void setDefaultGraphQueryResultFormat(RDFFormat format) {
		_defaultGraphQueryResultFormat = format;
	}

	// implements SesameService.login(String, String)
	public void login(String username, String password)
		throws AccessDeniedException, IOException
	{
		if (username == null || username.trim().length() == 0) {
			throw new IllegalArgumentException("Illegal user name: " + username);
		}

		// Build the query string
		HashMap postParams = new HashMap(2);
		postParams.put("user", username.trim());
		if (password != null && password.trim().length() > 0) {
			postParams.put("password", password.trim());
		}

		// Set up request
		HttpURLConnection conn = (HttpURLConnection)_loginURL.openConnection();
		HttpClientUtil.preparePostRequest(conn, postParams);

		// Send request
		conn.connect();

		// Get the response
		int responseCode = conn.getResponseCode();

		if (responseCode == HttpURLConnection.HTTP_OK) {
			_cookieManager.clear();
			_cookieManager.extractCookies(conn);
		}
		else if (responseCode == HttpURLConnection.HTTP_FORBIDDEN) {
			throw new AccessDeniedException(conn.getResponseMessage());
		}
		else {
			throw new IOException(_loginURL.toString()+": "+conn.getResponseMessage());
		}
	}

	// implements SesameService.logout()
	public void logout()
		throws IOException
	{
		// Set up request
		HttpURLConnection conn = (HttpURLConnection)_logoutURL.openConnection();
		HttpClientUtil.preparePostRequest(conn, new HashMap(0));

		// Send request
		conn.connect();

		// Assume logging out succeeded (can it fail?)
		_cookieManager.clear();

		conn.disconnect();
	}

	// implements SesameService.getRepositoryList()
	public RepositoryList getRepositoryList()
		throws IOException
	{
		RepositoryListBuilder listBuilder = new RepositoryListBuilder();

		_listRepositories(listBuilder);

		return listBuilder.getRepositoryList();
	}

	protected void _listRepositories(RepositoryListListener list)
		throws IOException
	{
		// Set up request
		HttpURLConnection conn = (HttpURLConnection)_listReposURL.openConnection();
		_cookieManager.setCookies(conn);
		HttpClientUtil.setAcceptGZIPEncoding(conn);

		// Send the request
		conn.connect();

		InputStream resultStream = HttpClientUtil.getInputStream(conn);

		// Parse the result
		try {
			XMLReader xmlReader = XMLReaderFactory.createXMLReader();
			RepositoryListReader rlReader = new RepositoryListReader(xmlReader);
			rlReader.read(resultStream, list);
			resultStream.close();
		}
		catch (SAXException e) {
			String msg;

			// Nested exception?
			Exception ne = e.getException();
			if (ne != null) {
				msg = ne.getMessage();
			}
			else {
				msg = e.getMessage();
			}

			IOException ioe = new IOException("Unable to parse query results from Sesame: " + msg);
			ioe.initCause(e);
			throw ioe;
		}
	}

	// implements SesameService.getRepository(String)
	public SesameRepository getRepository(String repositoryID)
		throws UnknownRepositoryException, IOException
	{
		// Check if the repository exists
		RepositoryList repList = getRepositoryList();
		if (repList.getRepository(repositoryID) == null) {
			throw new UnknownRepositoryException(repositoryID);
		}

		try {
			HTTPRepository result = new HTTPRepository(_serverURL, repositoryID, _cookieManager);

			if (_defaultTableQueryResultFormat != null) {
				result.setTableQueryResultFormat(_defaultTableQueryResultFormat);
			}
			if (_defaultGraphQueryResultFormat != null) {
				result.setGraphQueryResultFormat(_defaultGraphQueryResultFormat);
			}

			return result;
		}
		catch (MalformedURLException e) {
			// This should never happen
			throw new RuntimeException(e);
		}
	}


	// package protected: method also used by HTTPRepository
	static URL resolveURL(URL baseURL, String relPath)
		throws MalformedURLException
	{
		String protocol = baseURL.getProtocol();
		String host = baseURL.getHost();
		int port = baseURL.getPort();
		String file = baseURL.getFile();

		if (!file.endsWith("/")) {
			file = file + "/";
		}

		return new URL(protocol, host, port, file + relPath);
	}
}
